PEP: 315 Title: Enhanced While Loop Version: $Revision$ Last-Modified:
$Date$ Author: Raymond Hettinger <python@rcn.com>, W Isaac Carroll
<icarroll@pobox.com> Status: Rejected Type: Standards Track
Content-Type: text/x-rst Created: 25-Apr-2003 Python-Version: 2.5
Post-History:

Abstract
========

This PEP proposes adding an optional "do" clause to the beginning of the
while loop to make loop code clearer and reduce errors caused by code
duplication.

Notice
======

Rejected; see \[1\]\_.

This PEP has been deferred since 2006; see \[2\]\_.

Subsequent efforts to revive the PEP in April 2009 did not meet with
success because no syntax emerged that could compete with the following
form::

    while True:
        <setup code>
        if not <condition>:
            break
        <loop body>

A syntax alternative to the one proposed in the PEP was found for a
basic do-while loop but it gained little support because the condition
was at the top::

    do ... while <cond>:
        <loop body>

Users of the language are advised to use the while-True form with an
inner if-break when a do-while loop would have been appropriate.

Motivation
==========

It is often necessary for some code to be executed before each
evaluation of the while loop condition. This code is often duplicated
outside the loop, as setup code that executes once before entering the
loop::

    <setup code>
    while <condition>:
        <loop body>
        <setup code>

The problem is that duplicated code can be a source of errors if one
instance is changed but the other is not. Also, the purpose of the
second instance of the setup code is not clear because it comes at the
end of the loop.

It is possible to prevent code duplication by moving the loop condition
into a helper function, or an if statement in the loop body. However,
separating the loop condition from the while keyword makes the behavior
of the loop less clear::

    def helper(args):
        <setup code>
        return <condition>

    while helper(args):
        <loop body>

This last form has the additional drawback of requiring the loop's else
clause to be added to the body of the if statement, further obscuring
the loop's behavior::

    while True:
        <setup code>
        if not <condition>: break
        <loop body>

This PEP proposes to solve these problems by adding an optional clause
to the while loop, which allows the setup code to be expressed in a
natural way::

    do:
        <setup code>
    while <condition>:
        <loop body>

This keeps the loop condition with the while keyword where it belongs,
and does not require code to be duplicated.

Syntax
======

The syntax of the while statement ::

while\_stmt : "while" expression ":" suite \["else" ":" suite\]

is extended as follows::

    while_stmt : ["do" ":" suite]
                "while" expression ":" suite
                ["else" ":" suite]

Semantics of break and continue
===============================

In the do-while loop the break statement will behave the same as in the
standard while loop: It will immediately terminate the loop without
evaluating the loop condition or executing the else clause.

A continue statement in the do-while loop jumps to the while condition
check.

In general, when the while suite is empty (a pass statement), the
do-while loop and break and continue statements should match the
semantics of do-while in other languages.

Likewise, when the do suite is empty, the do-while loop and break and
continue statements should match behavior found in regular while loops.

Future Statement
================

Because of the new keyword "do", the statement ::

from **future** import do\_while

will initially be required to use the do-while form.

Implementation
==============

The first implementation of this PEP can compile the do-while loop as an
infinite loop with a test that exits the loop.

References
==========

.. \[1\] Guido van Rossum, PEP 315: do-while
https://mail.python.org/pipermail/python-ideas/2013-June/021610.html

.. \[2\] Raymond Hettinger, release plan for 2.5 ?
https://mail.python.org/pipermail/python-dev/2006-February/060718.html

Copyright
=========

This document is placed in the public domain.

.. Local Variables: mode: indented-text indent-tabs-mode: nil
sentence-end-double-space: t fill-column: 75 End:
