"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipals = false;
        this.myPrincipals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.myPrincipals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.myPrincipals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.myPrincipals)) {
            this.myPrincipals[prefix] = [];
        }
        this.myPrincipals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds one or more account principals to the statement
     *
     * @param accounts ID of the AWS account
     */
    forAccount(...accounts) {
        accounts.forEach((account) => this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`));
        return this;
    }
    /**
     * Adds one or more [federated](https://aws.amazon.com/identity/federation/) (web identity) principals to the statement
     *
     * @param providers ID of the AWS account
     */
    forFederated(...providers) {
        providers.forEach((provider) => this.addPrincipal(PrincipalType.FEDERATED, provider));
        return this;
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds one or more canonical user principals to the statement
     *
     * @param userIDs The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(...userIDs) {
        userIDs.forEach((userID) => this.addPrincipal(PrincipalType.CANONICAL_USER, userID));
        return this;
    }
    /**
     * Adds one or more federated SAML principals to the statement
     *
     * @param account ID of the AWS account
     * @param providerNames Name of the SAML provider
     */
    forSaml(account, ...providerNames) {
        providerNames.forEach((providerName) => this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`));
        return this;
    }
    /**
     * Adds one or more IAM user principals to the statement
     *
     * @param account ID of the AWS account
     * @param users Name of the IAM user
     */
    forUser(account, ...users) {
        users.forEach((user) => this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`));
        return this;
    }
    /**
     * Adds one or more IAM role principals to the statement
     *
     * @param account ID of the AWS account
     * @param roles Name of the IAM role
     */
    forRole(account, ...roles) {
        roles.forEach((role) => this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`));
        return this;
    }
    /**
     * Adds one or more specific assumed role session principals to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionNames Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, ...sessionNames) {
        sessionNames.forEach((sessionName) => {
            this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
        });
        return this;
    }
    /**
     * Adds one or more service principals to the statement
     *
     * @param services Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(...services) {
        services.forEach((service) => this.addPrincipal(PrincipalType.SERVICE, service));
        return this;
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithPrincipal[_a] = { fqn: "iam-floyd.PolicyStatementWithPrincipal", version: "0.355.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiNi1wcmluY2lwYWxzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiNi1wcmluY2lwYWxzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsMENBQXVEO0FBU3ZELElBQVksYUFLWDtBQUxELFdBQVksYUFBYTtJQUN2Qiw0QkFBVyxDQUFBO0lBQ1gsd0NBQXVCLENBQUE7SUFDdkIsaURBQWdDLENBQUE7SUFDaEMsb0NBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUxXLGFBQWEsR0FBYixxQkFBYSxLQUFiLHFCQUFhLFFBS3hCO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLDRCQUE2QixTQUFRLHFDQUF5QjtJQUEzRTs7UUFDWSxxQkFBZ0IsR0FBRyxLQUFLLENBQUM7UUFDekIsaUJBQVksR0FBZSxFQUFFLENBQUM7S0FzT3pDO0lBcE9DOzs7O09BSUc7SUFDSSxNQUFNO1FBQ1gsa0RBQWtEO1FBQ2xELElBQUksT0FBTyxJQUFJLENBQUMsWUFBWSxJQUFJLFVBQVUsRUFBRTtZQUMxQyxJQUFJLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztZQUMxQixPQUFPLEtBQUssQ0FBQyxNQUFNLEVBQUUsQ0FBQztTQUN2QjtRQUVELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUM7UUFDbEUsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1FBRWpDLElBQUksSUFBSSxDQUFDLGFBQWEsRUFBRSxFQUFFO1lBQ3hCLFNBQVMsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsWUFBWSxDQUFDO1NBQ3JDO1FBRUQsT0FBTyxTQUFTLENBQUM7SUFDbkIsQ0FBQztJQUVNLGVBQWU7UUFDcEIsSUFBSSxDQUFDLGtCQUFrQixFQUFFLENBQUM7UUFDMUIsa0RBQWtEO1FBQ2xELE9BQU8sS0FBSyxDQUFDLGVBQWUsRUFBRSxDQUFDO0lBQ2pDLENBQUM7SUFFUyxrQkFBa0IsS0FBSSxDQUFDO0lBRWpDOztPQUVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0ksYUFBYTtRQUNsQixPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ08sWUFBWSxDQUFDLE1BQWMsRUFBRSxTQUFpQjtRQUN0RCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsSUFBSSxDQUFDO1FBQzdCLElBQUksQ0FBQyxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUU7WUFDbEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUM7U0FDaEM7UUFDRCxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLEdBQUcsQ0FBQyxHQUFXLEVBQUUsTUFBZTtRQUNyQyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsTUFBTSxJQUFJLGFBQWEsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDN0QsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxVQUFVLENBQUMsR0FBRyxRQUFrQjtRQUNyQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FDM0IsSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFLGdCQUFnQixPQUFPLE9BQU8sQ0FBQyxDQUNyRSxDQUFDO1FBQ0YsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFlBQVksQ0FBQyxHQUFHLFNBQW1CO1FBQ3hDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxRQUFRLEVBQUUsRUFBRSxDQUM3QixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxTQUFTLEVBQUUsUUFBUSxDQUFDLENBQ3JELENBQUM7UUFDRixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7T0FFRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOztPQUVHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7T0FFRztJQUVJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCLENBQUMsR0FBRyxPQUFpQjtRQUMxQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FDekIsSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUN4RCxDQUFDO1FBQ0YsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxPQUFPLENBQUMsT0FBZSxFQUFFLEdBQUcsYUFBdUI7UUFDeEQsYUFBYSxDQUFDLE9BQU8sQ0FBQyxDQUFDLFlBQVksRUFBRSxFQUFFLENBQ3JDLElBQUksQ0FBQyxZQUFZLENBQUMsZ0JBQWdCLE9BQU8sa0JBQWtCLFlBQVksRUFBRSxDQUFDLENBQzNFLENBQUM7UUFDRixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE9BQU8sQ0FBQyxPQUFlLEVBQUUsR0FBRyxLQUFlO1FBQ2hELEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUNyQixJQUFJLENBQUMsWUFBWSxDQUNmLGFBQWEsQ0FBQyxHQUFHLEVBQ2pCLGdCQUFnQixPQUFPLFNBQVMsSUFBSSxFQUFFLENBQ3ZDLENBQ0YsQ0FBQztRQUNGLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksT0FBTyxDQUFDLE9BQWUsRUFBRSxHQUFHLEtBQWU7UUFDaEQsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLElBQUksRUFBRSxFQUFFLENBQ3JCLElBQUksQ0FBQyxZQUFZLENBQ2YsYUFBYSxDQUFDLEdBQUcsRUFDakIsZ0JBQWdCLE9BQU8sU0FBUyxJQUFJLEVBQUUsQ0FDdkMsQ0FDRixDQUFDO1FBQ0YsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCLENBQzFCLE9BQWUsRUFDZixRQUFnQixFQUNoQixHQUFHLFlBQXNCO1FBRXpCLFlBQVksQ0FBQyxPQUFPLENBQUMsQ0FBQyxXQUFXLEVBQUUsRUFBRTtZQUNuQyxJQUFJLENBQUMsWUFBWSxDQUNmLGFBQWEsQ0FBQyxHQUFHLEVBQ2pCLGdCQUFnQixPQUFPLGlCQUFpQixRQUFRLElBQUksV0FBVyxFQUFFLENBQ2xFLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUVJLFVBQVUsQ0FBQyxHQUFHLFFBQWtCO1FBQ3JDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUMzQixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQ2xELENBQUM7UUFDRixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDbkQsQ0FBQzs7QUF2T0gsb0VBd09DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50V2l0aEVmZmVjdCB9IGZyb20gJy4vNS1lZmZlY3QnO1xuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBQcmluY2lwYWwnc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFByaW5jaXBhbHMge1xuICBba2V5OiBzdHJpbmddOiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGVudW0gUHJpbmNpcGFsVHlwZSB7XG4gIEFXUyA9ICdBV1MnLFxuICBGRURFUkFURUQgPSAnRmVkZXJhdGVkJyxcbiAgQ0FOT05JQ0FMX1VTRVIgPSAnQ2Fub25pY2FsVXNlcicsXG4gIFNFUlZJQ0UgPSAnU2VydmljZScsXG59XG5cbi8qKlxuICogQWRkcyBcInByaW5jaXBhbFwiIGZ1bmN0aW9uYWxpdHkgdG8gdGhlIFBvbGljeSBTdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFBvbGljeVN0YXRlbWVudFdpdGhQcmluY2lwYWwgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnRXaXRoRWZmZWN0IHtcbiAgcHJvdGVjdGVkIHVzZU5vdFByaW5jaXBhbHMgPSBmYWxzZTtcbiAgcHJvdGVjdGVkIG15UHJpbmNpcGFsczogUHJpbmNpcGFscyA9IHt9O1xuXG4gIC8qKlxuICAgKiBJbmplY3RzIHByaW5jaXBhbHMgaW50byB0aGUgc3RhdGVtZW50LlxuICAgKlxuICAgKiBPbmx5IHJlbGV2YW50IGZvciB0aGUgbWFpbiBwYWNrYWdlLiBJbiBDREsgbW9kZSB0aGlzIG9ubHkgY2FsbHMgc3VwZXIuXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCk6IGFueSB7XG4gICAgLy8gQHRzLWlnbm9yZSBvbmx5IGF2YWlsYWJsZSBhZnRlciBzd2FwcGluZyAxLWJhc2VcbiAgICBpZiAodHlwZW9mIHRoaXMuYWRkUmVzb3VyY2VzID09ICdmdW5jdGlvbicpIHtcbiAgICAgIHRoaXMuY2RrQXBwbHlQcmluY2lwYWxzKCk7XG4gICAgICByZXR1cm4gc3VwZXIudG9KU09OKCk7XG4gICAgfVxuXG4gICAgY29uc3QgbW9kZSA9IHRoaXMudXNlTm90UHJpbmNpcGFscyA/ICdOb3RQcmluY2lwYWwnIDogJ1ByaW5jaXBhbCc7XG4gICAgY29uc3Qgc3RhdGVtZW50ID0gc3VwZXIudG9KU09OKCk7XG5cbiAgICBpZiAodGhpcy5oYXNQcmluY2lwYWxzKCkpIHtcbiAgICAgIHN0YXRlbWVudFttb2RlXSA9IHRoaXMubXlQcmluY2lwYWxzO1xuICAgIH1cblxuICAgIHJldHVybiBzdGF0ZW1lbnQ7XG4gIH1cblxuICBwdWJsaWMgdG9TdGF0ZW1lbnRKc29uKCk6IGFueSB7XG4gICAgdGhpcy5jZGtBcHBseVByaW5jaXBhbHMoKTtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIHJldHVybiBzdXBlci50b1N0YXRlbWVudEpzb24oKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjZGtBcHBseVByaW5jaXBhbHMoKSB7fVxuXG4gIC8qKlxuICAgKiBTd2l0Y2hlcyB0aGUgc3RhdGVtZW50IHRvIHVzZSBbYG5vdFByaW5jaXBhbGBdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfbm90cHJpbmNpcGFsLmh0bWwpLlxuICAgKi9cbiAgcHVibGljIG5vdFByaW5jaXBhbHMoKSB7XG4gICAgdGhpcy51c2VOb3RQcmluY2lwYWxzID0gdHJ1ZTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVja3Mgd2VhdGhlciBhIHByaW5jaXBhbCB3YXMgYXBwbGllZCB0byB0aGUgcG9saWN5XG4gICAqL1xuICBwdWJsaWMgaGFzUHJpbmNpcGFscygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gT2JqZWN0LmtleXModGhpcy5teVByaW5jaXBhbHMpLmxlbmd0aCA+IDA7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBwcmVmaXggT25lIG9mICoqQVdTKiosICoqRmVkZXJhdGVkKiosICoqQ2Fub25pY2FsVXNlcioqIG9yICoqU2VydmljZSoqXG4gICAqIEBwYXJhbSBwcmluY2lwYWwgVGhlIHByaW5jaXBhbCBzdHJpbmdcbiAgICovXG4gIHByb3RlY3RlZCBhZGRQcmluY2lwYWwocHJlZml4OiBzdHJpbmcsIHByaW5jaXBhbDogc3RyaW5nKSB7XG4gICAgdGhpcy5za2lwQXV0b1Jlc291cmNlID0gdHJ1ZTtcbiAgICBpZiAoIShwcmVmaXggaW4gdGhpcy5teVByaW5jaXBhbHMpKSB7XG4gICAgICB0aGlzLm15UHJpbmNpcGFsc1twcmVmaXhdID0gW107XG4gICAgfVxuICAgIHRoaXMubXlQcmluY2lwYWxzW3ByZWZpeF0ucHVzaChwcmluY2lwYWwpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW55IHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBhcm4gVGhlIEFSTiBvZiB0aGUgcHJpbmNpcGFsXG4gICAqIEBwYXJhbSBwcmVmaXggT25lIG9mICoqQVdTKiosICoqRmVkZXJhdGVkKiosICoqQ2Fub25pY2FsVXNlcioqIG9yICoqU2VydmljZSoqIC0gRGVmYXVsdDogKipBV1MqKlxuICAgKi9cbiAgcHVibGljIGZvcihhcm46IHN0cmluZywgcHJlZml4Pzogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKHByZWZpeCB8fCBQcmluY2lwYWxUeXBlLkFXUywgYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIG9uZSBvciBtb3JlIGFjY291bnQgcHJpbmNpcGFscyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBhY2NvdW50cyBJRCBvZiB0aGUgQVdTIGFjY291bnRcbiAgICovXG4gIHB1YmxpYyBmb3JBY2NvdW50KC4uLmFjY291bnRzOiBzdHJpbmdbXSkge1xuICAgIGFjY291bnRzLmZvckVhY2goKGFjY291bnQpID0+XG4gICAgICB0aGlzLmFkZFByaW5jaXBhbChQcmluY2lwYWxUeXBlLkFXUywgYGFybjphd3M6aWFtOjoke2FjY291bnR9OnJvb3RgKVxuICAgICk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBvbmUgb3IgbW9yZSBbZmVkZXJhdGVkXShodHRwczovL2F3cy5hbWF6b24uY29tL2lkZW50aXR5L2ZlZGVyYXRpb24vKSAod2ViIGlkZW50aXR5KSBwcmluY2lwYWxzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIHByb3ZpZGVycyBJRCBvZiB0aGUgQVdTIGFjY291bnRcbiAgICovXG4gIHB1YmxpYyBmb3JGZWRlcmF0ZWQoLi4ucHJvdmlkZXJzOiBzdHJpbmdbXSkge1xuICAgIHByb3ZpZGVycy5mb3JFYWNoKChwcm92aWRlcikgPT5cbiAgICAgIHRoaXMuYWRkUHJpbmNpcGFsKFByaW5jaXBhbFR5cGUuRkVERVJBVEVELCBwcm92aWRlcilcbiAgICApO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBmZWRlcmF0ZWQgW0FXUyBDb2duaXRvXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvaWRfcm9sZXNfcHJvdmlkZXJzX29pZGNfY29nbml0by5odG1sKSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgcHVibGljIGZvckZlZGVyYXRlZENvZ25pdG8oKSB7XG4gICAgcmV0dXJuIHRoaXMuZm9yRmVkZXJhdGVkKCdjb2duaXRvLWlkZW50aXR5LmFtYXpvbmF3cy5jb20nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgZmVkZXJhdGVkIFtBbWF6b25dKGh0dHBzOi8vbG9naW4uYW1hem9uLmNvbS8pIHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqL1xuICBwdWJsaWMgZm9yRmVkZXJhdGVkQW1hem9uKCkge1xuICAgIHJldHVybiB0aGlzLmZvckZlZGVyYXRlZCgnd3d3LmFtYXpvbi5jb20nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgZmVkZXJhdGVkIFtGYWNlYm9va10oaHR0cHM6Ly9kZXZlbG9wZXJzLmZhY2Vib29rLmNvbS9kb2NzL2ZhY2Vib29rLWxvZ2luKSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgcHVibGljIGZvckZlZGVyYXRlZEZhY2Vib29rKCkge1xuICAgIHJldHVybiB0aGlzLmZvckZlZGVyYXRlZCgnZ3JhcGguZmFjZWJvb2suY29tJyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGZlZGVyYXRlZCBbR29vZ2xlXShodHRwczovL2RldmVsb3BlcnMuZ29vZ2xlLmNvbS9pZGVudGl0eS9wcm90b2NvbHMvb2F1dGgyL29wZW5pZC1jb25uZWN0KSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKi9cblxuICBwdWJsaWMgZm9yRmVkZXJhdGVkR29vZ2xlKCkge1xuICAgIHJldHVybiB0aGlzLmZvckZlZGVyYXRlZCgnYWNjb3VudHMuZ29vZ2xlLmNvbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgb25lIG9yIG1vcmUgY2Fub25pY2FsIHVzZXIgcHJpbmNpcGFscyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSB1c2VySURzIFRoZSB1c2VyIElEXG4gICAqXG4gICAqIFlvdSBjYW4gW2ZpbmQgdGhlIGNhbm9uaWNhbCB1c2VyIElEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2VuZXJhbC9sYXRlc3QvZ3IvYWNjdC1pZGVudGlmaWVycy5odG1sI0ZpbmRpbmdDYW5vbmljYWxJZCkgZm9yIHlvdXIgQVdTIGFjY291bnQgaW4gdGhlIEFXUyBNYW5hZ2VtZW50IENvbnNvbGUuIFRoZSBjYW5vbmljYWwgdXNlciBJRCBmb3IgYW4gQVdTIGFjY291bnQgaXMgc3BlY2lmaWMgdG8gdGhlIGFjY291bnQuIFlvdSBjYW4gcmV0cmlldmUgdGhlIGNhbm9uaWNhbCB1c2VyIElEIGZvciB5b3VyIEFXUyBhY2NvdW50IGFzIGVpdGhlciB0aGUgcm9vdCB1c2VyIG9yIGFuIElBTSB1c2VyLlxuICAgKi9cbiAgcHVibGljIGZvckNhbm9uaWNhbFVzZXIoLi4udXNlcklEczogc3RyaW5nW10pIHtcbiAgICB1c2VySURzLmZvckVhY2goKHVzZXJJRCkgPT5cbiAgICAgIHRoaXMuYWRkUHJpbmNpcGFsKFByaW5jaXBhbFR5cGUuQ0FOT05JQ0FMX1VTRVIsIHVzZXJJRClcbiAgICApO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgb25lIG9yIG1vcmUgZmVkZXJhdGVkIFNBTUwgcHJpbmNpcGFscyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBhY2NvdW50IElEIG9mIHRoZSBBV1MgYWNjb3VudFxuICAgKiBAcGFyYW0gcHJvdmlkZXJOYW1lcyBOYW1lIG9mIHRoZSBTQU1MIHByb3ZpZGVyXG4gICAqL1xuICBwdWJsaWMgZm9yU2FtbChhY2NvdW50OiBzdHJpbmcsIC4uLnByb3ZpZGVyTmFtZXM6IHN0cmluZ1tdKSB7XG4gICAgcHJvdmlkZXJOYW1lcy5mb3JFYWNoKChwcm92aWRlck5hbWUpID0+XG4gICAgICB0aGlzLmZvckZlZGVyYXRlZChgYXJuOmF3czppYW06OiR7YWNjb3VudH06c2FtbC1wcm92aWRlci8ke3Byb3ZpZGVyTmFtZX1gKVxuICAgICk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBvbmUgb3IgbW9yZSBJQU0gdXNlciBwcmluY2lwYWxzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnQgSUQgb2YgdGhlIEFXUyBhY2NvdW50XG4gICAqIEBwYXJhbSB1c2VycyBOYW1lIG9mIHRoZSBJQU0gdXNlclxuICAgKi9cbiAgcHVibGljIGZvclVzZXIoYWNjb3VudDogc3RyaW5nLCAuLi51c2Vyczogc3RyaW5nW10pIHtcbiAgICB1c2Vycy5mb3JFYWNoKCh1c2VyKSA9PlxuICAgICAgdGhpcy5hZGRQcmluY2lwYWwoXG4gICAgICAgIFByaW5jaXBhbFR5cGUuQVdTLFxuICAgICAgICBgYXJuOmF3czppYW06OiR7YWNjb3VudH06dXNlci8ke3VzZXJ9YFxuICAgICAgKVxuICAgICk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBvbmUgb3IgbW9yZSBJQU0gcm9sZSBwcmluY2lwYWxzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnQgSUQgb2YgdGhlIEFXUyBhY2NvdW50XG4gICAqIEBwYXJhbSByb2xlcyBOYW1lIG9mIHRoZSBJQU0gcm9sZVxuICAgKi9cbiAgcHVibGljIGZvclJvbGUoYWNjb3VudDogc3RyaW5nLCAuLi5yb2xlczogc3RyaW5nW10pIHtcbiAgICByb2xlcy5mb3JFYWNoKChyb2xlKSA9PlxuICAgICAgdGhpcy5hZGRQcmluY2lwYWwoXG4gICAgICAgIFByaW5jaXBhbFR5cGUuQVdTLFxuICAgICAgICBgYXJuOmF3czppYW06OiR7YWNjb3VudH06cm9sZS8ke3JvbGV9YFxuICAgICAgKVxuICAgICk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBvbmUgb3IgbW9yZSBzcGVjaWZpYyBhc3N1bWVkIHJvbGUgc2Vzc2lvbiBwcmluY2lwYWxzIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnQgSUQgb2YgdGhlIEFXUyBhY2NvdW50XG4gICAqIEBwYXJhbSByb2xlTmFtZSBOYW1lIG9mIHRoZSBJQU0gcm9sZVxuICAgKiBAcGFyYW0gc2Vzc2lvbk5hbWVzIE5hbWUgb2YgdGhlIHNlc3Npb24uIFlvdSBjYW5ub3QgdXNlIGEgd2lsZGNhcmQgKGAqYCkgdG8gbWVhbiAqYWxsIHNlc3Npb25zKi4gUHJpbmNpcGFscyBtdXN0IGFsd2F5cyBuYW1lIGEgc3BlY2lmaWMgc2Vzc2lvblxuICAgKi9cbiAgcHVibGljIGZvckFzc3VtZWRSb2xlU2Vzc2lvbihcbiAgICBhY2NvdW50OiBzdHJpbmcsXG4gICAgcm9sZU5hbWU6IHN0cmluZyxcbiAgICAuLi5zZXNzaW9uTmFtZXM6IHN0cmluZ1tdXG4gICkge1xuICAgIHNlc3Npb25OYW1lcy5mb3JFYWNoKChzZXNzaW9uTmFtZSkgPT4ge1xuICAgICAgdGhpcy5hZGRQcmluY2lwYWwoXG4gICAgICAgIFByaW5jaXBhbFR5cGUuQVdTLFxuICAgICAgICBgYXJuOmF3czpzdHM6OiR7YWNjb3VudH06YXNzdW1lZC1yb2xlLyR7cm9sZU5hbWV9LyR7c2Vzc2lvbk5hbWV9YFxuICAgICAgKTtcbiAgICB9KTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIG9uZSBvciBtb3JlIHNlcnZpY2UgcHJpbmNpcGFscyB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBzZXJ2aWNlcyBMb25nIHZlcnNpb24gb2YgdGhlIHNlcnZpY2UgbmFtZS4gVXN1YWxseSBpbiB0aGUgZm9ybWF0OiBgbG9uZ19zZXJ2aWNlLW5hbWUuYW1hem9uYXdzLmNvbWBcbiAgICpcbiAgICogVGhlIHNlcnZpY2UgcHJpbmNpcGFsIGlzIGRlZmluZWQgYnkgdGhlIHNlcnZpY2UuIFRvIGxlYXJuIHRoZSBzZXJ2aWNlIHByaW5jaXBhbCBmb3IgYSBzZXJ2aWNlLCBzZWUgdGhlIGRvY3VtZW50YXRpb24gZm9yIHRoYXQgc2VydmljZS4gRm9yIHNvbWUgc2VydmljZXMsIHNlZSBbQVdTIFNlcnZpY2VzIFRoYXQgV29yayB3aXRoIElBTV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9hd3Mtc2VydmljZXMtdGhhdC13b3JrLXdpdGgtaWFtLmh0bWwpIGFuZCBsb29rIGZvciB0aGUgc2VydmljZXMgdGhhdCBoYXZlICoqWWVzKiogaW4gdGhlICoqU2VydmljZS1MaW5rZWQgUm9sZSoqIGNvbHVtbi4gQ2hvb3NlIGEgKipZZXMqKiB3aXRoIGEgbGluayB0byB2aWV3IHRoZSBzZXJ2aWNlLWxpbmtlZCByb2xlIGRvY3VtZW50YXRpb24gZm9yIHRoYXQgc2VydmljZS4gVmlldyB0aGUgKipTZXJ2aWNlLUxpbmtlZCBSb2xlIFBlcm1pc3Npb25zKiogc2VjdGlvbiBmb3IgdGhhdCBzZXJ2aWNlIHRvIHZpZXcgdGhlIHNlcnZpY2UgcHJpbmNpcGFsLlxuICAgKi9cblxuICBwdWJsaWMgZm9yU2VydmljZSguLi5zZXJ2aWNlczogc3RyaW5nW10pIHtcbiAgICBzZXJ2aWNlcy5mb3JFYWNoKChzZXJ2aWNlKSA9PlxuICAgICAgdGhpcy5hZGRQcmluY2lwYWwoUHJpbmNpcGFsVHlwZS5TRVJWSUNFLCBzZXJ2aWNlKVxuICAgICk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHB1YmxpYyBhc3Nlc1xuICAgKlxuICAgKiAqKkVWRVJZT05FIElOIFRIRSBXT1JMRCBIQVMgQUNDRVNTKipcbiAgICpcbiAgICogV2Ugc3Ryb25nbHkgcmVjb21tZW5kIHRoYXQgeW91IGRvIG5vdCB1c2UgYSB3aWxkY2FyZCBpbiB0aGUgUHJpbmNpcGFsIGVsZW1lbnQgaW4gYSByb2xlJ3MgdHJ1c3QgcG9saWN5IHVubGVzcyB5b3Ugb3RoZXJ3aXNlIHJlc3RyaWN0IGFjY2VzcyB0aHJvdWdoIGEgQ29uZGl0aW9uIGVsZW1lbnQgaW4gdGhlIHBvbGljeS4gT3RoZXJ3aXNlLCBhbnkgSUFNIHVzZXIgaW4gYW55IGFjY291bnQgaW4geW91ciBwYXJ0aXRpb24gY2FuIGFjY2VzcyB0aGUgcm9sZS5cbiAgICovXG4gIHB1YmxpYyBmb3JQdWJsaWMoKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKFByaW5jaXBhbFR5cGUuQVdTLCAnKicpO1xuICB9XG59XG4iXX0=