#!/usr/bin/python

# (c) 2018-2019, NetApp, Inc
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function

__metaclass__ = type

ANSIBLE_METADATA = {'metadata_version': '1.1',
                    'status': ['preview'],
                    'supported_by': 'certified'}

DOCUMENTATION = '''
module: na_ontap_vscan_on_demand_task
short_description: NetApp ONTAP Vscan on demand task configuration.
extends_documentation_fragment:
    - netapp.ontap.netapp.na_ontap
version_added: '2.8'
author: NetApp Ansible Team (@carchi8py) <ng-ansibleteam@netapp.com>
description:
- Configure on demand task for Vscan
options:
  state:
    description:
    - Whether a Vscan on demand task is present or not
    choices: ['present', 'absent']
    default: present

  vserver:
    description:
    - the name of the data vserver to use.
    required: true

  cross_junction:
    description:
    - Specifies whether the On-Demand task is allowed to cross volume junctions
    type: bool
    default: False

  directory_recursion:
    description:
    - Specifies whether the On-Demand task is allowed to recursively scan through sub-directories.
    type: bool
    default: False

  file_ext_to_exclude:
    description:
    - File-Extensions for which scanning must not be performed.
    - File whose extension matches with both inclusion and exclusion list is not considered for scanning.
    type: list

  file_ext_to_include:
    description:
    - File extensions for which scanning is considered.
    - The default value is '*', which means that all files are considered for scanning except those which are excluded from scanning.
    - File whose extension matches with both inclusion and exclusion list is not considered for scanning.
    type: list

  max_file_size:
    description:
    - Max file-size (in bytes) allowed for scanning. The default value of 10737418240 (10GB) is taken if not provided at the time of creating a task.

  paths_to_exclude:
    description:
    - File-paths for which scanning must not be performed.
    type: list

  report_directory:
    description:
    - Path from the vserver root where task report is created. The path must be a directory and provided in unix-format from the root of the Vserver.
    - Example /vol1/on-demand-reports.

  report_log_level:
    description:
    - Log level for the On-Demand report.
    choices: ['verbose', 'info', 'error']
    default: error

  request_timeout:
    description:
    - Total request-service time-limit in seconds. If the virus-scanner does not respond within the provided time, scan will be timedout.

  scan_files_with_no_ext:
    description:
    - Specifies whether files without any extension are considered for scanning or not.
    type: bool
    default: True

  scan_paths:
    description:
    - List of paths that need to be scanned. The path must be provided in unix-format and from the root of the Vserver.
    - Example /vol1/large_files.
    type: list

  scan_priority:
    description:
    - Priority of the On-Demand scan requests generated by this task.
    choices: ['low', 'normal']
    default: low

  schedule:
    description:
    - Schedule of the task. The task will be run as per the schedule.
    - For running the task immediately, vscan-on-demand-task-run api must be used after creating a task.

  task_name:
    description:
    - Name of the task.
    required: True
'''


EXAMPLES = """
    - name: Create Vscan On Demand Task
      na_ontap_vscan_on_demand_task:
        state: present
        username: '{{ netapp_username }}'
        password: '{{ netapp_password }}'
        hostname: '{{ netapp_hostname }}'
        vserver: carchi-vsim2
        task_name: carchiOnDemand
        scan_paths: /
        report_directory: /
        file_ext_to_exclude: ['py', 'yml']
        max_file_size: 10737418241
        paths_to_exclude: ['/tmp', '/var']
        report_log_level: info
        request_timeout: 60

    - name: Delete Vscan On Demand Task
      na_ontap_vscan_on_demand_task:
        state: absent
        username: '{{ netapp_username }}'
        password: '{{ netapp_password }}'
        hostname: '{{ netapp_hostname }}'
        vserver: carchi-vsim2
        task_name: carchiOnDemand
"""

RETURN = """

"""

import traceback

from ansible.module_utils.basic import AnsibleModule
from ansible.module_utils._text import to_native
import ansible_collections.netapp.ontap.plugins.module_utils.netapp as netapp_utils
from ansible_collections.netapp.ontap.plugins.module_utils.netapp_module import NetAppModule

HAS_NETAPP_LIB = netapp_utils.has_netapp_lib()


class NetAppOntapVscanOnDemandTask(object):
    def __init__(self):
        self.argument_spec = netapp_utils.na_ontap_host_argument_spec()
        self.argument_spec.update(dict(
            state=dict(choices=['present', 'absent'], default='present'),
            vserver=dict(required=True, type='str'),
            cross_junction=dict(required=False, type='bool', default=False),
            directory_recursion=dict(required=False, type='bool', default=False),
            file_ext_to_exclude=dict(required=False, type="list"),
            file_ext_to_include=dict(required=False, type="list"),
            max_file_size=dict(required=False, type="str"),
            paths_to_exclude=dict(required=False, type="list"),
            report_directory=dict(required=False, type='str'),
            report_log_level=dict(required=False, choices=['verbose', 'info', 'error'], default='error'),
            request_timeout=dict(required=False, type='str'),
            scan_files_with_no_ext=dict(required=False, type='bool', default=True),
            scan_paths=dict(required=False, type="list"),
            scan_priority=dict(required=False, choices=['low', 'normal'], default='low'),
            schedule=dict(required=False, type="str"),
            task_name=dict(required=True, type="str")
        ))
        self.module = AnsibleModule(
            argument_spec=self.argument_spec,
            supports_check_mode=True,
            required_if=[
                ["state", "present", ["report_directory", "scan_paths"]]
            ]
        )
        self.na_helper = NetAppModule()
        self.parameters = self.na_helper.set_parameters(self.module.params)

        if HAS_NETAPP_LIB is False:
            self.module.fail_json(msg="the python NetApp-Lib module is required")
        else:
            self.server = netapp_utils.setup_na_ontap_zapi(module=self.module, vserver=self.parameters['vserver'])

    def get_demand_task(self):
        """
        Get a demand task
        :return: A vscan-on-demand-task-info or None
        """
        demand_task_iter = netapp_utils.zapi.NaElement("vscan-on-demand-task-get-iter")
        demand_task_info = netapp_utils.zapi.NaElement("vscan-on-demand-task-info")
        demand_task_info.add_new_child('task-name', self.parameters['task_name'])
        query = netapp_utils.zapi.NaElement('query')
        query.add_child_elem(demand_task_info)
        demand_task_iter.add_child_elem(query)
        try:
            result = self.server.invoke_successfully(demand_task_iter, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error searching for Vscan on demand task %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())
        if result.get_child_by_name('num-records') and int(result.get_child_content('num-records')) >= 1:
            return result.get_child_by_name('attributes-list').get_child_by_name('vscan-on-demand-task-info')
        return None

    def create_demand_task(self):
        """
        Create a Demand Task
        :return: None
        """
        demand_task_obj = netapp_utils.zapi.NaElement("vscan-on-demand-task-create")
        # Required items first
        demand_task_obj.add_new_child('report-directory', self.parameters['report_directory'])
        demand_task_obj.add_new_child('task-name', self.parameters['task_name'])
        scan_paths = netapp_utils.zapi.NaElement("scan-paths")
        for scan_path in self.parameters['scan_paths']:
            scan_paths.add_new_child('string', scan_path)
        demand_task_obj.add_child_elem(scan_paths)
        # Optional items next
        if self.parameters.get('cross_junction'):
            demand_task_obj.add_new_child('cross-junction', str(self.parameters['cross_junction']).lower())
        if self.parameters.get('directory_recursion'):
            demand_task_obj.add_new_child('directory-recursion', str(self.parameters['directory_recursion']).lower())
        if self.parameters.get('file_ext_to_exclude'):
            ext_to_exclude_obj = netapp_utils.zapi.NaElement('file-ext-to-exclude')
            for exclude_file in self.parameters['file_ext_to_exclude']:
                ext_to_exclude_obj.add_new_child('file-extension', exclude_file)
            demand_task_obj.add_child_elem(ext_to_exclude_obj)
        if self.parameters.get('file_ext_to_include'):
            ext_to_include_obj = netapp_utils.zapi.NaElement('file-ext-to-include')
            for include_file in self.parameters['file_ext_to_exclude']:
                ext_to_include_obj.add_child_elem('file-extension', include_file)
            demand_task_obj.add_child_elem(ext_to_include_obj)
        if self.parameters.get('max_file_size'):
            demand_task_obj.add_new_child('max-file-size', self.parameters['max_file_size'])
        if self.parameters.get('paths_to_exclude'):
            exclude_paths = netapp_utils.zapi.NaElement('paths-to-exclude')
            for path in self.parameters['paths_to_exclude']:
                exclude_paths.add_new_child('string', path)
            demand_task_obj.add_child_elem(exclude_paths)
        if self.parameters.get('report_log_level'):
            demand_task_obj.add_new_child('report-log-level', self.parameters['report_log_level'])
        if self.parameters.get('request_timeout'):
            demand_task_obj.add_new_child('request-timeout', self.parameters['request_timeout'])
        if self.parameters.get('scan_files_with_no_ext'):
            demand_task_obj.add_new_child('scan-files-with-no-ext', str(self.parameters['scan_files_with_no_ext']).lower())
        if self.parameters.get('scan_priority'):
            demand_task_obj.add_new_child('scan-priority', self.parameters['scan_priority'].lower())
        if self.parameters.get('schedule'):
            demand_task_obj.add_new_child('schedule', self.parameters['schedule'])
        try:
            result = self.server.invoke_successfully(demand_task_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error creating on demand task %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def delete_demand_task(self):
        """
        Delete a Demand Task"
        :return:
        """
        demand_task_obj = netapp_utils.zapi.NaElement('vscan-on-demand-task-delete')
        demand_task_obj.add_new_child('task-name', self.parameters['task_name'])
        try:
            self.server.invoke_successfully(demand_task_obj, True)
        except netapp_utils.zapi.NaApiError as error:
            self.module.fail_json(msg='Error deleting on demand task, %s: %s' %
                                      (self.parameters['task_name'], to_native(error)),
                                  exception=traceback.format_exc())

    def asup_log_for_cserver(self, event_name):
        """
        Fetch admin vserver for the given cluster
        Create and Autosupport log event with the given module name
        :param event_name: Name of the event log
        :return: None
        """
        results = netapp_utils.get_cserver(self.server)
        cserver = netapp_utils.setup_na_ontap_zapi(module=self.module, vserver=results)
        netapp_utils.ems_log_event(event_name, cserver)

    def apply(self):
        self.asup_log_for_cserver("na_ontap_vscan_on_demand_task")
        current = self.get_demand_task()
        cd_action = self.na_helper.get_cd_action(current, self.parameters)
        if self.na_helper.changed:
            if self.module.check_mode:
                pass
            else:
                if cd_action == 'create':
                    self.create_demand_task()
                elif cd_action == 'delete':
                    self.delete_demand_task()
        self.module.exit_json(changed=self.na_helper.changed)


def main():
    """
    Execute action from playbook
    """
    command = NetAppOntapVscanOnDemandTask()
    command.apply()


if __name__ == '__main__':
    main()
