# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['glQiwiApi',
 'glQiwiApi.core',
 'glQiwiApi.core.builtin',
 'glQiwiApi.core.web_hooks',
 'glQiwiApi.ext',
 'glQiwiApi.qiwi',
 'glQiwiApi.types',
 'glQiwiApi.types.qiwi_types',
 'glQiwiApi.types.yoomoney_types',
 'glQiwiApi.utils',
 'glQiwiApi.yoo_money']

package_data = \
{'': ['*']}

install_requires = \
['aiofiles>=0.6.0,<0.7.0',
 'aiohttp>=3.7.4.post0,<4.0.0',
 'loguru>=0.5.3,<0.6.0',
 'pydantic==1.8.2',
 'pytz==2021.1',
 'wheel>=0.36.2,<0.37.0']

extras_require = \
{':python_version < "3.7"': ['contextvars>=2.4,<3.0']}

setup_kwargs = {
    'name': 'glqiwiapi',
    'version': '1.0.3',
    'description': "The fast and lightweight wrapper of QIWI and YooMoney API's",
    'long_description': '<h2 align="center">\n<img src="https://github.com/GLEF1X/glQiwiApi/blob/master/docs/static/logo.png" width="200"></img>\n\n\n[![PyPI version](https://img.shields.io/pypi/v/glQiwiApi.svg)](https://pypi.org/project/glQiwiApi/) [![Python](https://img.shields.io/badge/Python-3.7+-blue)](https://www.python.org/downloads/) [![Code Quality Score](https://www.code-inspector.com/project/20780/score/svg)](https://frontend.code-inspector.com/public/project/20780/glQiwiApi/dashboard) ![Code Grade](https://www.code-inspector.com/project/20780/status/svg) ![Downloads](https://img.shields.io/pypi/dm/glQiwiApi) ![docs](https://readthedocs.org/projects/pip/badge/?version=latest)\n[![Language grade: Python](https://img.shields.io/lgtm/grade/python/g/GLEF1X/glQiwiApi.svg?logo=lgtm&logoWidth=18)](https://lgtm.com/projects/g/GLEF1X/glQiwiApi/context:python) [![CodeFactor](https://www.codefactor.io/repository/github/glef1x/glqiwiapi/badge)](https://www.codefactor.io/repository/github/glef1x/glqiwiapi)\n![codecov](https://codecov.io/gh/GLEF1X/glQiwiApi/branch/dev-1.x/graph/badge.svg?token=OD538HKV15)\n![CI](https://github.com/GLEF1X/glQiwiApi/actions/workflows/python-package.yml/badge.svg) ![mypy](https://img.shields.io/badge/%20type_checker-mypy-%231674b1?style=flat)\n\n<img src="https://github.com/GLEF1X/glQiwiApi/blob/master/demo.gif"/>\n</h2>\n\n## 🌎Official api resources:\n\n* 🎓 __Docs: [here](https://glqiwiapi.readthedocs.io/en/master/index.html)__\n* 🖱️ __Developer\n  contacts: [![Dev-Telegram](https://img.shields.io/badge/Telegram-blue.svg?style=flat-square&logo=telegram)](https://t.me/GLEF1X)__\n\n### 📣Why glQiwiApi?\n\n* :boom:__It\'s working faster than other async libraries for qiwi__\n* :dizzy:__Frequent updates and edits__\n* :innocent: __The library developer will always help you with any problems you\n  might encounter__\n\n### 💾Installation\n\n```bash\npip install glQiwiApi==1.0.3b2\n```\n\n---\n\n## 🐦Dependencies\n\n| Library | Description                                    |\n|:-------:|:----------------------------------------------:|\n|aiohttp  | default http server                            |\n|aiofiles | saving receipts in pdf                         |\n|uvloop   | Optional(can boost API), but work only on Linux|\n|pydantic | Json data validator. Very fast instead of custom|\n|loguru   | library which aims to bring enjoyable logging in Python|\n\n---\n\n## 🧸Dive-in Examples\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n\nasync def main():\n    # If you want to use qiwi wrapper without async context just \n    # pass on "without_context=True"\n    wallet = QiwiWrapper(\n        api_access_token=\'your_token\',\n        phone_number=\'+number\',\n        without_context=True\n    )\n    print((await wallet.get_balance()).amount)\n    # OR(x3 performance boost with async context,\n    # because it use only 1 aiohttp session to get response for all requests\n    # in async with context manager)\n    async with QiwiWrapper(api_access_token=\'your_token\') as w:\n        w.phone_number = \'+number\'\n        # Данным вызовом вы получите текущий баланс кошелька. \n        print((await w.get_balance()).amount)\n\n\n# Also you can use it like here\nmy_wallet = QiwiWrapper(\n    api_access_token=\'your_token\',\n    phone_number=\'+phone_number\'\n)\n\n\nasync def main_2():\n    async with my_wallet as w:\n        print(await w.get_balance())\n\n\nasyncio.run(main())\n```\n\n---\n\n## 🌀Checking transactions\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n\nasync def main():\n    async with QiwiWrapper(api_access_token=\'your_token\') as w:\n        w.phone_number = \'+number\'\n        # Таким образом мы проверим, была ли транзакция на сумму 999 рублей с комментарием\n        # \'I like glQiwiApi!\' и отправителем с номером +7904832168\n        is_paid = await w.check_transaction(\n            amount=999,\n            comment=\'I like glQiwiApi!\',\n            sender_number=\'+7904832168\'\n        )\n        print(is_paid)\n\n\nasyncio.run(main())\n```\n\n## 🌱Create & check p2p bills\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n\nasync def main():\n  # You can pass on only p2p tokens, if you want to use only p2p api\n  async with QiwiWrapper(\n          secret_p2p="your_secret_p2p"\n  ) as w:\n    # Таким образом можно создать p2p счет\n    # В примере указан счёт на 1 рубль с комментарием some_comment\n    bill = await w.create_p2p_bill(\n      amount=1,\n      comment=\'my_comm\'\n    )\n    # Проверка на статус "оплачено" созданного p2p счёта\n    if (await w.check_p2p_bill_status(bill_id=bill.bill_id)) == \'PAID\':\n      print(\'Успешно оплачено\')\n    else:\n      print(\'Транзакция не найдена\')\n    # Или, начиная с версии апи 0.2.0\n    print(await bill.paid)  # This will print you bool answer\n\n\nasyncio.run(main())\n```\n\n#### That will create a form like that\n\n![form](https://i.ibb.co/T0C5RYz/2021-03-21-14-58-33.png)\n\n## ⛏Send to another wallet & get receipt(получение чека транзакции)\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n\nasync def main():\n    async with QiwiWrapper(api_access_token="token") as w:\n        w.phone_number = "+number"\n        # Так выглядит перевод на другой киви кошелек\n        # в примере перевод будет на номер +7904832168 с комментарием "На шоколадку" и суммой 1 рубль\n        trans_id = await w.to_wallet(\n            to_number=\'+7904832168\',\n            comment=\'На шоколадку\',\n            trans_sum=1\n        )\n        # В данном примере мы сохраним чек в директории, где вы запускаете скрипт как my_receipt.pdf\n        await w.get_receipt(\n            transaction_id=trans_id,\n            transaction_type=\'OUT\',\n            file_path=\'my_receipt\'\n        )\n\n\nasyncio.run(main())\n\n```\n\n## 🌟Webhooks & handlers\n\n```python\n\nfrom glQiwiApi import QiwiWrapper, types, BaseFilter\nfrom glQiwiApi.utils import executor\n\nwallet = QiwiWrapper(\n    api_access_token=\'token from https://qiwi.com/api/\',\n    secret_p2p=\'secret token from https://qiwi.com/p2p-admin/\'\n)\n\nclass CustomFilter(BaseFilter):\n    async def check(self, update: types.Transaction) -> bool:\n        # some stuff\n        return True\n\n\n@wallet.transaction_handler(CustomFilter())  # start with 1.0.3b2 you can use class-based filters, but also combine it with lambda statements, if you want\nasync def get_transaction(event: types.WebHook):\n    print(event)\n\n\n@wallet.bill_handler()\nasync def fetch_bill(notification: types.Notification):\n    print(notification)\n\n    \nexecutor.start_webhook(wallet, port=80)\n```\n\n## 🧑🏻🔬Polling updates\n\n```python\nfrom glQiwiApi import BaseFilter, QiwiWrapper, types\nfrom glQiwiApi.utils import executor\n\n# let\'s imagine that payload its a dictionary with your tokens =)\nwallet = QiwiWrapper(**payload)\n\n\nclass MyFirstFilter(BaseFilter):\n    async def check(self, update: types.Transaction) -> bool:\n        return True\n\n\nclass MySecondFilter(BaseFilter):\n\n    async def check(self, update: types.Transaction) -> bool:\n        return False\n\n\n@wallet.transaction_handler(MyFirstFilter(), lambda event: event is not None, ~MySecondFilter())\nasync def my_handler(event: types.Transaction):\n    ...\n\n\nexecutor.start_polling(wallet)\n```\n\n## 💳Send to card & check commission\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n\nasync def main():\n    async with QiwiWrapper(api_access_token="token") as w:\n        w.phone_number = "+number"\n        # Так можно отправлять средства на карты разных банков, получая при этом айди транзакции\n        trans_id = await w.to_card(\n            trans_sum=1,\n            to_card=\'4890494756089082\'\n        )\n        print(trans_id)\n        # Так можно предварительно расчитать комиссию за транзакцию\n        commission = await w.commission(\n            to_account=\'4890494756089082\',\n            pay_sum=1\n        )\n        print(commission.qiwi_commission.amount)\n\n\nasyncio.run(main())\n\n```\n\n## 🚀Query caching (beta)\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper\n\n# Кэширование по умолчанию отключено, так как\n# эта функция все ещё находиться в бета тестировании и\n# константа DEFAULT_CACHE_TIME = 0, чтобы это исправить и включить кэширование\n# нужно передать cache_time в конструктор класса QiwiWrapper\n# или YooMoneyAPI\nwallet = QiwiWrapper(\n    # Токен, полученный с https://qiwi.com/api\n    api_access_token=\'token\',\n    # Номер вашего мобильного номера с "+"\n    phone_number=\'+phone_number\',\n    # Время кэширование запроса в секундах(пока ещё в бета тестировании)\n    cache_time=5\n)\n\n\nasync def cache_test():\n    async with wallet:\n        # Результат заноситься в кэш\n        print(await wallet.transactions(rows_num=50))\n        # Этот запрос возьмется из кэша\n        print(await wallet.transactions(rows_num=50))\n\n        # Запросы ниже не будут браться из кэша,\n        # причиной тому есть разница в параметрах запроса\n        # Результат все также заноситься в кэш\n        print(len(await wallet.transactions(rows_num=30)) == 30)  # True\n        # Однако, повторный запрос к апи будет выполнен, поскольку\n        # при попытке взятие результата из кэша валидатор сравнивает\n        # параметры запроса, если они не совпадают, то\n        # кэш игнорируется\n        # Повторный запрос к апи\n        print(len(await wallet.transactions(rows_num=10)) == 10)  # True\n\n\nasyncio.run(cache_test())\n\n\n```\n\n## ⚠️Handling exceptions\n\n```python\nimport asyncio\n\nfrom glQiwiApi import QiwiWrapper, RequestError\n\n\nasync def main():\n    async with QiwiWrapper(api_access_token=\'your_token\') as w:\n        w.phone_number = \'+number\'\n        try:\n            await w.to_card(to_card="some_card", trans_sum=2)\n        except RequestError as ex:\n            # Its give u full traceback from api if response was bad\n            print(ex.json())\n\n\nasyncio.run(main())\n```\n\n---\n\n## 🗺QIWI terminals\n\n__glQiwiApi covers qiwi\'s MAPS api in QiwiMaps class__\n\n---\n\n# YooMoney API\n\n## Important. How to get YooMoney access token\n\n+ #### Регистрируем своё приложение в YooMoney по ссылке: [click](https://yoomoney.ru/myservices/new)\n\n![yoo_money_register_app](https://i.imgur.com/Mu6R8Po.png)\n\n+ #### Получаем client_id после регистрации и далее используем уже YooMoneyAPI:\n\n```python\nimport asyncio\n\nfrom glQiwiApi import YooMoneyAPI\n\n\nasync def get_url_to_auth() -> None:\n    # Получаем ссылку для авторизации, переходим по ней, если получаем invalid_request или какую-то ошибку\n    # значит либо неправильно передан scope параметр, нужно уменьшить список прав или попробовать пересоздать приложение\n    print(await YooMoneyAPI.build_url_for_auth(\n        # Для платежей, проверки аккаунта и истории платежей, нужно указать scope=["account-info", "operation-history", "operation-details", "payment-p2p"]\n        scope=["account-info", "operation-history"],\n        client_id=\'айди, полученный при регистрации приложения выше\',\n        redirect_uri=\'ссылка, указаная при регистрации выше в поле Redirect URI\'\n    ))\n\n\nasyncio.run(get_url_to_auth())\n```\n\n+ #### Теперь нужно получить временный код и МАКСИМАЛЬНО БЫСТРО получить токен, используя class method YooMoneyAPI:\n\n![reg](https://i2.paste.pics/7660ed1444d1b3fc74b08128c74dbcd4.png?trs=9bfa3b1c0203c2ffe9982e7813a27700d047bfbc7ed23b79b99c1c4ffdd34995)\n\n```python\nimport asyncio\n\nfrom glQiwiApi import YooMoneyAPI\n\n\nasync def get_token() -> None:\n    print(await YooMoneyAPI.get_access_token(\n        code=\'код полученный из ссылки, как на скрине выше\',\n        client_id=\'айди приложения, полученое при регистрации\',\n        redirect_uri=\'ссылка, указанная при регистрации\'\n    ))\n\n\nasyncio.run(get_token())\n```\n\n---\n\n## Dive-in Examples\n\n```python\nimport asyncio\n\nfrom glQiwiApi import YooMoneyAPI\n\nTOKEN = \'some_token\'\n\n\nasync def main():\n    wallet = YooMoneyAPI(\n        api_access_token=TOKEN,\n        without_context=True\n    )\n    transactions = await wallet.transactions()\n    print(transactions)\n    # OR(x3 performance boost)\n    async with YooMoneyAPI(api_access_token=TOKEN) as w:\n        print(await w.transactions(records=50))\n\n\nasyncio.run(main())\n```\n\n## Send money to another wallet and checking this transaction\n\n```python\n\nimport asyncio\n\nfrom glQiwiApi import YooMoneyAPI\n\nTOKEN = \'your_token\'\n\n\nasync def main():\n    w = YooMoneyAPI(TOKEN)\n    async with w:\n        # Так вы можете отослать средства на другой счет, в примере это перевод на аккаунт 4100116602400968\n        # на сумму 2 рубля с комментарием "I LOVE glQiwiApi"\n        payment = await w.send(\n            to_account=\'4100116602400968\',\n            comment=\'I LOVE glQiwiApi\',\n            amount=2\n        )\n        # Опционально, так вы можете проверить транзакцию, поступила ли она человеку на счёт\n        print(await w.check_transaction(amount=2, comment=\'I LOVE glQiwiApi\',\n                                        transaction_type=\'out\'))\n\n\nasyncio.run(main())\n\n```\n\n## Fetch account info\n\n```python\nimport asyncio\n\nfrom glQiwiApi import YooMoneyAPI\n\nTOKEN = \'your_token\'\n\n\nasync def main():\n    w = YooMoneyAPI(TOKEN)\n    async with w:\n        # Так вы получаете информацию об аккаунте в виде объекта AccountInfo\n        account_info = await w.account_info\n        print(account_info.account_status)\n        print(account_info.balance)\n\n\nasyncio.run(main())\n\n```\n',
    'author': 'Gleb Garanin',
    'author_email': 'glebgar567@gmail.com',
    'maintainer': 'GLEF1X',
    'maintainer_email': 'glebgar567@gmail.com',
    'url': 'https://github.com/GLEF1X/glQiwiApi',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
