# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pg_es_fdw']

package_data = \
{'': ['*']}

install_requires = \
['elasticsearch>=7.6.0,<8.0.0']

setup_kwargs = {
    'name': 'pg-es-fdw',
    'version': '0.8.0',
    'description': 'Connect PostgreSQL and Elastic Search with this Foreign Data Wrapper',
    'long_description': 'PostgreSQL Elastic Search foreign data wrapper\n==============================================\n\nThis allows you to index data in Elastic Search and then search it from\nPostgreSQL. You can write as well as read.\n\nSYNOPSIS\n--------\n\n### Supported Versions\n\n| Elastic Search | Dependency Installation Command |\n|----------------|---------------------------------|\n| 5 | `sudo pip install "elasticsearch>=5,<6"` |\n| 6 | `sudo pip install "elasticsearch>=6,<7"` |\n| 7 | `sudo pip install "elasticsearch>=7,<8"` |\n\n| PostgreSQL | Dependency Installation Command |\n|------------|---------------------------------|\n| 9.4 | `sudo apt-get install postgresql-9.4-python-multicorn` |\n| 9.5 | `sudo apt-get install postgresql-9.5-python-multicorn` |\n| 9.6 | `sudo apt-get install postgresql-9.6-python-multicorn` |\n| 10 | `sudo apt-get install postgresql-10-python-multicorn` |\n| 11 | `sudo apt-get install postgresql-11-python-multicorn` |\n| 12 | `sudo apt-get install postgresql-12-python3-multicorn` |\n\nPlease note that the Debian package for Multicorn on PostgreSQL 12 requires Python 3.\n\n### Installation\n\nThis requires installation on the PostgreSQL server, and has system level dependencies.\nYou can install the dependencies with:\n\n```\nsudo apt-get install python python-pip\n```\n\nYou should install the version of multicorn that is specific to your postgres\nversion. See the table in _Supported Versions_ for installation commands. The\nmulticorn package is also only available from Ubuntu Xenial (16.04) onwards. If\nyou cannot install multicorn in this way then you can use\n[pgxn](http://pgxnclient.projects.pgfoundry.org/) to install it.\n\nThis uses the Elastic Search client which has release versions that correspond\nto the major version of the Elastic Search server. You should install the\n`elasticsearch` dependency separately. See the table in _Supported Versions_\nfor installation commands.\n\nOnce the dependencies are installed you can install the foreign data wrapper\nusing pip:\n\n```\nsudo pip install pg_es_fdw\n```\n\n### Usage\n\nA running configuration for this can be found in the `docker-compose.yml`\nwithin this folder.\n\nThe basic steps are:\n\n * Load the extension\n * Create the server\n * Create the foreign table\n * Populate the foreign table\n * Query the foreign table...\n\n#### Load extension and Create server\n\n```sql\nCREATE EXTENSION multicorn;\n\nCREATE SERVER multicorn_es FOREIGN DATA WRAPPER multicorn\nOPTIONS (\n  wrapper \'pg_es_fdw.ElasticsearchFDW\'\n);\n```\n\n#### Create the foreign table\n\n```sql\nCREATE FOREIGN TABLE articles_es\n    (\n        id BIGINT,\n        title TEXT,\n        body TEXT,\n        metadata JSON,\n        query TEXT,\n        score NUMERIC\n    )\nSERVER multicorn_es\nOPTIONS\n    (\n        host \'elasticsearch\',\n        port \'9200\',\n        index \'article-index\',\n        type \'article\',\n        rowid_column \'id\',\n        query_column \'query\',\n        score_column \'score\',\n        timeout \'20\',\n        username \'elastic\',\n        password \'changeme\'\n    )\n;\n```\n\nElastic Search 7 and greater does not require the `type` option, which\ncorresponds to the `doc_type` used in prior versions of Elastic Search.\n\nThis corresponds to an Elastic Search index which contains a `title` and `body`\nfields. The other fields have special meaning:\n\n * The `id` field is mapped to the Elastic Search document id\n * The `query` field accepts Elastic Search queries to filter the rows\n * The `score` field returns the score for the document against the query\n * The `timeout` field specifies the connection timeout in seconds\n * The `username` field specifies the basic auth username used\n * The `password` field specifies the basic auth password used\n * Any other options are passed through to the elasticsearch client, use this to specify things like ssl\n\nThese are configured using the `rowid_column`, `query_column`,\n`score_column`, `timeout`, `username` and `password` options.\nAll of these are optional.\n\nTo use basic auth you must provide both a username and a password,\neven if the password is blank.\n\n##### JSON and JSONB\n\nWhen elasticsearch returns nested data it is serialized to TEXT as json before being returned.\nThis means you can create columns with JSON or JSONB types and the data will be correctly converted on read.\nIf you write to a JSON or JSONB column then the data is passed to elasticsearch as json.\n\nAs the data is converted on the fly per query the benefits of using JSONB over JSON are limited.\n\n##### Elastic Search Authentication\n\nCurrently basic auth is supported for authentication.\nYou can provide the username and password by setting the `username` and `password` options when creating the table.\nYou must provide both, even if the password is blank.\nIf you do not provide them then basic auth is disabled for the table.\n\nIf you need to use other forms of authentication then please open an issue.\n\n#### Populate the foreign table\n\n```sql\nINSERT INTO articles_es\n    (\n        id,\n        title,\n        body,\n        metadata\n    )\nVALUES\n    (\n        1,\n        \'foo\',\n        \'spike\',\n        \'{"score": 3}\'::json\n    );\n```\n\nIt is possible to write documents to Elastic Search using the foreign data\nwrapper. This feature was introduced in PostgreSQL 9.3.\n\n#### Query the foreign table\n\nTo select all documents:\n\n```sql\nSELECT\n    id,\n    title,\n    body,\n    metadata\nFROM\n    articles_es\n;\n```\n\nTo filter the documents using a query:\n\n```sql\nSELECT\n    id,\n    title,\n    body,\n    metadata,\n    score\nFROM\n    articles_es\nWHERE\n    query = \'body:chess\'\n;\n```\n\nThis uses the [URI Search](https://www.elastic.co/guide/en/elasticsearch/reference/current/search-uri-request.html) from Elastic Search.\n\nCaveats\n-------\n\nElastic Search does not support transactions, so the elasticsearch index\nis not guaranteed to be synchronized with the canonical version in PostgreSQL.\nUnfortunately this is the case even for serializable isolation level transactions.\nIt would however be possible to check against Elastic Search version field and locking.\n\nRollback is currently not supported.\n\nTests\n-----\n\nThere are end to end tests that use docker to create a PostgreSQL and Elastic\nSearch database. These are then populated with data and tests are run against\nthem.\n\nThese require docker and docker-compose. These also require python packages\nwhich you can install with:\n\n```bash\npip install -r tests/requirements.txt\n```\n\nThe makefile will test all versions if you run `make test`:\n\n```bash\n➜ make test\npoetry run tests/run.py --pg 9.4 9.5 9.6 10 11 12 --es 5 6 7\nTesting PostgreSQL 9.4 with Elasticsearch 5\nPostgreSQL 9.4 with Elasticsearch 5: Test read - PASS\nPostgreSQL 9.4 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 9.4 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 9.4 with Elasticsearch 6\nPostgreSQL 9.4 with Elasticsearch 6: Test read - PASS\nPostgreSQL 9.4 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 9.4 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 9.4 with Elasticsearch 7\nPostgreSQL 9.4 with Elasticsearch 7: Test read - PASS\nPostgreSQL 9.4 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 9.4 with Elasticsearch 7: Test query - PASS\nTesting PostgreSQL 9.5 with Elasticsearch 5\nPostgreSQL 9.5 with Elasticsearch 5: Test read - PASS\nPostgreSQL 9.5 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 9.5 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 9.5 with Elasticsearch 6\nPostgreSQL 9.5 with Elasticsearch 6: Test read - PASS\nPostgreSQL 9.5 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 9.5 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 9.5 with Elasticsearch 7\nPostgreSQL 9.5 with Elasticsearch 7: Test read - PASS\nPostgreSQL 9.5 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 9.5 with Elasticsearch 7: Test query - PASS\nTesting PostgreSQL 9.6 with Elasticsearch 5\nPostgreSQL 9.6 with Elasticsearch 5: Test read - PASS\nPostgreSQL 9.6 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 9.6 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 9.6 with Elasticsearch 6\nPostgreSQL 9.6 with Elasticsearch 6: Test read - PASS\nPostgreSQL 9.6 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 9.6 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 9.6 with Elasticsearch 7\nPostgreSQL 9.6 with Elasticsearch 7: Test read - PASS\nPostgreSQL 9.6 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 9.6 with Elasticsearch 7: Test query - PASS\nTesting PostgreSQL 10 with Elasticsearch 5\nPostgreSQL 10 with Elasticsearch 5: Test read - PASS\nPostgreSQL 10 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 10 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 10 with Elasticsearch 6\nPostgreSQL 10 with Elasticsearch 6: Test read - PASS\nPostgreSQL 10 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 10 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 10 with Elasticsearch 7\nPostgreSQL 10 with Elasticsearch 7: Test read - PASS\nPostgreSQL 10 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 10 with Elasticsearch 7: Test query - PASS\nTesting PostgreSQL 11 with Elasticsearch 5\nPostgreSQL 11 with Elasticsearch 5: Test read - PASS\nPostgreSQL 11 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 11 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 11 with Elasticsearch 6\nPostgreSQL 11 with Elasticsearch 6: Test read - PASS\nPostgreSQL 11 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 11 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 11 with Elasticsearch 7\nPostgreSQL 11 with Elasticsearch 7: Test read - PASS\nPostgreSQL 11 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 11 with Elasticsearch 7: Test query - PASS\nTesting PostgreSQL 12 with Elasticsearch 5\nPostgreSQL 12 with Elasticsearch 5: Test read - PASS\nPostgreSQL 12 with Elasticsearch 5: Test nested-read - PASS\nPostgreSQL 12 with Elasticsearch 5: Test query - PASS\nTesting PostgreSQL 12 with Elasticsearch 6\nPostgreSQL 12 with Elasticsearch 6: Test read - PASS\nPostgreSQL 12 with Elasticsearch 6: Test nested-read - PASS\nPostgreSQL 12 with Elasticsearch 6: Test query - PASS\nTesting PostgreSQL 12 with Elasticsearch 7\nPostgreSQL 12 with Elasticsearch 7: Test read - PASS\nPostgreSQL 12 with Elasticsearch 7: Test nested-read - PASS\nPostgreSQL 12 with Elasticsearch 7: Test query - PASS\nPASS\n```\n\nIf you want to run the tests for specific versions then you can then run the\ntests using `tests/run.py`.  This takes the PostgreSQL version(s) to test using\nthe `--pg` argument and the Elastic Search versions to test with the `--es`\nargument.  The currently supported versions of PostgreSQL are 9.4 through to 12.\nThe currently supported versions of Elastic Search are 5 to 7. You can pass\nmultiple versions to test it against all of them.\n\n### Test Failure Messages\n\n```\nError starting userland proxy: listen tcp 0.0.0.0:5432: bind: address already in use\n```\nYou are already running something that listens to 5432.\nTry stopping your running postgres server:\n```\nsudo /etc/init.d/postgresql stop\n```\n\n```\nmax virtual memory areas vm.max_map_count [65530] is too low, increase to at least [262144]\n```\nYour system does not have the appropriate limits in place to run a production ready instance of elasticsearch.\nTry increasing it:\n```\nsudo sysctl -w vm.max_map_count=262144\n```\nThis setting will revert after a reboot.\n\n### Migrating from <=0.6.0\n\nIn version 0.7.0 the TEXT representation of json objects changed from HSTORE to JSON.\nIf you have been mapping json objects to HSTORE columns then you should change the column type to JSON.\nThe arrow operator exists for json so queries should not need rewriting.\n',
    'author': 'Matthew Franglen',
    'author_email': 'matthew@franglen.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/matthewfranglen/postgres-elasticsearch-fdw',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
}


setup(**setup_kwargs)
