/*
 * RELIC is an Efficient LIbrary for Cryptography
 * Copyright (C) 2007-2019 RELIC Authors
 *
 * This file is part of RELIC. RELIC is legal property of its developers,
 * whose names are not listed here. Please refer to the COPYRIGHT file
 * for contact information.
 *
 * RELIC is free software; you can redistribute it and/or modify it under the
 * terms of the version 2.1 (or later) of the GNU Lesser General Public License
 * as published by the Free Software Foundation; or version 2.0 of the Apache
 * License as published by the Apache Software Foundation. See the LICENSE files
 * for more details.
 *
 * RELIC is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE. See the LICENSE files for more details.
 *
 * You should have received a copy of the GNU Lesser General Public or the
 * Apache License along with RELIC. If not, see <https://www.gnu.org/licenses/>
 * or <https://www.apache.org/licenses/>.
 */

/**
 * @file
 *
 * Tests for hash functions.
 *
 * @ingroup test
 */

#include <stdio.h>

#include "relic.h"
#include "relic_test.h"

#if ARCH == AVR || ARCH == MSP || ARCH == ARM
#define MSG_SIZE	112
#define TEST_MAX	2
#else
#define MSG_SIZE	1000001
#define TEST_MAX	3
#endif

/*
 *  Define patterns for testing
 */
#define TEST1   "abc"

#define TEST2a  "abcdbcdecdefdefgefghfghighijhi"
#define TEST2b  "jkijkljklmklmnlmnomnopnopq"

#define TEST3a	"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
#define TEST3b	"aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"

#define TEST4a	"abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
#define TEST4b	"hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu"

char *tests[3] = {
	TEST1,
	TEST2a TEST2b,
	TEST3a TEST3b,
};

long int count[3] = { 1, 1, 10000 };

#if MD_MAP == SH224 || !defined(STRIP)

uint8_t result224[3][28] = {
	{0x23, 0x09, 0x7D, 0x22, 0x34, 0x05, 0xD8, 0x22, 0x86, 0x42, 0xA4, 0x77,
			0xBD, 0xA2, 0x55, 0xB3, 0x2A, 0xAD, 0xBC, 0xE4, 0xBD, 0xA0,
			0xB3, 0xF7, 0xE3, 0x6C, 0x9D, 0xA7},
	{0x75, 0x38, 0x8B, 0x16, 0x51, 0x27, 0x76, 0xCC, 0x5D, 0xBA, 0x5D, 0xA1,
			0xFD, 0x89, 0x01, 0x50, 0xB0, 0xC6, 0x45, 0x5C, 0xB4, 0xF5,
			0x8B, 0x19, 0x52, 0x52, 0x25, 0x25},
	{0x20, 0x79, 0x46, 0x55, 0x98, 0x0C, 0x91, 0xD8, 0xBB, 0xB4, 0xC1, 0xEA,
			0x97, 0x61, 0x8A, 0x4B, 0xF0, 0x3F, 0x42, 0x58, 0x19, 0x48,
			0xB2, 0xEE, 0x4E, 0xE7, 0xAD, 0x67},
};

uint8_t result_xmd_sha224[4][75] = {
	{0x8A, 0x96, 0x42, 0x30, 0x80, 0x8F, 0xC6, 0xC7, 0xB3, 0xD9, 0xC4,
	 0x85, 0xC6, 0x84, 0xEC, 0xEE, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x65, 0x5C, 0x30, 0x4F, 0x73, 0x08, 0x4A, 0x36, 0xEE, 0xA7, 0xE7,
	 0x3B, 0xEC, 0x44, 0xE7, 0x42, 0x97, 0xEC, 0x7A, 0x51, 0xA1, 0x10,
	 0xBE, 0x2D, 0xF2, 0xCF, 0xF8, 0x0C, 0xBD, 0xF2, 0x50, 0xF0, 0x16,
	 0x96, 0x83, 0x55, 0x4B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x2D, 0xD5, 0x65, 0x81, 0x01, 0xF4, 0x37, 0x66, 0xE0, 0x1E, 0x1F,
	 0x85, 0x81, 0xAE, 0x0D, 0x42, 0x45, 0x3E, 0x41, 0x39, 0x71, 0xF6,
	 0xE0, 0x7B, 0xB0, 0xC3, 0xBE, 0x2A, 0x92, 0x82, 0x1C, 0x67, 0xD1,
	 0xE8, 0x4E, 0x78, 0x49, 0xB7, 0x5C, 0x4A, 0x27, 0x56, 0xB7, 0xDB,
	 0x15, 0xF4, 0xB4, 0x5C, 0x8D, 0x2C, 0x44, 0xAB, 0x4B, 0xD1, 0x89,
	 0x04, 0xA5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xF2, 0xE6, 0xD9, 0x6D, 0x43, 0x15, 0x41, 0x63, 0x56, 0x67, 0x55,
	 0xEF, 0x54, 0x9E, 0x63, 0x9E, 0x26, 0x6E, 0x78, 0x3E, 0x24, 0xFF,
	 0x6A, 0x91, 0x8E, 0x34, 0x05, 0x49, 0x3B, 0x20, 0x1E, 0xBA, 0xAF,
	 0x8F, 0xB5, 0x95, 0xBC, 0x24, 0x11, 0x17, 0x8D, 0x7C, 0xAC, 0xC5,
	 0xE9, 0x1F, 0x01, 0x50, 0x88, 0x45, 0x92, 0x04, 0x6C, 0x77, 0x27,
	 0x4D, 0xC8, 0x33, 0x2C, 0x1D, 0x50, 0x84, 0xD6, 0xDA, 0x95, 0x0A,
	 0x97, 0xE7, 0x02, 0x0A, 0x86, 0x0C, 0xE8, 0xEE, 0x9C},
};

static int sha224(void) {
	int code = RLC_ERR;
	int i, j;
	uint8_t message[MSG_SIZE], digest[28];

	TEST_ONCE("sha224 hash function is correct") {
		for (i = 0; i < TEST_MAX; i++) {
			memset(message, 0, MSG_SIZE);
			for (j = 0; j < count[i]; j++) {
				strcat((char *)message, tests[i]);
			}
			md_map_sh224(digest, message, strlen((char *)message));
			TEST_ASSERT(memcmp(digest, result224[i], 28) == 0, end);
		}
	}
	TEST_END;

	TEST_ONCE("sha224 xmd function is correct") {
		uint8_t buf[75] = {0,};
		md_xmd_sh224(buf, 16, (uint8_t *)TEST1, strlen(TEST1), (uint8_t *)"", 0);
		TEST_ASSERT(memcmp(buf, result_xmd_sha224[0], 16) == 0, end);
		md_xmd_sh224(buf, 37, (uint8_t *)TEST2a, strlen(TEST2a), (uint8_t *)TEST2b, strlen(TEST2b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha224[1], 37) == 0, end);
		md_xmd_sh224(buf, 57, (uint8_t *)TEST3a, strlen(TEST3a), (uint8_t *)TEST3b, strlen(TEST3b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha224[2], 57) == 0, end);
		md_xmd_sh224(buf, 75, (uint8_t *)TEST4a, strlen(TEST4a), (uint8_t *)TEST4b, strlen(TEST4b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha224[3], 75) == 0, end);
	}
	TEST_END;

	code = RLC_OK;

  end:
	return code;
}

#endif

#if MD_MAP == SH256 || !defined(STRIP)

uint8_t result256[3][32] = {
	{0xBA, 0x78, 0x16, 0xBF, 0x8F, 0x01, 0xCF, 0xEA, 0x41, 0x41, 0x40, 0xDE,
			0x5D, 0xAE, 0x22, 0x23, 0xB0, 0x03, 0x61, 0xA3, 0x96, 0x17,
			0x7A, 0x9C, 0xB4, 0x10, 0xFF, 0x61, 0xF2, 0x00, 0x15, 0xAD},
	{0x24, 0x8d, 0x6A, 0x61, 0xd2, 0x06, 0x38, 0xb8, 0xE5, 0xc0, 0x26, 0x93,
			0x0c, 0x3E, 0x60, 0x39, 0xA3, 0x3c, 0xE4, 0x59, 0x64, 0xFF,
			0x21, 0x67, 0xF6, 0xEc, 0xEd, 0xd4, 0x19, 0xdB, 0x06, 0xc1},
	{0xCD, 0xC7, 0x6E, 0x5C, 0x99, 0x14, 0xFB, 0x92, 0x81, 0xA1, 0xC7, 0xE2,
			0x84, 0xd7, 0x3E, 0x67, 0xF1, 0x80, 0x9A, 0x48, 0xA4, 0x97,
			0x20, 0x0E, 0x04, 0x6D, 0x39, 0xCC, 0xC7, 0x11, 0x2C, 0xD0}
};

uint8_t result_xmd_sha256[4][75] = {
	{0x94, 0x77, 0xB7, 0xE5, 0xCA, 0x5E, 0xF7, 0x57, 0x6C, 0xF8, 0xF1,
	 0x18, 0x26, 0xAA, 0x86, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xE8, 0x5E, 0x5B, 0x4E, 0x3F, 0x36, 0x71, 0xC9, 0x6D, 0xA0, 0xF9,
	 0x52, 0xFF, 0x55, 0x00, 0xDB, 0x95, 0x62, 0x1D, 0x0C, 0x89, 0x4F,
	 0xD0, 0x8F, 0x37, 0x78, 0x50, 0x9D, 0xF5, 0x44, 0x9D, 0x63, 0xB7,
	 0x6C, 0xEC, 0x88, 0xF3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x3D, 0x5B, 0x01, 0x5C, 0x89, 0x4E, 0x82, 0x4D, 0xDA, 0xCC, 0x0B,
	 0x87, 0x27, 0x4A, 0x62, 0x66, 0xCB, 0xF5, 0x55, 0xF7, 0xF3, 0x84,
	 0x58, 0x09, 0x4A, 0xED, 0xD6, 0xD6, 0x70, 0x86, 0xC0, 0x04, 0x24,
	 0xE6, 0xE3, 0x83, 0xAA, 0x0F, 0x35, 0xB7, 0x82, 0xF7, 0xAC, 0xFF,
	 0xE0, 0xE8, 0x51, 0x25, 0xDB, 0xB4, 0x8B, 0xC0, 0x29, 0xD7, 0xE3,
	 0x0E, 0x7B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xD9, 0x95, 0x00, 0xD5, 0xCE, 0x7B, 0x24, 0xB6, 0x7E, 0x38, 0x21,
	 0xC0, 0x02, 0x9A, 0xBE, 0xF6, 0x6D, 0x20, 0x67, 0x51, 0x18, 0xFF,
	 0xA1, 0xCF, 0xFB, 0x81, 0x62, 0xF2, 0xFB, 0x4D, 0x8F, 0x8A, 0xE2,
	 0x41, 0x35, 0xFA, 0x8B, 0x1C, 0xAD, 0xE8, 0x00, 0x28, 0xA3, 0x5A,
	 0x3D, 0xB6, 0xCF, 0xCB, 0xBB, 0x35, 0x04, 0xF8, 0x6C, 0xB0, 0xA3,
	 0x1B, 0x2F, 0x01, 0x12, 0xC6, 0x10, 0x2B, 0x70, 0x2B, 0xDF, 0xD2,
	 0x8D, 0xAC, 0x57, 0xE3, 0x8F, 0x57, 0xE3, 0xE9, 0xBE},
};

static int sha256(void) {
	int code = RLC_ERR;
	int i, j;
	uint8_t message[MSG_SIZE], digest[32];

	TEST_ONCE("sha256 hash function is correct") {
		for (i = 0; i < TEST_MAX; i++) {
			memset(message, 0, MSG_SIZE);
			for (j = 0; j < count[i]; j++) {
				strcat((char *)message, tests[i]);
			}
			md_map_sh256(digest, message, strlen((char *)message));
			TEST_ASSERT(memcmp(digest, result256[i], 32) == 0, end);
		}
	}
	TEST_END;

	TEST_ONCE("sha256 xmd function is correct") {
		uint8_t buf[75] = {0,};
		md_xmd_sh256(buf, 16, (uint8_t *)TEST1, strlen(TEST1), (uint8_t *)"", 0);
		TEST_ASSERT(memcmp(buf, result_xmd_sha256[0], 16) == 0, end);
		md_xmd_sh256(buf, 37, (uint8_t *)TEST2a, strlen(TEST2a), (uint8_t *)TEST2b, strlen(TEST2b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha256[1], 37) == 0, end);
		md_xmd_sh256(buf, 57, (uint8_t *)TEST3a, strlen(TEST3a), (uint8_t *)TEST3b, strlen(TEST3b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha256[2], 57) == 0, end);
		md_xmd_sh256(buf, 75, (uint8_t *)TEST4a, strlen(TEST4a), (uint8_t *)TEST4b, strlen(TEST4b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha256[3], 75) == 0, end);
	}
	TEST_END;

	code = RLC_OK;

  end:
	return code;
}

#endif

#if MD_MAP == SH384 || !defined(STRIP)

char *tests2[3] = {
	TEST1,
	TEST4a TEST4b,
	TEST3a TEST3b
};

long int count2[3] = { 1, 1, 10000 };

uint8_t result384[4][48] = {
	{0xCB, 0x00, 0x75, 0x3F, 0x45, 0xA3, 0x5E, 0x8B, 0xB5, 0xA0, 0x3D, 0x69,
			0x9A, 0xC6, 0x50, 0x07, 0x27, 0x2C, 0x32, 0xAB, 0x0E, 0xDE,
			0xD1, 0x63, 0x1A, 0x8B, 0x60, 0x5A, 0x43, 0xFF, 0x5B, 0xED,
			0x80, 0x86, 0x07, 0x2B, 0xA1, 0xE7, 0xCC, 0x23, 0x58, 0xBA,
			0xEC, 0xA1, 0x34, 0xC8, 0x25, 0xA7},
	{0x09, 0x33, 0x0C, 0x33, 0xF7, 0x11, 0x47, 0xE8, 0x3D, 0x19, 0x2F, 0xC7,
			0x82, 0xCD, 0x1B, 0x47, 0x53, 0x11, 0x1B, 0x17, 0x3B, 0x3B,
			0x05, 0xD2, 0x2F, 0xA0, 0x80, 0x86, 0xE3, 0xB0, 0xF7, 0x12,
			0xFC, 0xC7, 0xC7, 0x1A, 0x55, 0x7E, 0x2D, 0xB9, 0x66, 0xC3,
			0xE9, 0xFA, 0x91, 0x74, 0x60, 0x39},
	{0x9D, 0x0E, 0x18, 0x09, 0x71, 0x64, 0x74, 0xCB, 0x08, 0x6E, 0x83, 0x4E,
			0x31, 0x0A, 0x4A, 0x1C, 0xED, 0x14, 0x9E, 0x9C, 0x00, 0xF2,
			0x48, 0x52, 0x79, 0x72, 0xCE, 0xC5, 0x70, 0x4C, 0x2A, 0x5B,
			0x07, 0xB8, 0xB3, 0xDC, 0x38, 0xEC, 0xC4, 0xEB, 0xAE, 0x97,
			0xDD, 0xD8, 0x7F, 0x3D, 0x89, 0x85},
};

uint8_t result_xmd_sha384[4][75] = {
	{0x08, 0xB9, 0x33, 0x36, 0xAE, 0x29, 0x19, 0x5A, 0x13, 0x4F, 0x70,
	 0x7C, 0x96, 0x60, 0x3A, 0x8E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0xB7, 0x8A, 0x23, 0x64, 0xC4, 0xF5, 0x08, 0xC0, 0x79, 0x35, 0xA6,
	 0xAA, 0x90, 0xF4, 0xAF, 0xFB, 0x8A, 0x5B, 0x30, 0xB9, 0xA4, 0x98,
	 0x79, 0xA6, 0xB2, 0x13, 0x05, 0xB4, 0x0F, 0x35, 0x42, 0x3F, 0xF4,
	 0x06, 0x12, 0x5A, 0xE0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x4D, 0x58, 0xC4, 0x5B, 0x6A, 0xA5, 0x0D, 0xEC, 0xC1, 0x5B, 0xBA,
	 0xFA, 0x43, 0x03, 0x4A, 0x94, 0xDE, 0x57, 0x98, 0x50, 0xC6, 0xCD,
	 0xFB, 0x30, 0xE9, 0xB0, 0x2F, 0x2B, 0x81, 0xC7, 0x72, 0x26, 0xAC,
	 0x00, 0x49, 0x3D, 0x70, 0x74, 0x90, 0xD3, 0xD4, 0x1B, 0xFF, 0x33,
	 0x97, 0x6F, 0x8A, 0x0B, 0x29, 0xFF, 0xBF, 0x1F, 0x87, 0x5A, 0xB6,
	 0x88, 0xB9, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x24, 0x1B, 0xA6, 0xF3, 0x0A, 0x18, 0x51, 0x3D, 0x07, 0x07, 0x6A,
	 0x8E, 0x24, 0xCC, 0xCD, 0x29, 0xDE, 0x36, 0xA0, 0xD4, 0xD6, 0x28,
	 0xD9, 0x65, 0x3E, 0x02, 0x15, 0xFB, 0xCE, 0x3A, 0xE8, 0xE6, 0x50,
	 0xEF, 0x76, 0x06, 0xF4, 0xDF, 0xB1, 0x29, 0xEB, 0x3F, 0xD5, 0xDA,
	 0xDE, 0x7E, 0x2F, 0x6A, 0x1A, 0x6A, 0xAF, 0x2D, 0xFE, 0xE0, 0x5B,
	 0x5B, 0xDC, 0xF6, 0x1F, 0x14, 0xF3, 0xE9, 0x16, 0xF1, 0x38, 0xF7,
	 0xF8, 0xD1, 0x46, 0x34, 0xBF, 0x5A, 0x03, 0xDC, 0x73},
};

static int sha384(void) {
	int code = RLC_ERR;
	int i, j;
	uint8_t message[MSG_SIZE], digest[48];

	TEST_ONCE("sha384 hash function is correct") {
		for (i = 0; i < TEST_MAX; i++) {
			memset(message, 0, MSG_SIZE);
			for (j = 0; j < count2[i]; j++) {
				strcat((char *)message, tests2[i]);
			}
			md_map_sh384(digest, message, strlen((char *)message));
			TEST_ASSERT(memcmp(digest, result384[i], 48) == 0, end);
		}
	}
	TEST_END;

	TEST_ONCE("sha384 xmd function is correct") {
		uint8_t buf[75] = {0,};
		md_xmd_sh384(buf, 16, (uint8_t *)TEST1, strlen(TEST1), (uint8_t *)"", 0);
		TEST_ASSERT(memcmp(buf, result_xmd_sha384[0], 16) == 0, end);
		md_xmd_sh384(buf, 37, (uint8_t *)TEST2a, strlen(TEST2a), (uint8_t *)TEST2b, strlen(TEST2b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha384[1], 37) == 0, end);
		md_xmd_sh384(buf, 57, (uint8_t *)TEST3a, strlen(TEST3a), (uint8_t *)TEST3b, strlen(TEST3b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha384[2], 57) == 0, end);
		md_xmd_sh384(buf, 75, (uint8_t *)TEST4a, strlen(TEST4a), (uint8_t *)TEST4b, strlen(TEST4b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha384[3], 75) == 0, end);
	}
	TEST_END;

	code = RLC_OK;

  end:
	return code;
}

#endif

#if MD_MAP == SH512 || !defined(STRIP)

uint8_t result512[4][64] = {
	{0xDD, 0xAF, 0x35, 0xA1, 0x93, 0x61, 0x7A, 0xBA, 0xCC, 0x41, 0x73, 0x49,
			0xAE, 0x20, 0x41, 0x31, 0x12, 0xE6, 0xFA, 0x4E, 0x89, 0xA9,
			0x7E, 0xA2, 0x0A, 0x9E, 0xEE, 0xE6, 0x4B, 0x55, 0xD3, 0x9A,
			0x21, 0x92, 0x99, 0x2A, 0x27, 0x4F, 0xC1, 0xA8, 0x36, 0xBA,
			0x3C, 0x23, 0xA3, 0xFE, 0xEB, 0xBD, 0x45, 0x4D, 0x44, 0x23,
			0x64, 0x3C, 0xE8, 0x0E, 0x2A, 0x9A, 0xC9, 0x4F, 0xA5, 0x4C,
			0xA4, 0x9F},
	{0x8E, 0x95, 0x9B, 0x75, 0xDA, 0xE3, 0x13, 0xDA, 0x8C, 0xF4, 0xF7, 0x28,
			0x14, 0xFC, 0x14, 0x3F, 0x8F, 0x77, 0x79, 0xC6, 0xEB, 0x9F,
			0x7F, 0xA1, 0x72, 0x99, 0xAE, 0xAD, 0xB6, 0x88, 0x90, 0x18,
			0x50, 0x1D, 0x28, 0x9E, 0x49, 0x00, 0xF7, 0xE4, 0x33, 0x1B,
			0x99, 0xDE, 0xC4, 0xB5, 0x43, 0x3A, 0xC7, 0xD3, 0x29, 0xEE,
			0xB6, 0xDD, 0x26, 0x54, 0x5E, 0x96, 0xE5, 0x5B, 0x87, 0x4B,
			0xE9, 0x09},
	{0xE7, 0x18, 0x48, 0x3D, 0x0C, 0xE7, 0x69, 0x64, 0x4E, 0x2E, 0x42, 0xC7,
			0xBC, 0x15, 0xB4, 0x63, 0x8E, 0x1F, 0x98, 0xB1, 0x3B, 0x20,
			0x44, 0x28, 0x56, 0x32, 0xA8, 0x03, 0xAF, 0xA9, 0x73, 0xEB,
			0xDE, 0x0F, 0xF2, 0x44, 0x87, 0x7E, 0xA6, 0x0A, 0x4C, 0xB0,
			0x43, 0x2C, 0xE5, 0x77, 0xC3, 0x1B, 0xEB, 0x00, 0x9C, 0x5C,
			0x2C, 0x49, 0xAA, 0x2E, 0x4E, 0xAD, 0xB2, 0x17, 0xAD, 0x8C,
			0xC0, 0x9B},
};

uint8_t result_xmd_sha512[4][75] = {
	{0x59, 0xDB, 0xF2, 0x0C, 0x8A, 0x4C, 0xCA, 0xA9, 0x92, 0xA9, 0x69,
	 0xEE, 0xFA, 0x7B, 0x75, 0x25, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x42, 0x86, 0xE2, 0x61, 0xD3, 0x5C, 0xAD, 0x94, 0x6B, 0xA2, 0xFF,
	 0x55, 0x8D, 0x5C, 0x71, 0xE8, 0x85, 0x76, 0x74, 0x55, 0xF8, 0xAE,
	 0x1D, 0xB3, 0xE8, 0x7A, 0xF3, 0x2E, 0x54, 0x31, 0x22, 0x01, 0xA0,
	 0x61, 0x8A, 0xC4, 0xCF, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x2A, 0x86, 0x54, 0x63, 0x3F, 0x41, 0xDF, 0x61, 0x52, 0x07, 0xCE,
	 0x04, 0xA8, 0x98, 0x22, 0x15, 0xAA, 0xC3, 0x39, 0x73, 0x21, 0x8B,
	 0x84, 0x96, 0x71, 0xCA, 0x61, 0x34, 0x09, 0x6E, 0x4E, 0xBC, 0x1D,
	 0xD3, 0x60, 0x2A, 0xC1, 0xA5, 0xF1, 0xA2, 0x07, 0xD4, 0xB7, 0xE1,
	 0xEC, 0x03, 0xF1, 0x51, 0x57, 0x05, 0x0B, 0xD4, 0xE9, 0xB3, 0xE6,
	 0xAE, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00},
	{0x07, 0x63, 0x10, 0x42, 0x23, 0x90, 0x13, 0x01, 0x61, 0xA2, 0x95,
	 0xCE, 0xB4, 0xA4, 0xA6, 0x4C, 0xFE, 0xE9, 0xA5, 0x57, 0x62, 0xA6,
	 0xC2, 0x98, 0x2B, 0x6C, 0x78, 0x7A, 0x50, 0xFE, 0x14, 0xC1, 0x42,
	 0x48, 0x32, 0x67, 0x8B, 0x6E, 0x8F, 0x05, 0xB5, 0x9F, 0xE5, 0x83,
	 0x0C, 0xE5, 0x29, 0x7F, 0x0F, 0x99, 0xC3, 0xDF, 0x6E, 0xF6, 0x2D,
	 0x33, 0x46, 0xEA, 0xF7, 0xC9, 0xC5, 0x9F, 0xF7, 0xCB, 0x6D, 0x68,
	 0xDC, 0x7F, 0x06, 0xEB, 0x60, 0x17, 0x66, 0x24, 0x99},
};

static int sha512(void) {
	int code = RLC_ERR;
	int i, j;
	uint8_t message[MSG_SIZE], digest[64];

	TEST_ONCE("sha512 hash function is correct") {
		for (i = 0; i < TEST_MAX; i++) {
			memset(message, 0, MSG_SIZE);
			for (j = 0; j < count2[i]; j++) {
				strcat((char *)message, tests2[i]);
			}
			md_map_sh512(digest, message, strlen((char *)message));
			TEST_ASSERT(memcmp(digest, result512[i], 64) == 0, end);
		}
	}
	TEST_END;

	TEST_ONCE("sha512 xmd function is correct") {
		uint8_t buf[75] = {0,};
		md_xmd_sh512(buf, 16, (uint8_t *)TEST1, strlen(TEST1), (uint8_t *)"", 0);
		TEST_ASSERT(memcmp(buf, result_xmd_sha512[0], 16) == 0, end);
		md_xmd_sh512(buf, 37, (uint8_t *)TEST2a, strlen(TEST2a), (uint8_t *)TEST2b, strlen(TEST2b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha512[1], 37) == 0, end);
		md_xmd_sh512(buf, 57, (uint8_t *)TEST3a, strlen(TEST3a), (uint8_t *)TEST3b, strlen(TEST3b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha512[2], 57) == 0, end);
		md_xmd_sh512(buf, 75, (uint8_t *)TEST4a, strlen(TEST4a), (uint8_t *)TEST4b, strlen(TEST4b));
		TEST_ASSERT(memcmp(buf, result_xmd_sha512[3], 75) == 0, end);
	}
	TEST_END;

	code = RLC_OK;

  end:
	return code;
}

#endif

#if MD_MAP == SH256

/* Test vector #1, ANSI X9.63 */

uint8_t key2[] = {
	0x44, 0x30, 0x24, 0xC3, 0xDA, 0xE6, 0x6B, 0x95, 0xE6, 0xF5, 0x67, 0x06,
	0x01, 0x55, 0x8F, 0x71
};

#endif

static int kdf(void) {
	int code = RLC_ERR;
	uint8_t message[] = { 0x96, 0xC0, 0x56, 0x19, 0xD5, 0x6C, 0x32, 0x8A, 0xB9,
		0x5F, 0xE8, 0x4B, 0x18, 0x26, 0x4B, 0x08, 0x72, 0x5B, 0x85, 0xE3, 0x3F,
		0xD3, 0x4F, 0x08 };
	uint8_t key[32];

#if MD_MAP == SH256
	TEST_ONCE("key derivation function (sha256) is correct") {
		md_kdf(key, sizeof(key2), message, sizeof(message));
		TEST_ASSERT(memcmp(key, key2, sizeof(key2)) == 0, end);
	}
	TEST_END;

	code = RLC_OK;

  end:
	return code;

#else
	(void)message;
	(void)code;
	(void)key;
	return RLC_OK;
#endif
}

static int hmac(void) {
	int code = RLC_ERR;
	uint8_t mac[RLC_MD_LEN];

#if MD_MAP == SH256
	uint8_t key[131];
	char *msg[] = {
		"Hi There", "what do ya want for nothing?", "Test With Truncation",
		"Test Using Larger Than Block-Size Key - Hash Key First",
		"This is a test using a larger than block-size key and a larger "
				"than block-size data. The key needs to be hashed before being "
				"used by the HMAC algorithm."
	};

	uint8_t result[][64] = {
		{ 0xB0, 0x34, 0x4C, 0x61, 0xD8, 0xDB, 0x38, 0x53, 0x5C, 0xA8, 0xAF,
			0xCE, 0xAF, 0x0B, 0xF1, 0x2B, 0x88, 0x1D, 0xC2, 0x00, 0xC9, 0x83,
			0x3D, 0xA7, 0x26, 0xE9, 0x37, 0x6C, 0x2E, 0x32, 0xCF, 0xF7 },
		{ 0x5B, 0xDC, 0xC1, 0x46, 0xBF, 0x60, 0x75, 0x4E, 0x6A, 0x04, 0x24,
			0x26, 0x08, 0x95, 0x75, 0xC7, 0x5A, 0x00, 0x3F, 0x08, 0x9D, 0x27,
			0x39, 0x83, 0x9D, 0xEC, 0x58, 0xB9, 0x64, 0xEC, 0x38, 0x43 },
		{ 0x77, 0x3E, 0xA9, 0x1E, 0x36, 0x80, 0x0E, 0x46, 0x85, 0x4D, 0xB8,
			0xEB, 0xD0, 0x91, 0x81, 0xA7, 0x29, 0x59, 0x09, 0x8B, 0x3E, 0xF8,
			0xC1, 0x22, 0xD9, 0x63, 0x55, 0x14, 0xCE, 0xD5, 0x65, 0xFE },
		{ 0x82, 0x55, 0x8A, 0x38, 0x9A, 0x44, 0x3C, 0x0E, 0xA4, 0xCC, 0x81,
			0x98, 0x99, 0xF2, 0x08, 0x3a, 0x85, 0xF0, 0xFA, 0xA3, 0xE5, 0x78,
			0xF8, 0x07, 0x7A, 0x2E, 0x3F, 0xF4, 0x67, 0x29, 0x66, 0x5B },
		{ 0xA3, 0xB6, 0x16, 0x74, 0x73, 0x10, 0x0E, 0xE0, 0x6E, 0x0C, 0x79,
			0x6C, 0x29, 0x55, 0x55, 0x2B },
		{ 0x60, 0xE4, 0x31, 0x59, 0x1E, 0xE0, 0xB6, 0x7F, 0x0D, 0x8A, 0x26,
			0xAA, 0xCB, 0xF5, 0xB7, 0x7F, 0x8E, 0x0B, 0xC6, 0x21, 0x37, 0x28,
			0xC5, 0x14, 0x05, 0x46, 0x04, 0x0F, 0x0E, 0xE3, 0x7F, 0x54 },
		{ 0x9B, 0x09, 0xFF, 0xA7, 0x1B, 0x94, 0x2F, 0xCB, 0x27, 0x63, 0x5F, 0xBC,
			0xD5, 0xB0, 0xE9, 0x44, 0xBF, 0xDC, 0x63, 0x64, 0x4F, 0x07,
			0x13, 0x93, 0x8A, 0x7F, 0x51, 0x53, 0x5C, 0x3A, 0x35, 0xE2 },
	};

	TEST_ONCE("hmac (sha256) is correct") {
		memset(key, 0x0B, 20);
		md_hmac(mac, (uint8_t *)msg[0], strlen(msg[0]), key, 20);
		TEST_ASSERT(memcmp(mac, result[0], RLC_MD_LEN) == 0, end);
		strncpy((char *)key, "Jefe", 5);
		md_hmac(mac, (uint8_t *)msg[1], strlen(msg[1]), key, 4);
		TEST_ASSERT(memcmp(mac, result[1], RLC_MD_LEN) == 0, end);
		memset(key, 0xAA, 20);
		memset(key + 20, 0xDD, 50);
		md_hmac(mac, key + 20, 50, key, 20);
		TEST_ASSERT(memcmp(mac, result[2], RLC_MD_LEN) == 0, end);
		for (int j = 0; j < 25; j++) {
			key[j] = j + 1;
		}
		memset(key + 25, 0xCD, 50);
		md_hmac(mac, key + 25, 50, key, 25);
		TEST_ASSERT(memcmp(mac, result[3], RLC_MD_LEN) == 0, end);
		memset(key, 0x0C, 20);
		md_hmac(mac, (uint8_t *)msg[2], strlen(msg[2]), key, 20);
		TEST_ASSERT(memcmp(mac, result[4], 16) == 0, end);
		memset(key, 0xAA, 131);
		md_hmac(mac, (uint8_t *)msg[3], strlen(msg[3]), key, 131);
		TEST_ASSERT(memcmp(mac, result[5], RLC_MD_LEN) == 0, end);
		md_hmac(mac, (uint8_t *)msg[4], strlen(msg[4]), key, 131);
		TEST_ASSERT(memcmp(mac, result[6], RLC_MD_LEN) == 0, end);
	}
	TEST_END;
#endif

	code = RLC_OK;

  end:
	return code;
}

int main(void) {
	if (core_init() != RLC_OK) {
		core_clean();
		return 1;
	}

	util_banner("Tests for the MD module:\n", 0);

#if MD_MAP == SH224 || !defined(STRIP)
	if (sha224() != RLC_OK) {
		core_clean();
		return 1;
	}
#endif

#if MD_MAP == SH256 || !defined(STRIP)
	if (sha256() != RLC_OK) {
		core_clean();
		return 1;
	}
#endif

#if MD_MAP == SH384 || !defined(STRIP)
	if (sha384() != RLC_OK) {
		core_clean();
		return 1;
	}
#endif

#if MD_MAP == SH512 || !defined(STRIP)
	if (sha512() != RLC_OK) {
		core_clean();
		return 1;
	}
#endif

	if (kdf() != RLC_OK) {
		core_clean();
		return 1;
	}

	if (hmac() != RLC_OK) {
		core_clean();
		return 1;
	}

	util_banner("All tests have passed.\n", 0);

	core_clean();
	return 0;
}
