"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CognitoIdentity = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class CognitoIdentity extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cognito-identity](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitoidentity.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cognito-identity';
        this.accessLevelList = {
            "Write": [
                "CreateIdentityPool",
                "DeleteIdentities",
                "DeleteIdentityPool",
                "GetId",
                "MergeDeveloperIdentities",
                "SetIdentityPoolRoles",
                "UnlinkDeveloperIdentity",
                "UnlinkIdentity",
                "UpdateIdentityPool"
            ],
            "Read": [
                "DescribeIdentity",
                "DescribeIdentityPool",
                "GetCredentialsForIdentity",
                "GetIdentityPoolRoles",
                "GetOpenIdToken",
                "GetOpenIdTokenForDeveloperIdentity",
                "LookupDeveloperIdentity"
            ],
            "List": [
                "ListIdentities",
                "ListIdentityPools",
                "ListTagsForResource"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates a new identity pool.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_CreateIdentityPool.html
     */
    toCreateIdentityPool() {
        this.to('cognito-identity:CreateIdentityPool');
        return this;
    }
    /**
     * Deletes identities from an identity pool. You can specify a list of 1-60 identities that you want to delete.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentities.html
     */
    toDeleteIdentities() {
        this.to('cognito-identity:DeleteIdentities');
        return this;
    }
    /**
     * Deletes a user pool. Once a pool is deleted, users will not be able to authenticate with the pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DeleteIdentityPool.html
     */
    toDeleteIdentityPool() {
        this.to('cognito-identity:DeleteIdentityPool');
        return this;
    }
    /**
     * Returns metadata related to the given identity, including when the identity was created and any associated linked logins.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentity.html
     */
    toDescribeIdentity() {
        this.to('cognito-identity:DescribeIdentity');
        return this;
    }
    /**
     * Gets details about a particular identity pool, including the pool name, ID description, creation date, and current number of users.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_DescribeIdentityPool.html
     */
    toDescribeIdentityPool() {
        this.to('cognito-identity:DescribeIdentityPool');
        return this;
    }
    /**
     * Returns credentials for the provided identity ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetCredentialsForIdentity.html
     */
    toGetCredentialsForIdentity() {
        this.to('cognito-identity:GetCredentialsForIdentity');
        return this;
    }
    /**
     * Generates (or retrieves) a Cognito ID. Supplying multiple logins will create an implicit linked account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetId.html
     */
    toGetId() {
        this.to('cognito-identity:GetId');
        return this;
    }
    /**
     * Gets the roles for an identity pool.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetIdentityPoolRoles.html
     */
    toGetIdentityPoolRoles() {
        this.to('cognito-identity:GetIdentityPoolRoles');
        return this;
    }
    /**
     * Gets an OpenID token, using a known Cognito ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdToken.html
     */
    toGetOpenIdToken() {
        this.to('cognito-identity:GetOpenIdToken');
        return this;
    }
    /**
     * Registers (or retrieves) a Cognito IdentityId and an OpenID Connect token for a user authenticated by your backend authentication process.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_GetOpenIdTokenForDeveloperIdentity.html
     */
    toGetOpenIdTokenForDeveloperIdentity() {
        this.to('cognito-identity:GetOpenIdTokenForDeveloperIdentity');
        return this;
    }
    /**
     * Lists the identities in a pool.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentities.html
     */
    toListIdentities() {
        this.to('cognito-identity:ListIdentities');
        return this;
    }
    /**
     * Lists all of the Cognito identity pools registered for your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListIdentityPools.html
     */
    toListIdentityPools() {
        this.to('cognito-identity:ListIdentityPools');
        return this;
    }
    /**
     * Lists the tags that are assigned to an Amazon Cognito identity pool.
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('cognito-identity:ListTagsForResource');
        return this;
    }
    /**
     * Retrieves the IdentityID associated with a DeveloperUserIdentifier or the list of DeveloperUserIdentifiers associated with an IdentityId for an existing identity.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_LookupDeveloperIdentity.html
     */
    toLookupDeveloperIdentity() {
        this.to('cognito-identity:LookupDeveloperIdentity');
        return this;
    }
    /**
     * Merges two users having different IdentityIds, existing in the same identity pool, and identified by the same developer provider.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_MergeDeveloperIdentities.html
     */
    toMergeDeveloperIdentities() {
        this.to('cognito-identity:MergeDeveloperIdentities');
        return this;
    }
    /**
     * Sets the roles for an identity pool. These roles are used when making calls to GetCredentialsForIdentity action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_SetIdentityPoolRoles.html
     */
    toSetIdentityPoolRoles() {
        this.to('cognito-identity:SetIdentityPoolRoles');
        return this;
    }
    /**
     * Assigns a set of tags to an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('cognito-identity:TagResource');
        return this;
    }
    /**
     * Unlinks a DeveloperUserIdentifier from an existing identity.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkDeveloperIdentity.html
     */
    toUnlinkDeveloperIdentity() {
        this.to('cognito-identity:UnlinkDeveloperIdentity');
        return this;
    }
    /**
     * Unlinks a federated identity from an existing account.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UnlinkIdentity.html
     */
    toUnlinkIdentity() {
        this.to('cognito-identity:UnlinkIdentity');
        return this;
    }
    /**
     * Removes the specified tags from an Amazon Cognito identity pool.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('cognito-identity:UntagResource');
        return this;
    }
    /**
     * Updates a user pool.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/cognitoidentity/latest/APIReference/API_UpdateIdentityPool.html
     */
    toUpdateIdentityPool() {
        this.to('cognito-identity:UpdateIdentityPool');
        return this;
    }
    /**
     * Adds a resource of type identitypool to the statement
     *
     * https://docs.aws.amazon.com/cognito/latest/developerguide/identity-pools.html
     *
     * @param identityPoolId - Identifier for the identityPoolId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIdentitypool(identityPoolId, account, region, partition) {
        var arn = 'arn:${Partition}:cognito-identity:${Region}:${Account}:identitypool/${IdentityPoolId}';
        arn = arn.replace('${IdentityPoolId}', identityPoolId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.CognitoIdentity = CognitoIdentity;
//# sourceMappingURL=data:application/json;base64,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