"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Elasticache = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [elasticache](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticache.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Elasticache extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [elasticache](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticache.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'elasticache';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Write": [
                "AuthorizeCacheSecurityGroupIngress",
                "BatchApplyUpdateAction",
                "BatchStopUpdateAction",
                "CompleteMigration",
                "CopySnapshot",
                "CreateCacheCluster",
                "CreateCacheParameterGroup",
                "CreateCacheSecurityGroup",
                "CreateCacheSubnetGroup",
                "CreateGlobalReplicationGroup",
                "CreateReplicationGroup",
                "CreateSnapshot",
                "DecreaseNodeGroupsInGlobalReplicationGroup",
                "DecreaseReplicaCount",
                "DeleteCacheCluster",
                "DeleteCacheParameterGroup",
                "DeleteCacheSecurityGroup",
                "DeleteCacheSubnetGroup",
                "DeleteGlobalReplicationGroup",
                "DeleteReplicationGroup",
                "DeleteSnapshot",
                "DisassociateGlobalReplicationGroup",
                "FailoverGlobalReplicationGroup",
                "IncreaseNodeGroupsInGlobalReplicationGroup",
                "IncreaseReplicaCount",
                "ModifyCacheCluster",
                "ModifyCacheParameterGroup",
                "ModifyCacheSubnetGroup",
                "ModifyGlobalReplicationGroup",
                "ModifyReplicationGroup",
                "ModifyReplicationGroupShardConfiguration",
                "PurchaseReservedCacheNodesOffering",
                "RebalanceSlotsInGlobalReplicationGroup",
                "RebootCacheCluster",
                "ResetCacheParameterGroup",
                "RevokeCacheSecurityGroupIngress",
                "StartMigration",
                "TestFailover"
            ],
            "List": [
                "DescribeCacheClusters",
                "DescribeCacheEngineVersions",
                "DescribeCacheParameterGroups",
                "DescribeCacheParameters",
                "DescribeCacheSecurityGroups",
                "DescribeCacheSubnetGroups",
                "DescribeEngineDefaultParameters",
                "DescribeEvents",
                "DescribeGlobalReplicationGroups",
                "DescribeReplicationGroups",
                "DescribeReservedCacheNodes",
                "DescribeReservedCacheNodesOfferings",
                "DescribeServiceUpdates",
                "DescribeSnapshots",
                "DescribeUpdateActions",
                "ListAllowedNodeTypeModifications"
            ],
            "Read": [
                "ListTagsForResource"
            ]
        };
    }
    /**
     * The AddTagsToResource action adds up to 10 cost allocation tags to the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('elasticache:AddTagsToResource');
        return this;
    }
    /**
     * The AuthorizeCacheSecurityGroupIngress action allows network ingress to a cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:AuthorizeSecurityGroupIngress
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_AuthorizeCacheSecurityGroupIngress.html
     */
    toAuthorizeCacheSecurityGroupIngress() {
        this.to('elasticache:AuthorizeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Apply the service update.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchApplyUpdateAction.html
     */
    toBatchApplyUpdateAction() {
        this.to('elasticache:BatchApplyUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_BatchStopUpdateAction.html
     */
    toBatchStopUpdateAction() {
        this.to('elasticache:BatchStopUpdateAction');
        return this;
    }
    /**
     * Stop the service update.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CompleteMigration.html
     */
    toCompleteMigration() {
        this.to('elasticache:CompleteMigration');
        return this;
    }
    /**
     * The CopySnapshot action makes a copy of an existing snapshot.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CopySnapshot.html
     */
    toCopySnapshot() {
        this.to('elasticache:CopySnapshot');
        return this;
    }
    /**
     * The CreateCacheCluster action creates a cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheCluster.html
     */
    toCreateCacheCluster() {
        this.to('elasticache:CreateCacheCluster');
        return this;
    }
    /**
     * The CreateCacheParameterGroup action creates a new cache parameter group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheParameterGroup.html
     */
    toCreateCacheParameterGroup() {
        this.to('elasticache:CreateCacheParameterGroup');
        return this;
    }
    /**
     * The CreateCacheSecurityGroup action creates a new cache security group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSecurityGroup.html
     */
    toCreateCacheSecurityGroup() {
        this.to('elasticache:CreateCacheSecurityGroup');
        return this;
    }
    /**
     * The CreateCacheSubnetGroup action creates a new cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateCacheSubnetGroup.html
     */
    toCreateCacheSubnetGroup() {
        this.to('elasticache:CreateCacheSubnetGroup');
        return this;
    }
    /**
     * The CreateGlobalReplicationGroup action creates a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateGlobalReplicationGroup.html
     */
    toCreateGlobalReplicationGroup() {
        this.to('elasticache:CreateGlobalReplicationGroup');
        return this;
    }
    /**
     * The CreateReplicationGroup action creates a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     * - elasticache:AddTagsToResource
     * - s3:GetObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateReplicationGroup.html
     */
    toCreateReplicationGroup() {
        this.to('elasticache:CreateReplicationGroup');
        return this;
    }
    /**
     * The CreateSnapshot action creates a copy of an entire cache cluster at a specific moment in time.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     * - s3:DeleteObject
     * - s3:GetBucketAcl
     * - s3:PutObject
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('elasticache:CreateSnapshot');
        return this;
    }
    /**
     * The DecreaseNodeGroupsInGlobalReplicationGroup action dec a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toDecreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:DecreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The DecreaseReplicaCount action decreases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DecreaseReplicaCount.html
     */
    toDecreaseReplicaCount() {
        this.to('elasticache:DecreaseReplicaCount');
        return this;
    }
    /**
     * The DeleteCacheCluster action deletes a previously provisioned cache cluster.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheCluster.html
     */
    toDeleteCacheCluster() {
        this.to('elasticache:DeleteCacheCluster');
        return this;
    }
    /**
     * The DeleteCacheParameterGroup action deletes the specified cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheParameterGroup.html
     */
    toDeleteCacheParameterGroup() {
        this.to('elasticache:DeleteCacheParameterGroup');
        return this;
    }
    /**
     * The DeleteCacheSecurityGroup action deletes a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSecurityGroup.html
     */
    toDeleteCacheSecurityGroup() {
        this.to('elasticache:DeleteCacheSecurityGroup');
        return this;
    }
    /**
     * The DeleteCacheSubnetGroup action deletes a cache subnet group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteCacheSubnetGroup.html
     */
    toDeleteCacheSubnetGroup() {
        this.to('elasticache:DeleteCacheSubnetGroup');
        return this;
    }
    /**
     * The DeleteGlobalReplicationGroup action deletes a global datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteGlobalReplicationGroup.html
     */
    toDeleteGlobalReplicationGroup() {
        this.to('elasticache:DeleteGlobalReplicationGroup');
        return this;
    }
    /**
     * The DeleteReplicationGroup action deletes an existing replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteReplicationGroup.html
     */
    toDeleteReplicationGroup() {
        this.to('elasticache:DeleteReplicationGroup');
        return this;
    }
    /**
     * The DeleteSnapshot action deletes an existing snapshot.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DeleteSnapshot.html
     */
    toDeleteSnapshot() {
        this.to('elasticache:DeleteSnapshot');
        return this;
    }
    /**
     * The DescribeCacheClusters action returns information about all provisioned cache clusters if no cache cluster identifier is specified, or about a specific cache cluster if a cache cluster identifier is supplied.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheClusters.html
     */
    toDescribeCacheClusters() {
        this.to('elasticache:DescribeCacheClusters');
        return this;
    }
    /**
     * The DescribeCacheEngineVersions action returns a list of the available cache engines and their versions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheEngineVersions.html
     */
    toDescribeCacheEngineVersions() {
        this.to('elasticache:DescribeCacheEngineVersions');
        return this;
    }
    /**
     * The DescribeCacheParameterGroups action returns information about parameter groups for this account, or a particular parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameterGroups.html
     */
    toDescribeCacheParameterGroups() {
        this.to('elasticache:DescribeCacheParameterGroups');
        return this;
    }
    /**
     * The DescribeCacheParameters action returns the detailed parameter list for a particular cache parameter group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheParameters.html
     */
    toDescribeCacheParameters() {
        this.to('elasticache:DescribeCacheParameters');
        return this;
    }
    /**
     * The DescribeCacheSecurityGroups action returns a list of cache security group descriptions, or the description of the specified security group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSecurityGroups.html
     */
    toDescribeCacheSecurityGroups() {
        this.to('elasticache:DescribeCacheSecurityGroups');
        return this;
    }
    /**
     * The DescribeCacheSubnetGroups action returns a list of cache subnet group descriptions, or the description of the specified subnet group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeCacheSubnetGroups.html
     */
    toDescribeCacheSubnetGroups() {
        this.to('elasticache:DescribeCacheSubnetGroups');
        return this;
    }
    /**
     * The DescribeEngineDefaultParameters action returns the default engine and system parameter information for the specified cache engine.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEngineDefaultParameters.html
     */
    toDescribeEngineDefaultParameters() {
        this.to('elasticache:DescribeEngineDefaultParameters');
        return this;
    }
    /**
     * The DescribeEvents action returns events related to cache clusters, cache security groups, and cache parameter groups.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeEvents.html
     */
    toDescribeEvents() {
        this.to('elasticache:DescribeEvents');
        return this;
    }
    /**
     * The DescribeGlobalReplicationGroups action returns information about global datastores for this account, or a particular global datastore.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeGlobalReplicationGroups.html
     */
    toDescribeGlobalReplicationGroups() {
        this.to('elasticache:DescribeGlobalReplicationGroups');
        return this;
    }
    /**
     * The DescribeReplicationGroups action returns information about replication groups for this account, or a particular replication group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReplicationGroups.html
     */
    toDescribeReplicationGroups() {
        this.to('elasticache:DescribeReplicationGroups');
        return this;
    }
    /**
     * The DescribeReservedCacheNodes action returns information about reserved cache nodes for this account, or a particular reserved cache node.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodes.html
     */
    toDescribeReservedCacheNodes() {
        this.to('elasticache:DescribeReservedCacheNodes');
        return this;
    }
    /**
     * The DescribeReservedCacheNodesOfferings action lists available reserved cache node offerings.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeReservedCacheNodesOfferings.html
     */
    toDescribeReservedCacheNodesOfferings() {
        this.to('elasticache:DescribeReservedCacheNodesOfferings');
        return this;
    }
    /**
     * Returns details of the service updates
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeServiceUpdates.html
     */
    toDescribeServiceUpdates() {
        this.to('elasticache:DescribeServiceUpdates');
        return this;
    }
    /**
     * The DescribeSnapshots action returns information about cache cluster snapshots.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeSnapshots.html
     */
    toDescribeSnapshots() {
        this.to('elasticache:DescribeSnapshots');
        return this;
    }
    /**
     * Returns details of the update actions.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DescribeUpdateActions.html
     */
    toDescribeUpdateActions() {
        this.to('elasticache:DescribeUpdateActions');
        return this;
    }
    /**
     * The DisassociateGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_DisassociateGlobalReplicationGroup.html
     */
    toDisassociateGlobalReplicationGroup() {
        this.to('elasticache:DisassociateGlobalReplicationGroup');
        return this;
    }
    /**
     * The FailoverGlobalReplicationGroup action removes a secondary Replication Group from the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_FailoverGlobalReplicationGroup.html
     */
    toFailoverGlobalReplicationGroup() {
        this.to('elasticache:FailoverGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseNodeGroupsInGlobalReplicationGroup action increases the number of node groups in the Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseNodeGroupsInGlobalReplicationGroup.html
     */
    toIncreaseNodeGroupsInGlobalReplicationGroup() {
        this.to('elasticache:IncreaseNodeGroupsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The IncreaseReplicaCount action increases the number of replicas in a Redis replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_IncreaseReplicaCount.html
     */
    toIncreaseReplicaCount() {
        this.to('elasticache:IncreaseReplicaCount');
        return this;
    }
    /**
     * List Allowed Node Type Modifications
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListAllowedNodeTypeModifications.html
     */
    toListAllowedNodeTypeModifications() {
        this.to('elasticache:ListAllowedNodeTypeModifications');
        return this;
    }
    /**
     * The ListTagsForResource action lists all cost allocation tags currently on the named resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('elasticache:ListTagsForResource');
        return this;
    }
    /**
     * The ModifyCacheCluster action modifies the settings for a cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheCluster.html
     */
    toModifyCacheCluster() {
        this.to('elasticache:ModifyCacheCluster');
        return this;
    }
    /**
     * The ModifyCacheParameterGroup action modifies the parameters of a cache parameter group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheParameterGroup.html
     */
    toModifyCacheParameterGroup() {
        this.to('elasticache:ModifyCacheParameterGroup');
        return this;
    }
    /**
     * The ModifyCacheSubnetGroup action modifies an existing cache subnet group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyCacheSubnetGroup.html
     */
    toModifyCacheSubnetGroup() {
        this.to('elasticache:ModifyCacheSubnetGroup');
        return this;
    }
    /**
     * The ModifyGlobalReplicationGroup action modifies the settings for a Global Datastore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyGlobalReplicationGroup.html
     */
    toModifyGlobalReplicationGroup() {
        this.to('elasticache:ModifyGlobalReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroup action modifies the settings for a replication group.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroup.html
     */
    toModifyReplicationGroup() {
        this.to('elasticache:ModifyReplicationGroup');
        return this;
    }
    /**
     * The ModifyReplicationGroupShardConfiguration action allows you to add shards, remove shards, or rebalance the keyspaces among exisiting shards.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ModifyReplicationGroupShardConfiguration.html
     */
    toModifyReplicationGroupShardConfiguration() {
        this.to('elasticache:ModifyReplicationGroupShardConfiguration');
        return this;
    }
    /**
     * The PurchaseReservedCacheNodesOffering action allows you to purchase a reserved cache node offering.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - elasticache:AddTagsToResource
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_PurchaseReservedCacheNodesOffering.html
     */
    toPurchaseReservedCacheNodesOffering() {
        this.to('elasticache:PurchaseReservedCacheNodesOffering');
        return this;
    }
    /**
     * The RebalanceSlotsInGlobalReplicationGroup action redistributes slots to ensure uniform distribution across existing shards in the cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebalanceSlotsInGlobalReplicationGroup.html
     */
    toRebalanceSlotsInGlobalReplicationGroup() {
        this.to('elasticache:RebalanceSlotsInGlobalReplicationGroup');
        return this;
    }
    /**
     * The RebootCacheCluster action reboots some, or all, of the cache nodes within a provisioned cache cluster.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RebootCacheCluster.html
     */
    toRebootCacheCluster() {
        this.to('elasticache:RebootCacheCluster');
        return this;
    }
    /**
     * The RemoveTagsFromResource action removes the tags identified by the TagKeys list from the named resource.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('elasticache:RemoveTagsFromResource');
        return this;
    }
    /**
     * The ResetCacheParameterGroup action modifies the parameters of a cache parameter group to the engine or system default value.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_ResetCacheParameterGroup.html
     */
    toResetCacheParameterGroup() {
        this.to('elasticache:ResetCacheParameterGroup');
        return this;
    }
    /**
     * The RevokeCacheSecurityGroupIngress action revokes ingress from a cache security group.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_RevokeCacheSecurityGroupIngress.html
     */
    toRevokeCacheSecurityGroupIngress() {
        this.to('elasticache:RevokeCacheSecurityGroupIngress');
        return this;
    }
    /**
     * Start the migration of data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_StartMigration.html
     */
    toStartMigration() {
        this.to('elasticache:StartMigration');
        return this;
    }
    /**
     * The TestFailover action allows you to test automatic failover on a specified node group in a replication group
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - ec2:CreateNetworkInterface
     * - ec2:DeleteNetworkInterface
     * - ec2:DescribeNetworkInterfaces
     * - ec2:DescribeSubnets
     * - ec2:DescribeVpcs
     *
     * https://docs.aws.amazon.com/AmazonElastiCache/latest/APIReference/API_TestFailover.html
     */
    toTestFailover() {
        this.to('elasticache:TestFailover');
        return this;
    }
    /**
     * Adds a resource of type parametergroup to the statement
     *
     * @param cacheParameterGroupName - Identifier for the cacheParameterGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onParametergroup(cacheParameterGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:parametergroup:${CacheParameterGroupName}';
        arn = arn.replace('${CacheParameterGroupName}', cacheParameterGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type securitygroup to the statement
     *
     * @param cacheSecurityGroupName - Identifier for the cacheSecurityGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSecuritygroup(cacheSecurityGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:securitygroup:${CacheSecurityGroupName}';
        arn = arn.replace('${CacheSecurityGroupName}', cacheSecurityGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type subnetgroup to the statement
     *
     * @param cacheSubnetGroupName - Identifier for the cacheSubnetGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSubnetgroup(cacheSubnetGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:subnetgroup:${CacheSubnetGroupName}';
        arn = arn.replace('${CacheSubnetGroupName}', cacheSubnetGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type replicationgroup to the statement
     *
     * @param replicationGroupId - Identifier for the replicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReplicationgroup(replicationGroupId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:replicationgroup:${ReplicationGroupId}';
        arn = arn.replace('${ReplicationGroupId}', replicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type cluster to the statement
     *
     * @param cacheClusterId - Identifier for the cacheClusterId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCluster(cacheClusterId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:cluster:${CacheClusterId}';
        arn = arn.replace('${CacheClusterId}', cacheClusterId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type reserved-instance to the statement
     *
     * @param reservedCacheNodeId - Identifier for the reservedCacheNodeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onReservedInstance(reservedCacheNodeId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:reserved-instance:${ReservedCacheNodeId}';
        arn = arn.replace('${ReservedCacheNodeId}', reservedCacheNodeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * @param snapshotName - Identifier for the snapshotName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onSnapshot(snapshotName, account, region, partition) {
        var arn = 'arn:${Partition}:elasticache:${Region}:${Account}:snapshot:${SnapshotName}';
        arn = arn.replace('${SnapshotName}', snapshotName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type globalreplicationgroup to the statement
     *
     * @param globalReplicationGroupId - Identifier for the globalReplicationGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onGlobalreplicationgroup(globalReplicationGroupId, account, partition) {
        var arn = 'arn:${Partition}:elasticache::${Account}:globalreplicationgroup:${GlobalReplicationGroupId}';
        arn = arn.replace('${GlobalReplicationGroupId}', globalReplicationGroupId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Elasticache = Elasticache;
//# sourceMappingURL=data:application/json;base64,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