"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ebs = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ebs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ebs extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ebs](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticblockstore.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ebs';
        this.accessLevelList = {
            "Write": [
                "CompleteSnapshot",
                "PutSnapshotBlock",
                "StartSnapshot"
            ],
            "Read": [
                "GetSnapshotBlock",
                "ListChangedBlocks",
                "ListSnapshotBlocks"
            ]
        };
    }
    /**
     * Grants permission to seal and complete the snapshot after all of the required blocks of data have been written to it.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_CompleteSnapshot.html
     */
    toCompleteSnapshot() {
        this.to('ebs:CompleteSnapshot');
        return this;
    }
    /**
     * Grants permission to return the data of a block in an Amazon Elastic Block Store (EBS) snapshot
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_GetSnapshotBlock.html
     */
    toGetSnapshotBlock() {
        this.to('ebs:GetSnapshotBlock');
        return this;
    }
    /**
     * Grants permission to list the blocks that are different between two Amazon Elastic Block Store (EBS) snapshots of the same volume/snapshot lineage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListChangedBlocks.html
     */
    toListChangedBlocks() {
        this.to('ebs:ListChangedBlocks');
        return this;
    }
    /**
     * Grants permission to list the blocks in an Amazon Elastic Block Store (EBS) snapshot.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_ListSnapshotBlocks.html
     */
    toListSnapshotBlocks() {
        this.to('ebs:ListSnapshotBlocks');
        return this;
    }
    /**
     * Grants permission to write a block of data to a snapshot created by the StartSnapshot operation.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_PutSnapshotBlock.html
     */
    toPutSnapshotBlock() {
        this.to('ebs:PutSnapshotBlock');
        return this;
    }
    /**
     * Grants permission to create a new EBS snapshot.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/ebs/latest/APIReference/API_StartSnapshot.html
     */
    toStartSnapshot() {
        this.to('ebs:StartSnapshot');
        return this;
    }
    /**
     * Adds a resource of type snapshot to the statement
     *
     * https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/iam-policy-structure.html#EC2_ARN_Format
     *
     * @param snapshotId - Identifier for the snapshotId.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     * - .ifDescription()
     * - .ifParentSnapshot()
     * - .ifVolumeSize()
     */
    onSnapshot(snapshotId, region, partition) {
        var arn = 'arn:${Partition}:ec2:${Region}::snapshot/${SnapshotId}';
        arn = arn.replace('${SnapshotId}', snapshotId);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the description of the snapshot being created.
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDescription(value, operator) {
        return this.if(`ebs:Description`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the ID of the parent snapshot.
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifParentSnapshot(value, operator) {
        return this.if(`ebs:ParentSnapshot`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the size of the volume for the snapshot being created, in GiB.
     *
     * Applies to resource types:
     * - snapshot
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifVolumeSize(value, operator) {
        return this.if(`ebs:VolumeSize`, value, operator || 'NumericEquals');
    }
}
exports.Ebs = Ebs;
//# sourceMappingURL=data:application/json;base64,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