"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Events = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Events extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [events](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoneventbridge.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'events';
        this.accessLevelList = {
            "Write": [
                "ActivateEventSource",
                "CreateEventBus",
                "CreatePartnerEventSource",
                "DeactivateEventSource",
                "DeleteEventBus",
                "DeletePartnerEventSource",
                "DeleteRule",
                "DisableRule",
                "EnableRule",
                "PutEvents",
                "PutPartnerEvents",
                "PutPermission",
                "PutTargets",
                "RemovePermission",
                "RemoveTargets"
            ],
            "Read": [
                "DescribeEventBus",
                "DescribeEventSource",
                "DescribePartnerEventSource",
                "DescribeRule",
                "TestEventPattern"
            ],
            "List": [
                "ListEventBuses",
                "ListEventSources",
                "ListPartnerEventSourceAccounts",
                "ListPartnerEventSources",
                "ListRuleNamesByTarget",
                "ListRules",
                "ListTagsForResource",
                "ListTargetsByRule"
            ],
            "Tagging": [
                "PutRule",
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Activates a partner event source that has been deactivated. Once activated, your matching event bus will start receiving events from the event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ActivateEventSource.html
     */
    toActivateEventSource() {
        this.to('events:ActivateEventSource');
        return this;
    }
    /**
     * Creates a new event bus within your account. This can be a custom event bus which you can use to receive events from your own custom applications and services, or it can be a partner event bus which can be matched to a partner event source.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreateEventBus.html
     */
    toCreateEventBus() {
        this.to('events:CreateEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_CreatePartnerEventSource.html
     */
    toCreatePartnerEventSource() {
        this.to('events:CreatePartnerEventSource');
        return this;
    }
    /**
     * Called by an AWS partner to create a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeactivateEventSource.html
     */
    toDeactivateEventSource() {
        this.to('events:DeactivateEventSource');
        return this;
    }
    /**
     * Deletes the specified custom event bus or partner event bus. All rules associated with this event bus are also deleted. You can't delete your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeleteEventBus() {
        this.to('events:DeleteEventBus');
        return this;
    }
    /**
     * Called by an AWS partner to delete a partner event source.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeletePartnerEventSource
     */
    toDeletePartnerEventSource() {
        this.to('events:DeletePartnerEventSource');
        return this;
    }
    /**
     * Deletes a rule. You must remove all targets from a rule using RemoveTargets before you can delete the rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DeleteRule.html
     */
    toDeleteRule() {
        this.to('events:DeleteRule');
        return this;
    }
    /**
     * Displays the external AWS accounts that are permitted to write events to your account using your account's event bus, and the associated policy.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventBus.html
     */
    toDescribeEventBus() {
        this.to('events:DescribeEventBus');
        return this;
    }
    /**
     * Describes the details of the specified partner event source that is shared with your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeEventSource.html
     */
    toDescribeEventSource() {
        this.to('events:DescribeEventSource');
        return this;
    }
    /**
     * Called by an AWS partner describe the details of the specified partner event source that they have created.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribePartnerEventSource.html
     */
    toDescribePartnerEventSource() {
        this.to('events:DescribePartnerEventSource');
        return this;
    }
    /**
     * Describes the details of the specified rule.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DescribeRule.html
     */
    toDescribeRule() {
        this.to('events:DescribeRule');
        return this;
    }
    /**
     * Disables a rule. A disabled rule won't match any events, and won't self-trigger if it has a schedule expression.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_DisableRule.html
     */
    toDisableRule() {
        this.to('events:DisableRule');
        return this;
    }
    /**
     * Enables a rule. If the rule does not exist, the operation fails.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_EnableRule.html
     */
    toEnableRule() {
        this.to('events:EnableRule');
        return this;
    }
    /**
     * Lists all the event buses in your account, including the default event bus, custom event buses, and partner event buses.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventBuses.html
     */
    toListEventBuses() {
        this.to('events:ListEventBuses');
        return this;
    }
    /**
     * Lists the event sources shared with this account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListEventSources.html
     */
    toListEventSources() {
        this.to('events:ListEventSources');
        return this;
    }
    /**
     * Called by an AWS partner to display the AWS account ID that the specified partner event source is associated with.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSourceAccounts.html
     */
    toListPartnerEventSourceAccounts() {
        this.to('events:ListPartnerEventSourceAccounts');
        return this;
    }
    /**
     * Called by an AWS partner to list all the partner event sources that they have created.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListPartnerEventSources.html
     */
    toListPartnerEventSources() {
        this.to('events:ListPartnerEventSources');
        return this;
    }
    /**
     * Lists the names of the rules that the given target is put to.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRuleNamesByTarget.html
     */
    toListRuleNamesByTarget() {
        this.to('events:ListRuleNamesByTarget');
        return this;
    }
    /**
     * Lists the Amazon EventBridge rules in your account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListRules.html
     */
    toListRules() {
        this.to('events:ListRules');
        return this;
    }
    /**
     * This action lists tags for an Amazon EventBridge resource.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('events:ListTagsForResource');
        return this;
    }
    /**
     * Lists of targets assigned to the rule.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_ListTargetsByRule.html
     */
    toListTargetsByRule() {
        this.to('events:ListTargetsByRule');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutEvents.html
     */
    toPutEvents() {
        this.to('events:PutEvents');
        return this;
    }
    /**
     * Sends custom events to Amazon EventBridge so that they can be matched to rules.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPartnerEvents.html
     */
    toPutPartnerEvents() {
        this.to('events:PutPartnerEvents');
        return this;
    }
    /**
     * Running PutPermission permits the specified AWS account to put events to your account's default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutPermission.html
     */
    toPutPermission() {
        this.to('events:PutPermission');
        return this;
    }
    /**
     * Creates or updates a rule. Rules are enabled by default, or based on value of the State parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDetailUserIdentityPrincipalId()
     * - .ifDetailType()
     * - .ifSource()
     * - .ifDetailService()
     * - .ifDetailEventTypeCode()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutRule.html
     */
    toPutRule() {
        this.to('events:PutRule');
        return this;
    }
    /**
     * Adds target(s) to a rule. Targets are the resources that can be invoked when a rule is triggered.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifTargetArn()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_PutTargets.html
     */
    toPutTargets() {
        this.to('events:PutTargets');
        return this;
    }
    /**
     * Revokes the permission of another AWS account to be able to put events to your default event bus.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemovePermission.html
     */
    toRemovePermission() {
        this.to('events:RemovePermission');
        return this;
    }
    /**
     * Removes target(s) from a rule so that when the rule is triggered, those targets will no longer be invoked.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_RemoveTargets.html
     */
    toRemoveTargets() {
        this.to('events:RemoveTargets');
        return this;
    }
    /**
     * This action tags an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('events:TagResource');
        return this;
    }
    /**
     * Tests whether an event pattern matches the provided event.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_TestEventPattern.html
     */
    toTestEventPattern() {
        this.to('events:TestEventPattern');
        return this;
    }
    /**
     * This action removes a tag from an Amazon EventBridge resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/eventbridge/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('events:UntagResource');
        return this;
    }
    /**
     * Adds a resource of type event-source to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventSourceName - Identifier for the eventSourceName.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onEventSource(eventSourceName, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}::event-source/${EventSourceName}';
        arn = arn.replace('${EventSourceName}', eventSourceName);
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-bus to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param eventBusName - Identifier for the eventBusName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventBus(eventBusName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:event-bus/${EventBusName}';
        arn = arn.replace('${EventBusName}', eventBusName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/iam-access-control-identity-based-eventbridge.html#eventbridge-arn-format
     *
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:events:${Region}:${Account}:rule/${RuleName}';
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * The ARN of a target that can be put to a rule.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limiting-access-to-targets
     *
     * Applies to actions:
     * - .toPutTargets()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifTargetArn(value, operator) {
        return this.if(`events:TargetArn`, value, operator || 'ArnLike');
    }
    /**
     * Matches the literal string of the detail-type filed of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-pattern-detail-type
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailType(value, operator) {
        return this.if(`events:detail-type`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.eventTypeCode field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-type-code
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailEventTypeCode(value, operator) {
        return this.if(`events:detail.eventTypeCode`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.service field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#limit-rule-by-service
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailService(value, operator) {
        return this.if(`events:detail.service`, value, operator || 'StringLike');
    }
    /**
     * Matches the literal string for the detail.useridentity.principalid field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#consume-specific-events
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDetailUserIdentityPrincipalId(value, operator) {
        return this.if(`events:detail.userIdentity.principalId`, value, operator || 'StringLike');
    }
    /**
     * The AWS service or AWS partner event source that generated the event. Matches the literal string of the source field of the event.
     *
     * https://docs.aws.amazon.com/eventbridge/latest/userguide/policy-keys-eventbridge.html#events-limit-access-control
     *
     * Applies to actions:
     * - .toPutRule()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSource(value, operator) {
        return this.if(`events:source`, value, operator || 'StringLike');
    }
}
exports.Events = Events;
//# sourceMappingURL=data:application/json;base64,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