"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Frauddetector = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Frauddetector extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [frauddetector](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonfrauddetector.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'frauddetector';
        this.accessLevelList = {
            "Write": [
                "BatchCreateVariable",
                "CreateDetectorVersion",
                "CreateModel",
                "CreateModelVersion",
                "CreateRule",
                "CreateVariable",
                "DeleteDetector",
                "DeleteDetectorVersion",
                "DeleteEvent",
                "DeleteRule",
                "PutDetector",
                "PutEntityType",
                "PutEventType",
                "PutExternalModel",
                "PutKMSEncryptionKey",
                "PutLabel",
                "PutOutcome",
                "UpdateDetectorVersion",
                "UpdateDetectorVersionMetadata",
                "UpdateDetectorVersionStatus",
                "UpdateModel",
                "UpdateModelVersion",
                "UpdateModelVersionStatus",
                "UpdateRuleMetadata",
                "UpdateRuleVersion",
                "UpdateVariable"
            ],
            "List": [
                "BatchGetVariable",
                "GetDetectorVersion",
                "GetDetectors",
                "GetEntityTypes",
                "GetEventTypes",
                "GetExternalModels",
                "GetLabels",
                "GetModelVersion",
                "GetModels",
                "GetOutcomes",
                "GetRules",
                "GetVariables",
                "ListTagsForResource"
            ],
            "Read": [
                "DescribeDetector",
                "DescribeModelVersions",
                "GetEventPrediction",
                "GetKMSEncryptionKey"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Creates a batch of variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchCreateVariable
     */
    toBatchCreateVariable() {
        this.to('frauddetector:BatchCreateVariable');
        return this;
    }
    /**
     * Gets a batch of variables.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_BatchGetVariable
     */
    toBatchGetVariable() {
        this.to('frauddetector:BatchGetVariable');
        return this;
    }
    /**
     * Creates a detector version. The detector version starts in a DRAFT status.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateDetectorVersion
     */
    toCreateDetectorVersion() {
        this.to('frauddetector:CreateDetectorVersion');
        return this;
    }
    /**
     * Creates a model using the specified model type.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModel
     */
    toCreateModel() {
        this.to('frauddetector:CreateModel');
        return this;
    }
    /**
     * Creates a version of the model using the specified model type and model id.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateModelVersion
     */
    toCreateModelVersion() {
        this.to('frauddetector:CreateModelVersion');
        return this;
    }
    /**
     * Creates a rule for use with the specified detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateRule.html
     */
    toCreateRule() {
        this.to('frauddetector:CreateRule');
        return this;
    }
    /**
     * Creates a variable.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_CreateVariable.html
     */
    toCreateVariable() {
        this.to('frauddetector:CreateVariable');
        return this;
    }
    /**
     * Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions associated with the detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetector
     */
    toDeleteDetector() {
        this.to('frauddetector:DeleteDetector');
        return this;
    }
    /**
     * Deletes the detector version. You cannot delete detector versions that are in ACTIVE status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteDetectorVersion
     */
    toDeleteDetectorVersion() {
        this.to('frauddetector:DeleteDetectorVersion');
        return this;
    }
    /**
     * Deletes the specified event.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteEvent
     */
    toDeleteEvent() {
        this.to('frauddetector:DeleteEvent');
        return this;
    }
    /**
     * Deletes the rule. You cannot delete a rule if it is used by an ACTIVE or INACTIVE detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DeleteRule
     */
    toDeleteRule() {
        this.to('frauddetector:DeleteRule');
        return this;
    }
    /**
     * Gets all versions for a specified detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeDetector
     */
    toDescribeDetector() {
        this.to('frauddetector:DescribeDetector');
        return this;
    }
    /**
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can also get details for a single, specified model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_DescribeModelVersions
     */
    toDescribeModelVersions() {
        this.to('frauddetector:DescribeModelVersions');
        return this;
    }
    /**
     * Gets a particular detector version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectorVersion
     */
    toGetDetectorVersion() {
        this.to('frauddetector:GetDetectorVersion');
        return this;
    }
    /**
     * Gets all detectors or a single detector if a detectorId is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetDetectorsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetDetectors
     */
    toGetDetectors() {
        this.to('frauddetector:GetDetectors');
        return this;
    }
    /**
     * Gets all entity types or a specific entity type if a name is specified. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEntityTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEntityTypes
     */
    toGetEntityTypes() {
        this.to('frauddetector:GetEntityTypes');
        return this;
    }
    /**
     * Evaluates an event against a detector version. If a version ID is not provided, the detector’s (ACTIVE) version is used.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventPrediction
     */
    toGetEventPrediction() {
        this.to('frauddetector:GetEventPrediction');
        return this;
    }
    /**
     * Gets all event types or a specific event type if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetEventTypesResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetEventTypes
     */
    toGetEventTypes() {
        this.to('frauddetector:GetEventTypes');
        return this;
    }
    /**
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 10 records per page. If you provide a maxResults, the value must be between 5 and 10. To get the next page results, provide the pagination token from the GetExternalModelsResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetExternalModels
     */
    toGetExternalModels() {
        this.to('frauddetector:GetExternalModels');
        return this;
    }
    /**
     * Gets the encryption key if a Key Management Service (KMS) customer master key (CMK) has been specified to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetKMSEncryptionKey
     */
    toGetKMSEncryptionKey() {
        this.to('frauddetector:GetKMSEncryptionKey');
        return this;
    }
    /**
     * Gets all labels or a specific label if name is provided. This is a paginated API. If you provide a null maxResults, this action retrieves a maximum of 50 records per page. If you provide a maxResults, the value must be between 10 and 50. To get the next page results, provide the pagination token from the GetGetLabelsResponse as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetLabels
     */
    toGetLabels() {
        this.to('frauddetector:GetLabels');
        return this;
    }
    /**
     * Gets the details of the specified model version.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModelVersion
     */
    toGetModelVersion() {
        this.to('frauddetector:GetModelVersion');
        return this;
    }
    /**
     * Gets one or more models. Gets all models for the AWS account if no model type and no model id provided. Gets all models for the AWS account and model type, if the model type is specified but model id is not provided. Gets a specific model if (model type, model id) tuple is specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetModels
     */
    toGetModels() {
        this.to('frauddetector:GetModels');
        return this;
    }
    /**
     * Gets one or more outcomes. This is a paginated API. If you provide a null maxResults, this actions retrieves a maximum of 100 records per page. If you provide a maxResults, the value must be between 50 and 100. To get the next page results, provide the pagination token from the GetOutcomesResult as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetOutcomes
     */
    toGetOutcomes() {
        this.to('frauddetector:GetOutcomes');
        return this;
    }
    /**
     * Get all rules for a detector (paginated) if ruleId and ruleVersion are not specified. Gets all rules for the detector and the ruleId if present (paginated). Gets a specific rule if both the ruleId and the ruleVersion are specified.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetRules.html
     */
    toGetRules() {
        this.to('frauddetector:GetRules');
        return this;
    }
    /**
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null maxSizePerPage results in retrieving maximum of 100 records per page. If you provide maxSizePerPage the value must be between 50 and 100. To get the next page result, a provide a pagination token from GetVariablesResult as part of your request. Null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_GetVariables
     */
    toGetVariables() {
        this.to('frauddetector:GetVariables');
        return this;
    }
    /**
     * Lists all tags associated with the resource. This is a paginated API. To get the next page results, provide the pagination token from the response as part of your request. A null pagination token fetches the records from the beginning.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('frauddetector:ListTagsForResource');
        return this;
    }
    /**
     * Creates or updates a detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutDetector.html
     */
    toPutDetector() {
        this.to('frauddetector:PutDetector');
        return this;
    }
    /**
     * Creates or updates an entity type. An entity represents who is performing the event. As part of a fraud prediction, you pass the entity ID to indicate the specific entity who performed the event. An entity type classifies the entity. Example classifications include customer, merchant, or account.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEntityType.html
     */
    toPutEntityType() {
        this.to('frauddetector:PutEntityType');
        return this;
    }
    /**
     * Creates or updates an event type. An event is a business activity that is evaluated for fraud risk. With Amazon Fraud Detector, you generate fraud predictions for events. An event type defines the structure for an event sent to Amazon Fraud Detector. This includes the variables sent as part of the event, the entity performing the event (such as a customer), and the labels that classify the event. Example event types include online payment transactions, account registrations, and authentications.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutEventType.html
     */
    toPutEventType() {
        this.to('frauddetector:PutEventType');
        return this;
    }
    /**
     * Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration of the model endpoint, including the IAM role and/or the mapped variables.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutExternalModel.html
     */
    toPutExternalModel() {
        this.to('frauddetector:PutExternalModel');
        return this;
    }
    /**
     * Specifies the Key Management Service (KMS) customer master key (CMK) to be used to encrypt content in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutKMSEncryptionKey.html
     */
    toPutKMSEncryptionKey() {
        this.to('frauddetector:PutKMSEncryptionKey');
        return this;
    }
    /**
     * Creates or updates label. A label classifies an event as fraudulent or legitimate. Labels are associated with event types and used to train supervised machine learning models in Amazon Fraud Detector.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutLabel.html
     */
    toPutLabel() {
        this.to('frauddetector:PutLabel');
        return this;
    }
    /**
     * Creates or updates an outcome.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_PutOutcome.html
     */
    toPutOutcome() {
        this.to('frauddetector:PutOutcome');
        return this;
    }
    /**
     * Assigns tags to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_TagResource.html
     */
    toTagResource() {
        this.to('frauddetector:TagResource');
        return this;
    }
    /**
     * Removes tags from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UntagResource.html
     */
    toUntagResource() {
        this.to('frauddetector:UntagResource');
        return this;
    }
    /**
     * Updates a detector version. The detector version attributes that you can update include models, external model endpoints, rules, rule execution mode, and description. You can only update a DRAFT detector version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersion
     */
    toUpdateDetectorVersion() {
        this.to('frauddetector:UpdateDetectorVersion');
        return this;
    }
    /**
     * Updates the detector version's description. You can update the metadata for any detector version (DRAFT, ACTIVE, or INACTIVE).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionMetadata
     */
    toUpdateDetectorVersionMetadata() {
        this.to('frauddetector:UpdateDetectorVersionMetadata');
        return this;
    }
    /**
     * Updates the detector version’s status. You can perform the following promotions or demotions using UpdateDetectorVersionStatus: DRAFT to ACTIVE, ACTIVE to INACTIVE, and INACTIVE to ACTIVE.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateDetectorVersionStatus
     */
    toUpdateDetectorVersionStatus() {
        this.to('frauddetector:UpdateDetectorVersionStatus');
        return this;
    }
    /**
     * Updates a model. You can update the description attribute using this action.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModel
     */
    toUpdateModel() {
        this.to('frauddetector:UpdateModel');
        return this;
    }
    /**
     * Updates a model version. Updating a model version retrains an existing model version using updated training data and produces a new minor version of the model. You can update the training data set location and data access role attributes using this action. This action creates and trains a new minor version of the model, for example version 1.01, 1.02, 1.03.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersion
     */
    toUpdateModelVersion() {
        this.to('frauddetector:UpdateModelVersion');
        return this;
    }
    /**
     * Updates the status of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateModelVersionStatus
     */
    toUpdateModelVersionStatus() {
        this.to('frauddetector:UpdateModelVersionStatus');
        return this;
    }
    /**
     * Updates a rule's metadata. The description attribute can be updated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleMetadata.html
     */
    toUpdateRuleMetadata() {
        this.to('frauddetector:UpdateRuleMetadata');
        return this;
    }
    /**
     * Updates a rule version resulting in a new rule version. Updates a rule version resulting in a new rule version (version 1, 2, 3 ...).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateRuleVersion.html
     */
    toUpdateRuleVersion() {
        this.to('frauddetector:UpdateRuleVersion');
        return this;
    }
    /**
     * Updates a variable.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/frauddetector/latest/api/API_UpdateVariable.html
     */
    toUpdateVariable() {
        this.to('frauddetector:UpdateVariable');
        return this;
    }
    /**
     * Adds a resource of type detector to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetector(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type detector-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDetectorVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:detector-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type entity-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEntityType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:entity-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type external-model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onExternalModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:external-model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type event-type to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onEventType(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:event-type/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type label to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onLabel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:label/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModel(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type model-version to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onModelVersion(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:model-version/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type outcome to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onOutcome(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:outcome/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rule to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRule(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:rule/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type variable to the statement
     *
     * @param resourcePath - Identifier for the resourcePath.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVariable(resourcePath, account, region, partition) {
        var arn = 'arn:${Partition}:frauddetector:${Region}:${Account}:variable/${ResourcePath}';
        arn = arn.replace('${ResourcePath}', resourcePath);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Frauddetector = Frauddetector;
//# sourceMappingURL=data:application/json;base64,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