"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Firehose = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Firehose extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [firehose](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisfirehose.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'firehose';
        this.accessLevelList = {
            "Write": [
                "CreateDeliveryStream",
                "DeleteDeliveryStream",
                "PutRecord",
                "PutRecordBatch",
                "StartDeliveryStreamEncryption",
                "StopDeliveryStreamEncryption",
                "TagDeliveryStream",
                "UntagDeliveryStream",
                "UpdateDestination"
            ],
            "List": [
                "DescribeDeliveryStream",
                "ListDeliveryStreams",
                "ListTagsForDeliveryStream"
            ]
        };
    }
    /**
     * Creates a delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_CreateDeliveryStream.html
     */
    toCreateDeliveryStream() {
        this.to('firehose:CreateDeliveryStream');
        return this;
    }
    /**
     * Deletes a delivery stream and its data.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DeleteDeliveryStream.html
     */
    toDeleteDeliveryStream() {
        this.to('firehose:DeleteDeliveryStream');
        return this;
    }
    /**
     * Describes the specified delivery stream and gets the status.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_DescribeDeliveryStream.html
     */
    toDescribeDeliveryStream() {
        this.to('firehose:DescribeDeliveryStream');
        return this;
    }
    /**
     * Lists your delivery streams.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListDeliveryStreams.html
     */
    toListDeliveryStreams() {
        this.to('firehose:ListDeliveryStreams');
        return this;
    }
    /**
     * Lists the tags for the specified delivery stream.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_ListTagsForDeliveryStream.html
     */
    toListTagsForDeliveryStream() {
        this.to('firehose:ListTagsForDeliveryStream');
        return this;
    }
    /**
     * Writes a single data record into an Amazon Kinesis Firehose delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecord.html
     */
    toPutRecord() {
        this.to('firehose:PutRecord');
        return this;
    }
    /**
     * Writes multiple data records into a delivery stream in a single call, which can achieve higher throughput per producer than when writing single records.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_PutRecordBatch.html
     */
    toPutRecordBatch() {
        this.to('firehose:PutRecordBatch');
        return this;
    }
    /**
     * Enables server-side encryption (SSE) for the delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StartDeliveryStreamEncryption.html
     */
    toStartDeliveryStreamEncryption() {
        this.to('firehose:StartDeliveryStreamEncryption');
        return this;
    }
    /**
     * Disables the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_StopDeliveryStreamEncryption.html
     */
    toStopDeliveryStreamEncryption() {
        this.to('firehose:StopDeliveryStreamEncryption');
        return this;
    }
    /**
     * Adds or updates tags for the specified delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_TagDeliveryStream.html
     */
    toTagDeliveryStream() {
        this.to('firehose:TagDeliveryStream');
        return this;
    }
    /**
     * Removes tags from the specified delivery stream.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UntagDeliveryStream.html
     */
    toUntagDeliveryStream() {
        this.to('firehose:UntagDeliveryStream');
        return this;
    }
    /**
     * Updates the specified destination of the specified delivery stream.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/firehose/latest/APIReference/API_UpdateDestination.html
     */
    toUpdateDestination() {
        this.to('firehose:UpdateDestination');
        return this;
    }
    /**
     * Adds a resource of type deliverystream to the statement
     *
     * https://docs.aws.amazon.com/firehose/latest/dev/basic-create.html
     *
     * @param deliveryStreamName - Identifier for the deliveryStreamName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDeliverystream(deliveryStreamName, account, region, partition) {
        var arn = 'arn:${Partition}:firehose:${Region}:${Account}:deliverystream/${DeliveryStreamName}';
        arn = arn.replace('${DeliveryStreamName}', deliveryStreamName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Firehose = Firehose;
//# sourceMappingURL=data:application/json;base64,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