"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Kinesisvideo = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisvideostreams.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Kinesisvideo extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [kinesisvideo](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonkinesisvideostreams.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'kinesisvideo';
        this.accessLevelList = {
            "Write": [
                "ConnectAsMaster",
                "ConnectAsViewer",
                "CreateSignalingChannel",
                "CreateStream",
                "DeleteSignalingChannel",
                "DeleteStream",
                "PutMedia",
                "SendAlexaOfferToMaster",
                "UpdateDataRetention",
                "UpdateSignalingChannel",
                "UpdateStream"
            ],
            "List": [
                "DescribeSignalingChannel",
                "DescribeStream",
                "ListFragments",
                "ListSignalingChannels",
                "ListStreams"
            ],
            "Read": [
                "GetClip",
                "GetDASHStreamingSessionURL",
                "GetDataEndpoint",
                "GetHLSStreamingSessionURL",
                "GetIceServerConfig",
                "GetMedia",
                "GetMediaForFragmentList",
                "GetSignalingChannelEndpoint",
                "ListTagsForResource",
                "ListTagsForStream"
            ],
            "Tagging": [
                "TagResource",
                "TagStream",
                "UntagResource",
                "UntagStream"
            ]
        };
    }
    /**
     * Grants permission to connect as a master to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsMaster.html
     */
    toConnectAsMaster() {
        this.to('kinesisvideo:ConnectAsMaster');
        return this;
    }
    /**
     * Grants permission to connect as a viewer to the signaling channel specified by the endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ConnectAsViewer.html
     */
    toConnectAsViewer() {
        this.to('kinesisvideo:ConnectAsViewer');
        return this;
    }
    /**
     * Grants permission to create a signaling channel
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateSignalingChannel.html
     */
    toCreateSignalingChannel() {
        this.to('kinesisvideo:CreateSignalingChannel');
        return this;
    }
    /**
     * Grants permission to create a Kinesis video stream
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_CreateStream.html
     */
    toCreateStream() {
        this.to('kinesisvideo:CreateStream');
        return this;
    }
    /**
     * Grants permission to delete an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteSignalingChannel.html
     */
    toDeleteSignalingChannel() {
        this.to('kinesisvideo:DeleteSignalingChannel');
        return this;
    }
    /**
     * Grants permission to delete an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DeleteStream.html
     */
    toDeleteStream() {
        this.to('kinesisvideo:DeleteStream');
        return this;
    }
    /**
     * Grants permission to describe the specified signaling channel
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeSignalingChannel.html
     */
    toDescribeSignalingChannel() {
        this.to('kinesisvideo:DescribeSignalingChannel');
        return this;
    }
    /**
     * Grants permission to describe the specified Kinesis video stream
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_DescribeStream.html
     */
    toDescribeStream() {
        this.to('kinesisvideo:DescribeStream');
        return this;
    }
    /**
     * Grants permission to get a media clip from a video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetClip.html
     */
    toGetClip() {
        this.to('kinesisvideo:GetClip');
        return this;
    }
    /**
     * Grants permission to create a URL for MPEG-DASH video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDASHStreamingSessionURL.html
     */
    toGetDASHStreamingSessionURL() {
        this.to('kinesisvideo:GetDASHStreamingSessionURL');
        return this;
    }
    /**
     * Grants permission to get an endpoint for a specified stream for either reading or writing media data to Kinesis Video Streams
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetDataEndpoint.html
     */
    toGetDataEndpoint() {
        this.to('kinesisvideo:GetDataEndpoint');
        return this;
    }
    /**
     * Grants permission to create a URL for HLS video streaming
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetHLSStreamingSessionURL.html
     */
    toGetHLSStreamingSessionURL() {
        this.to('kinesisvideo:GetHLSStreamingSessionURL');
        return this;
    }
    /**
     * Grants permission to get the ICE server configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetIceServerConfig.html
     */
    toGetIceServerConfig() {
        this.to('kinesisvideo:GetIceServerConfig');
        return this;
    }
    /**
     * Grants permission to return media content of a Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMedia.html
     */
    toGetMedia() {
        this.to('kinesisvideo:GetMedia');
        return this;
    }
    /**
     * Grants permission to read and return media data only from persisted storage
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetMediaForFragmentList.html
     */
    toGetMediaForFragmentList() {
        this.to('kinesisvideo:GetMediaForFragmentList');
        return this;
    }
    /**
     * Grants permission to get endpoints for a specified combination of protocol and role for a signaling channel
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_GetSignalingChannelEndpoint.html
     */
    toGetSignalingChannelEndpoint() {
        this.to('kinesisvideo:GetSignalingChannelEndpoint');
        return this;
    }
    /**
     * Grants permission to list the fragments from archival storage based on the pagination token or selector type with range specified
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListFragments.html
     */
    toListFragments() {
        this.to('kinesisvideo:ListFragments');
        return this;
    }
    /**
     * Grants permission to list your signaling channels
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListSignalingChannels.html
     */
    toListSignalingChannels() {
        this.to('kinesisvideo:ListSignalingChannels');
        return this;
    }
    /**
     * Grants permission to list your Kinesis video streams
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListStreams.html
     */
    toListStreams() {
        this.to('kinesisvideo:ListStreams');
        return this;
    }
    /**
     * Grants permission to fetch the tags associated with your resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('kinesisvideo:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to fetch the tags associated with Kinesis video stream
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_ListTagsForStream.html
     */
    toListTagsForStream() {
        this.to('kinesisvideo:ListTagsForStream');
        return this;
    }
    /**
     * Grants permission to send media data to a Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_PutMedia.html
     */
    toPutMedia() {
        this.to('kinesisvideo:PutMedia');
        return this;
    }
    /**
     * Grants permission to send the Alexa SDP offer to the master
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_SendAlexaOfferToMaster.html
     */
    toSendAlexaOfferToMaster() {
        this.to('kinesisvideo:SendAlexaOfferToMaster');
        return this;
    }
    /**
     * Grants permission to attach set of tags to your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagResource.html
     */
    toTagResource() {
        this.to('kinesisvideo:TagResource');
        return this;
    }
    /**
     * Grants permission to attach set of tags to your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_TagStream.html
     */
    toTagStream() {
        this.to('kinesisvideo:TagStream');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from your resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagResource.html
     */
    toUntagResource() {
        this.to('kinesisvideo:UntagResource');
        return this;
    }
    /**
     * Grants permission to remove one or more tags from your Kinesis video streams
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UntagStream.html
     */
    toUntagStream() {
        this.to('kinesisvideo:UntagStream');
        return this;
    }
    /**
     * Grants permission to update the data retention period of your Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateDataRetention.html
     */
    toUpdateDataRetention() {
        this.to('kinesisvideo:UpdateDataRetention');
        return this;
    }
    /**
     * Grants permission to update an existing signaling channel
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateSignalingChannel.html
     */
    toUpdateSignalingChannel() {
        this.to('kinesisvideo:UpdateSignalingChannel');
        return this;
    }
    /**
     * Grants permission to update an existing Kinesis video stream
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/API_UpdateStream.html
     */
    toUpdateStream() {
        this.to('kinesisvideo:UpdateStream');
        return this;
    }
    /**
     * Adds a resource of type stream to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/how-it-works.html
     *
     * @param streamName - Identifier for the streamName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStream(streamName, creationTime, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisvideo:${Region}:${Account}:stream/${StreamName}/${CreationTime}';
        arn = arn.replace('${StreamName}', streamName);
        arn = arn.replace('${CreationTime}', creationTime);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type channel to the statement
     *
     * https://docs.aws.amazon.com/kinesisvideostreams/latest/dg/kinesisvideostreams-webrtc-dg/latest/devguide/kvswebrtc-how-it-works.html
     *
     * @param channelName - Identifier for the channelName.
     * @param creationTime - Identifier for the creationTime.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onChannel(channelName, creationTime, account, region, partition) {
        var arn = 'arn:${Partition}:kinesisvideo:${Region}:${Account}:channel/${ChannelName}/${CreationTime}';
        arn = arn.replace('${ChannelName}', channelName);
        arn = arn.replace('${CreationTime}', creationTime);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Kinesisvideo = Kinesisvideo;
//# sourceMappingURL=data:application/json;base64,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