"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mq = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [mq](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmq.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Mq extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [mq](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonmq.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'mq';
        this.accessLevelList = {
            "Write": [
                "CreateBroker",
                "CreateConfiguration",
                "CreateTags",
                "CreateUser",
                "DeleteBroker",
                "DeleteTags",
                "DeleteUser",
                "RebootBroker",
                "UpdateBroker",
                "UpdateConfiguration",
                "UpdateUser"
            ],
            "Read": [
                "DescribeBroker",
                "DescribeBrokerEngineTypes",
                "DescribeBrokerInstanceOptions",
                "DescribeConfiguration",
                "DescribeConfigurationRevision",
                "DescribeUser"
            ],
            "List": [
                "ListBrokers",
                "ListConfigurationRevisions",
                "ListConfigurations",
                "ListTags",
                "ListUsers"
            ]
        };
    }
    /**
     * Grants permission to create a broker.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-post
     */
    toCreateBroker() {
        this.to('mq:CreateBroker');
        return this;
    }
    /**
     * Grants permission to create a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the engine type and engine version).
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-post
     */
    toCreateConfiguration() {
        this.to('mq:CreateConfiguration');
        return this;
    }
    /**
     * Grants permission to create tags.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-post
     */
    toCreateTags() {
        this.to('mq:CreateTags');
        return this;
    }
    /**
     * Grants permission to create an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-post
     */
    toCreateUser() {
        this.to('mq:CreateUser');
        return this;
    }
    /**
     * Grants permission to delete a broker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-delete
     */
    toDeleteBroker() {
        this.to('mq:DeleteBroker');
        return this;
    }
    /**
     * Grants permission to delete tags.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-delete
     */
    toDeleteTags() {
        this.to('mq:DeleteTags');
        return this;
    }
    /**
     * Grants permission to delete an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-delete
     */
    toDeleteUser() {
        this.to('mq:DeleteUser');
        return this;
    }
    /**
     * Grants permission to return information about the specified broker.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toDescribeBroker() {
        this.to('mq:DescribeBroker');
        return this;
    }
    /**
     * Grants permission to return information about broker engines.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-engine-types.html#broker-engine-types-http-methods
     */
    toDescribeBrokerEngineTypes() {
        this.to('mq:DescribeBrokerEngineTypes');
        return this;
    }
    /**
     * Grants permission to return information about the broker instance options
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/broker-instance-options.html#broker-engine-types-http-methods
     */
    toDescribeBrokerInstanceOptions() {
        this.to('mq:DescribeBrokerInstanceOptions');
        return this;
    }
    /**
     * Grants permission to return information about the specified configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-get
     */
    toDescribeConfiguration() {
        this.to('mq:DescribeConfiguration');
        return this;
    }
    /**
     * Grants permission to return the specified configuration revision for the specified configuration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration-revision.html#rest-api-configuration-revision-methods-get
     */
    toDescribeConfigurationRevision() {
        this.to('mq:DescribeConfigurationRevision');
        return this;
    }
    /**
     * Grants permission to return information about an ActiveMQ user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-get
     */
    toDescribeUser() {
        this.to('mq:DescribeUser');
        return this;
    }
    /**
     * Grants permission to return a list of all brokers.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-brokers.html#rest-api-brokers-methods-get
     */
    toListBrokers() {
        this.to('mq:ListBrokers');
        return this;
    }
    /**
     * Grants permission to return a list of all existing revisions for the specified configuration.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-revisions.html#rest-api-revisions-methods-get
     */
    toListConfigurationRevisions() {
        this.to('mq:ListConfigurationRevisions');
        return this;
    }
    /**
     * Grants permission to return a list of all configurations.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configurations.html#rest-api-configurations-methods-get
     */
    toListConfigurations() {
        this.to('mq:ListConfigurations');
        return this;
    }
    /**
     * Grants permission to return a list of tags.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-tags.html#rest-api-tags-methods-get
     */
    toListTags() {
        this.to('mq:ListTags');
        return this;
    }
    /**
     * Grants permission to return a list of all ActiveMQ users.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-users.html#rest-api-users-methods-get
     */
    toListUsers() {
        this.to('mq:ListUsers');
        return this;
    }
    /**
     * Grants permission to reboot a broker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-restart.html#rest-api-reboot-methods-post
     */
    toRebootBroker() {
        this.to('mq:RebootBroker');
        return this;
    }
    /**
     * Grants permission to add a pending configuration change to a broker.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-broker.html#rest-api-broker-methods-get
     */
    toUpdateBroker() {
        this.to('mq:UpdateBroker');
        return this;
    }
    /**
     * Grants permission to update the specified configuration.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-configuration.html#rest-api-configuration-methods-put
     */
    toUpdateConfiguration() {
        this.to('mq:UpdateConfiguration');
        return this;
    }
    /**
     * Grants permission to update the information for an ActiveMQ user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/amazon-mq/latest/api-reference/rest-api-username.html#rest-api-username-methods-put
     */
    toUpdateUser() {
        this.to('mq:UpdateUser');
        return this;
    }
    /**
     * Adds a resource of type brokers to the statement
     *
     * @param brokerId - Identifier for the brokerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onBrokers(brokerId, account, region, partition) {
        var arn = 'arn:${Partition}:mq:${Region}:${Account}:broker:${Broker-id}';
        arn = arn.replace('${Broker-id}', brokerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type configurations to the statement
     *
     * @param configurationId - Identifier for the configurationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onConfigurations(configurationId, account, region, partition) {
        var arn = 'arn:${Partition}:mq:${Region}:${Account}:configuration:${Configuration-id}';
        arn = arn.replace('${Configuration-id}', configurationId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Mq = Mq;
//# sourceMappingURL=data:application/json;base64,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