"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rekognition = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Rekognition extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [rekognition](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonrekognition.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'rekognition';
        this.accessLevelList = {
            "Read": [
                "CompareFaces",
                "DescribeCollection",
                "DescribeProjectVersions",
                "DescribeProjects",
                "DescribeStreamProcessor",
                "DetectCustomLabels",
                "DetectFaces",
                "DetectLabels",
                "DetectModerationLabels",
                "DetectText",
                "GetCelebrityInfo",
                "GetCelebrityRecognition",
                "GetContentModeration",
                "GetFaceDetection",
                "GetFaceSearch",
                "GetLabelDetection",
                "GetPersonTracking",
                "GetSegmentDetection",
                "GetTextDetection",
                "ListCollections",
                "ListFaces",
                "RecognizeCelebrities",
                "SearchFaces",
                "SearchFacesByImage"
            ],
            "Write": [
                "CreateCollection",
                "CreateProject",
                "CreateProjectVersion",
                "CreateStreamProcessor",
                "DeleteCollection",
                "DeleteFaces",
                "DeleteProject",
                "DeleteProjectVersion",
                "DeleteStreamProcessor",
                "IndexFaces",
                "StartCelebrityRecognition",
                "StartContentModeration",
                "StartFaceDetection",
                "StartFaceSearch",
                "StartLabelDetection",
                "StartPersonTracking",
                "StartProjectVersion",
                "StartSegmentDetection",
                "StartStreamProcessor",
                "StartTextDetection",
                "StopProjectVersion",
                "StopStreamProcessor"
            ],
            "List": [
                "ListStreamProcessors"
            ]
        };
    }
    /**
     * Compares a face in source input image with each face detected in the target input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CompareFaces.html
     */
    toCompareFaces() {
        this.to('rekognition:CompareFaces');
        return this;
    }
    /**
     * Creates a collection in an AWS region. You can then add faces to the collection using the IndexFaces API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateCollection.html
     */
    toCreateCollection() {
        this.to('rekognition:CreateCollection');
        return this;
    }
    /**
     * Creates a new Amazon Rekognition Custom Labels project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProject.html
     */
    toCreateProject() {
        this.to('rekognition:CreateProject');
        return this;
    }
    /**
     * Creates a new version of a model and begins training.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateProjectVersion.html
     */
    toCreateProjectVersion() {
        this.to('rekognition:CreateProjectVersion');
        return this;
    }
    /**
     * Creates an Amazon Rekognition stream processor that you can use to detect and recognize faces in a streaming video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_CreateStreamProcessor.html
     */
    toCreateStreamProcessor() {
        this.to('rekognition:CreateStreamProcessor');
        return this;
    }
    /**
     * Deletes the specified collection. Note that this operation removes all faces in the collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteCollection.html
     */
    toDeleteCollection() {
        this.to('rekognition:DeleteCollection');
        return this;
    }
    /**
     * Deletes faces from a collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteFaces.html
     */
    toDeleteFaces() {
        this.to('rekognition:DeleteFaces');
        return this;
    }
    /**
     * Deletes a project.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProject.html
     */
    toDeleteProject() {
        this.to('rekognition:DeleteProject');
        return this;
    }
    /**
     * Deletes a model.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteProjectVersion.html
     */
    toDeleteProjectVersion() {
        this.to('rekognition:DeleteProjectVersion');
        return this;
    }
    /**
     * Deletes the stream processor identified by Name.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DeleteStreamProcessor.html
     */
    toDeleteStreamProcessor() {
        this.to('rekognition:DeleteStreamProcessor');
        return this;
    }
    /**
     * Describes the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeCollection.html
     */
    toDescribeCollection() {
        this.to('rekognition:DescribeCollection');
        return this;
    }
    /**
     * Lists and describes the model versions in an Amazon Rekognition Custom Labels project.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjectVersions.html
     */
    toDescribeProjectVersions() {
        this.to('rekognition:DescribeProjectVersions');
        return this;
    }
    /**
     * Lists and gets information about your Amazon Rekognition Custom Labels projects.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeProjects.html
     */
    toDescribeProjects() {
        this.to('rekognition:DescribeProjects');
        return this;
    }
    /**
     * Provides information about a stream processor created by CreateStreamProcessor.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DescribeStreamProcessorh.html
     */
    toDescribeStreamProcessor() {
        this.to('rekognition:DescribeStreamProcessor');
        return this;
    }
    /**
     * Detects custom labels in a supplied image by using an Amazon Rekognition Custom Labels model version.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectCustomLabels.html
     */
    toDetectCustomLabels() {
        this.to('rekognition:DetectCustomLabels');
        return this;
    }
    /**
     * Detects human faces within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectFaces.html
     */
    toDetectFaces() {
        this.to('rekognition:DetectFaces');
        return this;
    }
    /**
     * Detects instances of real-world labels within an image (JPEG or PNG) provided as input.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectLabels.html
     */
    toDetectLabels() {
        this.to('rekognition:DetectLabels');
        return this;
    }
    /**
     * Detects moderation labels within input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectModerationLabels.html
     */
    toDetectModerationLabels() {
        this.to('rekognition:DetectModerationLabels');
        return this;
    }
    /**
     * Detects text in the input image and converts it into machine-readable text.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_DetectText.html
     */
    toDetectText() {
        this.to('rekognition:DetectText');
        return this;
    }
    /**
     * Gets the name and additional information about a celebrity based on his or her Rekognition ID.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityInfo.html
     */
    toGetCelebrityInfo() {
        this.to('rekognition:GetCelebrityInfo');
        return this;
    }
    /**
     * Gets the celebrity recognition results for a Rekognition Video analysis started by StartCelebrityRecognition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetCelebrityRecognition.html
     */
    toGetCelebrityRecognition() {
        this.to('rekognition:GetCelebrityRecognition');
        return this;
    }
    /**
     * Gets the content moderation analysis results for a Rekognition Video analysis started by StartContentModeration.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetContentModeration.html
     */
    toGetContentModeration() {
        this.to('rekognition:GetContentModeration');
        return this;
    }
    /**
     * Gets face detection results for a Rekognition Video analysis started by StartFaceDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceDetection.html
     */
    toGetFaceDetection() {
        this.to('rekognition:GetFaceDetection');
        return this;
    }
    /**
     * Gets the face search results for Rekognition Video face search started by StartFaceSearch.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetFaceSearch.html
     */
    toGetFaceSearch() {
        this.to('rekognition:GetFaceSearch');
        return this;
    }
    /**
     * Gets the label detection results of a Rekognition Video analysis started by StartLabelDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetLabelDetection.html
     */
    toGetLabelDetection() {
        this.to('rekognition:GetLabelDetection');
        return this;
    }
    /**
     * Gets information about people detected within a video.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetPersonTracking.html
     */
    toGetPersonTracking() {
        this.to('rekognition:GetPersonTracking');
        return this;
    }
    /**
     * Gets segment detection results for a Rekognition Video analysis started by StartSegmentDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetSegmentDetection.html
     */
    toGetSegmentDetection() {
        this.to('rekognition:GetSegmentDetection');
        return this;
    }
    /**
     * Gets text detection results for a Rekognition Video analysis started by StartTextDetection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_GetTextDetection.html
     */
    toGetTextDetection() {
        this.to('rekognition:GetTextDetection');
        return this;
    }
    /**
     * Detects faces in the input image and adds them to the specified collection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_IndexFaces.html
     */
    toIndexFaces() {
        this.to('rekognition:IndexFaces');
        return this;
    }
    /**
     * Returns a list of collection IDs in your account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListCollections.html
     */
    toListCollections() {
        this.to('rekognition:ListCollections');
        return this;
    }
    /**
     * Returns metadata for faces in the specified collection.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListFaces.html
     */
    toListFaces() {
        this.to('rekognition:ListFaces');
        return this;
    }
    /**
     * Gets a list of stream processors that you have created with CreateStreamProcessor.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_ListStreamProcessors.html
     */
    toListStreamProcessors() {
        this.to('rekognition:ListStreamProcessors');
        return this;
    }
    /**
     * Returns an array of celebrities recognized in the input image.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_RecognizeCelebrities.html
     */
    toRecognizeCelebrities() {
        this.to('rekognition:RecognizeCelebrities');
        return this;
    }
    /**
     * For a given input face ID, searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFaces.html
     */
    toSearchFaces() {
        this.to('rekognition:SearchFaces');
        return this;
    }
    /**
     * For a given input image, first detects the largest face in the image, and then searches the specified collection for matching faces.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_SearchFacesByImage.html
     */
    toSearchFacesByImage() {
        this.to('rekognition:SearchFacesByImage');
        return this;
    }
    /**
     * Starts asynchronous recognition of celebrities in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartCelebrityRecognition.html
     */
    toStartCelebrityRecognition() {
        this.to('rekognition:StartCelebrityRecognition');
        return this;
    }
    /**
     * Starts asynchronous detection of explicit or suggestive adult content in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartContentModeration.html
     */
    toStartContentModeration() {
        this.to('rekognition:StartContentModeration');
        return this;
    }
    /**
     * Starts asynchronous detection of faces in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceDetection.html
     */
    toStartFaceDetection() {
        this.to('rekognition:StartFaceDetection');
        return this;
    }
    /**
     * Starts the asynchronous search for faces in a collection that match the faces of persons detected in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartFaceSearch.html
     */
    toStartFaceSearch() {
        this.to('rekognition:StartFaceSearch');
        return this;
    }
    /**
     * Starts asynchronous detection of labels in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartLabelDetection.html
     */
    toStartLabelDetection() {
        this.to('rekognition:StartLabelDetection');
        return this;
    }
    /**
     * Starts the asynchronous tracking of persons in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartPersonTracking.html
     */
    toStartPersonTracking() {
        this.to('rekognition:StartPersonTracking');
        return this;
    }
    /**
     * Starts the deployment of a model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartProjectVersion.html
     */
    toStartProjectVersion() {
        this.to('rekognition:StartProjectVersion');
        return this;
    }
    /**
     * Starts asynchronous detection of segments in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartSegmentDetection.html
     */
    toStartSegmentDetection() {
        this.to('rekognition:StartSegmentDetection');
        return this;
    }
    /**
     * Starts processing a stream processor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartStreamProcessor.html
     */
    toStartStreamProcessor() {
        this.to('rekognition:StartStreamProcessor');
        return this;
    }
    /**
     * Starts asynchronous detection of text in a video.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StartTextDetection.html
     */
    toStartTextDetection() {
        this.to('rekognition:StartTextDetection');
        return this;
    }
    /**
     * Stops a deployed model version.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopProjectVersion.html
     */
    toStopProjectVersion() {
        this.to('rekognition:StopProjectVersion');
        return this;
    }
    /**
     * Stops a running stream processor that was created by CreateStreamProcessor.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/API_StopStreamProcessor.html
     */
    toStopStreamProcessor() {
        this.to('rekognition:StopStreamProcessor');
        return this;
    }
    /**
     * Adds a resource of type collection to the statement
     *
     * https://docs.aws.amazon.com/rekognition/latest/dg/howitworks-collection.html
     *
     * @param collectionId - Identifier for the collectionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCollection(collectionId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:collection/${CollectionId}';
        arn = arn.replace('${CollectionId}', collectionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type streamprocessor to the statement
     *
     * @param streamprocessorId - Identifier for the streamprocessorId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onStreamprocessor(streamprocessorId, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:streamprocessor/${StreamprocessorId}';
        arn = arn.replace('${StreamprocessorId}', streamprocessorId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type project to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProject(projectName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type projectversion to the statement
     *
     * @param projectName - Identifier for the projectName.
     * @param versionName - Identifier for the versionName.
     * @param creationTimestamp - Identifier for the creationTimestamp.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onProjectversion(projectName, versionName, creationTimestamp, account, region, partition) {
        var arn = 'arn:${Partition}:rekognition:${Region}:${Account}:project/${ProjectName}/version/${VersionName}/${CreationTimestamp}';
        arn = arn.replace('${ProjectName}', projectName);
        arn = arn.replace('${VersionName}', versionName);
        arn = arn.replace('${CreationTimestamp}', creationTimestamp);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Rekognition = Rekognition;
//# sourceMappingURL=data:application/json;base64,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