"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class S3 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [s3](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazons3.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 's3';
        this.accessLevelList = {
            "Write": [
                "AbortMultipartUpload",
                "CreateAccessPoint",
                "CreateBucket",
                "CreateJob",
                "DeleteAccessPoint",
                "DeleteBucket",
                "DeleteBucketWebsite",
                "DeleteObject",
                "DeleteObjectVersion",
                "PutAccelerateConfiguration",
                "PutAnalyticsConfiguration",
                "PutBucketCORS",
                "PutBucketLogging",
                "PutBucketNotification",
                "PutBucketObjectLockConfiguration",
                "PutBucketRequestPayment",
                "PutBucketVersioning",
                "PutBucketWebsite",
                "PutEncryptionConfiguration",
                "PutInventoryConfiguration",
                "PutLifecycleConfiguration",
                "PutMetricsConfiguration",
                "PutObject",
                "PutObjectLegalHold",
                "PutObjectRetention",
                "PutReplicationConfiguration",
                "ReplicateDelete",
                "ReplicateObject",
                "RestoreObject",
                "UpdateJobPriority",
                "UpdateJobStatus"
            ],
            "Permissions management": [
                "BypassGovernanceRetention",
                "DeleteAccessPointPolicy",
                "DeleteBucketPolicy",
                "ObjectOwnerOverrideToBucketOwner",
                "PutAccessPointPolicy",
                "PutAccountPublicAccessBlock",
                "PutBucketAcl",
                "PutBucketPolicy",
                "PutBucketPublicAccessBlock",
                "PutObjectAcl",
                "PutObjectVersionAcl"
            ],
            "Tagging": [
                "DeleteJobTagging",
                "DeleteObjectTagging",
                "DeleteObjectVersionTagging",
                "PutBucketTagging",
                "PutJobTagging",
                "PutObjectTagging",
                "PutObjectVersionTagging",
                "ReplicateTags"
            ],
            "Read": [
                "DescribeJob",
                "GetAccelerateConfiguration",
                "GetAccessPoint",
                "GetAccessPointPolicy",
                "GetAccessPointPolicyStatus",
                "GetAccountPublicAccessBlock",
                "GetAnalyticsConfiguration",
                "GetBucketAcl",
                "GetBucketCORS",
                "GetBucketLocation",
                "GetBucketLogging",
                "GetBucketNotification",
                "GetBucketObjectLockConfiguration",
                "GetBucketPolicy",
                "GetBucketPolicyStatus",
                "GetBucketPublicAccessBlock",
                "GetBucketRequestPayment",
                "GetBucketTagging",
                "GetBucketVersioning",
                "GetBucketWebsite",
                "GetEncryptionConfiguration",
                "GetInventoryConfiguration",
                "GetJobTagging",
                "GetLifecycleConfiguration",
                "GetMetricsConfiguration",
                "GetObject",
                "GetObjectAcl",
                "GetObjectLegalHold",
                "GetObjectRetention",
                "GetObjectTagging",
                "GetObjectTorrent",
                "GetObjectVersion",
                "GetObjectVersionAcl",
                "GetObjectVersionForReplication",
                "GetObjectVersionTagging",
                "GetObjectVersionTorrent",
                "GetReplicationConfiguration",
                "ListAccessPoints",
                "ListBucketMultipartUploads",
                "ListBucketVersions",
                "ListJobs",
                "ListMultipartUploadParts"
            ],
            "List": [
                "ListAllMyBuckets",
                "ListBucket"
            ]
        };
    }
    /**
     * Grants permission to abort a multipart upload
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_AbortMultipartUpload.html
     */
    toAbortMultipartUpload() {
        this.to('s3:AbortMultipartUpload');
        return this;
    }
    /**
     * Grants permission to allow circumvention of governance-mode object retention settings
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-bypass
     */
    toBypassGovernanceRetention() {
        this.to('s3:BypassGovernanceRetention');
        return this;
    }
    /**
     * Grants permission to create a new access point
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateAccessPoint.html
     */
    toCreateAccessPoint() {
        this.to('s3:CreateAccessPoint');
        return this;
    }
    /**
     * Grants permission to create a new bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifLocationconstraint()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CreateBucket.html
     */
    toCreateBucket() {
        this.to('s3:CreateBucket');
        return this;
    }
    /**
     * Grants permission to create a new Amazon S3 Batch Operations job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifRequestJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_CreateJob.html
     */
    toCreateJob() {
        this.to('s3:CreateJob');
        return this;
    }
    /**
     * Grants permission to delete the access point named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPoint.html
     */
    toDeleteAccessPoint() {
        this.to('s3:DeleteAccessPoint');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointArn()
     * - .ifDataAccessPointAccount()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteAccessPointPolicy.html
     */
    toDeleteAccessPointPolicy() {
        this.to('s3:DeleteAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to delete the bucket named in the URI
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucket.html
     */
    toDeleteBucket() {
        this.to('s3:DeleteBucket');
        return this;
    }
    /**
     * Grants permission to delete the policy on a specified bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketPolicy.html
     */
    toDeleteBucketPolicy() {
        this.to('s3:DeleteBucketPolicy');
        return this;
    }
    /**
     * Grants permission to remove the website configuration for a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteBucketWebsite.html
     */
    toDeleteBucketWebsite() {
        this.to('s3:DeleteBucketWebsite');
        return this;
    }
    /**
     * Grants permission to remove tags from an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DeleteJobTagging.html
     */
    toDeleteJobTagging() {
        this.to('s3:DeleteJobTagging');
        return this;
    }
    /**
     * Grants permission to remove the null version of an object and insert a delete marker, which becomes the current version of the object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObject() {
        this.to('s3:DeleteObject');
        return this;
    }
    /**
     * Grants permission to use the tagging subresource to remove the entire tag set from the specified object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectTagging() {
        this.to('s3:DeleteObjectTagging');
        return this;
    }
    /**
     * Grants permission to remove a specific version of an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObject.html
     */
    toDeleteObjectVersion() {
        this.to('s3:DeleteObjectVersion');
        return this;
    }
    /**
     * Grants permission to remove the entire tag set for a specific version of the object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_DeleteObjectTagging.html
     */
    toDeleteObjectVersionTagging() {
        this.to('s3:DeleteObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to retrieve the configuration parameters and status for a batch operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_DescribeJob.html
     */
    toDescribeJob() {
        this.to('s3:DescribeJob');
        return this;
    }
    /**
     * Grants permission to uses the accelerate subresource to return the Transfer Acceleration state of a bucket, which is either Enabled or Suspended
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAccelerateConfiguration.html
     */
    toGetAccelerateConfiguration() {
        this.to('s3:GetAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to return configuration information about the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPoint.html
     */
    toGetAccessPoint() {
        this.to('s3:GetAccessPoint');
        return this;
    }
    /**
     * Grants permission to returns the access point policy associated with the specified access point
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicy.html
     */
    toGetAccessPointPolicy() {
        this.to('s3:GetAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to return the policy status for a specific access point policy
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetAccessPointPolicyStatus.html
     */
    toGetAccessPointPolicyStatus() {
        this.to('s3:GetAccessPointPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetPublicAccessBlock.html
     */
    toGetAccountPublicAccessBlock() {
        this.to('s3:GetAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to get an analytics configuration from an Amazon S3 bucket, identified by the analytics configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAnalyticsConfiguration.html
     */
    toGetAnalyticsConfiguration() {
        this.to('s3:GetAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to return the access control list (ACL) of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketAcl.html
     */
    toGetBucketAcl() {
        this.to('s3:GetBucketAcl');
        return this;
    }
    /**
     * Grants permission to return the CORS configuration information set for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketCors.html
     */
    toGetBucketCORS() {
        this.to('s3:GetBucketCORS');
        return this;
    }
    /**
     * Grants permission to return the Region that an Amazon S3 bucket resides in
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLocation.html
     */
    toGetBucketLocation() {
        this.to('s3:GetBucketLocation');
        return this;
    }
    /**
     * Grants permission to return the logging status of an Amazon S3 bucket and the permissions users have to view or modify that status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLogging.html
     */
    toGetBucketLogging() {
        this.to('s3:GetBucketLogging');
        return this;
    }
    /**
     * Grants permission to get the notification configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketNotification.html
     */
    toGetBucketNotification() {
        this.to('s3:GetBucketNotification');
        return this;
    }
    /**
     * Grants permission to get the Object Lock configuration of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLockConfiguration.html
     */
    toGetBucketObjectLockConfiguration() {
        this.to('s3:GetBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to return the policy of the specified bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicy.html
     */
    toGetBucketPolicy() {
        this.to('s3:GetBucketPolicy');
        return this;
    }
    /**
     * Grants permission to retrieve the policy status for a specific Amazon S3 bucket, which indicates whether the bucket is public
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketPolicyStatus.html
     */
    toGetBucketPolicyStatus() {
        this.to('s3:GetBucketPolicyStatus');
        return this;
    }
    /**
     * Grants permission to retrieve the PublicAccessBlock configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetPublicAccessBlock.html
     */
    toGetBucketPublicAccessBlock() {
        this.to('s3:GetBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to return the request payment configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketRequestPayment.html
     */
    toGetBucketRequestPayment() {
        this.to('s3:GetBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to return the tag set associated with an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketTagging.html
     */
    toGetBucketTagging() {
        this.to('s3:GetBucketTagging');
        return this;
    }
    /**
     * Grants permission to return the versioning state of an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketVersioning.html
     */
    toGetBucketVersioning() {
        this.to('s3:GetBucketVersioning');
        return this;
    }
    /**
     * Grants permission to return the website configuration for an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketWebsite.html
     */
    toGetBucketWebsite() {
        this.to('s3:GetBucketWebsite');
        return this;
    }
    /**
     * Grants permission to return the default encryption configuration an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketEncryption.html
     */
    toGetEncryptionConfiguration() {
        this.to('s3:GetEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to return an inventory configuration from an Amazon S3 bucket, identified by the inventory configuration ID
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketInventoryConfiguration.html
     */
    toGetInventoryConfiguration() {
        this.to('s3:GetInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to return the tag set of an existing Amazon S3 Batch Operations job
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_GetJobTagging.html
     */
    toGetJobTagging() {
        this.to('s3:GetJobTagging');
        return this;
    }
    /**
     * Grants permission to return the lifecycle configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketLifecycleConfiguration.html
     */
    toGetLifecycleConfiguration() {
        this.to('s3:GetLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to get a metrics configuration from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketMetricsConfiguration.html
     */
    toGetMetricsConfiguration() {
        this.to('s3:GetMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve objects from Amazon S3
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObject() {
        this.to('s3:GetObject');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectAcl() {
        this.to('s3:GetObjectAcl');
        return this;
    }
    /**
     * Grants permission to get an object's current Legal Hold status
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectLegalHold.html
     */
    toGetObjectLegalHold() {
        this.to('s3:GetObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to retrieve the retention settings for an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectRetention.html
     */
    toGetObjectRetention() {
        this.to('s3:GetObjectRetention');
        return this;
    }
    /**
     * Grants permission to return the tag set of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTagging.html
     */
    toGetObjectTagging() {
        this.to('s3:GetObjectTagging');
        return this;
    }
    /**
     * Grants permission to return torrent files from an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectTorrent() {
        this.to('s3:GetObjectTorrent');
        return this;
    }
    /**
     * Grants permission to retrieve a specific version of an object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObject.html
     */
    toGetObjectVersion() {
        this.to('s3:GetObjectVersion');
        return this;
    }
    /**
     * Grants permission to return the access control list (ACL) of a specific object version
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectAcl.html
     */
    toGetObjectVersionAcl() {
        this.to('s3:GetObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to replicate both unencrypted objects and objects encrypted with SSE-S3 or SSE-KMS
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     */
    toGetObjectVersionForReplication() {
        this.to('s3:GetObjectVersionForReplication');
        return this;
    }
    /**
     * Grants permission to return the tag set for a specific version of the object
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     */
    toGetObjectVersionTagging() {
        this.to('s3:GetObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to get Torrent files about a different version using the versionId subresource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetObjectTorrent.html
     */
    toGetObjectVersionTorrent() {
        this.to('s3:GetObjectVersionTorrent');
        return this;
    }
    /**
     * Grants permission to get the replication configuration information set on an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_GetBucketReplication.html
     */
    toGetReplicationConfiguration() {
        this.to('s3:GetReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to list access points
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListAccessPoints.html
     */
    toListAccessPoints() {
        this.to('s3:ListAccessPoints');
        return this;
    }
    /**
     * Grants permission to list all buckets owned by the authenticated sender of the request
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListBuckets.html
     */
    toListAllMyBuckets() {
        this.to('s3:ListAllMyBuckets');
        return this;
    }
    /**
     * Grants permission to list some or all of the objects in an Amazon S3 bucket (up to 1000)
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectsV2.html
     */
    toListBucket() {
        this.to('s3:ListBucket');
        return this;
    }
    /**
     * Grants permission to list in-progress multipart uploads
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListMultipartUploads.html
     */
    toListBucketMultipartUploads() {
        this.to('s3:ListBucketMultipartUploads');
        return this;
    }
    /**
     * Grants permission to list metadata about all the versions of objects in an Amazon S3 bucket
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifDelimiter()
     * - .ifMaxKeys()
     * - .ifPrefix()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListObjectVersions.html
     */
    toListBucketVersions() {
        this.to('s3:ListBucketVersions');
        return this;
    }
    /**
     * Grants permission to list current jobs and jobs that have ended recently
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_ListJobs.html
     */
    toListJobs() {
        this.to('s3:ListJobs');
        return this;
    }
    /**
     * Grants permission to list the parts that have been uploaded for a specific multipart upload
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_ListParts.html
     */
    toListMultipartUploadParts() {
        this.to('s3:ListMultipartUploadParts');
        return this;
    }
    /**
     * Grants permission to change replica ownership
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/replication-change-owner.html#repl-ownership-add-role-permission
     */
    toObjectOwnerOverrideToBucketOwner() {
        this.to('s3:ObjectOwnerOverrideToBucketOwner');
        return this;
    }
    /**
     * Grants permission to use the accelerate subresource to set the Transfer Acceleration state of an existing S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAccelerateConfiguration.html
     */
    toPutAccelerateConfiguration() {
        this.to('s3:PutAccelerateConfiguration');
        return this;
    }
    /**
     * Grants permission to associate an access policy with a specified access point
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutAccessPointPolicy.html
     */
    toPutAccessPointPolicy() {
        this.to('s3:PutAccessPointPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for an AWS account
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutPublicAccessBlock.html
     */
    toPutAccountPublicAccessBlock() {
        this.to('s3:PutAccountPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set an analytics configuration for the bucket, specified by the analytics configuration ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAnalyticsConfiguration.html
     */
    toPutAnalyticsConfiguration() {
        this.to('s3:PutAnalyticsConfiguration');
        return this;
    }
    /**
     * Grants permission to set the permissions on an existing bucket using access control lists (ACLs)
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketAcl.html
     */
    toPutBucketAcl() {
        this.to('s3:PutBucketAcl');
        return this;
    }
    /**
     * Grants permission to set the CORS configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketCors.html
     */
    toPutBucketCORS() {
        this.to('s3:PutBucketCORS');
        return this;
    }
    /**
     * Grants permission to set the logging parameters for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLogging.html
     */
    toPutBucketLogging() {
        this.to('s3:PutBucketLogging');
        return this;
    }
    /**
     * Grants permission to receive notifications when certain events happen in an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketNotification.html
     */
    toPutBucketNotification() {
        this.to('s3:PutBucketNotification');
        return this;
    }
    /**
     * Grants permission to put Object Lock configuration on a specific bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLockConfiguration.html
     */
    toPutBucketObjectLockConfiguration() {
        this.to('s3:PutBucketObjectLockConfiguration');
        return this;
    }
    /**
     * Grants permission to add or replace a bucket policy on a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketPolicy.html
     */
    toPutBucketPolicy() {
        this.to('s3:PutBucketPolicy');
        return this;
    }
    /**
     * Grants permission to create or modify the PublicAccessBlock configuration for a specific Amazon S3 bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutPublicAccessBlock.html
     */
    toPutBucketPublicAccessBlock() {
        this.to('s3:PutBucketPublicAccessBlock');
        return this;
    }
    /**
     * Grants permission to set the request payment configuration of a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketRequestPayment.html
     */
    toPutBucketRequestPayment() {
        this.to('s3:PutBucketRequestPayment');
        return this;
    }
    /**
     * Grants permission to add a set of tags to an existing Amazon S3 bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketTagging.html
     */
    toPutBucketTagging() {
        this.to('s3:PutBucketTagging');
        return this;
    }
    /**
     * Grants permission to set the versioning state of an existing Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketVersioning.html
     */
    toPutBucketVersioning() {
        this.to('s3:PutBucketVersioning');
        return this;
    }
    /**
     * Grants permission to set the configuration of the website that is specified in the website subresource
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketWebsite.html
     */
    toPutBucketWebsite() {
        this.to('s3:PutBucketWebsite');
        return this;
    }
    /**
     * Grants permission to set the encryption configuration for an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketEncryption.html
     */
    toPutEncryptionConfiguration() {
        this.to('s3:PutEncryptionConfiguration');
        return this;
    }
    /**
     * Grants permission to add an inventory configuration to the bucket, identified by the inventory ID
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketInventoryConfiguration.html
     */
    toPutInventoryConfiguration() {
        this.to('s3:PutInventoryConfiguration');
        return this;
    }
    /**
     * Grants permission to replace tags on an existing Amazon S3 Batch Operations job
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_PutJobTagging.html
     */
    toPutJobTagging() {
        this.to('s3:PutJobTagging');
        return this;
    }
    /**
     * Grants permission to create a new lifecycle configuration for the bucket or replace an existing lifecycle configuration
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketLifecycleConfiguration.html
     */
    toPutLifecycleConfiguration() {
        this.to('s3:PutLifecycleConfiguration');
        return this;
    }
    /**
     * Grants permission to set or update a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketMetricsConfiguration.html
     */
    toPutMetricsConfiguration() {
        this.to('s3:PutMetricsConfiguration');
        return this;
    }
    /**
     * Grants permission to add an object to a bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzCopySource()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzMetadataDirective()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     * - .ifXAmzStorageClass()
     * - .ifXAmzWebsiteRedirectLocation()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObject.html
     */
    toPutObject() {
        this.to('s3:PutObject');
        return this;
    }
    /**
     * Grants permission to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectAcl() {
        this.to('s3:PutObjectAcl');
        return this;
    }
    /**
     * Grants permission to apply a Legal Hold configuration to the specified object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockLegalHold()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectLegalHold.html
     */
    toPutObjectLegalHold() {
        this.to('s3:PutObjectLegalHold');
        return this;
    }
    /**
     * Grants permission to place an Object Retention configuration on an object
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifObjectLockMode()
     * - .ifObjectLockRetainUntilDate()
     * - .ifObjectLockRemainingRetentionDays()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectRetention.html
     */
    toPutObjectRetention() {
        this.to('s3:PutObjectRetention');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set to an object that already exists in a bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectTagging() {
        this.to('s3:PutObjectTagging');
        return this;
    }
    /**
     * Grants permission to use the acl subresource to set the access control list (ACL) permissions for an object that already exists in a bucket
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzAcl()
     * - .ifXAmzContentSha256()
     * - .ifXAmzGrantFullControl()
     * - .ifXAmzGrantRead()
     * - .ifXAmzGrantReadAcp()
     * - .ifXAmzGrantWrite()
     * - .ifXAmzGrantWriteAcp()
     * - .ifXAmzStorageClass()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectAcl.html
     */
    toPutObjectVersionAcl() {
        this.to('s3:PutObjectVersionAcl');
        return this;
    }
    /**
     * Grants permission to set the supplied tag-set for a specific version of an object
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifExistingObjectTag()
     * - .ifRequestObjectTag()
     * - .ifRequestObjectTagKeys()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifVersionid()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutObjectTagging.html
     */
    toPutObjectVersionTagging() {
        this.to('s3:PutObjectVersionTagging');
        return this;
    }
    /**
     * Grants permission to create a new replication configuration or replace an existing one
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * Dependent actions:
     * - iam:PassRole
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_PutBucketReplication.html
     */
    toPutReplicationConfiguration() {
        this.to('s3:PutReplicationConfiguration');
        return this;
    }
    /**
     * Grants permission to replicate delete markers to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateDelete() {
        this.to('s3:ReplicateDelete');
        return this;
    }
    /**
     * Grants permission to replicate objects and object tags to the destination bucket
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifXAmzServerSideEncryption()
     * - .ifXAmzServerSideEncryptionAwsKmsKeyId()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateObject() {
        this.to('s3:ReplicateObject');
        return this;
    }
    /**
     * Grants permission to replicate object tags to the destination bucket
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/setting-repl-config-perm-overview.html
     */
    toReplicateTags() {
        this.to('s3:ReplicateTags');
        return this;
    }
    /**
     * Grants permission to restore an archived copy of an object back into Amazon S3
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifDataAccessPointAccount()
     * - .ifDataAccessPointArn()
     * - .ifAccessPointNetworkOrigin()
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_RestoreObject.html
     */
    toRestoreObject() {
        this.to('s3:RestoreObject');
        return this;
    }
    /**
     * Grants permission to update the priority of an existing job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifRequestJobPriority()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobPriority.html
     */
    toUpdateJobPriority() {
        this.to('s3:UpdateJobPriority');
        return this;
    }
    /**
     * Grants permission to update the status for the specified job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAuthType()
     * - .ifSignatureAge()
     * - .ifSignatureversion()
     * - .ifXAmzContentSha256()
     * - .ifExistingJobPriority()
     * - .ifExistingJobOperation()
     * - .ifJobSuspendedCause()
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_control_UpdateJobStatus.html
     */
    toUpdateJobStatus() {
        this.to('s3:UpdateJobStatus');
        return this;
    }
    /**
     * Adds a resource of type accesspoint to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/access-points.html
     *
     * @param accessPointName - Identifier for the accessPointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccesspoint(accessPointName, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:accesspoint/${AccessPointName}';
        arn = arn.replace('${AccessPointName}', accessPointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type bucket to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingBucket.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onBucket(bucketName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type object to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingObjects.html
     *
     * @param bucketName - Identifier for the bucketName.
     * @param objectName - Identifier for the objectName.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onObject(bucketName, objectName, partition) {
        var arn = 'arn:${Partition}:s3:::${BucketName}/${ObjectName}';
        arn = arn.replace('${BucketName}', bucketName);
        arn = arn.replace('${ObjectName}', objectName);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-managing-jobs.html
     *
     * @param jobId - Identifier for the jobId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJob(jobId, account, region, partition) {
        var arn = 'arn:${Partition}:s3:${Region}:${Account}:job/${JobId}';
        arn = arn.replace('${JobId}', jobId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Filters access by the network origin (Internet or VPC)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAccessPointNetworkOrigin(value, operator) {
        return this.if(`s3:AccessPointNetworkOrigin`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the AWS Account ID that owns the access point
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/creating-access-points.html#access-points-policies
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointAccount(value, operator) {
        return this.if(`s3:DataAccessPointAccount`, value, operator || 'StringLike');
    }
    /**
     * Filters access by an access point Amazon Resource Name (ARN)
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListMultipartUploadParts()
     * - .toPutAccessPointPolicy()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toRestoreObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDataAccessPointArn(value, operator) {
        return this.if(`s3:DataAccessPointArn`, value, operator || 'StringLike');
    }
    /**
     * Filters access to updating the job priority by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingJobOperation(value, operator) {
        return this.if(`s3:ExistingJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling existing jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toDeleteJobTagging()
     * - .toPutJobTagging()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifExistingJobPriority(value, operator) {
        return this.if(`s3:ExistingJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by existing object tag key and value
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toPutObjectAcl()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifExistingObjectTag(key, value, operator) {
        return this.if(`s3:ExistingObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access to cancelling suspended jobs by a specific job suspended cause (for example, AWAITING_CONFIRMATION)
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifJobSuspendedCause(value, operator) {
        return this.if(`s3:JobSuspendedCause`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationConstraint(value, operator) {
        return this.if(`s3:LocationConstraint`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating jobs by operation
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestJobOperation(value, operator) {
        return this.if(`s3:RequestJobOperation`, value, operator || 'StringLike');
    }
    /**
     * Filters access to creating new jobs by priority range
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/batch-ops-job-tags-examples.html
     *
     * Applies to actions:
     * - .toCreateJob()
     * - .toUpdateJobPriority()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifRequestJobPriority(value, operator) {
        return this.if(`s3:RequestJobPriority`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the tag keys and values to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param key The tag key to check
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTag(key, value, operator) {
        return this.if(`s3:RequestObjectTag/${key}`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the tag keys to be added to objects
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-tagging.html#tagging-and-policies
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifRequestObjectTagKeys(value, operator) {
        return this.if(`s3:RequestObjectTagKeys`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionId(value, operator) {
        return this.if(`s3:VersionId`, value, operator || 'StringLike');
    }
    /**
     * Filters access by authentication method
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAuthType(value, operator) {
        return this.if(`s3:authType`, value, operator || 'StringLike');
    }
    /**
     * Filters access by delimiter parameter
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/walkthrough1.html
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifDelimiter(value, operator) {
        return this.if(`s3:delimiter`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific Region
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-1
     *
     * Applies to actions:
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLocationconstraint(value, operator) {
        return this.if(`s3:locationconstraint`, value, operator || 'StringLike');
    }
    /**
     * Filters access by maximum number of keys returned in a ListBucket request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#example-numeric-condition-operators
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifMaxKeys(value, operator) {
        return this.if(`s3:max-keys`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by object legal hold status
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-legal-holds
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectLegalHold()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockLegalHold(value, operator) {
        return this.if(`s3:object-lock-legal-hold`, value, operator || 'StringLike');
    }
    /**
     * Filters access by object retention mode (COMPLIANCE or GOVERNANCE)
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockMode(value, operator) {
        return this.if(`s3:object-lock-mode`, value, operator || 'StringLike');
    }
    /**
     * Filters access by remaining object retention days
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-managing.html#object-lock-managing-retention-limits
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRemainingRetentionDays(value, operator) {
        return this.if(`s3:object-lock-remaining-retention-days`, value, operator || 'StringLike');
    }
    /**
     * Filters access by object retain-until date
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/object-lock-overview.html#object-lock-retention-periods
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectRetention()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifObjectLockRetainUntilDate(value, operator) {
        return this.if(`s3:object-lock-retain-until-date`, value, operator || 'StringLike');
    }
    /**
     * Filters access by key name prefix
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#condition-key-bucket-ops-2
     *
     * Applies to actions:
     * - .toListBucket()
     * - .toListBucketVersions()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifPrefix(value, operator) {
        return this.if(`s3:prefix`, value, operator || 'StringLike');
    }
    /**
     * Filters access by the age in milliseconds of the request signature
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [numeric operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_Numeric). **Default:** `NumericEquals`
     */
    ifSignatureAge(value, operator) {
        return this.if(`s3:signatureAge`, value, operator || 'NumericEquals');
    }
    /**
     * Filters access by the version of AWS Signature used on the request
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketObjectLockConfiguration()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketObjectLockConfiguration()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifSignatureversion(value, operator) {
        return this.if(`s3:signatureversion`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific object version
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#getobjectversion-limit-access-to-specific-version-3
     *
     * Applies to actions:
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifVersionid(value, operator) {
        return this.if(`s3:versionid`, value, operator || 'StringLike');
    }
    /**
     * Filters access by canned ACL in the request's x-amz-acl header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzAcl(value, operator) {
        return this.if(`s3:x-amz-acl`, value, operator || 'StringLike');
    }
    /**
     * Filters access to unsigned content in your bucket
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/bucket-policy-s3-sigv4-conditions.html
     *
     * Applies to actions:
     * - .toAbortMultipartUpload()
     * - .toBypassGovernanceRetention()
     * - .toCreateAccessPoint()
     * - .toCreateBucket()
     * - .toCreateJob()
     * - .toDeleteAccessPoint()
     * - .toDeleteAccessPointPolicy()
     * - .toDeleteBucket()
     * - .toDeleteBucketPolicy()
     * - .toDeleteBucketWebsite()
     * - .toDeleteJobTagging()
     * - .toDeleteObject()
     * - .toDeleteObjectTagging()
     * - .toDeleteObjectVersion()
     * - .toDeleteObjectVersionTagging()
     * - .toDescribeJob()
     * - .toGetAccelerateConfiguration()
     * - .toGetAccessPoint()
     * - .toGetAccessPointPolicy()
     * - .toGetAccessPointPolicyStatus()
     * - .toGetAccountPublicAccessBlock()
     * - .toGetAnalyticsConfiguration()
     * - .toGetBucketAcl()
     * - .toGetBucketCORS()
     * - .toGetBucketLogging()
     * - .toGetBucketNotification()
     * - .toGetBucketPolicy()
     * - .toGetBucketPolicyStatus()
     * - .toGetBucketPublicAccessBlock()
     * - .toGetBucketRequestPayment()
     * - .toGetBucketTagging()
     * - .toGetBucketVersioning()
     * - .toGetBucketWebsite()
     * - .toGetEncryptionConfiguration()
     * - .toGetInventoryConfiguration()
     * - .toGetJobTagging()
     * - .toGetLifecycleConfiguration()
     * - .toGetMetricsConfiguration()
     * - .toGetObject()
     * - .toGetObjectAcl()
     * - .toGetObjectLegalHold()
     * - .toGetObjectRetention()
     * - .toGetObjectTagging()
     * - .toGetObjectTorrent()
     * - .toGetObjectVersion()
     * - .toGetObjectVersionAcl()
     * - .toGetObjectVersionForReplication()
     * - .toGetObjectVersionTagging()
     * - .toGetObjectVersionTorrent()
     * - .toGetReplicationConfiguration()
     * - .toListAccessPoints()
     * - .toListAllMyBuckets()
     * - .toListBucket()
     * - .toListBucketMultipartUploads()
     * - .toListBucketVersions()
     * - .toListJobs()
     * - .toListMultipartUploadParts()
     * - .toObjectOwnerOverrideToBucketOwner()
     * - .toPutAccelerateConfiguration()
     * - .toPutAccessPointPolicy()
     * - .toPutAccountPublicAccessBlock()
     * - .toPutAnalyticsConfiguration()
     * - .toPutBucketAcl()
     * - .toPutBucketCORS()
     * - .toPutBucketLogging()
     * - .toPutBucketNotification()
     * - .toPutBucketPolicy()
     * - .toPutBucketPublicAccessBlock()
     * - .toPutBucketRequestPayment()
     * - .toPutBucketTagging()
     * - .toPutBucketVersioning()
     * - .toPutBucketWebsite()
     * - .toPutEncryptionConfiguration()
     * - .toPutInventoryConfiguration()
     * - .toPutJobTagging()
     * - .toPutLifecycleConfiguration()
     * - .toPutMetricsConfiguration()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectLegalHold()
     * - .toPutObjectRetention()
     * - .toPutObjectTagging()
     * - .toPutObjectVersionAcl()
     * - .toPutObjectVersionTagging()
     * - .toPutReplicationConfiguration()
     * - .toReplicateDelete()
     * - .toReplicateObject()
     * - .toReplicateTags()
     * - .toRestoreObject()
     * - .toUpdateJobPriority()
     * - .toUpdateJobStatus()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzContentSha256(value, operator) {
        return this.if(`s3:x-amz-content-sha256`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with a specific bucket, prefix, or object as the copy source
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/amazon-s3-policy-keys.html#putobject-limit-copy-source-3
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzCopySource(value, operator) {
        return this.if(`s3:x-amz-copy-source`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with the x-amz-grant-full-control (full control) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantFullControl(value, operator) {
        return this.if(`s3:x-amz-grant-full-control`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with the x-amz-grant-read (read access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantRead(value, operator) {
        return this.if(`s3:x-amz-grant-read`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with the x-amz-grant-read-acp (read permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantReadAcp(value, operator) {
        return this.if(`s3:x-amz-grant-read-acp`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with the x-amz-grant-write (write access) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWrite(value, operator) {
        return this.if(`s3:x-amz-grant-write`, value, operator || 'StringLike');
    }
    /**
     * Filters access to requests with the x-amz-grant-write-acp (write permissions for the ACL) header
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#permissions
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toCreateBucket()
     * - .toPutBucketAcl()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzGrantWriteAcp(value, operator) {
        return this.if(`s3:x-amz-grant-write-acp`, value, operator || 'StringLike');
    }
    /**
     * Filters access by object metadata behavior (COPY or REPLACE) when objects are copied
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/API/API_CopyObject.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzMetadataDirective(value, operator) {
        return this.if(`s3:x-amz-metadata-directive`, value, operator || 'StringLike');
    }
    /**
     * Filters access by server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryption(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption`, value, operator || 'StringLike');
    }
    /**
     * Filters access by AWS KMS customer managed CMK for server-side encryption
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html#require-sse-kms
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toReplicateObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzServerSideEncryptionAwsKmsKeyId(value, operator) {
        return this.if(`s3:x-amz-server-side-encryption-aws-kms-key-id`, value, operator || 'StringLike');
    }
    /**
     * Filters access by storage class
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-class-intro.html#sc-howtoset
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     * - .toPutObjectAcl()
     * - .toPutObjectVersionAcl()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzStorageClass(value, operator) {
        return this.if(`s3:x-amz-storage-class`, value, operator || 'StringLike');
    }
    /**
     * Filters access by a specific website redirect location for buckets that are configured as static websites
     *
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html#page-redirect-using-rest-api
     *
     * Applies to actions:
     * - .toBypassGovernanceRetention()
     * - .toPutObject()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifXAmzWebsiteRedirectLocation(value, operator) {
        return this.if(`s3:x-amz-website-redirect-location`, value, operator || 'StringLike');
    }
}
exports.S3 = S3;
//# sourceMappingURL=data:application/json;base64,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