"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Storagegateway = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [storagegateway](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonstoragegateway.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Storagegateway extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [storagegateway](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonstoragegateway.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'storagegateway';
        this.accessLevelList = {
            "Write": [
                "ActivateGateway",
                "AddCache",
                "AddUploadBuffer",
                "AddWorkingStorage",
                "AttachVolume",
                "CancelArchival",
                "CancelRetrieval",
                "CreateCachediSCSIVolume",
                "CreateNFSFileShare",
                "CreateSMBFileShare",
                "CreateSnapshot",
                "CreateSnapshotFromVolumeRecoveryPoint",
                "CreateStorediSCSIVolume",
                "CreateTapeWithBarcode",
                "CreateTapes",
                "DeleteBandwidthRateLimit",
                "DeleteChapCredentials",
                "DeleteFileShare",
                "DeleteGateway",
                "DeleteSnapshotSchedule",
                "DeleteTape",
                "DeleteTapeArchive",
                "DeleteVolume",
                "DetachVolume",
                "DisableGateway",
                "JoinDomain",
                "NotifyWhenUploaded",
                "RefreshCache",
                "ResetCache",
                "RetrieveTapeArchive",
                "RetrieveTapeRecoveryPoint",
                "SetLocalConsolePassword",
                "SetSMBGuestPassword",
                "ShutdownGateway",
                "StartGateway",
                "UpdateBandwidthRateLimit",
                "UpdateChapCredentials",
                "UpdateGatewayInformation",
                "UpdateGatewaySoftwareNow",
                "UpdateMaintenanceStartTime",
                "UpdateNFSFileShare",
                "UpdateSMBFileShare",
                "UpdateSnapshotSchedule",
                "UpdateVTLDeviceType"
            ],
            "Tagging": [
                "AddTagsToResource",
                "RemoveTagsFromResource"
            ],
            "Read": [
                "DescribeBandwidthRateLimit",
                "DescribeCache",
                "DescribeCachediSCSIVolumes",
                "DescribeChapCredentials",
                "DescribeGatewayInformation",
                "DescribeMaintenanceStartTime",
                "DescribeNFSFileShares",
                "DescribeSMBFileShares",
                "DescribeSMBSettings",
                "DescribeSnapshotSchedule",
                "DescribeStorediSCSIVolumes",
                "DescribeTapeArchives",
                "DescribeTapeRecoveryPoints",
                "DescribeTapes",
                "DescribeUploadBuffer",
                "DescribeVTLDevices",
                "DescribeWorkingStorage",
                "ListTagsForResource",
                "ListTapes",
                "ListVolumeInitiators"
            ],
            "List": [
                "ListFileShares",
                "ListGateways",
                "ListLocalDisks",
                "ListVolumeRecoveryPoints",
                "ListVolumes"
            ]
        };
    }
    /**
     * This operation activates the gateway you previously deployed on your host.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ActivateGateway.html
     */
    toActivateGateway() {
        this.to('storagegateway:ActivateGateway');
        return this;
    }
    /**
     * This operation configures one or more gateway local disks as cache for a cached-volume gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddCache.html
     */
    toAddCache() {
        this.to('storagegateway:AddCache');
        return this;
    }
    /**
     * This operation adds one or more tags to the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddTagsToResource.html
     */
    toAddTagsToResource() {
        this.to('storagegateway:AddTagsToResource');
        return this;
    }
    /**
     * This operation configures one or more gateway local disks as upload buffer for a specified gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddUploadBuffer.html
     */
    toAddUploadBuffer() {
        this.to('storagegateway:AddUploadBuffer');
        return this;
    }
    /**
     * This operation configures one or more gateway local disks as working storage for a gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AddWorkingStorage.html
     */
    toAddWorkingStorage() {
        this.to('storagegateway:AddWorkingStorage');
        return this;
    }
    /**
     * This operation connects a volume to an iSCSI connection and then attaches the volume to the specified gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_AttachVolume.html
     */
    toAttachVolume() {
        this.to('storagegateway:AttachVolume');
        return this;
    }
    /**
     * Cancels archiving of a virtual tape to the virtual tape shelf (VTS) after the archiving process is initiated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelArchival.html
     */
    toCancelArchival() {
        this.to('storagegateway:CancelArchival');
        return this;
    }
    /**
     * Cancels retrieval of a virtual tape from the virtual tape shelf (VTS) to a gateway after the retrieval process is initiated.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CancelRetrieval.html
     */
    toCancelRetrieval() {
        this.to('storagegateway:CancelRetrieval');
        return this;
    }
    /**
     * This operation creates a cached volume on a specified cached gateway. This operation is supported only for the gateway-cached volume architecture.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateCachediSCSIVolume.html
     */
    toCreateCachediSCSIVolume() {
        this.to('storagegateway:CreateCachediSCSIVolume');
        return this;
    }
    /**
     * This operation creates a NFS file share on an existing file gateway.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateNFSFileShare.html
     */
    toCreateNFSFileShare() {
        this.to('storagegateway:CreateNFSFileShare');
        return this;
    }
    /**
     * This operation creates a SMB file share on an existing file gateway.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSMBFileShare.html
     */
    toCreateSMBFileShare() {
        this.to('storagegateway:CreateSMBFileShare');
        return this;
    }
    /**
     * This operation initiates a snapshot of a volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshot.html
     */
    toCreateSnapshot() {
        this.to('storagegateway:CreateSnapshot');
        return this;
    }
    /**
     * This operation initiates a snapshot of a gateway from a volume recovery point.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateSnapshotFromVolumeRecoveryPoint.html
     */
    toCreateSnapshotFromVolumeRecoveryPoint() {
        this.to('storagegateway:CreateSnapshotFromVolumeRecoveryPoint');
        return this;
    }
    /**
     * This operation creates a volume on a specified gateway.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateStorediSCSIVolume.html
     */
    toCreateStorediSCSIVolume() {
        this.to('storagegateway:CreateStorediSCSIVolume');
        return this;
    }
    /**
     * Creates a virtual tape by using your own barcode.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapeWithBarcode.html
     */
    toCreateTapeWithBarcode() {
        this.to('storagegateway:CreateTapeWithBarcode');
        return this;
    }
    /**
     * Creates one or more virtual tapes. You write data to the virtual tapes and then archive the tapes.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_CreateTapes.html
     */
    toCreateTapes() {
        this.to('storagegateway:CreateTapes');
        return this;
    }
    /**
     * This operation deletes the bandwidth rate limits of a gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteBandwidthRateLimit.html
     */
    toDeleteBandwidthRateLimit() {
        this.to('storagegateway:DeleteBandwidthRateLimit');
        return this;
    }
    /**
     * This operation deletes Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target and initiator pair.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteChapCredentials.html
     */
    toDeleteChapCredentials() {
        this.to('storagegateway:DeleteChapCredentials');
        return this;
    }
    /**
     * This operation deletes a file share from a file gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteFileShare.html
     */
    toDeleteFileShare() {
        this.to('storagegateway:DeleteFileShare');
        return this;
    }
    /**
     * This operation deletes a gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteGateway.html
     */
    toDeleteGateway() {
        this.to('storagegateway:DeleteGateway');
        return this;
    }
    /**
     * This operation deletes a snapshot of a volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteSnapshotSchedule.html
     */
    toDeleteSnapshotSchedule() {
        this.to('storagegateway:DeleteSnapshotSchedule');
        return this;
    }
    /**
     * Deletes the specified virtual tape.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTape.html
     */
    toDeleteTape() {
        this.to('storagegateway:DeleteTape');
        return this;
    }
    /**
     * Deletes the specified virtual tape from the virtual tape shelf (VTS).
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteTapeArchive.html
     */
    toDeleteTapeArchive() {
        this.to('storagegateway:DeleteTapeArchive');
        return this;
    }
    /**
     * This operation deletes the specified gateway volume that you previously created using the CreateCachediSCSIVolume or CreateStorediSCSIVolume API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DeleteVolume.html
     */
    toDeleteVolume() {
        this.to('storagegateway:DeleteVolume');
        return this;
    }
    /**
     * This operation returns the bandwidth rate limits of a gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeBandwidthRateLimit.html
     */
    toDescribeBandwidthRateLimit() {
        this.to('storagegateway:DescribeBandwidthRateLimit');
        return this;
    }
    /**
     * This operation returns information about the cache of a gateway. This operation is supported only for the gateway-cached volume architecture.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCache.html
     */
    toDescribeCache() {
        this.to('storagegateway:DescribeCache');
        return this;
    }
    /**
     * This operation returns a description of the gateway volumes specified in the request. This operation is supported only for the gateway-cached volume architecture.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeCachediSCSIVolumes.html
     */
    toDescribeCachediSCSIVolumes() {
        this.to('storagegateway:DescribeCachediSCSIVolumes');
        return this;
    }
    /**
     * This operation returns an array of Challenge-Handshake Authentication Protocol (CHAP) credentials information for a specified iSCSI target, one for each target-initiator pair.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeChapCredentials.html
     */
    toDescribeChapCredentials() {
        this.to('storagegateway:DescribeChapCredentials');
        return this;
    }
    /**
     * This operation returns metadata about a gateway such as its name, network interfaces, configured time zone, and the state (whether the gateway is running or not).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeGatewayInformation.html
     */
    toDescribeGatewayInformation() {
        this.to('storagegateway:DescribeGatewayInformation');
        return this;
    }
    /**
     * This operation returns your gateway's weekly maintenance start time including the day and time of the week.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeMaintenanceStartTime.html
     */
    toDescribeMaintenanceStartTime() {
        this.to('storagegateway:DescribeMaintenanceStartTime');
        return this;
    }
    /**
     * This operation gets a description for one or more file shares from a file gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeNFSFileShares.html
     */
    toDescribeNFSFileShares() {
        this.to('storagegateway:DescribeNFSFileShares');
        return this;
    }
    /**
     * This operation gets a description for one or more file shares from a file gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBFileShares.html
     */
    toDescribeSMBFileShares() {
        this.to('storagegateway:DescribeSMBFileShares');
        return this;
    }
    /**
     * This operation gets a description of a Server Message Block (SMB) file share settings from a file gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSMBSettings.html
     */
    toDescribeSMBSettings() {
        this.to('storagegateway:DescribeSMBSettings');
        return this;
    }
    /**
     * This operation describes the snapshot schedule for the specified gateway volume.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeSnapshotSchedule.html
     */
    toDescribeSnapshotSchedule() {
        this.to('storagegateway:DescribeSnapshotSchedule');
        return this;
    }
    /**
     * This operation returns the description of the gateway volumes specified in the request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeStorediSCSIVolumes.html
     */
    toDescribeStorediSCSIVolumes() {
        this.to('storagegateway:DescribeStorediSCSIVolumes');
        return this;
    }
    /**
     * Returns a description of specified virtual tapes in the virtual tape shelf (VTS).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeArchives.html
     */
    toDescribeTapeArchives() {
        this.to('storagegateway:DescribeTapeArchives');
        return this;
    }
    /**
     * Returns a list of virtual tape recovery points that are available for the specified gateway-VTL.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapeRecoveryPoints.html
     */
    toDescribeTapeRecoveryPoints() {
        this.to('storagegateway:DescribeTapeRecoveryPoints');
        return this;
    }
    /**
     * Returns a description of the specified Amazon Resource Name (ARN) of virtual tapes.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeTapes.html
     */
    toDescribeTapes() {
        this.to('storagegateway:DescribeTapes');
        return this;
    }
    /**
     * This operation returns information about the upload buffer of a gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeUploadBuffer.html
     */
    toDescribeUploadBuffer() {
        this.to('storagegateway:DescribeUploadBuffer');
        return this;
    }
    /**
     * Returns a description of virtual tape library (VTL) devices for the specified gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeVTLDevices.html
     */
    toDescribeVTLDevices() {
        this.to('storagegateway:DescribeVTLDevices');
        return this;
    }
    /**
     * This operation returns information about the working storage of a gateway.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DescribeWorkingStorage.html
     */
    toDescribeWorkingStorage() {
        this.to('storagegateway:DescribeWorkingStorage');
        return this;
    }
    /**
     * This operation disconnects a volume from an iSCSI connection and then detaches the volume from the specified gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DetachVolume.html
     */
    toDetachVolume() {
        this.to('storagegateway:DetachVolume');
        return this;
    }
    /**
     * Disables a gateway when the gateway is no longer functioning.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_DisableGateway.html
     */
    toDisableGateway() {
        this.to('storagegateway:DisableGateway');
        return this;
    }
    /**
     * This operation enables you to join an Active Directory Domain.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_JoinDomain.html
     */
    toJoinDomain() {
        this.to('storagegateway:JoinDomain');
        return this;
    }
    /**
     * This operation gets a list of the file shares for a specific file gateway, or the list of file shares that belong to the calling user account.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListFileShares.html
     */
    toListFileShares() {
        this.to('storagegateway:ListFileShares');
        return this;
    }
    /**
     * This operation lists gateways owned by an AWS account in a region specified in the request. The returned list is ordered by gateway Amazon Resource Name (ARN).
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListGateways.html
     */
    toListGateways() {
        this.to('storagegateway:ListGateways');
        return this;
    }
    /**
     * This operation returns a list of the gateway's local disks.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListLocalDisks.html
     */
    toListLocalDisks() {
        this.to('storagegateway:ListLocalDisks');
        return this;
    }
    /**
     * This operation lists the tags that have been added to the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('storagegateway:ListTagsForResource');
        return this;
    }
    /**
     * Lists virtual tapes in your virtual tape library (VTL) and your virtual tape shelf (VTS).
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListTapes.html
     */
    toListTapes() {
        this.to('storagegateway:ListTapes');
        return this;
    }
    /**
     * This operation lists iSCSI initiators that are connected to a volume.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeInitiators.html
     */
    toListVolumeInitiators() {
        this.to('storagegateway:ListVolumeInitiators');
        return this;
    }
    /**
     * This operation lists the recovery points for a specified gateway.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumeRecoveryPoints.html
     */
    toListVolumeRecoveryPoints() {
        this.to('storagegateway:ListVolumeRecoveryPoints');
        return this;
    }
    /**
     * This operation lists the iSCSI stored volumes of a gateway.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ListVolumes.html
     */
    toListVolumes() {
        this.to('storagegateway:ListVolumes');
        return this;
    }
    /**
     * This action sends you a notification through CloudWatch Events when all files written to your NFS file share have been uploaded to Amazon S3.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_NotifyWhenUploaded.html
     */
    toNotifyWhenUploaded() {
        this.to('storagegateway:NotifyWhenUploaded');
        return this;
    }
    /**
     * This operation refreshes the cache for the specified file share.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RefreshCache.html
     */
    toRefreshCache() {
        this.to('storagegateway:RefreshCache');
        return this;
    }
    /**
     * This operation removes one or more tags from the specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RemoveTagsFromResource.html
     */
    toRemoveTagsFromResource() {
        this.to('storagegateway:RemoveTagsFromResource');
        return this;
    }
    /**
     * This operation resets all cache disks that have encountered a error and makes the disks available for reconfiguration as cache storage.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ResetCache.html
     */
    toResetCache() {
        this.to('storagegateway:ResetCache');
        return this;
    }
    /**
     * Retrieves an archived virtual tape from the virtual tape shelf (VTS) to a gateway-VTL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeArchive.html
     */
    toRetrieveTapeArchive() {
        this.to('storagegateway:RetrieveTapeArchive');
        return this;
    }
    /**
     * Retrieves the recovery point for the specified virtual tape.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_RetrieveTapeRecoveryPoint.html
     */
    toRetrieveTapeRecoveryPoint() {
        this.to('storagegateway:RetrieveTapeRecoveryPoint');
        return this;
    }
    /**
     * Sets the password for your VM local console.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetLocalConsolePassword.html
     */
    toSetLocalConsolePassword() {
        this.to('storagegateway:SetLocalConsolePassword');
        return this;
    }
    /**
     * Sets the password for SMB Guest user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_SetSMBGuestPassword.html
     */
    toSetSMBGuestPassword() {
        this.to('storagegateway:SetSMBGuestPassword');
        return this;
    }
    /**
     * This operation shuts down a gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_ShutdownGateway.html
     */
    toShutdownGateway() {
        this.to('storagegateway:ShutdownGateway');
        return this;
    }
    /**
     * This operation starts a gateway that you previously shut down.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_StartGateway.html
     */
    toStartGateway() {
        this.to('storagegateway:StartGateway');
        return this;
    }
    /**
     * This operation updates the bandwidth rate limits of a gateway.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateBandwidthRateLimit.html
     */
    toUpdateBandwidthRateLimit() {
        this.to('storagegateway:UpdateBandwidthRateLimit');
        return this;
    }
    /**
     * This operation updates the Challenge-Handshake Authentication Protocol (CHAP) credentials for a specified iSCSI target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateChapCredentials.html
     */
    toUpdateChapCredentials() {
        this.to('storagegateway:UpdateChapCredentials');
        return this;
    }
    /**
     * This operation updates a gateway's metadata, which includes the gateway's name and time zone.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewayInformation.html
     */
    toUpdateGatewayInformation() {
        this.to('storagegateway:UpdateGatewayInformation');
        return this;
    }
    /**
     * This operation updates the gateway virtual machine (VM) software.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateGatewaySoftwareNow.html
     */
    toUpdateGatewaySoftwareNow() {
        this.to('storagegateway:UpdateGatewaySoftwareNow');
        return this;
    }
    /**
     * This operation updates a gateway's weekly maintenance start time information, including day and time of the week. The maintenance time is the time in your gateway's time zone.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateMaintenanceStartTime.html
     */
    toUpdateMaintenanceStartTime() {
        this.to('storagegateway:UpdateMaintenanceStartTime');
        return this;
    }
    /**
     * This operation updates a NFS file share.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateNFSFileShare.html
     */
    toUpdateNFSFileShare() {
        this.to('storagegateway:UpdateNFSFileShare');
        return this;
    }
    /**
     * This operation updates a SMB file share.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSMBFileShare.html
     */
    toUpdateSMBFileShare() {
        this.to('storagegateway:UpdateSMBFileShare');
        return this;
    }
    /**
     * This operation updates a snapshot schedule configured for a gateway volume.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateSnapshotSchedule.html
     */
    toUpdateSnapshotSchedule() {
        this.to('storagegateway:UpdateSnapshotSchedule');
        return this;
    }
    /**
     * This operation updates the type of medium changer in a gateway-VTL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/storagegateway/latest/APIReference/API_UpdateVTLDeviceType.html
     */
    toUpdateVTLDeviceType() {
        this.to('storagegateway:UpdateVTLDeviceType');
        return this;
    }
    /**
     * Adds a resource of type device to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/resource_vtl-devices.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param vtldevice - Identifier for the vtldevice.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDevice(gatewayId, vtldevice, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/device/${Vtldevice}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${Vtldevice}', vtldevice);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type gateway to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGateway(gatewayId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type share to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateFileShare.html
     *
     * @param shareId - Identifier for the shareId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onShare(shareId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:share/${ShareId}';
        arn = arn.replace('${ShareId}', shareId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tape to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#storage-gateway-vtl-concepts
     *
     * @param tapeBarcode - Identifier for the tapeBarcode.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTape(tapeBarcode, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:tape/${TapeBarcode}';
        arn = arn.replace('${TapeBarcode}', tapeBarcode);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type target to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/GettingStartedCreateVolumes.html
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param iscsiTarget - Identifier for the iscsiTarget.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTarget(gatewayId, iscsiTarget, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/target/${IscsiTarget}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${IscsiTarget}', iscsiTarget);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type volume to the statement
     *
     * https://docs.aws.amazon.com/storagegateway/latest/userguide/StorageGatewayConcepts.html#volume-gateway-concepts
     *
     * @param gatewayId - Identifier for the gatewayId.
     * @param volumeId - Identifier for the volumeId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVolume(gatewayId, volumeId, account, region, partition) {
        var arn = 'arn:${Partition}:storagegateway:${Region}:${Account}:gateway/${GatewayId}/volume/${VolumeId}';
        arn = arn.replace('${GatewayId}', gatewayId);
        arn = arn.replace('${VolumeId}', volumeId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Storagegateway = Storagegateway;
//# sourceMappingURL=data:application/json;base64,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