"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Appsync = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [appsync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Appsync extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [appsync](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsappsync.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'appsync';
        this.accessLevelList = {
            "Write": [
                "CreateApiKey",
                "CreateDataSource",
                "CreateFunction",
                "CreateResolver",
                "CreateType",
                "DeleteApiKey",
                "DeleteDataSource",
                "DeleteFunction",
                "DeleteGraphqlApi",
                "DeleteResolver",
                "DeleteType",
                "GraphQL",
                "StartSchemaCreation",
                "UpdateApiKey",
                "UpdateDataSource",
                "UpdateFunction",
                "UpdateGraphqlApi",
                "UpdateResolver",
                "UpdateType"
            ],
            "Tagging": [
                "CreateGraphqlApi",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "GetDataSource",
                "GetFunction",
                "GetGraphqlApi",
                "GetIntrospectionSchema",
                "GetResolver",
                "GetSchemaCreationStatus",
                "GetType",
                "ListTagsForResource"
            ],
            "List": [
                "ListApiKeys",
                "ListDataSources",
                "ListFunctions",
                "ListGraphqlApis",
                "ListResolvers",
                "ListResolversByFunction",
                "ListTypes"
            ]
        };
    }
    /**
     * Creates a unique key that you can distribute to clients who are executing your API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateApiKey.html
     */
    toCreateApiKey() {
        this.to('appsync:CreateApiKey');
        return this;
    }
    /**
     * Creates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateDataSource.html
     */
    toCreateDataSource() {
        this.to('appsync:CreateDataSource');
        return this;
    }
    /**
     * Create a new Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateFunction.html
     */
    toCreateFunction() {
        this.to('appsync:CreateFunction');
        return this;
    }
    /**
     * Creates a GraphqlApi object, which is the top level AppSync resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateGraphqlApi.html
     */
    toCreateGraphqlApi() {
        this.to('appsync:CreateGraphqlApi');
        return this;
    }
    /**
     * Creates a Resolver object. A resolver converts incoming requests into a format that a data source can understand, and converts the data source's responses into GraphQL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateResolver.html
     */
    toCreateResolver() {
        this.to('appsync:CreateResolver');
        return this;
    }
    /**
     * Creates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_CreateType.html
     */
    toCreateType() {
        this.to('appsync:CreateType');
        return this;
    }
    /**
     * Deletes an API key.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteApiKey.html
     */
    toDeleteApiKey() {
        this.to('appsync:DeleteApiKey');
        return this;
    }
    /**
     * Deletes a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteDataSource.html
     */
    toDeleteDataSource() {
        this.to('appsync:DeleteDataSource');
        return this;
    }
    /**
     * Deletes a Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteFunction.html
     */
    toDeleteFunction() {
        this.to('appsync:DeleteFunction');
        return this;
    }
    /**
     * Deletes a GraphqlApi object. This will also clean up every AppSync resource below that API.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteGraphqlApi.html
     */
    toDeleteGraphqlApi() {
        this.to('appsync:DeleteGraphqlApi');
        return this;
    }
    /**
     * Deletes a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteResolver.html
     */
    toDeleteResolver() {
        this.to('appsync:DeleteResolver');
        return this;
    }
    /**
     * Deletes a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_DeleteType.html
     */
    toDeleteType() {
        this.to('appsync:DeleteType');
        return this;
    }
    /**
     * Retrieves a DataSource object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetDataSource.html
     */
    toGetDataSource() {
        this.to('appsync:GetDataSource');
        return this;
    }
    /**
     * Retrieves a Function object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetFunction.html
     */
    toGetFunction() {
        this.to('appsync:GetFunction');
        return this;
    }
    /**
     * Retrieves a GraphqlApi object.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetGraphqlApi.html
     */
    toGetGraphqlApi() {
        this.to('appsync:GetGraphqlApi');
        return this;
    }
    /**
     * Retrieves the introspection schema for a GraphQL API.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetIntrospectionSchema.html
     */
    toGetIntrospectionSchema() {
        this.to('appsync:GetIntrospectionSchema');
        return this;
    }
    /**
     * Retrieves a Resolver object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetResolver.html
     */
    toGetResolver() {
        this.to('appsync:GetResolver');
        return this;
    }
    /**
     * Retrieves the current status of a schema creation operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetSchemaCreationStatus.html
     */
    toGetSchemaCreationStatus() {
        this.to('appsync:GetSchemaCreationStatus');
        return this;
    }
    /**
     * Retrieves a Type object.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_GetType.html
     */
    toGetType() {
        this.to('appsync:GetType');
        return this;
    }
    /**
     * Sends a GraphQL query to a GraphQL API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/using-your-api.html
     */
    toGraphQL() {
        this.to('appsync:GraphQL');
        return this;
    }
    /**
     * Lists the API keys for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListApiKeys.html
     */
    toListApiKeys() {
        this.to('appsync:ListApiKeys');
        return this;
    }
    /**
     * Lists the data sources for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListDataSources.html
     */
    toListDataSources() {
        this.to('appsync:ListDataSources');
        return this;
    }
    /**
     * Lists the functions for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListFunctions.html
     */
    toListFunctions() {
        this.to('appsync:ListFunctions');
        return this;
    }
    /**
     * Lists your GraphQL APIs.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListGraphqlApis.html
     */
    toListGraphqlApis() {
        this.to('appsync:ListGraphqlApis');
        return this;
    }
    /**
     * Lists the resolvers for a given API and type.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolvers.html
     */
    toListResolvers() {
        this.to('appsync:ListResolvers');
        return this;
    }
    /**
     * List the resolvers that are associated with a specific function.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListResolversByFunction.html
     */
    toListResolversByFunction() {
        this.to('appsync:ListResolversByFunction');
        return this;
    }
    /**
     * List the tags for a resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('appsync:ListTagsForResource');
        return this;
    }
    /**
     * Lists the types for a given API.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        this.to('appsync:ListTypes');
        return this;
    }
    /**
     * Adds a new schema to your GraphQL API. This operation is asynchronous - GetSchemaCreationStatus can show when it has completed.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_StartSchemaCreation.html
     */
    toStartSchemaCreation() {
        this.to('appsync:StartSchemaCreation');
        return this;
    }
    /**
     * Tag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('appsync:TagResource');
        return this;
    }
    /**
     * Untag a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('appsync:UntagResource');
        return this;
    }
    /**
     * Updates an API key for a given API.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateApiKey.html
     */
    toUpdateApiKey() {
        this.to('appsync:UpdateApiKey');
        return this;
    }
    /**
     * Updates a DataSource object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateDataSource.html
     */
    toUpdateDataSource() {
        this.to('appsync:UpdateDataSource');
        return this;
    }
    /**
     * Updates an existing Function object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateFunction.html
     */
    toUpdateFunction() {
        this.to('appsync:UpdateFunction');
        return this;
    }
    /**
     * Updates a GraphqlApi object.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateGraphqlApi.html
     */
    toUpdateGraphqlApi() {
        this.to('appsync:UpdateGraphqlApi');
        return this;
    }
    /**
     * Updates a Resolver object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateResolver.html
     */
    toUpdateResolver() {
        this.to('appsync:UpdateResolver');
        return this;
    }
    /**
     * Updates a Type object.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/appsync/latest/APIReference/API_UpdateType.html
     */
    toUpdateType() {
        this.to('appsync:UpdateType');
        return this;
    }
    /**
     * Adds a resource of type datasource to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/attaching-a-data-source.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param datasourceName - Identifier for the datasourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatasource(graphQLAPIId, datasourceName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/datasources/${DatasourceName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${DatasourceName}', datasourceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type graphqlapi to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-a-graphql-api.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onGraphqlapi(graphQLAPIId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type field to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/configuring-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param fieldName - Identifier for the fieldName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onField(graphQLAPIId, typeName, fieldName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}/fields/${FieldName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${FieldName}', fieldName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type type to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/designing-your-schema.html#adding-a-root-query-type
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param typeName - Identifier for the typeName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onType(graphQLAPIId, typeName, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/types/${TypeName}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${TypeName}', typeName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type function to the statement
     *
     * https://docs.aws.amazon.com/appsync/latest/devguide/pipeline-resolvers.html
     *
     * @param graphQLAPIId - Identifier for the graphQLAPIId.
     * @param functionId - Identifier for the functionId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onFunction(graphQLAPIId, functionId, account, region, partition) {
        var arn = 'arn:${Partition}:appsync:${Region}:${Account}:apis/${GraphQLAPIId}/functions/${FunctionId}';
        arn = arn.replace('${GraphQLAPIId}', graphQLAPIId);
        arn = arn.replace('${FunctionId}', functionId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Appsync = Appsync;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzYXBwc3luYy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImF3c2FwcHN5bmMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQ0Esc0NBQTRDO0FBRTVDOzs7O0dBSUc7QUFDSCxNQUFhLE9BQVEsU0FBUSx3QkFBZTtJQUcxQzs7OztPQUlHO0lBQ0gsWUFBYSxHQUFZO1FBQ3ZCLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztRQVJOLGtCQUFhLEdBQUcsU0FBUyxDQUFDO1FBcWV2QixvQkFBZSxHQUFvQjtZQUMzQyxPQUFPLEVBQUU7Z0JBQ1AsY0FBYztnQkFDZCxrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsZ0JBQWdCO2dCQUNoQixZQUFZO2dCQUNaLGNBQWM7Z0JBQ2Qsa0JBQWtCO2dCQUNsQixnQkFBZ0I7Z0JBQ2hCLGtCQUFrQjtnQkFDbEIsZ0JBQWdCO2dCQUNoQixZQUFZO2dCQUNaLFNBQVM7Z0JBQ1QscUJBQXFCO2dCQUNyQixjQUFjO2dCQUNkLGtCQUFrQjtnQkFDbEIsZ0JBQWdCO2dCQUNoQixrQkFBa0I7Z0JBQ2xCLGdCQUFnQjtnQkFDaEIsWUFBWTthQUNiO1lBQ0QsU0FBUyxFQUFFO2dCQUNULGtCQUFrQjtnQkFDbEIsYUFBYTtnQkFDYixlQUFlO2FBQ2hCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLGVBQWU7Z0JBQ2YsYUFBYTtnQkFDYixlQUFlO2dCQUNmLHdCQUF3QjtnQkFDeEIsYUFBYTtnQkFDYix5QkFBeUI7Z0JBQ3pCLFNBQVM7Z0JBQ1QscUJBQXFCO2FBQ3RCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLGFBQWE7Z0JBQ2IsaUJBQWlCO2dCQUNqQixlQUFlO2dCQUNmLGlCQUFpQjtnQkFDakIsZUFBZTtnQkFDZix5QkFBeUI7Z0JBQ3pCLFdBQVc7YUFDWjtTQUNGLENBQUM7SUExZ0JGLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdCQUFnQjtRQUNyQixJQUFJLENBQUMsRUFBRSxDQUFDLHdCQUF3QixDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFlBQVk7UUFDakIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQy9CLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7T0FTRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdCQUF3QjtRQUM3QixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsaUNBQWlDLENBQUMsQ0FBQztRQUMzQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxTQUFTO1FBQ2QsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUJBQWlCO1FBQ3RCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsdUJBQXVCLENBQUMsQ0FBQztRQUNqQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxpQkFBaUI7UUFDdEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDO1FBQ2pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLGlDQUFpQyxDQUFDLENBQUM7UUFDM0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxXQUFXO1FBQ2hCLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsRUFBRSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFDL0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLHVCQUF1QixDQUFDLENBQUM7UUFDakMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0JBQWtCO1FBQ3ZCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDO1FBQ2xDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLDBCQUEwQixDQUFDLENBQUM7UUFDcEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUM5QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFrREQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFlBQVksQ0FBQyxZQUFvQixFQUFFLGNBQXNCLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDckgsSUFBSSxHQUFHLEdBQUcsa0dBQWtHLENBQUM7UUFDN0csR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDbkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsbUJBQW1CLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFDdkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxZQUFZLENBQUMsWUFBb0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUM3RixJQUFJLEdBQUcsR0FBRyxvRUFBb0UsQ0FBQztRQUMvRSxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNuRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsT0FBTyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ2hELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRSxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7UUFDOUMsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsY0FBYyxFQUFFLFNBQVMsSUFBSSxLQUFLLENBQUMsQ0FBQztRQUN0RCxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDdEIsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksT0FBTyxDQUFDLFlBQW9CLEVBQUUsUUFBZ0IsRUFBRSxTQUFpQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQzdILElBQUksR0FBRyxHQUFHLDBHQUEwRyxDQUFDO1FBQ3JILEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxDQUFDO1FBQ25ELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGFBQWEsRUFBRSxRQUFRLENBQUMsQ0FBQztRQUMzQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDN0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksTUFBTSxDQUFDLFlBQW9CLEVBQUUsUUFBZ0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUN6RyxJQUFJLEdBQUcsR0FBRyxzRkFBc0YsQ0FBQztRQUNqRyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNuRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxhQUFhLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDM0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksVUFBVSxDQUFDLFlBQW9CLEVBQUUsVUFBa0IsRUFBRSxPQUFnQixFQUFFLE1BQWUsRUFBRSxTQUFrQjtRQUMvRyxJQUFJLEdBQUcsR0FBRyw0RkFBNEYsQ0FBQztRQUN2RyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUNuRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxlQUFlLEVBQUUsVUFBVSxDQUFDLENBQUM7UUFDL0MsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7Q0FDRjtBQWpvQkQsMEJBaW9CQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gXCIuLi9zaGFyZWQvYWNjZXNzLWxldmVsXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQgfSBmcm9tIFwiLi4vc2hhcmVkXCI7XG5cbi8qKlxuICogU3RhdGVtZW50IHByb3ZpZGVyIGZvciBzZXJ2aWNlIFthcHBzeW5jXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NhcHBzeW5jLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQXBwc3luYyBleHRlbmRzIFBvbGljeVN0YXRlbWVudCB7XG4gIHB1YmxpYyBzZXJ2aWNlUHJlZml4ID0gJ2FwcHN5bmMnO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW2FwcHN5bmNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2FwcHN5bmMuaHRtbCkuXG4gICAqXG4gICAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgY29uc3RydWN0b3IgKHNpZD86IHN0cmluZykge1xuICAgIHN1cGVyKHNpZCk7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIHVuaXF1ZSBrZXkgdGhhdCB5b3UgY2FuIGRpc3RyaWJ1dGUgdG8gY2xpZW50cyB3aG8gYXJlIGV4ZWN1dGluZyB5b3VyIEFQSS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVBcGlLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQXBpS2V5KCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6Q3JlYXRlQXBpS2V5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIERhdGFTb3VyY2Ugb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZURhdGFTb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlRGF0YVNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkNyZWF0ZURhdGFTb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBuZXcgRnVuY3Rpb24gb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0NyZWF0ZUZ1bmN0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUZ1bmN0aW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6Q3JlYXRlRnVuY3Rpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGEgR3JhcGhxbEFwaSBvYmplY3QsIHdoaWNoIGlzIHRoZSB0b3AgbGV2ZWwgQXBwU3luYyByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBUYWdnaW5nXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVxdWVzdFRhZygpXG4gICAqIC0gLmlmQXdzVGFnS2V5cygpXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIGlhbTpDcmVhdGVTZXJ2aWNlTGlua2VkUm9sZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVHcmFwaHFsQXBpLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUdyYXBocWxBcGkoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpDcmVhdGVHcmFwaHFsQXBpJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlcyBhIFJlc29sdmVyIG9iamVjdC4gQSByZXNvbHZlciBjb252ZXJ0cyBpbmNvbWluZyByZXF1ZXN0cyBpbnRvIGEgZm9ybWF0IHRoYXQgYSBkYXRhIHNvdXJjZSBjYW4gdW5kZXJzdGFuZCwgYW5kIGNvbnZlcnRzIHRoZSBkYXRhIHNvdXJjZSdzIHJlc3BvbnNlcyBpbnRvIEdyYXBoUUwuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlUmVzb2x2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUmVzb2x2ZXIoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpDcmVhdGVSZXNvbHZlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYSBUeXBlIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVUeXBlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVR5cGUoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpDcmVhdGVUeXBlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhbiBBUEkga2V5LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUFwaUtleS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBcGlLZXkoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpEZWxldGVBcGlLZXknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGEgRGF0YVNvdXJjZSBvYmplY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRGF0YVNvdXJjZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVEYXRhU291cmNlKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6RGVsZXRlRGF0YVNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBGdW5jdGlvbiBvYmplY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlRnVuY3Rpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlRnVuY3Rpb24oKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpEZWxldGVGdW5jdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBHcmFwaHFsQXBpIG9iamVjdC4gVGhpcyB3aWxsIGFsc28gY2xlYW4gdXAgZXZlcnkgQXBwU3luYyByZXNvdXJjZSBiZWxvdyB0aGF0IEFQSS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlR3JhcGhxbEFwaS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVHcmFwaHFsQXBpKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6RGVsZXRlR3JhcGhxbEFwaScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBSZXNvbHZlciBvYmplY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVsZXRlUmVzb2x2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUmVzb2x2ZXIoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpEZWxldGVSZXNvbHZlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgYSBUeXBlIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVUeXBlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVR5cGUoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpEZWxldGVUeXBlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGEgRGF0YVNvdXJjZSBvYmplY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9HZXREYXRhU291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldERhdGFTb3VyY2UoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpHZXREYXRhU291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmVzIGEgRnVuY3Rpb24gb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0RnVuY3Rpb24uaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0RnVuY3Rpb24oKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpHZXRGdW5jdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhIEdyYXBocWxBcGkgb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldEdyYXBocWxBcGkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0R3JhcGhxbEFwaSgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkdldEdyYXBocWxBcGknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIGludHJvc3BlY3Rpb24gc2NoZW1hIGZvciBhIEdyYXBoUUwgQVBJLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0SW50cm9zcGVjdGlvblNjaGVtYS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9HZXRJbnRyb3NwZWN0aW9uU2NoZW1hKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6R2V0SW50cm9zcGVjdGlvblNjaGVtYScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlcyBhIFJlc29sdmVyIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFJlc29sdmVyLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFJlc29sdmVyKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6R2V0UmVzb2x2ZXInKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgdGhlIGN1cnJlbnQgc3RhdHVzIG9mIGEgc2NoZW1hIGNyZWF0aW9uIG9wZXJhdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFNjaGVtYUNyZWF0aW9uU3RhdHVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0dldFNjaGVtYUNyZWF0aW9uU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6R2V0U2NoZW1hQ3JlYXRpb25TdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZXMgYSBUeXBlIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0dldFR5cGUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR2V0VHlwZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkdldFR5cGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZW5kcyBhIEdyYXBoUUwgcXVlcnkgdG8gYSBHcmFwaFFMIEFQSS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvZGV2Z3VpZGUvdXNpbmcteW91ci1hcGkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvR3JhcGhRTCgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkdyYXBoUUwnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgQVBJIGtleXMgZm9yIGEgZ2l2ZW4gQVBJLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEFwaUtleXMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFwaUtleXMoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpMaXN0QXBpS2V5cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBkYXRhIHNvdXJjZXMgZm9yIGEgZ2l2ZW4gQVBJLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdERhdGFTb3VyY2VzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3REYXRhU291cmNlcygpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkxpc3REYXRhU291cmNlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBmdW5jdGlvbnMgZm9yIGEgZ2l2ZW4gQVBJLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEZ1bmN0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0RnVuY3Rpb25zKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6TGlzdEZ1bmN0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHlvdXIgR3JhcGhRTCBBUElzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdEdyYXBocWxBcGlzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RHcmFwaHFsQXBpcygpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkxpc3RHcmFwaHFsQXBpcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSByZXNvbHZlcnMgZm9yIGEgZ2l2ZW4gQVBJIGFuZCB0eXBlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFJlc29sdmVycy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UmVzb2x2ZXJzKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6TGlzdFJlc29sdmVycycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3QgdGhlIHJlc29sdmVycyB0aGF0IGFyZSBhc3NvY2lhdGVkIHdpdGggYSBzcGVjaWZpYyBmdW5jdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBMaXN0XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RSZXNvbHZlcnNCeUZ1bmN0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RSZXNvbHZlcnNCeUZ1bmN0aW9uKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6TGlzdFJlc29sdmVyc0J5RnVuY3Rpb24nKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0IHRoZSB0YWdzIGZvciBhIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXNvdXJjZVRhZygpXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RUYWdzRm9yUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvTGlzdFRhZ3NGb3JSZXNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkxpc3RUYWdzRm9yUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgdHlwZXMgZm9yIGEgZ2l2ZW4gQVBJLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfTGlzdFR5cGVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RUeXBlcygpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOkxpc3RUeXBlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBuZXcgc2NoZW1hIHRvIHlvdXIgR3JhcGhRTCBBUEkuIFRoaXMgb3BlcmF0aW9uIGlzIGFzeW5jaHJvbm91cyAtIEdldFNjaGVtYUNyZWF0aW9uU3RhdHVzIGNhbiBzaG93IHdoZW4gaXQgaGFzIGNvbXBsZXRlZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9TdGFydFNjaGVtYUNyZWF0aW9uLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0U2NoZW1hQ3JlYXRpb24oKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpTdGFydFNjaGVtYUNyZWF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGFnIGEgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1JlcXVlc3RUYWcoKVxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVGFnUmVzb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVudGFnIGEgcmVzb3VyY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogVGFnZ2luZ1xuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1RhZ0tleXMoKVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VbnRhZ1Jlc291cmNlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpVbnRhZ1Jlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBBUEkga2V5IGZvciBhIGdpdmVuIEFQSS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVBcGlLZXkuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBpS2V5KCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6VXBkYXRlQXBpS2V5Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIERhdGFTb3VyY2Ugb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1VwZGF0ZURhdGFTb3VyY2UuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRGF0YVNvdXJjZSgpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOlVwZGF0ZURhdGFTb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGFuIGV4aXN0aW5nIEZ1bmN0aW9uIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVGdW5jdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVGdW5jdGlvbigpIHtcbiAgICB0aGlzLnRvKCdhcHBzeW5jOlVwZGF0ZUZ1bmN0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhIEdyYXBocWxBcGkgb2JqZWN0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIFBvc3NpYmxlIGNvbmRpdGlvbnM6XG4gICAqIC0gLmlmQXdzUmVzb3VyY2VUYWcoKVxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBpYW06Q3JlYXRlU2VydmljZUxpbmtlZFJvbGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlR3JhcGhxbEFwaS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVHcmFwaHFsQXBpKCkge1xuICAgIHRoaXMudG8oJ2FwcHN5bmM6VXBkYXRlR3JhcGhxbEFwaScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBSZXNvbHZlciBvYmplY3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlUmVzb2x2ZXIuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlUmVzb2x2ZXIoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpVcGRhdGVSZXNvbHZlcicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZXMgYSBUeXBlIG9iamVjdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYXBwc3luYy9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVUeXBlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVR5cGUoKSB7XG4gICAgdGhpcy50bygnYXBwc3luYzpVcGRhdGVUeXBlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgXCJXcml0ZVwiOiBbXG4gICAgICBcIkNyZWF0ZUFwaUtleVwiLFxuICAgICAgXCJDcmVhdGVEYXRhU291cmNlXCIsXG4gICAgICBcIkNyZWF0ZUZ1bmN0aW9uXCIsXG4gICAgICBcIkNyZWF0ZVJlc29sdmVyXCIsXG4gICAgICBcIkNyZWF0ZVR5cGVcIixcbiAgICAgIFwiRGVsZXRlQXBpS2V5XCIsXG4gICAgICBcIkRlbGV0ZURhdGFTb3VyY2VcIixcbiAgICAgIFwiRGVsZXRlRnVuY3Rpb25cIixcbiAgICAgIFwiRGVsZXRlR3JhcGhxbEFwaVwiLFxuICAgICAgXCJEZWxldGVSZXNvbHZlclwiLFxuICAgICAgXCJEZWxldGVUeXBlXCIsXG4gICAgICBcIkdyYXBoUUxcIixcbiAgICAgIFwiU3RhcnRTY2hlbWFDcmVhdGlvblwiLFxuICAgICAgXCJVcGRhdGVBcGlLZXlcIixcbiAgICAgIFwiVXBkYXRlRGF0YVNvdXJjZVwiLFxuICAgICAgXCJVcGRhdGVGdW5jdGlvblwiLFxuICAgICAgXCJVcGRhdGVHcmFwaHFsQXBpXCIsXG4gICAgICBcIlVwZGF0ZVJlc29sdmVyXCIsXG4gICAgICBcIlVwZGF0ZVR5cGVcIlxuICAgIF0sXG4gICAgXCJUYWdnaW5nXCI6IFtcbiAgICAgIFwiQ3JlYXRlR3JhcGhxbEFwaVwiLFxuICAgICAgXCJUYWdSZXNvdXJjZVwiLFxuICAgICAgXCJVbnRhZ1Jlc291cmNlXCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkdldERhdGFTb3VyY2VcIixcbiAgICAgIFwiR2V0RnVuY3Rpb25cIixcbiAgICAgIFwiR2V0R3JhcGhxbEFwaVwiLFxuICAgICAgXCJHZXRJbnRyb3NwZWN0aW9uU2NoZW1hXCIsXG4gICAgICBcIkdldFJlc29sdmVyXCIsXG4gICAgICBcIkdldFNjaGVtYUNyZWF0aW9uU3RhdHVzXCIsXG4gICAgICBcIkdldFR5cGVcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0QXBpS2V5c1wiLFxuICAgICAgXCJMaXN0RGF0YVNvdXJjZXNcIixcbiAgICAgIFwiTGlzdEZ1bmN0aW9uc1wiLFxuICAgICAgXCJMaXN0R3JhcGhxbEFwaXNcIixcbiAgICAgIFwiTGlzdFJlc29sdmVyc1wiLFxuICAgICAgXCJMaXN0UmVzb2x2ZXJzQnlGdW5jdGlvblwiLFxuICAgICAgXCJMaXN0VHlwZXNcIlxuICAgIF1cbiAgfTtcblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZGF0YXNvdXJjZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9hdHRhY2hpbmctYS1kYXRhLXNvdXJjZS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncmFwaFFMQVBJSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JhcGhRTEFQSUlkLlxuICAgKiBAcGFyYW0gZGF0YXNvdXJjZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZGF0YXNvdXJjZU5hbWUuXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25EYXRhc291cmNlKGdyYXBoUUxBUElJZDogc3RyaW5nLCBkYXRhc291cmNlTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTphcHBzeW5jOiR7UmVnaW9ufToke0FjY291bnR9OmFwaXMvJHtHcmFwaFFMQVBJSWR9L2RhdGFzb3VyY2VzLyR7RGF0YXNvdXJjZU5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtHcmFwaFFMQVBJSWR9JywgZ3JhcGhRTEFQSUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtEYXRhc291cmNlTmFtZX0nLCBkYXRhc291cmNlTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7QWNjb3VudH0nLCBhY2NvdW50IHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmVnaW9ufScsIHJlZ2lvbiB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1BhcnRpdGlvbn0nLCBwYXJ0aXRpb24gfHwgJ2F3cycpO1xuICAgIHJldHVybiB0aGlzLm9uKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHJlc291cmNlIG9mIHR5cGUgZ3JhcGhxbGFwaSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9hcHBzeW5jL2xhdGVzdC9kZXZndWlkZS9kZXNpZ25pbmctYS1ncmFwaHFsLWFwaS5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncmFwaFFMQVBJSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JhcGhRTEFQSUlkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvbkdyYXBocWxhcGkoZ3JhcGhRTEFQSUlkOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmFwcHN5bmM6JHtSZWdpb259OiR7QWNjb3VudH06YXBpcy8ke0dyYXBoUUxBUElJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0dyYXBoUUxBUElJZH0nLCBncmFwaFFMQVBJSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZpZWxkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL2NvbmZpZ3VyaW5nLXJlc29sdmVycy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncmFwaFFMQVBJSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JhcGhRTEFQSUlkLlxuICAgKiBAcGFyYW0gdHlwZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgdHlwZU5hbWUuXG4gICAqIEBwYXJhbSBmaWVsZE5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgZmllbGROYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uRmllbGQoZ3JhcGhRTEFQSUlkOiBzdHJpbmcsIHR5cGVOYW1lOiBzdHJpbmcsIGZpZWxkTmFtZTogc3RyaW5nLCBhY2NvdW50Pzogc3RyaW5nLCByZWdpb24/OiBzdHJpbmcsIHBhcnRpdGlvbj86IHN0cmluZykge1xuICAgIHZhciBhcm4gPSAnYXJuOiR7UGFydGl0aW9ufTphcHBzeW5jOiR7UmVnaW9ufToke0FjY291bnR9OmFwaXMvJHtHcmFwaFFMQVBJSWR9L3R5cGVzLyR7VHlwZU5hbWV9L2ZpZWxkcy8ke0ZpZWxkTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0dyYXBoUUxBUElJZH0nLCBncmFwaFFMQVBJSWQpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1R5cGVOYW1lfScsIHR5cGVOYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtGaWVsZE5hbWV9JywgZmllbGROYW1lKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSB0eXBlIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL2Rlc2lnbmluZy15b3VyLXNjaGVtYS5odG1sI2FkZGluZy1hLXJvb3QtcXVlcnktdHlwZVxuICAgKlxuICAgKiBAcGFyYW0gZ3JhcGhRTEFQSUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGdyYXBoUUxBUElJZC5cbiAgICogQHBhcmFtIHR5cGVOYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIHR5cGVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uVHlwZShncmFwaFFMQVBJSWQ6IHN0cmluZywgdHlwZU5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06YXBwc3luYzoke1JlZ2lvbn06JHtBY2NvdW50fTphcGlzLyR7R3JhcGhRTEFQSUlkfS90eXBlcy8ke1R5cGVOYW1lfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7R3JhcGhRTEFQSUlkfScsIGdyYXBoUUxBUElJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7VHlwZU5hbWV9JywgdHlwZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGZ1bmN0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FwcHN5bmMvbGF0ZXN0L2Rldmd1aWRlL3BpcGVsaW5lLXJlc29sdmVycy5odG1sXG4gICAqXG4gICAqIEBwYXJhbSBncmFwaFFMQVBJSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgZ3JhcGhRTEFQSUlkLlxuICAgKiBAcGFyYW0gZnVuY3Rpb25JZCAtIElkZW50aWZpZXIgZm9yIHRoZSBmdW5jdGlvbklkLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uRnVuY3Rpb24oZ3JhcGhRTEFQSUlkOiBzdHJpbmcsIGZ1bmN0aW9uSWQ6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06YXBwc3luYzoke1JlZ2lvbn06JHtBY2NvdW50fTphcGlzLyR7R3JhcGhRTEFQSUlkfS9mdW5jdGlvbnMvJHtGdW5jdGlvbklkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7R3JhcGhRTEFQSUlkfScsIGdyYXBoUUxBUElJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7RnVuY3Rpb25JZH0nLCBmdW5jdGlvbklkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50fScsIGFjY291bnQgfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtSZWdpb259JywgcmVnaW9uIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxufVxuIl19