"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Batch = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [batch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Batch extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [batch](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'batch';
        this.accessLevelList = {
            "Write": [
                "CancelJob",
                "CreateComputeEnvironment",
                "CreateJobQueue",
                "DeleteComputeEnvironment",
                "DeleteJobQueue",
                "DeregisterJobDefinition",
                "RegisterJobDefinition",
                "SubmitJob",
                "TerminateJob",
                "UpdateComputeEnvironment",
                "UpdateJobQueue"
            ],
            "Read": [
                "DescribeComputeEnvironments",
                "DescribeJobDefinitions",
                "DescribeJobQueues",
                "DescribeJobs"
            ],
            "List": [
                "ListJobs"
            ]
        };
    }
    /**
     * Cancels a job in an AWS Batch job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CancelJob.html
     */
    toCancelJob() {
        this.to('batch:CancelJob');
        return this;
    }
    /**
     * Creates an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateComputeEnvironment.html
     */
    toCreateComputeEnvironment() {
        this.to('batch:CreateComputeEnvironment');
        return this;
    }
    /**
     * Creates an AWS Batch job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_CreateJobQueue.html
     */
    toCreateJobQueue() {
        this.to('batch:CreateJobQueue');
        return this;
    }
    /**
     * Deletes an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteComputeEnvironment.html
     */
    toDeleteComputeEnvironment() {
        this.to('batch:DeleteComputeEnvironment');
        return this;
    }
    /**
     * Deletes the specified job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeleteJobQueue.html
     */
    toDeleteJobQueue() {
        this.to('batch:DeleteJobQueue');
        return this;
    }
    /**
     * Deregisters an AWS Batch job definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DeregisterJobDefinition.html
     */
    toDeregisterJobDefinition() {
        this.to('batch:DeregisterJobDefinition');
        return this;
    }
    /**
     * Describes one or more of your compute environments.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeComputeEnvironments.html
     */
    toDescribeComputeEnvironments() {
        this.to('batch:DescribeComputeEnvironments');
        return this;
    }
    /**
     * Describes a list of job definitions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobDefinitions.html
     */
    toDescribeJobDefinitions() {
        this.to('batch:DescribeJobDefinitions');
        return this;
    }
    /**
     * Describes one or more of your job queues.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobQueues.html
     */
    toDescribeJobQueues() {
        this.to('batch:DescribeJobQueues');
        return this;
    }
    /**
     * Describes a list of AWS Batch jobs.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_DescribeJobs.html
     */
    toDescribeJobs() {
        this.to('batch:DescribeJobs');
        return this;
    }
    /**
     * Returns a list of task jobs for a specified job queue.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_ListJobs.html
     */
    toListJobs() {
        this.to('batch:ListJobs');
        return this;
    }
    /**
     * Registers an AWS Batch job definition.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifUser()
     * - .ifPrivileged()
     * - .ifImage()
     * - .ifLogDriver()
     * - .ifAWSLogsGroup()
     * - .ifAWSLogsRegion()
     * - .ifAWSLogsStreamPrefix()
     * - .ifAWSLogsCreateGroup()
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_RegisterJobDefinition.html
     */
    toRegisterJobDefinition() {
        this.to('batch:RegisterJobDefinition');
        return this;
    }
    /**
     * Submits an AWS Batch job from a job definition.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_SubmitJob.html
     */
    toSubmitJob() {
        this.to('batch:SubmitJob');
        return this;
    }
    /**
     * Terminates jobs in a job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_TerminateJob.html
     */
    toTerminateJob() {
        this.to('batch:TerminateJob');
        return this;
    }
    /**
     * Updates an AWS Batch compute environment.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html
     */
    toUpdateComputeEnvironment() {
        this.to('batch:UpdateComputeEnvironment');
        return this;
    }
    /**
     * Updates a job queue.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateJobQueue.html
     */
    toUpdateJobQueue() {
        this.to('batch:UpdateJobQueue');
        return this;
    }
    /**
     * Adds a resource of type compute-environment to the statement
     *
     * @param computeEnvironmentName - Identifier for the computeEnvironmentName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onComputeEnvironment(computeEnvironmentName, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:compute-environment/${ComputeEnvironmentName}';
        arn = arn.replace('${ComputeEnvironmentName}', computeEnvironmentName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job-queue to the statement
     *
     * @param jobQueueName - Identifier for the jobQueueName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobQueue(jobQueueName, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:job-queue/${JobQueueName}';
        arn = arn.replace('${JobQueueName}', jobQueueName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job-definition to the statement
     *
     * @param jobDefinitionName - Identifier for the jobDefinitionName.
     * @param revision - Identifier for the revision.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onJobDefinition(jobDefinitionName, revision, account, region, partition) {
        var arn = 'arn:${Partition}:batch:${Region}:${Account}:job-definition/${JobDefinitionName}:${Revision}';
        arn = arn.replace('${JobDefinitionName}', jobDefinitionName);
        arn = arn.replace('${Revision}', revision);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * When this parameter is true, the awslogs-group will be created for the logs.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifAWSLogsCreateGroup(value) {
        return this.if(`batch:AWSLogsCreateGroup`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The awslogs group where the logs are located.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsGroup(value, operator) {
        return this.if(`batch:AWSLogsGroup`, value, operator || 'StringLike');
    }
    /**
     * The region where the logs are sent to.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsRegion(value, operator) {
        return this.if(`batch:AWSLogsRegion`, value, operator || 'StringLike');
    }
    /**
     * The awslogs log stream prefix.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAWSLogsStreamPrefix(value, operator) {
        return this.if(`batch:AWSLogsStreamPrefix`, value, operator || 'StringLike');
    }
    /**
     * The image used to start a container.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImage(value, operator) {
        return this.if(`batch:Image`, value, operator || 'StringLike');
    }
    /**
     * The log driver used for the container.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifLogDriver(value, operator) {
        return this.if(`batch:LogDriver`, value, operator || 'StringLike');
    }
    /**
     * When this parameter is true, the container is given elevated privileges on the host container instance (similar to the root user).
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value `true` or `false`. **Default:** `true`
     */
    ifPrivileged(value) {
        return this.if(`batch:Privileged`, (typeof value !== 'undefined' ? value : true), 'Bool');
    }
    /**
     * The user name or numeric uid to use inside the container.
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsbatch.html#awsbatch-policy-keys
     *
     * Applies to actions:
     * - .toRegisterJobDefinition()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifUser(value, operator) {
        return this.if(`batch:User`, value, operator || 'StringLike');
    }
}
exports.Batch = Batch;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzYmF0Y2guanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhd3NiYXRjaC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxzQ0FBc0Q7QUFFdEQ7Ozs7R0FJRztBQUNILE1BQWEsS0FBTSxTQUFRLHdCQUFlO0lBR3hDOzs7O09BSUc7SUFDSCxZQUFhLEdBQVk7UUFDdkIsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBUk4sa0JBQWEsR0FBRyxPQUFPLENBQUM7UUFxTnJCLG9CQUFlLEdBQW9CO1lBQzNDLE9BQU8sRUFBRTtnQkFDUCxXQUFXO2dCQUNYLDBCQUEwQjtnQkFDMUIsZ0JBQWdCO2dCQUNoQiwwQkFBMEI7Z0JBQzFCLGdCQUFnQjtnQkFDaEIseUJBQXlCO2dCQUN6Qix1QkFBdUI7Z0JBQ3ZCLFdBQVc7Z0JBQ1gsY0FBYztnQkFDZCwwQkFBMEI7Z0JBQzFCLGdCQUFnQjthQUNqQjtZQUNELE1BQU0sRUFBRTtnQkFDTiw2QkFBNkI7Z0JBQzdCLHdCQUF3QjtnQkFDeEIsbUJBQW1CO2dCQUNuQixjQUFjO2FBQ2Y7WUFDRCxNQUFNLEVBQUU7Z0JBQ04sVUFBVTthQUNYO1NBQ0YsQ0FBQztJQW5PRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksV0FBVztRQUNoQixJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDM0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztRQUMxQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnQkFBZ0I7UUFDckIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5QkFBeUI7UUFDOUIsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDZCQUE2QjtRQUNsQyxJQUFJLENBQUMsRUFBRSxDQUFDLG1DQUFtQyxDQUFDLENBQUM7UUFDN0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxtQkFBbUI7UUFDeEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsRUFBRSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDMUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7T0FnQkc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQyxDQUFDO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFdBQVc7UUFDaEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQzlCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDBCQUEwQjtRQUMvQixJQUFJLENBQUMsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCO1FBQ3JCLElBQUksQ0FBQyxFQUFFLENBQUMsc0JBQXNCLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUEyQkQ7Ozs7Ozs7T0FPRztJQUNJLG9CQUFvQixDQUFDLHNCQUE4QixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQy9HLElBQUksR0FBRyxHQUFHLDJGQUEyRixDQUFDO1FBQ3RHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLDJCQUEyQixFQUFFLHNCQUFzQixDQUFDLENBQUM7UUFDdkUsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7OztPQU9HO0lBQ0ksVUFBVSxDQUFDLFlBQW9CLEVBQUUsT0FBZ0IsRUFBRSxNQUFlLEVBQUUsU0FBa0I7UUFDM0YsSUFBSSxHQUFHLEdBQUcsdUVBQXVFLENBQUM7UUFDbEYsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsaUJBQWlCLEVBQUUsWUFBWSxDQUFDLENBQUM7UUFDbkQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsWUFBWSxFQUFFLE9BQU8sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQzlDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7Ozs7T0FRRztJQUNJLGVBQWUsQ0FBQyxpQkFBeUIsRUFBRSxRQUFnQixFQUFFLE9BQWdCLEVBQUUsTUFBZSxFQUFFLFNBQWtCO1FBQ3ZILElBQUksR0FBRyxHQUFHLDZGQUE2RixDQUFDO1FBQ3hHLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLHNCQUFzQixFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDN0QsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsYUFBYSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzNDLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLFlBQVksRUFBRSxPQUFPLElBQUksR0FBRyxDQUFDLENBQUM7UUFDaEQsR0FBRyxHQUFHLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE1BQU0sSUFBSSxHQUFHLENBQUMsQ0FBQztRQUM5QyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksb0JBQW9CLENBQUMsS0FBZTtRQUN6QyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDcEcsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxjQUFjLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUMxRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsb0JBQW9CLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUN4RSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLGVBQWUsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQzNFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxxQkFBcUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0kscUJBQXFCLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNqRixPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsMkJBQTJCLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLE9BQU8sQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ25FLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsS0FBSyxFQUFFLFFBQVEsSUFBSSxZQUFZLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7T0FVRztJQUNJLFdBQVcsQ0FBQyxLQUF3QixFQUFFLFFBQTRCO1FBQ3ZFLE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxLQUFLLEVBQUUsUUFBUSxJQUFJLFlBQVksQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZLENBQUMsS0FBZTtRQUNqQyxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLEVBQUUsQ0FBQyxPQUFPLEtBQUssS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsTUFBTSxDQUFDLENBQUM7SUFDNUYsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxNQUFNLENBQUMsS0FBd0IsRUFBRSxRQUE0QjtRQUNsRSxPQUFPLElBQUksQ0FBQyxFQUFFLENBQUMsWUFBWSxFQUFFLEtBQUssRUFBRSxRQUFRLElBQUksWUFBWSxDQUFDLENBQUM7SUFDaEUsQ0FBQztDQUNGO0FBelpELHNCQXlaQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEFjY2Vzc0xldmVsTGlzdCB9IGZyb20gXCIuLi9zaGFyZWQvYWNjZXNzLWxldmVsXCI7XG5pbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIE9wZXJhdG9yIH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYmF0Y2hdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2JhdGNoLmh0bWwpLlxuICpcbiAqIEBwYXJhbSBzaWQgW1NJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19zaWQuaHRtbCkgb2YgdGhlIHN0YXRlbWVudFxuICovXG5leHBvcnQgY2xhc3MgQmF0Y2ggZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdiYXRjaCc7XG5cbiAgLyoqXG4gICAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbYmF0Y2hdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2JhdGNoLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbmNlbHMgYSBqb2IgaW4gYW4gQVdTIEJhdGNoIGpvYiBxdWV1ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ2FuY2VsSm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0NhbmNlbEpvYigpIHtcbiAgICB0aGlzLnRvKCdiYXRjaDpDYW5jZWxKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGVzIGFuIEFXUyBCYXRjaCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9DcmVhdGVDb21wdXRlRW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlQ29tcHV0ZUVudmlyb25tZW50KCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkNyZWF0ZUNvbXB1dGVFbnZpcm9ubWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZXMgYW4gQVdTIEJhdGNoIGpvYiBxdWV1ZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfQ3JlYXRlSm9iUXVldWUuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlSm9iUXVldWUoKSB7XG4gICAgdGhpcy50bygnYmF0Y2g6Q3JlYXRlSm9iUXVldWUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGVzIGFuIEFXUyBCYXRjaCBjb21wdXRlIGVudmlyb25tZW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZWxldGVDb21wdXRlRW52aXJvbm1lbnQuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlQ29tcHV0ZUVudmlyb25tZW50KCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkRlbGV0ZUNvbXB1dGVFbnZpcm9ubWVudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIHNwZWNpZmllZCBqb2IgcXVldWUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0RlbGV0ZUpvYlF1ZXVlLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUpvYlF1ZXVlKCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkRlbGV0ZUpvYlF1ZXVlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVyZWdpc3RlcnMgYW4gQVdTIEJhdGNoIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXJlZ2lzdGVySm9iRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXJlZ2lzdGVySm9iRGVmaW5pdGlvbigpIHtcbiAgICB0aGlzLnRvKCdiYXRjaDpEZXJlZ2lzdGVySm9iRGVmaW5pdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyBvbmUgb3IgbW9yZSBvZiB5b3VyIGNvbXB1dGUgZW52aXJvbm1lbnRzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlQ29tcHV0ZUVudmlyb25tZW50cy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUNvbXB1dGVFbnZpcm9ubWVudHMoKSB7XG4gICAgdGhpcy50bygnYmF0Y2g6RGVzY3JpYmVDb21wdXRlRW52aXJvbm1lbnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGEgbGlzdCBvZiBqb2IgZGVmaW5pdGlvbnMuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfRGVzY3JpYmVKb2JEZWZpbml0aW9ucy5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZUpvYkRlZmluaXRpb25zKCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkRlc2NyaWJlSm9iRGVmaW5pdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgb25lIG9yIG1vcmUgb2YgeW91ciBqb2IgcXVldWVzLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0Rlc2NyaWJlSm9iUXVldWVzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlSm9iUXVldWVzKCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkRlc2NyaWJlSm9iUXVldWVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIGEgbGlzdCBvZiBBV1MgQmF0Y2ggam9icy5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9EZXNjcmliZUpvYnMuaHRtbFxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVKb2JzKCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkRlc2NyaWJlSm9icycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBsaXN0IG9mIHRhc2sgam9icyBmb3IgYSBzcGVjaWZpZWQgam9iIHF1ZXVlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX0xpc3RKb2JzLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RKb2JzKCkge1xuICAgIHRoaXMudG8oJ2JhdGNoOkxpc3RKb2JzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmVnaXN0ZXJzIGFuIEFXUyBCYXRjaCBqb2IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZlVzZXIoKVxuICAgKiAtIC5pZlByaXZpbGVnZWQoKVxuICAgKiAtIC5pZkltYWdlKClcbiAgICogLSAuaWZMb2dEcml2ZXIoKVxuICAgKiAtIC5pZkFXU0xvZ3NHcm91cCgpXG4gICAqIC0gLmlmQVdTTG9nc1JlZ2lvbigpXG4gICAqIC0gLmlmQVdTTG9nc1N0cmVhbVByZWZpeCgpXG4gICAqIC0gLmlmQVdTTG9nc0NyZWF0ZUdyb3VwKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1JlZ2lzdGVySm9iRGVmaW5pdGlvbi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9SZWdpc3RlckpvYkRlZmluaXRpb24oKSB7XG4gICAgdGhpcy50bygnYmF0Y2g6UmVnaXN0ZXJKb2JEZWZpbml0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogU3VibWl0cyBhbiBBV1MgQmF0Y2ggam9iIGZyb20gYSBqb2IgZGVmaW5pdGlvbi5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfU3VibWl0Sm9iLmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1N1Ym1pdEpvYigpIHtcbiAgICB0aGlzLnRvKCdiYXRjaDpTdWJtaXRKb2InKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBUZXJtaW5hdGVzIGpvYnMgaW4gYSBqb2IgcXVldWUuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2JhdGNoL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1Rlcm1pbmF0ZUpvYi5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9UZXJtaW5hdGVKb2IoKSB7XG4gICAgdGhpcy50bygnYmF0Y2g6VGVybWluYXRlSm9iJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlcyBhbiBBV1MgQmF0Y2ggY29tcHV0ZSBlbnZpcm9ubWVudC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfVXBkYXRlQ29tcHV0ZUVudmlyb25tZW50Lmh0bWxcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUNvbXB1dGVFbnZpcm9ubWVudCgpIHtcbiAgICB0aGlzLnRvKCdiYXRjaDpVcGRhdGVDb21wdXRlRW52aXJvbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGVzIGEgam9iIHF1ZXVlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvQVBJUmVmZXJlbmNlL0FQSV9VcGRhdGVKb2JRdWV1ZS5odG1sXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVKb2JRdWV1ZSgpIHtcbiAgICB0aGlzLnRvKCdiYXRjaDpVcGRhdGVKb2JRdWV1ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcHJvdGVjdGVkIGFjY2Vzc0xldmVsTGlzdDogQWNjZXNzTGV2ZWxMaXN0ID0ge1xuICAgIFwiV3JpdGVcIjogW1xuICAgICAgXCJDYW5jZWxKb2JcIixcbiAgICAgIFwiQ3JlYXRlQ29tcHV0ZUVudmlyb25tZW50XCIsXG4gICAgICBcIkNyZWF0ZUpvYlF1ZXVlXCIsXG4gICAgICBcIkRlbGV0ZUNvbXB1dGVFbnZpcm9ubWVudFwiLFxuICAgICAgXCJEZWxldGVKb2JRdWV1ZVwiLFxuICAgICAgXCJEZXJlZ2lzdGVySm9iRGVmaW5pdGlvblwiLFxuICAgICAgXCJSZWdpc3RlckpvYkRlZmluaXRpb25cIixcbiAgICAgIFwiU3VibWl0Sm9iXCIsXG4gICAgICBcIlRlcm1pbmF0ZUpvYlwiLFxuICAgICAgXCJVcGRhdGVDb21wdXRlRW52aXJvbm1lbnRcIixcbiAgICAgIFwiVXBkYXRlSm9iUXVldWVcIlxuICAgIF0sXG4gICAgXCJSZWFkXCI6IFtcbiAgICAgIFwiRGVzY3JpYmVDb21wdXRlRW52aXJvbm1lbnRzXCIsXG4gICAgICBcIkRlc2NyaWJlSm9iRGVmaW5pdGlvbnNcIixcbiAgICAgIFwiRGVzY3JpYmVKb2JRdWV1ZXNcIixcbiAgICAgIFwiRGVzY3JpYmVKb2JzXCJcbiAgICBdLFxuICAgIFwiTGlzdFwiOiBbXG4gICAgICBcIkxpc3RKb2JzXCJcbiAgICBdXG4gIH07XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGNvbXB1dGUtZW52aXJvbm1lbnQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gY29tcHV0ZUVudmlyb25tZW50TmFtZSAtIElkZW50aWZpZXIgZm9yIHRoZSBjb21wdXRlRW52aXJvbm1lbnROYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uQ29tcHV0ZUVudmlyb25tZW50KGNvbXB1dGVFbnZpcm9ubWVudE5hbWU6IHN0cmluZywgYWNjb3VudD86IHN0cmluZywgcmVnaW9uPzogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06YmF0Y2g6JHtSZWdpb259OiR7QWNjb3VudH06Y29tcHV0ZS1lbnZpcm9ubWVudC8ke0NvbXB1dGVFbnZpcm9ubWVudE5hbWV9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtDb21wdXRlRW52aXJvbm1lbnROYW1lfScsIGNvbXB1dGVFbnZpcm9ubWVudE5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYi1xdWV1ZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBqb2JRdWV1ZU5hbWUgLSBJZGVudGlmaWVyIGZvciB0aGUgam9iUXVldWVOYW1lLlxuICAgKiBAcGFyYW0gYWNjb3VudCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHJlZ2lvbiAtIFJlZ2lvbiBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIHJlZ2lvbnMuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uSm9iUXVldWUoam9iUXVldWVOYW1lOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmJhdGNoOiR7UmVnaW9ufToke0FjY291bnR9OmpvYi1xdWV1ZS8ke0pvYlF1ZXVlTmFtZX0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0pvYlF1ZXVlTmFtZX0nLCBqb2JRdWV1ZU5hbWUpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSByZXNvdXJjZSBvZiB0eXBlIGpvYi1kZWZpbml0aW9uIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25OYW1lIC0gSWRlbnRpZmllciBmb3IgdGhlIGpvYkRlZmluaXRpb25OYW1lLlxuICAgKiBAcGFyYW0gcmV2aXNpb24gLSBJZGVudGlmaWVyIGZvciB0aGUgcmV2aXNpb24uXG4gICAqIEBwYXJhbSBhY2NvdW50IC0gQWNjb3VudCBvZiB0aGUgcmVzb3VyY2U7IGRlZmF1bHRzIHRvIGVtcHR5IHN0cmluZzogYWxsIGFjY291bnRzLlxuICAgKiBAcGFyYW0gcmVnaW9uIC0gUmVnaW9uIG9mIHRoZSByZXNvdXJjZTsgZGVmYXVsdHMgdG8gZW1wdHkgc3RyaW5nOiBhbGwgcmVnaW9ucy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25Kb2JEZWZpbml0aW9uKGpvYkRlZmluaXRpb25OYW1lOiBzdHJpbmcsIHJldmlzaW9uOiBzdHJpbmcsIGFjY291bnQ/OiBzdHJpbmcsIHJlZ2lvbj86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OmJhdGNoOiR7UmVnaW9ufToke0FjY291bnR9OmpvYi1kZWZpbml0aW9uLyR7Sm9iRGVmaW5pdGlvbk5hbWV9OiR7UmV2aXNpb259JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtKb2JEZWZpbml0aW9uTmFtZX0nLCBqb2JEZWZpbml0aW9uTmFtZSk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UmV2aXNpb259JywgcmV2aXNpb24pO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0FjY291bnR9JywgYWNjb3VudCB8fCAnKicpO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke1JlZ2lvbn0nLCByZWdpb24gfHwgJyonKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIFdoZW4gdGhpcyBwYXJhbWV0ZXIgaXMgdHJ1ZSwgdGhlIGF3c2xvZ3MtZ3JvdXAgd2lsbCBiZSBjcmVhdGVkIGZvciB0aGUgbG9ncy5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIGB0cnVlYCBvciBgZmFsc2VgLiAqKkRlZmF1bHQ6KiogYHRydWVgXG4gICAqL1xuICBwdWJsaWMgaWZBV1NMb2dzQ3JlYXRlR3JvdXAodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGJhdGNoOkFXU0xvZ3NDcmVhdGVHcm91cGAsICh0eXBlb2YgdmFsdWUgIT09ICd1bmRlZmluZWQnID8gdmFsdWUgOiB0cnVlKSwgJ0Jvb2wnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgYXdzbG9ncyBncm91cCB3aGVyZSB0aGUgbG9ncyBhcmUgbG9jYXRlZC5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZBV1NMb2dzR3JvdXAodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGJhdGNoOkFXU0xvZ3NHcm91cGAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSByZWdpb24gd2hlcmUgdGhlIGxvZ3MgYXJlIHNlbnQgdG8uXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmQVdTTG9nc1JlZ2lvbih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYmF0Y2g6QVdTTG9nc1JlZ2lvbmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBhd3Nsb2dzIGxvZyBzdHJlYW0gcHJlZml4LlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkFXU0xvZ3NTdHJlYW1QcmVmaXgodmFsdWU6IHN0cmluZyB8IHN0cmluZ1tdLCBvcGVyYXRvcj86IE9wZXJhdG9yIHwgc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGJhdGNoOkFXU0xvZ3NTdHJlYW1QcmVmaXhgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgaW1hZ2UgdXNlZCB0byBzdGFydCBhIGNvbnRhaW5lci5cbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzYmF0Y2guaHRtbCNhd3NiYXRjaC1wb2xpY3kta2V5c1xuICAgKlxuICAgKiBBcHBsaWVzIHRvIGFjdGlvbnM6XG4gICAqIC0gLnRvUmVnaXN0ZXJKb2JEZWZpbml0aW9uKClcbiAgICpcbiAgICogQHBhcmFtIHZhbHVlIFRoZSB2YWx1ZShzKSB0byBjaGVja1xuICAgKiBAcGFyYW0gb3BlcmF0b3IgV29ya3Mgd2l0aCBbc3RyaW5nIG9wZXJhdG9yc10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL3JlZmVyZW5jZV9wb2xpY2llc19lbGVtZW50c19jb25kaXRpb25fb3BlcmF0b3JzLmh0bWwjQ29uZGl0aW9uc19TdHJpbmcpLiAqKkRlZmF1bHQ6KiogYFN0cmluZ0xpa2VgXG4gICAqL1xuICBwdWJsaWMgaWZJbWFnZSh2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYmF0Y2g6SW1hZ2VgLCB2YWx1ZSwgb3BlcmF0b3IgfHwgJ1N0cmluZ0xpa2UnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgbG9nIGRyaXZlciB1c2VkIGZvciB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgVGhlIHZhbHVlKHMpIHRvIGNoZWNrXG4gICAqIEBwYXJhbSBvcGVyYXRvciBXb3JrcyB3aXRoIFtzdHJpbmcgb3BlcmF0b3JzXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX2NvbmRpdGlvbl9vcGVyYXRvcnMuaHRtbCNDb25kaXRpb25zX1N0cmluZykuICoqRGVmYXVsdDoqKiBgU3RyaW5nTGlrZWBcbiAgICovXG4gIHB1YmxpYyBpZkxvZ0RyaXZlcih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYmF0Y2g6TG9nRHJpdmVyYCwgdmFsdWUsIG9wZXJhdG9yIHx8ICdTdHJpbmdMaWtlJyk7XG4gIH1cblxuICAvKipcbiAgICogV2hlbiB0aGlzIHBhcmFtZXRlciBpcyB0cnVlLCB0aGUgY29udGFpbmVyIGlzIGdpdmVuIGVsZXZhdGVkIHByaXZpbGVnZXMgb24gdGhlIGhvc3QgY29udGFpbmVyIGluc3RhbmNlIChzaW1pbGFyIHRvIHRoZSByb290IHVzZXIpLlxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3NiYXRjaC5odG1sI2F3c2JhdGNoLXBvbGljeS1rZXlzXG4gICAqXG4gICAqIEFwcGxpZXMgdG8gYWN0aW9uczpcbiAgICogLSAudG9SZWdpc3RlckpvYkRlZmluaXRpb24oKVxuICAgKlxuICAgKiBAcGFyYW0gdmFsdWUgYHRydWVgIG9yIGBmYWxzZWAuICoqRGVmYXVsdDoqKiBgdHJ1ZWBcbiAgICovXG4gIHB1YmxpYyBpZlByaXZpbGVnZWQodmFsdWU/OiBib29sZWFuKSB7XG4gICAgcmV0dXJuIHRoaXMuaWYoYGJhdGNoOlByaXZpbGVnZWRgLCAodHlwZW9mIHZhbHVlICE9PSAndW5kZWZpbmVkJyA/IHZhbHVlIDogdHJ1ZSksICdCb29sJyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHVzZXIgbmFtZSBvciBudW1lcmljIHVpZCB0byB1c2UgaW5zaWRlIHRoZSBjb250YWluZXIuXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9saXN0X2F3c2JhdGNoLmh0bWwjYXdzYmF0Y2gtcG9saWN5LWtleXNcbiAgICpcbiAgICogQXBwbGllcyB0byBhY3Rpb25zOlxuICAgKiAtIC50b1JlZ2lzdGVySm9iRGVmaW5pdGlvbigpXG4gICAqXG4gICAqIEBwYXJhbSB2YWx1ZSBUaGUgdmFsdWUocykgdG8gY2hlY2tcbiAgICogQHBhcmFtIG9wZXJhdG9yIFdvcmtzIHdpdGggW3N0cmluZyBvcGVyYXRvcnNdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfY29uZGl0aW9uX29wZXJhdG9ycy5odG1sI0NvbmRpdGlvbnNfU3RyaW5nKS4gKipEZWZhdWx0OioqIGBTdHJpbmdMaWtlYFxuICAgKi9cbiAgcHVibGljIGlmVXNlcih2YWx1ZTogc3RyaW5nIHwgc3RyaW5nW10sIG9wZXJhdG9yPzogT3BlcmF0b3IgfCBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5pZihgYmF0Y2g6VXNlcmAsIHZhbHVlLCBvcGVyYXRvciB8fCAnU3RyaW5nTGlrZScpO1xuICB9XG59XG4iXX0=