"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cloudformation = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [cloudformation](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudformation.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Cloudformation extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [cloudformation](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscloudformation.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'cloudformation';
        this.accessLevelList = {
            "Write": [
                "CancelUpdateStack",
                "ContinueUpdateRollback",
                "CreateChangeSet",
                "CreateStack",
                "CreateStackInstances",
                "CreateStackSet",
                "CreateUploadBucket",
                "DeleteChangeSet",
                "DeleteStack",
                "DeleteStackInstances",
                "DeleteStackSet",
                "DeregisterType",
                "ExecuteChangeSet",
                "RegisterType",
                "SetTypeDefaultVersion",
                "SignalResource",
                "StopStackSetOperation",
                "UpdateStack",
                "UpdateStackInstances",
                "UpdateStackSet",
                "UpdateTerminationProtection"
            ],
            "Read": [
                "DescribeAccountLimits",
                "DescribeChangeSet",
                "DescribeStackDriftDetectionStatus",
                "DescribeStackEvents",
                "DescribeStackInstance",
                "DescribeStackResource",
                "DescribeStackResourceDrifts",
                "DescribeStackResources",
                "DescribeStackSet",
                "DescribeStackSetOperation",
                "DescribeType",
                "DescribeTypeRegistration",
                "DetectStackDrift",
                "DetectStackResourceDrift",
                "DetectStackSetDrift",
                "EstimateTemplateCost",
                "GetStackPolicy",
                "GetTemplate",
                "GetTemplateSummary",
                "ValidateTemplate"
            ],
            "List": [
                "DescribeStacks",
                "ListChangeSets",
                "ListExports",
                "ListImports",
                "ListStackInstances",
                "ListStackResources",
                "ListStackSetOperationResults",
                "ListStackSetOperations",
                "ListStackSets",
                "ListStacks",
                "ListTypeRegistrations",
                "ListTypeVersions",
                "ListTypes"
            ],
            "Permissions management": [
                "SetStackPolicy"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Cancels an update on the specified stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CancelUpdateStack.html
     */
    toCancelUpdateStack() {
        this.to('cloudformation:CancelUpdateStack');
        return this;
    }
    /**
     * For a specified stack that is in the UPDATE_ROLLBACK_FAILED state, continues rolling it back to the UPDATE_ROLLBACK_COMPLETE state.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ContinueUpdateRollback.html
     */
    toContinueUpdateRollback() {
        this.to('cloudformation:ContinueUpdateRollback');
        return this;
    }
    /**
     * Creates a list of changes for a stack.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     * - .ifResourceTypes()
     * - .ifImportResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateChangeSet.html
     */
    toCreateChangeSet() {
        this.to('cloudformation:CreateChangeSet');
        return this;
    }
    /**
     * Creates a stack as specified in the template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStack.html
     */
    toCreateStack() {
        this.to('cloudformation:CreateStack');
        return this;
    }
    /**
     * Creates stack instances for the specified accounts, within the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackInstances.html
     */
    toCreateStackInstances() {
        this.to('cloudformation:CreateStackInstances');
        return this;
    }
    /**
     * Creates a stackset as specified in the template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_CreateStackSet.html
     */
    toCreateStackSet() {
        this.to('cloudformation:CreateStackSet');
        return this;
    }
    /**
     *
     *
     * Access Level: Write
     */
    toCreateUploadBucket() {
        this.to('cloudformation:CreateUploadBucket');
        return this;
    }
    /**
     * Deletes the specified change set. Deleting change sets ensures that no one executes the wrong change set.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteChangeSet.html
     */
    toDeleteChangeSet() {
        this.to('cloudformation:DeleteChangeSet');
        return this;
    }
    /**
     * Deletes a specified stack.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStack.html
     */
    toDeleteStack() {
        this.to('cloudformation:DeleteStack');
        return this;
    }
    /**
     * Deletes stack instances for the specified accounts, in the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackInstances.html
     */
    toDeleteStackInstances() {
        this.to('cloudformation:DeleteStackInstances');
        return this;
    }
    /**
     * Deletes a specified stackset.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeleteStackSet.html
     */
    toDeleteStackSet() {
        this.to('cloudformation:DeleteStackSet');
        return this;
    }
    /**
     * Deregisters an existing CloudFormation type or type version
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DeregisterType.html
     */
    toDeregisterType() {
        this.to('cloudformation:DeregisterType');
        return this;
    }
    /**
     * Retrieves your account's AWS CloudFormation limits.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeAccountLimits.html
     */
    toDescribeAccountLimits() {
        this.to('cloudformation:DescribeAccountLimits');
        return this;
    }
    /**
     * Returns the description for the specified change set.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeChangeSet.html
     */
    toDescribeChangeSet() {
        this.to('cloudformation:DescribeChangeSet');
        return this;
    }
    /**
     * Returns information about a stack drift detection operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackDriftDetectionStatus.html
     */
    toDescribeStackDriftDetectionStatus() {
        this.to('cloudformation:DescribeStackDriftDetectionStatus');
        return this;
    }
    /**
     * Returns all stack related events for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackEvents.html
     */
    toDescribeStackEvents() {
        this.to('cloudformation:DescribeStackEvents');
        return this;
    }
    /**
     * Returns the stack instance that's associated with the specified stack set, AWS account, and region.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackInstance.html
     */
    toDescribeStackInstance() {
        this.to('cloudformation:DescribeStackInstance');
        return this;
    }
    /**
     * Returns a description of the specified resource in the specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResource.html
     */
    toDescribeStackResource() {
        this.to('cloudformation:DescribeStackResource');
        return this;
    }
    /**
     * Returns drift information for the resources that have been checked for drift in the specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResourceDrifts.html
     */
    toDescribeStackResourceDrifts() {
        this.to('cloudformation:DescribeStackResourceDrifts');
        return this;
    }
    /**
     * Returns AWS resource descriptions for running and deleted stacks.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackResources.html
     */
    toDescribeStackResources() {
        this.to('cloudformation:DescribeStackResources');
        return this;
    }
    /**
     * Returns the description of the specified stack set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSet.html
     */
    toDescribeStackSet() {
        this.to('cloudformation:DescribeStackSet');
        return this;
    }
    /**
     * Returns the description of the specified stack set operation.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStackSetOperation.html
     */
    toDescribeStackSetOperation() {
        this.to('cloudformation:DescribeStackSetOperation');
        return this;
    }
    /**
     * Returns the description for the specified stack.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeStacks.html
     */
    toDescribeStacks() {
        this.to('cloudformation:DescribeStacks');
        return this;
    }
    /**
     * Returns information about the CloudFormation type requested
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeType.html
     */
    toDescribeType() {
        this.to('cloudformation:DescribeType');
        return this;
    }
    /**
     * Returns information about the registration process for a CloudFormation type
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DescribeTypeRegistration.html
     */
    toDescribeTypeRegistration() {
        this.to('cloudformation:DescribeTypeRegistration');
        return this;
    }
    /**
     * Detects whether a stack's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackDrift.html
     */
    toDetectStackDrift() {
        this.to('cloudformation:DetectStackDrift');
        return this;
    }
    /**
     * Returns information about whether a resource's actual configuration differs, or has drifted, from it's expected configuration, as defined in the stack template and any values specified as template parameters.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackResourceDrift.html
     */
    toDetectStackResourceDrift() {
        this.to('cloudformation:DetectStackResourceDrift');
        return this;
    }
    /**
     * Enables users to detect drift on a stack set and the stack instances that belong to that stack set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_DetectStackSetDrift.html
     */
    toDetectStackSetDrift() {
        this.to('cloudformation:DetectStackSetDrift');
        return this;
    }
    /**
     * Returns the estimated monthly cost of a template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_EstimateTemplateCost.html
     */
    toEstimateTemplateCost() {
        this.to('cloudformation:EstimateTemplateCost');
        return this;
    }
    /**
     * Updates a stack using the input information that was provided when the specified change set was created.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifChangeSetName()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ExecuteChangeSet.html
     */
    toExecuteChangeSet() {
        this.to('cloudformation:ExecuteChangeSet');
        return this;
    }
    /**
     * Returns the stack policy for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetStackPolicy.html
     */
    toGetStackPolicy() {
        this.to('cloudformation:GetStackPolicy');
        return this;
    }
    /**
     * Returns the template body for a specified stack.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplate.html
     */
    toGetTemplate() {
        this.to('cloudformation:GetTemplate');
        return this;
    }
    /**
     * Returns information about a new or existing template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_GetTemplateSummary.html
     */
    toGetTemplateSummary() {
        this.to('cloudformation:GetTemplateSummary');
        return this;
    }
    /**
     * Returns the ID and status of each active change set for a stack. For example, AWS CloudFormation lists change sets that are in the CREATE_IN_PROGRESS or CREATE_PENDING state.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListChangeSets.html
     */
    toListChangeSets() {
        this.to('cloudformation:ListChangeSets');
        return this;
    }
    /**
     * Lists all exported output values in the account and region in which you call this action.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListExports.html
     */
    toListExports() {
        this.to('cloudformation:ListExports');
        return this;
    }
    /**
     * Lists all stacks that are importing an exported output value.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListImports.html
     */
    toListImports() {
        this.to('cloudformation:ListImports');
        return this;
    }
    /**
     * Returns summary information about stack instances that are associated with the specified stack set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackInstances() {
        this.to('cloudformation:ListStackInstances');
        return this;
    }
    /**
     * Returns descriptions of all resources of the specified stack.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackResources.html
     */
    toListStackResources() {
        this.to('cloudformation:ListStackResources');
        return this;
    }
    /**
     * Returns summary information about the results of a stack set operation.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperationResults.html
     */
    toListStackSetOperationResults() {
        this.to('cloudformation:ListStackSetOperationResults');
        return this;
    }
    /**
     * Returns summary information about operations performed on a stack set.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSetOperations.html
     */
    toListStackSetOperations() {
        this.to('cloudformation:ListStackSetOperations');
        return this;
    }
    /**
     * Returns summary information about stack sets that are associated with the user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStackSets.html
     */
    toListStackSets() {
        this.to('cloudformation:ListStackSets');
        return this;
    }
    /**
     * Returns the summary information for stacks whose status matches the specified StackStatusFilter.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListStacks.html
     */
    toListStacks() {
        this.to('cloudformation:ListStacks');
        return this;
    }
    /**
     * Lists CloudFormation type registration attempts
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeRegistrations.html
     */
    toListTypeRegistrations() {
        this.to('cloudformation:ListTypeRegistrations');
        return this;
    }
    /**
     * Lists versions of a particular CloudFormation type
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypeVersions.html
     */
    toListTypeVersions() {
        this.to('cloudformation:ListTypeVersions');
        return this;
    }
    /**
     * Lists available CloudFormation types
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ListTypes.html
     */
    toListTypes() {
        this.to('cloudformation:ListTypes');
        return this;
    }
    /**
     * Registers a new CloudFormation type
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_RegisterType.html
     */
    toRegisterType() {
        this.to('cloudformation:RegisterType');
        return this;
    }
    /**
     * Sets a stack policy for a specified stack.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifStackPolicyUrl()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetStackPolicy.html
     */
    toSetStackPolicy() {
        this.to('cloudformation:SetStackPolicy');
        return this;
    }
    /**
     * Sets which version of a CloudFormation type applies to CloudFormation operations
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SetTypeDefaultVersion.html
     */
    toSetTypeDefaultVersion() {
        this.to('cloudformation:SetTypeDefaultVersion');
        return this;
    }
    /**
     * Sends a signal to the specified resource with a success or failure status.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_SignalResource.html
     */
    toSignalResource() {
        this.to('cloudformation:SignalResource');
        return this;
    }
    /**
     * Stops an in-progress operation on a stack set and its associated stack instances.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_StopStackSetOperation.html
     */
    toStopStackSetOperation() {
        this.to('cloudformation:StopStackSetOperation');
        return this;
    }
    /**
     * Tagging cloudformation resources.
     *
     * Access Level: Tagging
     */
    toTagResource() {
        this.to('cloudformation:TagResource');
        return this;
    }
    /**
     * Untagging cloudformation resources.
     *
     * Access Level: Tagging
     */
    toUntagResource() {
        this.to('cloudformation:UntagResource');
        return this;
    }
    /**
     * Updates a stack as specified in the template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifResourceTypes()
     * - .ifRoleArn()
     * - .ifStackPolicyUrl()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStack.html
     */
    toUpdateStack() {
        this.to('cloudformation:UpdateStack');
        return this;
    }
    /**
     * Updates the parameter values for stack instances for the specified accounts, within the specified regions.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackInstances.html
     */
    toUpdateStackInstances() {
        this.to('cloudformation:UpdateStackInstances');
        return this;
    }
    /**
     * Updates a stackset as specified in the template.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifRoleArn()
     * - .ifTemplateUrl()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateStackSet.html
     */
    toUpdateStackSet() {
        this.to('cloudformation:UpdateStackSet');
        return this;
    }
    /**
     * Updates termination protection for the specified stack.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_UpdateTerminationProtection.html
     */
    toUpdateTerminationProtection() {
        this.to('cloudformation:UpdateTerminationProtection');
        return this;
    }
    /**
     * Validates a specified template.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_ValidateTemplate.html
     */
    toValidateTemplate() {
        this.to('cloudformation:ValidateTemplate');
        return this;
    }
    /**
     * Adds a resource of type stack to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15b9
     *
     * @param stackName - Identifier for the stackName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStack(stackName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:stack/${StackName}/${Id}';
        arn = arn.replace('${StackName}', stackName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type stackset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/stacksets-concepts.html#stacksets-concepts-stackset
     *
     * @param stackSetName - Identifier for the stackSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onStackset(stackSetName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:stackset/${StackSetName}:${Id}';
        arn = arn.replace('${StackSetName}', stackSetName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type changeset to the statement
     *
     * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-whatis-concepts.html#w2ab1b5c15c11
     *
     * @param changeSetName - Identifier for the changeSetName.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChangeset(changeSetName, id, account, region, partition) {
        var arn = 'arn:${Partition}:cloudformation:${Region}:${Account}:changeSet/${ChangeSetName}:${Id}';
        arn = arn.replace('${ChangeSetName}', changeSetName);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * An AWS CloudFormation change set name. Use to control which change sets IAM users can execute or delete.
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toDeleteChangeSet()
     * - .toDescribeChangeSet()
     * - .toExecuteChangeSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifChangeSetName(value, operator) {
        return this.if(`cloudformation:ChangeSetName`, value, operator || 'StringLike');
    }
    /**
     * The template resource types, such as `AWS::EC2::Instance`. Use to control which resource types IAM users can work with when they want to import a resource into a stack.
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifImportResourceTypes(value, operator) {
        return this.if(`cloudformation:ImportResourceTypes`, value, operator || 'StringLike');
    }
    /**
     * The template resource types, such as `AWS::EC2::Instance`. Use to control which resource types IAM users can work with when they create or update a stack.
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifResourceTypes(value, operator) {
        return this.if(`cloudformation:ResourceTypes`, value, operator || 'StringLike');
    }
    /**
     * The ARN of an IAM service role. Use to control which service role IAM users can use to work with stacks or change sets.
     *
     * Applies to actions:
     * - .toContinueUpdateRollback()
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toDeleteStack()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [arn operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_ARN). **Default:** `ArnLike`
     */
    ifRoleArn(value, operator) {
        return this.if(`cloudformation:RoleArn`, value, operator || 'ArnLike');
    }
    /**
     * An Amazon S3 stack policy URL. Use to control which stack policies IAM users can associate with a stack during a create or update stack action.
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toSetStackPolicy()
     * - .toUpdateStack()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStackPolicyUrl(value, operator) {
        return this.if(`cloudformation:StackPolicyUrl`, value, operator || 'StringLike');
    }
    /**
     * An Amazon S3 template URL. Use to control which templates IAM users can use when they create or update stacks.
     *
     * Applies to actions:
     * - .toCreateChangeSet()
     * - .toCreateStack()
     * - .toCreateStackSet()
     * - .toUpdateStack()
     * - .toUpdateStackSet()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifTemplateUrl(value, operator) {
        return this.if(`cloudformation:TemplateUrl`, value, operator || 'StringLike');
    }
}
exports.Cloudformation = Cloudformation;
//# sourceMappingURL=data:application/json;base64,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