"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Codedeploy = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [codedeploy](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodedeploy.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Codedeploy extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [codedeploy](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscodedeploy.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'codedeploy';
        this.accessLevelList = {
            "Tagging": [
                "AddTagsToOnPremisesInstances",
                "RemoveTagsFromOnPremisesInstances",
                "TagResource",
                "UntagResource"
            ],
            "Read": [
                "BatchGetApplicationRevisions",
                "BatchGetApplications",
                "BatchGetDeploymentGroups",
                "BatchGetDeploymentInstances",
                "BatchGetDeploymentTargets",
                "BatchGetDeployments",
                "BatchGetOnPremisesInstances",
                "GetDeploymentTarget"
            ],
            "Write": [
                "ContinueDeployment",
                "CreateApplication",
                "CreateCloudFormationDeployment",
                "CreateDeployment",
                "CreateDeploymentConfig",
                "CreateDeploymentGroup",
                "DeleteApplication",
                "DeleteDeploymentConfig",
                "DeleteDeploymentGroup",
                "DeleteGitHubAccountToken",
                "DeleteResourcesByExternalId",
                "DeregisterOnPremisesInstance",
                "PutLifecycleEventHookExecutionStatus",
                "RegisterApplicationRevision",
                "RegisterOnPremisesInstance",
                "SkipWaitTimeForInstanceTermination",
                "StopDeployment",
                "UpdateApplication",
                "UpdateDeploymentGroup"
            ],
            "List": [
                "GetApplication",
                "GetApplicationRevision",
                "GetDeployment",
                "GetDeploymentConfig",
                "GetDeploymentGroup",
                "GetDeploymentInstance",
                "GetOnPremisesInstance",
                "ListApplicationRevisions",
                "ListApplications",
                "ListDeploymentConfigs",
                "ListDeploymentGroups",
                "ListDeploymentInstances",
                "ListDeploymentTargets",
                "ListDeployments",
                "ListGitHubAccountTokenNames",
                "ListOnPremisesInstances",
                "ListTagsForResource"
            ]
        };
    }
    /**
     * Add tags to one or more on-premises instances.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_AddTagsToOnPremisesInstances.html
     */
    toAddTagsToOnPremisesInstances() {
        this.to('codedeploy:AddTagsToOnPremisesInstances');
        return this;
    }
    /**
     * Gets information about one or more application revisions.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplicationRevisions.html
     */
    toBatchGetApplicationRevisions() {
        this.to('codedeploy:BatchGetApplicationRevisions');
        return this;
    }
    /**
     * Get information about multiple applications associated with the IAM user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetApplications.html
     */
    toBatchGetApplications() {
        this.to('codedeploy:BatchGetApplications');
        return this;
    }
    /**
     * Get information about one or more deployment groups.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentGroups.html
     */
    toBatchGetDeploymentGroups() {
        this.to('codedeploy:BatchGetDeploymentGroups');
        return this;
    }
    /**
     * Gets information about one or more instance that are part of a deployment group.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentInstances.html
     */
    toBatchGetDeploymentInstances() {
        this.to('codedeploy:BatchGetDeploymentInstances');
        return this;
    }
    /**
     * Returns an array of one or more targets associated with a deployment. This method works with all compute types and should be used instead of the deprecated BatchGetDeploymentInstances. The maximum number of targets that can be returned is 25.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeploymentTargets.html
     */
    toBatchGetDeploymentTargets() {
        this.to('codedeploy:BatchGetDeploymentTargets');
        return this;
    }
    /**
     * Get information about multiple deployments associated with the IAM user.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetDeployments.html
     */
    toBatchGetDeployments() {
        this.to('codedeploy:BatchGetDeployments');
        return this;
    }
    /**
     * Get information about one or more on-premises instances.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_BatchGetOnPremisesInstances.html
     */
    toBatchGetOnPremisesInstances() {
        this.to('codedeploy:BatchGetOnPremisesInstances');
        return this;
    }
    /**
     * Starts the process of rerouting traffic from instances in the original environment to instances in thereplacement environment without waiting for a specified wait time to elapse.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ContinueDeployment.html
     */
    toContinueDeployment() {
        this.to('codedeploy:ContinueDeployment');
        return this;
    }
    /**
     * Create an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateApplication.html
     */
    toCreateApplication() {
        this.to('codedeploy:CreateApplication');
        return this;
    }
    /**
     * Create CloudFormation deployment to cooperate ochestration for a CloudFormation stack update.
     *
     * Access Level: Write
     */
    toCreateCloudFormationDeployment() {
        this.to('codedeploy:CreateCloudFormationDeployment');
        return this;
    }
    /**
     * Create a deployment for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeployment.html
     */
    toCreateDeployment() {
        this.to('codedeploy:CreateDeployment');
        return this;
    }
    /**
     * Create a custom deployment configuration associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentConfig.html
     */
    toCreateDeploymentConfig() {
        this.to('codedeploy:CreateDeploymentConfig');
        return this;
    }
    /**
     * Create a deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_CreateDeploymentGroup.html
     */
    toCreateDeploymentGroup() {
        this.to('codedeploy:CreateDeploymentGroup');
        return this;
    }
    /**
     * Delete an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteApplication.html
     */
    toDeleteApplication() {
        this.to('codedeploy:DeleteApplication');
        return this;
    }
    /**
     * Delete a custom deployment configuration associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentConfig.html
     */
    toDeleteDeploymentConfig() {
        this.to('codedeploy:DeleteDeploymentConfig');
        return this;
    }
    /**
     * Delete a deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteDeploymentGroup.html
     */
    toDeleteDeploymentGroup() {
        this.to('codedeploy:DeleteDeploymentGroup');
        return this;
    }
    /**
     * Deletes a GitHub account connection.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteGitHubAccountToken.html
     */
    toDeleteGitHubAccountToken() {
        this.to('codedeploy:DeleteGitHubAccountToken');
        return this;
    }
    /**
     * Delete resources associated with the given external Id.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeleteResourcesByExternalId.html
     */
    toDeleteResourcesByExternalId() {
        this.to('codedeploy:DeleteResourcesByExternalId');
        return this;
    }
    /**
     * Deregister an on-premises instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_DeregisterOnPremisesInstance.html
     */
    toDeregisterOnPremisesInstance() {
        this.to('codedeploy:DeregisterOnPremisesInstance');
        return this;
    }
    /**
     * Get information about a single application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplication.html
     */
    toGetApplication() {
        this.to('codedeploy:GetApplication');
        return this;
    }
    /**
     * Get information about a single application revision for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetApplicationRevision.html
     */
    toGetApplicationRevision() {
        this.to('codedeploy:GetApplicationRevision');
        return this;
    }
    /**
     * Get information about a single deployment to a deployment group for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeployment.html
     */
    toGetDeployment() {
        this.to('codedeploy:GetDeployment');
        return this;
    }
    /**
     * Get information about a single deployment configuration associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentConfig.html
     */
    toGetDeploymentConfig() {
        this.to('codedeploy:GetDeploymentConfig');
        return this;
    }
    /**
     * Get information about a single deployment group for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentGroup.html
     */
    toGetDeploymentGroup() {
        this.to('codedeploy:GetDeploymentGroup');
        return this;
    }
    /**
     * Get information about a single instance in a deployment associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentInstance.html
     */
    toGetDeploymentInstance() {
        this.to('codedeploy:GetDeploymentInstance');
        return this;
    }
    /**
     * Returns information about a deployment target.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetDeploymentTarget.html
     */
    toGetDeploymentTarget() {
        this.to('codedeploy:GetDeploymentTarget');
        return this;
    }
    /**
     * Get information about a single on-premises instance.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_GetOnPremisesInstance.html
     */
    toGetOnPremisesInstance() {
        this.to('codedeploy:GetOnPremisesInstance');
        return this;
    }
    /**
     * Get information about all application revisions for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplicationRevisions.html
     */
    toListApplicationRevisions() {
        this.to('codedeploy:ListApplicationRevisions');
        return this;
    }
    /**
     * Get information about all applications associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListApplications.html
     */
    toListApplications() {
        this.to('codedeploy:ListApplications');
        return this;
    }
    /**
     * Get information about all deployment configurations associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentConfigs.html
     */
    toListDeploymentConfigs() {
        this.to('codedeploy:ListDeploymentConfigs');
        return this;
    }
    /**
     * Get information about all deployment groups for an application associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentGroups.html
     */
    toListDeploymentGroups() {
        this.to('codedeploy:ListDeploymentGroups');
        return this;
    }
    /**
     * Get information about all instances in a deployment associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentInstances.html
     */
    toListDeploymentInstances() {
        this.to('codedeploy:ListDeploymentInstances');
        return this;
    }
    /**
     * Returns an array of target IDs that are associated a deployment.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeploymentTargets.html
     */
    toListDeploymentTargets() {
        this.to('codedeploy:ListDeploymentTargets');
        return this;
    }
    /**
     * Get information about all deployments to a deployment group associated with the IAM user, or to get all deployments associated with the IAM user.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListDeployments.html
     */
    toListDeployments() {
        this.to('codedeploy:ListDeployments');
        return this;
    }
    /**
     * Lists the names of stored connections to GitHub accounts.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListGitHubAccountTokenNames.html
     */
    toListGitHubAccountTokenNames() {
        this.to('codedeploy:ListGitHubAccountTokenNames');
        return this;
    }
    /**
     * Get a list of one or more on-premises instance names.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ListOnPremisesInstances.html
     */
    toListOnPremisesInstances() {
        this.to('codedeploy:ListOnPremisesInstances');
        return this;
    }
    /**
     * Returns a list of tags for the resource identified by a specified ARN. Tags are used to organize and categorize your CodeDeploy resources.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('codedeploy:ListTagsForResource');
        return this;
    }
    /**
     * Notify a lifecycle event hook execution status for associated deployment with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_PutLifecycleEventHookExecutionStatus.html
     */
    toPutLifecycleEventHookExecutionStatus() {
        this.to('codedeploy:PutLifecycleEventHookExecutionStatus');
        return this;
    }
    /**
     * Register information about an application revision for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterApplicationRevision.html
     */
    toRegisterApplicationRevision() {
        this.to('codedeploy:RegisterApplicationRevision');
        return this;
    }
    /**
     * Register an on-premises instance.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RegisterOnPremisesInstance.html
     */
    toRegisterOnPremisesInstance() {
        this.to('codedeploy:RegisterOnPremisesInstance');
        return this;
    }
    /**
     * Remove tags from one or more on-premises instances.
     *
     * Access Level: Tagging
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_RemoveTagsFromOnPremisesInstances.html
     */
    toRemoveTagsFromOnPremisesInstances() {
        this.to('codedeploy:RemoveTagsFromOnPremisesInstances');
        return this;
    }
    /**
     * In a blue/green deployment, overrides any specified wait time and starts terminating instances immediately after the traffic routing is complete.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_ SkipWaitTimeForInstanceTermination.html
     */
    toSkipWaitTimeForInstanceTermination() {
        this.to('codedeploy:SkipWaitTimeForInstanceTermination');
        return this;
    }
    /**
     * Description for StopDeployment
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_StopDeployment.html
     */
    toStopDeployment() {
        this.to('codedeploy:StopDeployment');
        return this;
    }
    /**
     * Associates the list of tags in the input Tags parameter with the resource identified by the ResourceArn input parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('codedeploy:TagResource');
        return this;
    }
    /**
     * Disassociates a resource from a list of tags. The resource is identified by the ResourceArn input parameter. The tags are identfied by the list of keys in the TagKeys input parameter.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('codedeploy:UntagResource');
        return this;
    }
    /**
     * Description for UpdateApplication
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateApplication.html
     */
    toUpdateApplication() {
        this.to('codedeploy:UpdateApplication');
        return this;
    }
    /**
     * Change information about a single deployment group for an application associated with the IAM user.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/codedeploy/latest/APIReference/API_UpdateDeploymentGroup.html
     */
    toUpdateDeploymentGroup() {
        this.to('codedeploy:UpdateDeploymentGroup');
        return this;
    }
    /**
     * Adds a resource of type application to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApplication(applicationName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:application:${ApplicationName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentconfig to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param deploymentConfigurationName - Identifier for the deploymentConfigurationName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeploymentconfig(deploymentConfigurationName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:deploymentconfig:${DeploymentConfigurationName}';
        arn = arn.replace('${DeploymentConfigurationName}', deploymentConfigurationName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type deploymentgroup to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param applicationName - Identifier for the applicationName.
     * @param deploymentGroupName - Identifier for the deploymentGroupName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDeploymentgroup(applicationName, deploymentGroupName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:deploymentgroup:${ApplicationName}/${DeploymentGroupName}';
        arn = arn.replace('${ApplicationName}', applicationName);
        arn = arn.replace('${DeploymentGroupName}', deploymentGroupName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/codedeploy/latest/userguide/auth-and-access-control-permissions-reference.html
     *
     * @param instanceName - Identifier for the instanceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceName, account, region, partition) {
        var arn = 'arn:${Partition}:codedeploy:${Region}:${Account}:instance:${InstanceName}';
        arn = arn.replace('${InstanceName}', instanceName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Codedeploy = Codedeploy;
//# sourceMappingURL=data:application/json;base64,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