"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ce = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [ce](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscostexplorerservice.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Ce extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [ce](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awscostexplorerservice.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'ce';
        this.accessLevelList = {
            "Write": [
                "CreateAnomalyMonitor",
                "CreateAnomalySubscription",
                "CreateCostCategoryDefinition",
                "DeleteAnomalyMonitor",
                "DeleteAnomalySubscription",
                "DeleteCostCategoryDefinition",
                "ProvideAnomalyFeedback",
                "UpdateAnomalyMonitor",
                "UpdateAnomalySubscription",
                "UpdateCostCategoryDefinition"
            ],
            "Read": [
                "DescribeCostCategoryDefinition",
                "GetAnomalies",
                "GetAnomalyMonitors",
                "GetAnomalySubscriptions",
                "GetCostAndUsage",
                "GetCostAndUsageWithResources",
                "GetCostForecast",
                "GetDimensionValues",
                "GetReservationCoverage",
                "GetReservationPurchaseRecommendation",
                "GetReservationUtilization",
                "GetRightsizingRecommendation",
                "GetSavingsPlansCoverage",
                "GetSavingsPlansPurchaseRecommendation",
                "GetSavingsPlansUtilization",
                "GetSavingsPlansUtilizationDetails",
                "GetTags",
                "GetUsageForecast"
            ],
            "List": [
                "ListCostCategoryDefinitions"
            ]
        };
    }
    /**
     * Grants permission to create a new Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalyMonitor.html
     */
    toCreateAnomalyMonitor() {
        this.to('ce:CreateAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to create a new Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateAnomalySubscription.html
     */
    toCreateAnomalySubscription() {
        this.to('ce:CreateAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to create a new Cost Category with the requested name and rules
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_CreateCostCategoryDefinition.html
     */
    toCreateCostCategoryDefinition() {
        this.to('ce:CreateCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to delete an Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalyMonitor.html
     */
    toDeleteAnomalyMonitor() {
        this.to('ce:DeleteAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to delete an Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteAnomalySubscription.html
     */
    toDeleteAnomalySubscription() {
        this.to('ce:DeleteAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to delete a Cost Category
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DeleteCostCategoryDefinition.html
     */
    toDeleteCostCategoryDefinition() {
        this.to('ce:DeleteCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve descriptions such as the name, ARN, rules, definition, and effective dates of a Cost Category
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_DescribeCostCategoryDefinition.html
     */
    toDescribeCostCategoryDefinition() {
        this.to('ce:DescribeCostCategoryDefinition');
        return this;
    }
    /**
     * Grants permission to retrieve anomalies
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalies.html
     */
    toGetAnomalies() {
        this.to('ce:GetAnomalies');
        return this;
    }
    /**
     * Grants permission to query Anomaly Monitors
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalyMonitors.html
     */
    toGetAnomalyMonitors() {
        this.to('ce:GetAnomalyMonitors');
        return this;
    }
    /**
     * Grants permission to query Anomaly Subscriptions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetAnomalySubscriptions.html
     */
    toGetAnomalySubscriptions() {
        this.to('ce:GetAnomalySubscriptions');
        return this;
    }
    /**
     * Grants permission to retrieve the cost and usage metrics for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsage.html
     */
    toGetCostAndUsage() {
        this.to('ce:GetCostAndUsage');
        return this;
    }
    /**
     * Grants permission to retrieve the cost and usage metrics with resources for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostAndUsageWithResources.html
     */
    toGetCostAndUsageWithResources() {
        this.to('ce:GetCostAndUsageWithResources');
        return this;
    }
    /**
     * Grants permission to retrieve a cost forecast for a forecast time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetCostForecast.html
     */
    toGetCostForecast() {
        this.to('ce:GetCostForecast');
        return this;
    }
    /**
     * Grants permission to retrieve all available filter values for a filter for a period of time
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html
     */
    toGetDimensionValues() {
        this.to('ce:GetDimensionValues');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationCoverage.html
     */
    toGetReservationCoverage() {
        this.to('ce:GetReservationCoverage');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationPurchaseRecommendation.html
     */
    toGetReservationPurchaseRecommendation() {
        this.to('ce:GetReservationPurchaseRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the reservation utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetReservationUtilization.html
     */
    toGetReservationUtilization() {
        this.to('ce:GetReservationUtilization');
        return this;
    }
    /**
     * Grants permission to retrieve the rightsizing recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetRightsizingRecommendation.html
     */
    toGetRightsizingRecommendation() {
        this.to('ce:GetRightsizingRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans coverage for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansCoverage.html
     */
    toGetSavingsPlansCoverage() {
        this.to('ce:GetSavingsPlansCoverage');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans recommendations for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansPurchaseRecommendation.html
     */
    toGetSavingsPlansPurchaseRecommendation() {
        this.to('ce:GetSavingsPlansPurchaseRecommendation');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilization.html
     */
    toGetSavingsPlansUtilization() {
        this.to('ce:GetSavingsPlansUtilization');
        return this;
    }
    /**
     * Grants permission to retrieve the Savings Plans utilization details for your account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetSavingsPlansUtilizationDetails.html
     */
    toGetSavingsPlansUtilizationDetails() {
        this.to('ce:GetSavingsPlansUtilizationDetails');
        return this;
    }
    /**
     * Grants permission to query tags for a specified time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetTags.html
     */
    toGetTags() {
        this.to('ce:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a usage forecast for a forecast time period
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetUsageForecast.html
     */
    toGetUsageForecast() {
        this.to('ce:GetUsageForecast');
        return this;
    }
    /**
     * Grants permission to retrieve names, ARN, and effective dates for all Cost Categories
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ListCostCategoryDefinitions.html
     */
    toListCostCategoryDefinitions() {
        this.to('ce:ListCostCategoryDefinitions');
        return this;
    }
    /**
     * Grants permission to provide feedback on detected anomalies
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_ProvideAnomalyFeedback.html
     */
    toProvideAnomalyFeedback() {
        this.to('ce:ProvideAnomalyFeedback');
        return this;
    }
    /**
     * Grants permission to update an existing Anomaly Monitor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalyMonitor.html
     */
    toUpdateAnomalyMonitor() {
        this.to('ce:UpdateAnomalyMonitor');
        return this;
    }
    /**
     * Grants permission to update an existing Anomaly Subscription
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateAnomalySubscription.html
     */
    toUpdateAnomalySubscription() {
        this.to('ce:UpdateAnomalySubscription');
        return this;
    }
    /**
     * Grants permission to update an existing Cost Category
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_UpdateCostCategoryDefinition.html
     */
    toUpdateCostCategoryDefinition() {
        this.to('ce:UpdateCostCategoryDefinition');
        return this;
    }
}
exports.Ce = Ce;
//# sourceMappingURL=data:application/json;base64,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