"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Glue = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Glue extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [glue](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awsglue.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'glue';
        this.accessLevelList = {
            "Write": [
                "BatchCreatePartition",
                "BatchDeleteConnection",
                "BatchDeletePartition",
                "BatchDeleteTable",
                "BatchStopJobRun",
                "CancelMLTaskRun",
                "CreateClassifier",
                "CreateConnection",
                "CreateCrawler",
                "CreateDatabase",
                "CreateDevEndpoint",
                "CreateJob",
                "CreateMLTransform",
                "CreatePartition",
                "CreateScript",
                "CreateSecurityConfiguration",
                "CreateTable",
                "CreateTrigger",
                "CreateUserDefinedFunction",
                "CreateWorkflow",
                "DeleteClassifier",
                "DeleteConnection",
                "DeleteCrawler",
                "DeleteDatabase",
                "DeleteDevEndpoint",
                "DeleteJob",
                "DeleteMLTransform",
                "DeletePartition",
                "DeleteResourcePolicy",
                "DeleteSecurityConfiguration",
                "DeleteTable",
                "DeleteTrigger",
                "DeleteUserDefinedFunction",
                "DeleteWorkflow",
                "GetMapping",
                "ImportCatalogToGlue",
                "PutDataCatalogEncryptionSettings",
                "PutResourcePolicy",
                "PutWorkflowRunProperties",
                "ResetJobBookmark",
                "StartCrawler",
                "StartCrawlerSchedule",
                "StartExportLabelsTaskRun",
                "StartImportLabelsTaskRun",
                "StartJobRun",
                "StartMLEvaluationTaskRun",
                "StartMLLabelingSetGenerationTaskRun",
                "StartTrigger",
                "StartWorkflowRun",
                "StopCrawler",
                "StopCrawlerSchedule",
                "StopTrigger",
                "UpdateClassifier",
                "UpdateConnection",
                "UpdateCrawler",
                "UpdateCrawlerSchedule",
                "UpdateDatabase",
                "UpdateDevEndpoint",
                "UpdateJob",
                "UpdateMLTransform",
                "UpdatePartition",
                "UpdateTable",
                "UpdateTrigger",
                "UpdateUserDefinedFunction",
                "UpdateWorkflow",
                "UseMLTransforms"
            ],
            "Read": [
                "BatchDeleteTableVersion",
                "BatchGetCrawlers",
                "BatchGetDevEndpoints",
                "BatchGetJobs",
                "BatchGetPartition",
                "BatchGetTriggers",
                "BatchGetWorkflows",
                "DeleteTableVersion",
                "GetCatalogImportStatus",
                "GetClassifier",
                "GetClassifiers",
                "GetConnection",
                "GetConnections",
                "GetCrawler",
                "GetCrawlerMetrics",
                "GetCrawlers",
                "GetDataCatalogEncryptionSettings",
                "GetDatabase",
                "GetDatabases",
                "GetDataflowGraph",
                "GetDevEndpoint",
                "GetDevEndpoints",
                "GetJob",
                "GetJobBookmark",
                "GetJobRun",
                "GetJobRuns",
                "GetJobs",
                "GetMLTaskRun",
                "GetMLTransform",
                "GetPartition",
                "GetPartitions",
                "GetPlan",
                "GetResourcePolicy",
                "GetSecurityConfiguration",
                "GetSecurityConfigurations",
                "GetTable",
                "GetTableVersion",
                "GetTableVersions",
                "GetTables",
                "GetTags",
                "GetTrigger",
                "GetTriggers",
                "GetUserDefinedFunction",
                "GetUserDefinedFunctions",
                "GetWorkflow",
                "GetWorkflowRun",
                "GetWorkflowRunProperties",
                "GetWorkflowRuns",
                "SearchTables"
            ],
            "List": [
                "GetMLTaskRuns",
                "GetMLTransforms",
                "ListCrawlers",
                "ListDevEndpoints",
                "ListJobs",
                "ListMLTransforms",
                "ListTriggers",
                "ListWorkflows"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchCreatePartition
     */
    toBatchCreatePartition() {
        this.to('glue:BatchCreatePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more connections
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-BatchDeleteConnection
     */
    toBatchDeleteConnection() {
        this.to('glue:BatchDeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete one or more partitions
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchDeletePartition
     */
    toBatchDeletePartition() {
        this.to('glue:BatchDeletePartition');
        return this;
    }
    /**
     * Grants permission to delete one or more tables
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-BatchDeleteTable
     */
    toBatchDeleteTable() {
        this.to('glue:BatchDeleteTable');
        return this;
    }
    /**
     * Grants permission to delete one or more versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toBatchDeleteTableVersion() {
        this.to('glue:BatchDeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve one or more crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-BatchGetCrawlers
     */
    toBatchGetCrawlers() {
        this.to('glue:BatchGetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-BatchGetDevEndpoints
     */
    toBatchGetDevEndpoints() {
        this.to('glue:BatchGetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve one or more jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-BatchGetJobs
     */
    toBatchGetJobs() {
        this.to('glue:BatchGetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve one or more partitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-BatchGetPartition
     */
    toBatchGetPartition() {
        this.to('glue:BatchGetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve one or more triggers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-BatchGetTriggers
     */
    toBatchGetTriggers() {
        this.to('glue:BatchGetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve one or more workflows
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-BatchGetWorkflows
     */
    toBatchGetWorkflows() {
        this.to('glue:BatchGetWorkflows');
        return this;
    }
    /**
     * Grants permission to stop one or more job runs for a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-BatchStopStartJobRun
     */
    toBatchStopJobRun() {
        this.to('glue:BatchStopJobRun');
        return this;
    }
    /**
     * Grants permission to stop a running ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CancelMLTaskRun
     */
    toCancelMLTaskRun() {
        this.to('glue:CancelMLTaskRun');
        return this;
    }
    /**
     * Grants permission to create a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-CreateClassifier
     */
    toCreateClassifier() {
        this.to('glue:CreateClassifier');
        return this;
    }
    /**
     * Grants permission to create a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-CreateConnection
     */
    toCreateConnection() {
        this.to('glue:CreateConnection');
        return this;
    }
    /**
     * Grants permission to create a crawler
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-CreateCrawler
     */
    toCreateCrawler() {
        this.to('glue:CreateCrawler');
        return this;
    }
    /**
     * Grants permission to create a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-CreateDatabase
     */
    toCreateDatabase() {
        this.to('glue:CreateDatabase');
        return this;
    }
    /**
     * Grants permission to create a development endpoint
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-CreateDevEndpoint
     */
    toCreateDevEndpoint() {
        this.to('glue:CreateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to create a job
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-CreateJob
     */
    toCreateJob() {
        this.to('glue:CreateJob');
        return this;
    }
    /**
     * Grants permission to create an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-CreateMLTransform
     */
    toCreateMLTransform() {
        this.to('glue:CreateMLTransform');
        return this;
    }
    /**
     * Grants permission to create a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-CreatePartition
     */
    toCreatePartition() {
        this.to('glue:CreatePartition');
        return this;
    }
    /**
     * Grants permission to create a script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-CreateScript
     */
    toCreateScript() {
        this.to('glue:CreateScript');
        return this;
    }
    /**
     * Grants permission to create a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-CreateSecurityConfiguration
     */
    toCreateSecurityConfiguration() {
        this.to('glue:CreateSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to create a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-CreateTable
     */
    toCreateTable() {
        this.to('glue:CreateTable');
        return this;
    }
    /**
     * Grants permission to create a trigger
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-CreateTrigger
     */
    toCreateTrigger() {
        this.to('glue:CreateTrigger');
        return this;
    }
    /**
     * Grants permission to create a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-CreateUserDefinedFunction
     */
    toCreateUserDefinedFunction() {
        this.to('glue:CreateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to create a workflow
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-CreateWorkflow
     */
    toCreateWorkflow() {
        this.to('glue:CreateWorkflow');
        return this;
    }
    /**
     * Grants permission to delete a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-DeleteClassifier
     */
    toDeleteClassifier() {
        this.to('glue:DeleteClassifier');
        return this;
    }
    /**
     * Grants permission to delete a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-DeleteConnection
     */
    toDeleteConnection() {
        this.to('glue:DeleteConnection');
        return this;
    }
    /**
     * Grants permission to delete a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-DeleteCrawler
     */
    toDeleteCrawler() {
        this.to('glue:DeleteCrawler');
        return this;
    }
    /**
     * Grants permission to delete a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-DeleteDatabase
     */
    toDeleteDatabase() {
        this.to('glue:DeleteDatabase');
        return this;
    }
    /**
     * Grants permission to delete a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-DeleteDevEndpoint
     */
    toDeleteDevEndpoint() {
        this.to('glue:DeleteDevEndpoint');
        return this;
    }
    /**
     * Grants permission to delete a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-DeleteJob
     */
    toDeleteJob() {
        this.to('glue:DeleteJob');
        return this;
    }
    /**
     * Grants permission to delete an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-DeleteMLTransform
     */
    toDeleteMLTransform() {
        this.to('glue:DeleteMLTransform');
        return this;
    }
    /**
     * Grants permission to delete a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-DeletePartition
     */
    toDeletePartition() {
        this.to('glue:DeletePartition');
        return this;
    }
    /**
     * Grants permission to delete a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteResourcePolicy
     */
    toDeleteResourcePolicy() {
        this.to('glue:DeleteResourcePolicy');
        return this;
    }
    /**
     * Grants permission to delete a security configuration
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-DeleteSecurityConfiguration
     */
    toDeleteSecurityConfiguration() {
        this.to('glue:DeleteSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to delete a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTable
     */
    toDeleteTable() {
        this.to('glue:DeleteTable');
        return this;
    }
    /**
     * Grants permission to delete a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-DeleteTableVersion
     */
    toDeleteTableVersion() {
        this.to('glue:DeleteTableVersion');
        return this;
    }
    /**
     * Grants permission to delete a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-DeleteTrigger
     */
    toDeleteTrigger() {
        this.to('glue:DeleteTrigger');
        return this;
    }
    /**
     * Grants permission to delete a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-DeleteUserDefinedFunction
     */
    toDeleteUserDefinedFunction() {
        this.to('glue:DeleteUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to delete a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-DeleteWorkflow
     */
    toDeleteWorkflow() {
        this.to('glue:DeleteWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve the catalog import status
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-GetCatalogImportStatus
     */
    toGetCatalogImportStatus() {
        this.to('glue:GetCatalogImportStatus');
        return this;
    }
    /**
     * Grants permission to retrieve a classifier
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifier
     */
    toGetClassifier() {
        this.to('glue:GetClassifier');
        return this;
    }
    /**
     * Grants permission to list all classifiers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-GetClassifiers
     */
    toGetClassifiers() {
        this.to('glue:GetClassifiers');
        return this;
    }
    /**
     * Grants permission to retrieve a connection
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnection
     */
    toGetConnection() {
        this.to('glue:GetConnection');
        return this;
    }
    /**
     * Grants permission to retrieve a list of connections
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-GetConnections
     */
    toGetConnections() {
        this.to('glue:GetConnections');
        return this;
    }
    /**
     * Grants permission to retrieve a crawler
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawler
     */
    toGetCrawler() {
        this.to('glue:GetCrawler');
        return this;
    }
    /**
     * Grants permission to retrieve metrics about crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlerMetrics
     */
    toGetCrawlerMetrics() {
        this.to('glue:GetCrawlerMetrics');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-GetCrawlers
     */
    toGetCrawlers() {
        this.to('glue:GetCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve catalog encryption settings
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetDataCatalogEncryptionSettings
     */
    toGetDataCatalogEncryptionSettings() {
        this.to('glue:GetDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to retrieve a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabase
     */
    toGetDatabase() {
        this.to('glue:GetDatabase');
        return this;
    }
    /**
     * Grants permission to retrieve all databases
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-GetDatabases
     */
    toGetDatabases() {
        this.to('glue:GetDatabases');
        return this;
    }
    /**
     * Grants permission to transform a script into a directed acyclic graph (DAG)
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetDataflowGraph
     */
    toGetDataflowGraph() {
        this.to('glue:GetDataflowGraph');
        return this;
    }
    /**
     * Grants permission to retrieve a development endpoint
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoint
     */
    toGetDevEndpoint() {
        this.to('glue:GetDevEndpoint');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-GetDevEndpoints
     */
    toGetDevEndpoints() {
        this.to('glue:GetDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJob
     */
    toGetJob() {
        this.to('glue:GetJob');
        return this;
    }
    /**
     * Grants permission to retrieve a job bookmark
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-job-GetJobBookmark
     */
    toGetJobBookmark() {
        this.to('glue:GetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve a job run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRun
     */
    toGetJobRun() {
        this.to('glue:GetJobRun');
        return this;
    }
    /**
     * Grants permission to retrieve all job runs of a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-GetJobRuns
     */
    toGetJobRuns() {
        this.to('glue:GetJobRuns');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-GetJobs
     */
    toGetJobs() {
        this.to('glue:GetJobs');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Task Run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRun
     */
    toGetMLTaskRun() {
        this.to('glue:GetMLTaskRun');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Task Runs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTaskRuns
     */
    toGetMLTaskRuns() {
        this.to('glue:GetMLTaskRuns');
        return this;
    }
    /**
     * Grants permission to retrieve an ML Transform
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransform
     */
    toGetMLTransform() {
        this.to('glue:GetMLTransform');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-GetMLTransforms
     */
    toGetMLTransforms() {
        this.to('glue:GetMLTransforms');
        return this;
    }
    /**
     * Grants permission to create a mapping
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetMapping
     */
    toGetMapping() {
        this.to('glue:GetMapping');
        return this;
    }
    /**
     * Grants permission to retrieve a partition
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartition
     */
    toGetPartition() {
        this.to('glue:GetPartition');
        return this;
    }
    /**
     * Grants permission to retrieve the partitions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-GetPartitions
     */
    toGetPartitions() {
        this.to('glue:GetPartitions');
        return this;
    }
    /**
     * Grants permission to retrieve a mapping for a script
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-etl-script-generation.html#aws-glue-api-etl-script-generation-GetPlan
     */
    toGetPlan() {
        this.to('glue:GetPlan');
        return this;
    }
    /**
     * Grants permission to retrieve a resource policy
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetResourcePolicy
     */
    toGetResourcePolicy() {
        this.to('glue:GetResourcePolicy');
        return this;
    }
    /**
     * Grants permission to retrieve a security configuration
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfiguration
     */
    toGetSecurityConfiguration() {
        this.to('glue:GetSecurityConfiguration');
        return this;
    }
    /**
     * Grants permission to retrieve one or more security configurations
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-GetSecurityConfigurations
     */
    toGetSecurityConfigurations() {
        this.to('glue:GetSecurityConfigurations');
        return this;
    }
    /**
     * Grants permission to retrieve a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTable
     */
    toGetTable() {
        this.to('glue:GetTable');
        return this;
    }
    /**
     * Grants permission to retrieve a version of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersion
     */
    toGetTableVersion() {
        this.to('glue:GetTableVersion');
        return this;
    }
    /**
     * Grants permission to retrieve a list of versions of a table
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTableVersions
     */
    toGetTableVersions() {
        this.to('glue:GetTableVersions');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in a database
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-GetTables
     */
    toGetTables() {
        this.to('glue:GetTables');
        return this;
    }
    /**
     * Grants permission to retrieve all tags associated with a resource
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toGetTags() {
        this.to('glue:GetTags');
        return this;
    }
    /**
     * Grants permission to retrieve a trigger
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTrigger
     */
    toGetTrigger() {
        this.to('glue:GetTrigger');
        return this;
    }
    /**
     * Grants permission to retrieve the triggers associated with a job
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-GetTriggers
     */
    toGetTriggers() {
        this.to('glue:GetTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve a function definition.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunction
     */
    toGetUserDefinedFunction() {
        this.to('glue:GetUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to retrieve multiple function definitions
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-GetUserDefinedFunctions
     */
    toGetUserDefinedFunctions() {
        this.to('glue:GetUserDefinedFunctions');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflow
     */
    toGetWorkflow() {
        this.to('glue:GetWorkflow');
        return this;
    }
    /**
     * Grants permission to retrieve a workflow run
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRun
     */
    toGetWorkflowRun() {
        this.to('glue:GetWorkflowRun');
        return this;
    }
    /**
     * Grants permission to retrieve workflow run properties
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRunProperties
     */
    toGetWorkflowRunProperties() {
        this.to('glue:GetWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to retrieve all runs of a workflow
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-GetWorkflowRuns
     */
    toGetWorkflowRuns() {
        this.to('glue:GetWorkflowRuns');
        return this;
    }
    /**
     * Grants permission to import an Athena data catalog into AWS Glue
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-migration.html#aws-glue-api-catalog-migration-ImportCatalogToGlue
     */
    toImportCatalogToGlue() {
        this.to('glue:ImportCatalogToGlue');
        return this;
    }
    /**
     * Grants permission to retrieve all crawlers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-ListCrawlers
     */
    toListCrawlers() {
        this.to('glue:ListCrawlers');
        return this;
    }
    /**
     * Grants permission to retrieve all development endpoints
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-ListDevEndpoints
     */
    toListDevEndpoints() {
        this.to('glue:ListDevEndpoints');
        return this;
    }
    /**
     * Grants permission to retrieve all current jobs
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-ListJobs
     */
    toListJobs() {
        this.to('glue:ListJobs');
        return this;
    }
    /**
     * Grants permission to retrieve all ML Transforms
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-ListMLTransforms
     */
    toListMLTransforms() {
        this.to('glue:ListMLTransforms');
        return this;
    }
    /**
     * Grants permission to retrieve all triggers
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-ListTriggers
     */
    toListTriggers() {
        this.to('glue:ListTriggers');
        return this;
    }
    /**
     * Grants permission to retrieve all workflows
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-ListWorkflows
     */
    toListWorkflows() {
        this.to('glue:ListWorkflows');
        return this;
    }
    /**
     * Grants permission to update catalog encryption settings
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutDataCatalogEncryptionSettings
     */
    toPutDataCatalogEncryptionSettings() {
        this.to('glue:PutDataCatalogEncryptionSettings');
        return this;
    }
    /**
     * Grants permission to update a resource policy
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-security.html#aws-glue-api-jobs-security-PutResourcePolicy
     */
    toPutResourcePolicy() {
        this.to('glue:PutResourcePolicy');
        return this;
    }
    /**
     * Grants permission to update workflow run properties
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-PutWorkflowRunProperties
     */
    toPutWorkflowRunProperties() {
        this.to('glue:PutWorkflowRunProperties');
        return this;
    }
    /**
     * Grants permission to reset a job bookmark
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-ResetJobBookmark
     */
    toResetJobBookmark() {
        this.to('glue:ResetJobBookmark');
        return this;
    }
    /**
     * Grants permission to retrieve the tables in the catalog
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-SearchTables
     */
    toSearchTables() {
        this.to('glue:SearchTables');
        return this;
    }
    /**
     * Grants permission to start a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StartCrawler
     */
    toStartCrawler() {
        this.to('glue:StartCrawler');
        return this;
    }
    /**
     * Grants permission to change the schedule state of a crawler to SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StartCrawlerSchedule
     */
    toStartCrawlerSchedule() {
        this.to('glue:StartCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to start an Export Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartExportLabelsTaskRun
     */
    toStartExportLabelsTaskRun() {
        this.to('glue:StartExportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start an Import Labels ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartImportLabelsTaskRun
     */
    toStartImportLabelsTaskRun() {
        this.to('glue:StartImportLabelsTaskRun');
        return this;
    }
    /**
     * Grants permission to start running a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-runs.html#aws-glue-api-jobs-runs-StartJobRun
     */
    toStartJobRun() {
        this.to('glue:StartJobRun');
        return this;
    }
    /**
     * Grants permission to start an Evaluation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLEvaluationTaskRun
     */
    toStartMLEvaluationTaskRun() {
        this.to('glue:StartMLEvaluationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a Labeling Set Generation ML Task Run
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-StartMLLabelingSetGenerationTaskRun
     */
    toStartMLLabelingSetGenerationTaskRun() {
        this.to('glue:StartMLLabelingSetGenerationTaskRun');
        return this;
    }
    /**
     * Grants permission to start a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StartTrigger
     */
    toStartTrigger() {
        this.to('glue:StartTrigger');
        return this;
    }
    /**
     * Grants permission to start running a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-StartWorkflowRun
     */
    toStartWorkflowRun() {
        this.to('glue:StartWorkflowRun');
        return this;
    }
    /**
     * Grants permission to stop a running crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-StopCrawler
     */
    toStopCrawler() {
        this.to('glue:StopCrawler');
        return this;
    }
    /**
     * Grants permission to set the schedule state of a crawler to NOT_SCHEDULED
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-StopCrawlerSchedule
     */
    toStopCrawlerSchedule() {
        this.to('glue:StopCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to stop a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-StopTrigger
     */
    toStopTrigger() {
        this.to('glue:StopTrigger');
        return this;
    }
    /**
     * Grants permission to add tags to a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-TagResource
     */
    toTagResource() {
        this.to('glue:TagResource');
        return this;
    }
    /**
     * Grants permission to remove tags associated with a resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-tags.html#aws-glue-api-tags-UntagResource
     */
    toUntagResource() {
        this.to('glue:UntagResource');
        return this;
    }
    /**
     * Grants permission to update a classifier
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-classifiers.html#aws-glue-api-crawler-classifiers-UpdateClassifier
     */
    toUpdateClassifier() {
        this.to('glue:UpdateClassifier');
        return this;
    }
    /**
     * Grants permission to update a connection
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-connections.html#aws-glue-api-catalog-connections-UpdateConnection
     */
    toUpdateConnection() {
        this.to('glue:UpdateConnection');
        return this;
    }
    /**
     * Grants permission to update a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-crawling.html#aws-glue-api-crawler-crawling-UpdateCrawler
     */
    toUpdateCrawler() {
        this.to('glue:UpdateCrawler');
        return this;
    }
    /**
     * Grants permission to update the schedule of a crawler
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-crawler-scheduler.html#aws-glue-api-crawler-scheduler-UpdateCrawlerSchedule
     */
    toUpdateCrawlerSchedule() {
        this.to('glue:UpdateCrawlerSchedule');
        return this;
    }
    /**
     * Grants permission to update a database
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-databases.html#aws-glue-api-catalog-databases-UpdateDatabase
     */
    toUpdateDatabase() {
        this.to('glue:UpdateDatabase');
        return this;
    }
    /**
     * Grants permission to update a development endpoint
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-dev-endpoint.html#aws-glue-api-dev-endpoint-UpdateDevEndpoint
     */
    toUpdateDevEndpoint() {
        this.to('glue:UpdateDevEndpoint');
        return this;
    }
    /**
     * Grants permission to update a job
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-job.html#aws-glue-api-jobs-job-UpdateJob
     */
    toUpdateJob() {
        this.to('glue:UpdateJob');
        return this;
    }
    /**
     * Grants permission to update an ML Transform
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html#aws-glue-api-machine-learning-api-UpdateMLTransform
     */
    toUpdateMLTransform() {
        this.to('glue:UpdateMLTransform');
        return this;
    }
    /**
     * Grants permission to update a partition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-partitions.html#aws-glue-api-catalog-partitions-UpdatePartition
     */
    toUpdatePartition() {
        this.to('glue:UpdatePartition');
        return this;
    }
    /**
     * Grants permission to update a table
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-tables.html#aws-glue-api-catalog-tables-UpdateTable
     */
    toUpdateTable() {
        this.to('glue:UpdateTable');
        return this;
    }
    /**
     * Grants permission to update a trigger
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-jobs-trigger.html#aws-glue-api-jobs-trigger-UpdateTrigger
     */
    toUpdateTrigger() {
        this.to('glue:UpdateTrigger');
        return this;
    }
    /**
     * Grants permission to update a function definition
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-catalog-functions.html#aws-glue-api-catalog-functions-UpdateUserDefinedFunction
     */
    toUpdateUserDefinedFunction() {
        this.to('glue:UpdateUserDefinedFunction');
        return this;
    }
    /**
     * Grants permission to update a workflow
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-workflow.html#aws-glue-api-workflow-UpdateWorkflow
     */
    toUpdateWorkflow() {
        this.to('glue:UpdateWorkflow');
        return this;
    }
    /**
     * Grants permission to use an ML Transform from within a Glue ETL Script
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/glue/latest/dg/aws-glue-api-machine-learning-api.html
     */
    toUseMLTransforms() {
        this.to('glue:UseMLTransforms');
        return this;
    }
    /**
     * Adds a resource of type catalog to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onCatalog(account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:catalog';
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type database to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onDatabase(databaseName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:database/${DatabaseName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type table to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTable(databaseName, tableName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:table/${DatabaseName}/${TableName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type tableversion to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param tableName - Identifier for the tableName.
     * @param tableVersionName - Identifier for the tableVersionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onTableversion(databaseName, tableName, tableVersionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:tableVersion/${DatabaseName}/${TableName}/${TableVersionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${TableName}', tableName);
        arn = arn.replace('${TableVersionName}', tableVersionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type connection to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param connectionName - Identifier for the connectionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onConnection(connectionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:connection/${ConnectionName}';
        arn = arn.replace('${ConnectionName}', connectionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type userdefinedfunction to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param databaseName - Identifier for the databaseName.
     * @param userDefinedFunctionName - Identifier for the userDefinedFunctionName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onUserdefinedfunction(databaseName, userDefinedFunctionName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:userDefinedFunction/${DatabaseName}/${UserDefinedFunctionName}';
        arn = arn.replace('${DatabaseName}', databaseName);
        arn = arn.replace('${UserDefinedFunctionName}', userDefinedFunctionName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type devendpoint to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param devEndpointName - Identifier for the devEndpointName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onDevendpoint(devEndpointName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:devendpoint/${DevEndpointName}';
        arn = arn.replace('${DevEndpointName}', devEndpointName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type job to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param jobName - Identifier for the jobName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onJob(jobName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:job/${JobName}';
        arn = arn.replace('${JobName}', jobName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type trigger to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param triggerName - Identifier for the triggerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onTrigger(triggerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:trigger/${TriggerName}';
        arn = arn.replace('${TriggerName}', triggerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type crawler to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param crawlerName - Identifier for the crawlerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onCrawler(crawlerName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:crawler/${CrawlerName}';
        arn = arn.replace('${CrawlerName}', crawlerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type workflow to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param workflowName - Identifier for the workflowName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWorkflow(workflowName, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:workflow/${WorkflowName}';
        arn = arn.replace('${WorkflowName}', workflowName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type mlTransform to the statement
     *
     * https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html
     *
     * @param transformId - Identifier for the transformId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onMlTransform(transformId, account, region, partition) {
        var arn = 'arn:${Partition}:glue:${Region}:${Account}:mlTransform/${TransformId}';
        arn = arn.replace('${TransformId}', transformId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Glue = Glue;
//# sourceMappingURL=data:application/json;base64,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