"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Sso = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [sso](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssso.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Sso extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [sso](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awssso.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'sso';
        this.accessLevelList = {
            "Write": [
                "AssociateDirectory",
                "AssociateProfile",
                "AttachManagedPolicyToPermissionSet",
                "CreateAccountAssignment",
                "CreateApplicationInstance",
                "CreateApplicationInstanceCertificate",
                "CreateManagedApplicationInstance",
                "CreatePermissionSet",
                "CreateProfile",
                "CreateTrust",
                "DeleteAccountAssignment",
                "DeleteApplicationInstance",
                "DeleteApplicationInstanceCertificate",
                "DeleteInlinePolicyFromPermissionSet",
                "DeleteManagedApplicationInstance",
                "DeletePermissionSet",
                "DeletePermissionsPolicy",
                "DeleteProfile",
                "DetachManagedPolicyFromPermissionSet",
                "DisassociateDirectory",
                "DisassociateProfile",
                "ImportApplicationInstanceServiceProviderMetadata",
                "ProvisionPermissionSet",
                "PutInlinePolicyToPermissionSet",
                "PutMfaDeviceManagementForDirectory",
                "PutPermissionsPolicy",
                "StartSSO",
                "UpdateApplicationInstanceActiveCertificate",
                "UpdateApplicationInstanceDisplayData",
                "UpdateApplicationInstanceResponseConfiguration",
                "UpdateApplicationInstanceResponseSchemaConfiguration",
                "UpdateApplicationInstanceSecurityConfiguration",
                "UpdateApplicationInstanceServiceProviderConfiguration",
                "UpdateApplicationInstanceStatus",
                "UpdateDirectoryAssociation",
                "UpdateManagedApplicationInstanceStatus",
                "UpdatePermissionSet",
                "UpdateProfile",
                "UpdateSSOConfiguration",
                "UpdateTrust"
            ],
            "Read": [
                "DescribeAccountAssignmentCreationStatus",
                "DescribeAccountAssignmentDeletionStatus",
                "DescribePermissionSet",
                "DescribePermissionSetProvisioningStatus",
                "DescribePermissionsPolicies",
                "DescribeRegisteredRegions",
                "GetApplicationInstance",
                "GetApplicationTemplate",
                "GetInlinePolicyForPermissionSet",
                "GetManagedApplicationInstance",
                "GetMfaDeviceManagementForDirectory",
                "GetPermissionSet",
                "GetPermissionsPolicy",
                "GetProfile",
                "GetSSOStatus",
                "GetSharedSsoConfiguration",
                "GetSsoConfiguration",
                "GetTrust",
                "ListAccountAssignmentCreationStatus",
                "ListAccountAssignmentDeletionStatus",
                "ListAccountAssignments",
                "ListAccountsForProvisionedPermissionSet",
                "ListApplicationInstanceCertificates",
                "ListApplicationTemplates",
                "ListApplications",
                "ListDirectoryAssociations",
                "ListInstances",
                "ListManagedPoliciesInPermissionSet",
                "ListPermissionSetProvisioningStatus",
                "ListPermissionSets",
                "ListPermissionSetsProvisionedToAccount",
                "ListProfileAssociations",
                "ListProfiles",
                "ListTagsForResource"
            ],
            "List": [
                "ListApplicationInstances"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Connect a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateDirectory() {
        this.to('sso:AssociateDirectory');
        return this;
    }
    /**
     * Create an association between a directory user or group and a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAssociateProfile() {
        this.to('sso:AssociateProfile');
        return this;
    }
    /**
     * Attaches an AWS managed policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toAttachManagedPolicyToPermissionSet() {
        this.to('sso:AttachManagedPolicyToPermissionSet');
        return this;
    }
    /**
     * Assigns access to a Principal for a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateAccountAssignment() {
        this.to('sso:CreateAccountAssignment');
        return this;
    }
    /**
     * Add an application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstance() {
        this.to('sso:CreateApplicationInstance');
        return this;
    }
    /**
     * Add a new certificate for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateApplicationInstanceCertificate() {
        this.to('sso:CreateApplicationInstanceCertificate');
        return this;
    }
    /**
     * Add a managed application instance to AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateManagedApplicationInstance() {
        this.to('sso:CreateManagedApplicationInstance');
        return this;
    }
    /**
     * Create a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreatePermissionSet() {
        this.to('sso:CreatePermissionSet');
        return this;
    }
    /**
     * Create a profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateProfile() {
        this.to('sso:CreateProfile');
        return this;
    }
    /**
     * Create a federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toCreateTrust() {
        this.to('sso:CreateTrust');
        return this;
    }
    /**
     * Deletes a Principal's access from a specified AWS account using a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteAccountAssignment() {
        this.to('sso:DeleteAccountAssignment');
        return this;
    }
    /**
     * Delete the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstance() {
        this.to('sso:DeleteApplicationInstance');
        return this;
    }
    /**
     * Delete an inactive or expired certificate from the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteApplicationInstanceCertificate() {
        this.to('sso:DeleteApplicationInstanceCertificate');
        return this;
    }
    /**
     * Deletes the inline policy from a specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteInlinePolicyFromPermissionSet() {
        this.to('sso:DeleteInlinePolicyFromPermissionSet');
        return this;
    }
    /**
     * Delete the managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteManagedApplicationInstance() {
        this.to('sso:DeleteManagedApplicationInstance');
        return this;
    }
    /**
     * Delete a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionSet() {
        this.to('sso:DeletePermissionSet');
        return this;
    }
    /**
     * Delete the permission policy associated with a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeletePermissionsPolicy() {
        this.to('sso:DeletePermissionsPolicy');
        return this;
    }
    /**
     * Delete the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDeleteProfile() {
        this.to('sso:DeleteProfile');
        return this;
    }
    /**
     * Describes the status of the assignment creation request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentCreationStatus() {
        this.to('sso:DescribeAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Describes the status of an assignment deletion request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeAccountAssignmentDeletionStatus() {
        this.to('sso:DescribeAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Describes a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSet() {
        this.to('sso:DescribePermissionSet');
        return this;
    }
    /**
     * Describes the status for the given Permission Set Provisioning request.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionSetProvisioningStatus() {
        this.to('sso:DescribePermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all the permissions policies associated with a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribePermissionsPolicies() {
        this.to('sso:DescribePermissionsPolicies');
        return this;
    }
    /**
     * Obtains the regions where your organization has enabled AWS Single Sign-on
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDescribeRegisteredRegions() {
        this.to('sso:DescribeRegisteredRegions');
        return this;
    }
    /**
     * Detaches the attached AWS managed policy from the specified permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDetachManagedPolicyFromPermissionSet() {
        this.to('sso:DetachManagedPolicyFromPermissionSet');
        return this;
    }
    /**
     * Disassociate a directory to be used by AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateDirectory() {
        this.to('sso:DisassociateDirectory');
        return this;
    }
    /**
     * Disassociate a directory user or group from a profile
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toDisassociateProfile() {
        this.to('sso:DisassociateProfile');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationInstance() {
        this.to('sso:GetApplicationInstance');
        return this;
    }
    /**
     * Retrieve application template details
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetApplicationTemplate() {
        this.to('sso:GetApplicationTemplate');
        return this;
    }
    /**
     * Obtains the inline policy assigned to the permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetInlinePolicyForPermissionSet() {
        this.to('sso:GetInlinePolicyForPermissionSet');
        return this;
    }
    /**
     * Retrieve details for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetManagedApplicationInstance() {
        this.to('sso:GetManagedApplicationInstance');
        return this;
    }
    /**
     * Retrieve Mfa Device Management settings for the directory
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetMfaDeviceManagementForDirectory() {
        this.to('sso:GetMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Retrieve details of a permission set
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionSet() {
        this.to('sso:GetPermissionSet');
        return this;
    }
    /**
     * Retrieve all permission policies associated with a permission set
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:DescribePermissionsPolicies
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetPermissionsPolicy() {
        this.to('sso:GetPermissionsPolicy');
        return this;
    }
    /**
     * Retrieve a profile for an application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetProfile() {
        this.to('sso:GetProfile');
        return this;
    }
    /**
     * Check if AWS Single Sign-On is enabled
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSSOStatus() {
        this.to('sso:GetSSOStatus');
        return this;
    }
    /**
     * Retrieve shared configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSharedSsoConfiguration() {
        this.to('sso:GetSharedSsoConfiguration');
        return this;
    }
    /**
     * Retrieve configuration for the current SSO instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetSsoConfiguration() {
        this.to('sso:GetSsoConfiguration');
        return this;
    }
    /**
     * Retrieve the federation trust in a target account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toGetTrust() {
        this.to('sso:GetTrust');
        return this;
    }
    /**
     * Update the application instance by uploading an application SAML metadata file provided by the service provider
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toImportApplicationInstanceServiceProviderMetadata() {
        this.to('sso:ImportApplicationInstanceServiceProviderMetadata');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment creation requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentCreationStatus() {
        this.to('sso:ListAccountAssignmentCreationStatus');
        return this;
    }
    /**
     * Lists the status of the AWS account assignment deletion requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignmentDeletionStatus() {
        this.to('sso:ListAccountAssignmentDeletionStatus');
        return this;
    }
    /**
     * Lists the assignee of the specified AWS account with the specified permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountAssignments() {
        this.to('sso:ListAccountAssignments');
        return this;
    }
    /**
     * Lists all the AWS accounts where the specified permission set is provisioned.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListAccountsForProvisionedPermissionSet() {
        this.to('sso:ListAccountsForProvisionedPermissionSet');
        return this;
    }
    /**
     * Retrieve all of the certificates for a given application instance
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstanceCertificates() {
        this.to('sso:ListApplicationInstanceCertificates');
        return this;
    }
    /**
     * Retrieve all application instances
     *
     * Access Level: List
     *
     * Dependent actions:
     * - sso:GetApplicationInstance
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationInstances() {
        this.to('sso:ListApplicationInstances');
        return this;
    }
    /**
     * Retrieve all supported application templates
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:GetApplicationTemplate
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplicationTemplates() {
        this.to('sso:ListApplicationTemplates');
        return this;
    }
    /**
     * Retrieve all supported applications
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListApplications() {
        this.to('sso:ListApplications');
        return this;
    }
    /**
     * Retrieve details about the directory connected to AWS Single Sign-On
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListDirectoryAssociations() {
        this.to('sso:ListDirectoryAssociations');
        return this;
    }
    /**
     * Lists the SSO Instances that the caller has access to.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListInstances() {
        this.to('sso:ListInstances');
        return this;
    }
    /**
     * Lists the AWS managed policies that are attached to a specified permission set.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListManagedPoliciesInPermissionSet() {
        this.to('sso:ListManagedPoliciesInPermissionSet');
        return this;
    }
    /**
     * Lists the status of the Permission Set Provisioning requests for a specified SSO instance.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetProvisioningStatus() {
        this.to('sso:ListPermissionSetProvisioningStatus');
        return this;
    }
    /**
     * Retrieve all permission sets
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSets() {
        this.to('sso:ListPermissionSets');
        return this;
    }
    /**
     * Lists all the permission sets that are provisioned to a specified AWS account.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListPermissionSetsProvisionedToAccount() {
        this.to('sso:ListPermissionSetsProvisionedToAccount');
        return this;
    }
    /**
     * Retrieve the directory user or group associated with the profile
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfileAssociations() {
        this.to('sso:ListProfileAssociations');
        return this;
    }
    /**
     * Retrieve all profiles for an application instance
     *
     * Access Level: Read
     *
     * Dependent actions:
     * - sso:GetProfile
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListProfiles() {
        this.to('sso:ListProfiles');
        return this;
    }
    /**
     * Lists the tags that are attached to a specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toListTagsForResource() {
        this.to('sso:ListTagsForResource');
        return this;
    }
    /**
     * The process by which a specified permission set is provisioned to the specified target.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toProvisionPermissionSet() {
        this.to('sso:ProvisionPermissionSet');
        return this;
    }
    /**
     * Attaches an IAM inline policy to a permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutInlinePolicyToPermissionSet() {
        this.to('sso:PutInlinePolicyToPermissionSet');
        return this;
    }
    /**
     * Put Mfa Device Management settings for the directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutMfaDeviceManagementForDirectory() {
        this.to('sso:PutMfaDeviceManagementForDirectory');
        return this;
    }
    /**
     * Add a policy to a permission set
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toPutPermissionsPolicy() {
        this.to('sso:PutPermissionsPolicy');
        return this;
    }
    /**
     * Initialize AWS Single Sign-On
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toStartSSO() {
        this.to('sso:StartSSO');
        return this;
    }
    /**
     * Associates a set of tags with a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toTagResource() {
        this.to('sso:TagResource');
        return this;
    }
    /**
     * Disassociates a set of tags from a specified resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUntagResource() {
        this.to('sso:UntagResource');
        return this;
    }
    /**
     * Set a certificate as the active one for this application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceActiveCertificate() {
        this.to('sso:UpdateApplicationInstanceActiveCertificate');
        return this;
    }
    /**
     * Update display data of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceDisplayData() {
        this.to('sso:UpdateApplicationInstanceDisplayData');
        return this;
    }
    /**
     * Update federation response configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseConfiguration() {
        this.to('sso:UpdateApplicationInstanceResponseConfiguration');
        return this;
    }
    /**
     * Update federation response schema configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceResponseSchemaConfiguration() {
        this.to('sso:UpdateApplicationInstanceResponseSchemaConfiguration');
        return this;
    }
    /**
     * Update security details for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceSecurityConfiguration() {
        this.to('sso:UpdateApplicationInstanceSecurityConfiguration');
        return this;
    }
    /**
     * Update service provider related configuration for the application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceServiceProviderConfiguration() {
        this.to('sso:UpdateApplicationInstanceServiceProviderConfiguration');
        return this;
    }
    /**
     * Update the status of an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateApplicationInstanceStatus() {
        this.to('sso:UpdateApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the user attribute mappings for your connected directory
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateDirectoryAssociation() {
        this.to('sso:UpdateDirectoryAssociation');
        return this;
    }
    /**
     * Update the status of a managed application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateManagedApplicationInstanceStatus() {
        this.to('sso:UpdateManagedApplicationInstanceStatus');
        return this;
    }
    /**
     * Update the permission set.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdatePermissionSet() {
        this.to('sso:UpdatePermissionSet');
        return this;
    }
    /**
     * Update the profile for an application instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateProfile() {
        this.to('sso:UpdateProfile');
        return this;
    }
    /**
     * Update the configuration for the current SSO instance
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateSSOConfiguration() {
        this.to('sso:UpdateSSOConfiguration');
        return this;
    }
    /**
     * Update the federation trust in a target account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/singlesignon/latest/userguide/iam-auth-access-using-id-policies.html#policyexample
     */
    toUpdateTrust() {
        this.to('sso:UpdateTrust');
        return this;
    }
    /**
     * Adds a resource of type PermissionSet to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param permissionSetId - Identifier for the permissionSetId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onPermissionSet(instanceId, permissionSetId, partition) {
        var arn = 'arn:${Partition}:sso:::permissionSet/${InstanceId}/${PermissionSetId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${PermissionSetId}', permissionSetId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Account to the statement
     *
     * @param accountId - Account of the resource; defaults to empty string: all accounts.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onAccount(accountId, partition) {
        var arn = 'arn:${Partition}:sso:::account/${AccountId}';
        arn = arn.replace('${AccountId}', accountId || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type Instance to the statement
     *
     * @param instanceId - Identifier for the instanceId.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onInstance(instanceId, partition) {
        var arn = 'arn:${Partition}:sso:::instance/${InstanceId}';
        arn = arn.replace('${InstanceId}', instanceId);
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Sso = Sso;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzc3NvLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXdzc3NvLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLHNDQUE0QztBQUU1Qzs7OztHQUlHO0FBQ0gsTUFBYSxHQUFJLFNBQVEsd0JBQWU7SUFHdEM7Ozs7T0FJRztJQUNILFlBQWEsR0FBWTtRQUN2QixLQUFLLENBQUMsR0FBRyxDQUFDLENBQUM7UUFSTixrQkFBYSxHQUFHLEtBQUssQ0FBQztRQTI3Qm5CLG9CQUFlLEdBQW9CO1lBQzNDLE9BQU8sRUFBRTtnQkFDUCxvQkFBb0I7Z0JBQ3BCLGtCQUFrQjtnQkFDbEIsb0NBQW9DO2dCQUNwQyx5QkFBeUI7Z0JBQ3pCLDJCQUEyQjtnQkFDM0Isc0NBQXNDO2dCQUN0QyxrQ0FBa0M7Z0JBQ2xDLHFCQUFxQjtnQkFDckIsZUFBZTtnQkFDZixhQUFhO2dCQUNiLHlCQUF5QjtnQkFDekIsMkJBQTJCO2dCQUMzQixzQ0FBc0M7Z0JBQ3RDLHFDQUFxQztnQkFDckMsa0NBQWtDO2dCQUNsQyxxQkFBcUI7Z0JBQ3JCLHlCQUF5QjtnQkFDekIsZUFBZTtnQkFDZixzQ0FBc0M7Z0JBQ3RDLHVCQUF1QjtnQkFDdkIscUJBQXFCO2dCQUNyQixrREFBa0Q7Z0JBQ2xELHdCQUF3QjtnQkFDeEIsZ0NBQWdDO2dCQUNoQyxvQ0FBb0M7Z0JBQ3BDLHNCQUFzQjtnQkFDdEIsVUFBVTtnQkFDViw0Q0FBNEM7Z0JBQzVDLHNDQUFzQztnQkFDdEMsZ0RBQWdEO2dCQUNoRCxzREFBc0Q7Z0JBQ3RELGdEQUFnRDtnQkFDaEQsdURBQXVEO2dCQUN2RCxpQ0FBaUM7Z0JBQ2pDLDRCQUE0QjtnQkFDNUIsd0NBQXdDO2dCQUN4QyxxQkFBcUI7Z0JBQ3JCLGVBQWU7Z0JBQ2Ysd0JBQXdCO2dCQUN4QixhQUFhO2FBQ2Q7WUFDRCxNQUFNLEVBQUU7Z0JBQ04seUNBQXlDO2dCQUN6Qyx5Q0FBeUM7Z0JBQ3pDLHVCQUF1QjtnQkFDdkIseUNBQXlDO2dCQUN6Qyw2QkFBNkI7Z0JBQzdCLDJCQUEyQjtnQkFDM0Isd0JBQXdCO2dCQUN4Qix3QkFBd0I7Z0JBQ3hCLGlDQUFpQztnQkFDakMsK0JBQStCO2dCQUMvQixvQ0FBb0M7Z0JBQ3BDLGtCQUFrQjtnQkFDbEIsc0JBQXNCO2dCQUN0QixZQUFZO2dCQUNaLGNBQWM7Z0JBQ2QsMkJBQTJCO2dCQUMzQixxQkFBcUI7Z0JBQ3JCLFVBQVU7Z0JBQ1YscUNBQXFDO2dCQUNyQyxxQ0FBcUM7Z0JBQ3JDLHdCQUF3QjtnQkFDeEIseUNBQXlDO2dCQUN6QyxxQ0FBcUM7Z0JBQ3JDLDBCQUEwQjtnQkFDMUIsa0JBQWtCO2dCQUNsQiwyQkFBMkI7Z0JBQzNCLGVBQWU7Z0JBQ2Ysb0NBQW9DO2dCQUNwQyxxQ0FBcUM7Z0JBQ3JDLG9CQUFvQjtnQkFDcEIsd0NBQXdDO2dCQUN4Qyx5QkFBeUI7Z0JBQ3pCLGNBQWM7Z0JBQ2QscUJBQXFCO2FBQ3RCO1lBQ0QsTUFBTSxFQUFFO2dCQUNOLDBCQUEwQjthQUMzQjtZQUNELFNBQVMsRUFBRTtnQkFDVCxhQUFhO2dCQUNiLGVBQWU7YUFDaEI7U0FDRixDQUFDO0lBeGdDRixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxrQkFBa0I7UUFDdkIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLG9DQUFvQztRQUN6QyxJQUFJLENBQUMsRUFBRSxDQUFDLHdDQUF3QyxDQUFDLENBQUM7UUFDbEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUJBQXlCO1FBQzlCLElBQUksQ0FBQyxFQUFFLENBQUMsNkJBQTZCLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHNDQUFzQztRQUMzQyxJQUFJLENBQUMsRUFBRSxDQUFDLDBDQUEwQyxDQUFDLENBQUM7UUFDcEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGVBQWU7UUFDcEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDO1FBQzdCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQ0FBc0M7UUFDM0MsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0NBQWtDO1FBQ3ZDLElBQUksQ0FBQyxFQUFFLENBQUMsc0NBQXNDLENBQUMsQ0FBQztRQUNoRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQkFBcUI7UUFDMUIsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO1FBQ25DLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5Q0FBeUM7UUFDOUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVCQUF1QjtRQUM1QixJQUFJLENBQUMsRUFBRSxDQUFDLDJCQUEyQixDQUFDLENBQUM7UUFDckMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kseUNBQXlDO1FBQzlDLElBQUksQ0FBQyxFQUFFLENBQUMsNkNBQTZDLENBQUMsQ0FBQztRQUN2RCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw2QkFBNkI7UUFDbEMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLDJCQUEyQjtRQUNoQyxJQUFJLENBQUMsRUFBRSxDQUFDLCtCQUErQixDQUFDLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0NBQXNDO1FBQzNDLElBQUksQ0FBQyxFQUFFLENBQUMsMENBQTBDLENBQUMsQ0FBQztRQUNwRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGlDQUFpQztRQUN0QyxJQUFJLENBQUMsRUFBRSxDQUFDLHFDQUFxQyxDQUFDLENBQUM7UUFDL0MsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksK0JBQStCO1FBQ3BDLElBQUksQ0FBQyxFQUFFLENBQUMsbUNBQW1DLENBQUMsQ0FBQztRQUM3QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksc0JBQXNCO1FBQzNCLElBQUksQ0FBQyxFQUFFLENBQUMsMEJBQTBCLENBQUMsQ0FBQztRQUNwQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxZQUFZO1FBQ2pCLElBQUksQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMxQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxjQUFjO1FBQ25CLElBQUksQ0FBQyxFQUFFLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUM1QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSwyQkFBMkI7UUFDaEMsSUFBSSxDQUFDLEVBQUUsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1FBQ3pDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFCQUFxQjtRQUMxQixJQUFJLENBQUMsRUFBRSxDQUFDLHlCQUF5QixDQUFDLENBQUM7UUFDbkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksVUFBVTtRQUNmLElBQUksQ0FBQyxFQUFFLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDeEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksa0RBQWtEO1FBQ3ZELElBQUksQ0FBQyxFQUFFLENBQUMsc0RBQXNELENBQUMsQ0FBQztRQUNoRSxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxxQ0FBcUM7UUFDMUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1FBQ25ELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksd0JBQXdCO1FBQzdCLElBQUksQ0FBQyxFQUFFLENBQUMsNEJBQTRCLENBQUMsQ0FBQztRQUN0QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx5Q0FBeUM7UUFDOUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyw2Q0FBNkMsQ0FBQyxDQUFDO1FBQ3ZELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksMEJBQTBCO1FBQy9CLElBQUksQ0FBQyxFQUFFLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUN4QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSSwwQkFBMEI7UUFDL0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGtCQUFrQjtRQUN2QixJQUFJLENBQUMsRUFBRSxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDaEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksMkJBQTJCO1FBQ2hDLElBQUksQ0FBQyxFQUFFLENBQUMsK0JBQStCLENBQUMsQ0FBQztRQUN6QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxvQ0FBb0M7UUFDekMsSUFBSSxDQUFDLEVBQUUsQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1FBQ2xELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHFDQUFxQztRQUMxQyxJQUFJLENBQUMsRUFBRSxDQUFDLHlDQUF5QyxDQUFDLENBQUM7UUFDbkQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0JBQW9CO1FBQ3pCLElBQUksQ0FBQyxFQUFFLENBQUMsd0JBQXdCLENBQUMsQ0FBQztRQUNsQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3Q0FBd0M7UUFDN0MsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHlCQUF5QjtRQUM5QixJQUFJLENBQUMsRUFBRSxDQUFDLDZCQUE2QixDQUFDLENBQUM7UUFDdkMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7Ozs7OztPQVNHO0lBQ0ksY0FBYztRQUNuQixJQUFJLENBQUMsRUFBRSxDQUFDLGtCQUFrQixDQUFDLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdDQUFnQztRQUNyQyxJQUFJLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxDQUFDLENBQUM7UUFDOUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksb0NBQW9DO1FBQ3pDLElBQUksQ0FBQyxFQUFFLENBQUMsd0NBQXdDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQkFBc0I7UUFDM0IsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFVBQVU7UUFDZixJQUFJLENBQUMsRUFBRSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQ3hCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7Ozs7O09BVUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxFQUFFLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUMzQixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZUFBZTtRQUNwQixJQUFJLENBQUMsRUFBRSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksNENBQTRDO1FBQ2pELElBQUksQ0FBQyxFQUFFLENBQUMsZ0RBQWdELENBQUMsQ0FBQztRQUMxRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxzQ0FBc0M7UUFDM0MsSUFBSSxDQUFDLEVBQUUsQ0FBQywwQ0FBMEMsQ0FBQyxDQUFDO1FBQ3BELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGdEQUFnRDtRQUNyRCxJQUFJLENBQUMsRUFBRSxDQUFDLG9EQUFvRCxDQUFDLENBQUM7UUFDOUQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksc0RBQXNEO1FBQzNELElBQUksQ0FBQyxFQUFFLENBQUMsMERBQTBELENBQUMsQ0FBQztRQUNwRSxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxnREFBZ0Q7UUFDckQsSUFBSSxDQUFDLEVBQUUsQ0FBQyxvREFBb0QsQ0FBQyxDQUFDO1FBQzlELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHVEQUF1RDtRQUM1RCxJQUFJLENBQUMsRUFBRSxDQUFDLDJEQUEyRCxDQUFDLENBQUM7UUFDckUsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksaUNBQWlDO1FBQ3RDLElBQUksQ0FBQyxFQUFFLENBQUMscUNBQXFDLENBQUMsQ0FBQztRQUMvQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSw0QkFBNEI7UUFDakMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxnQ0FBZ0MsQ0FBQyxDQUFDO1FBQzFDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLHdDQUF3QztRQUM3QyxJQUFJLENBQUMsRUFBRSxDQUFDLDRDQUE0QyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCO1FBQzFCLElBQUksQ0FBQyxFQUFFLENBQUMseUJBQXlCLENBQUMsQ0FBQztRQUNuQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxFQUFFLENBQUMsbUJBQW1CLENBQUMsQ0FBQztRQUM3QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSx3QkFBd0I7UUFDN0IsSUFBSSxDQUFDLEVBQUUsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO1FBQ3RDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLGFBQWE7UUFDbEIsSUFBSSxDQUFDLEVBQUUsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBQzNCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQTBGRDs7Ozs7Ozs7O09BU0c7SUFDSSxlQUFlLENBQUMsVUFBa0IsRUFBRSxlQUF1QixFQUFFLFNBQWtCO1FBQ3BGLElBQUksR0FBRyxHQUFHLHVFQUF1RSxDQUFDO1FBQ2xGLEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxlQUFlLENBQUMsQ0FBQztRQUN6RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxTQUFTLENBQUMsU0FBa0IsRUFBRSxTQUFrQjtRQUNyRCxJQUFJLEdBQUcsR0FBRyw2Q0FBNkMsQ0FBQztRQUN4RCxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1FBQ3BELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxTQUFTLElBQUksS0FBSyxDQUFDLENBQUM7UUFDdEQsT0FBTyxJQUFJLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ3RCLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFVBQVUsQ0FBQyxVQUFrQixFQUFFLFNBQWtCO1FBQ3RELElBQUksR0FBRyxHQUFHLCtDQUErQyxDQUFDO1FBQzFELEdBQUcsR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUMvQyxHQUFHLEdBQUcsR0FBRyxDQUFDLE9BQU8sQ0FBQyxjQUFjLEVBQUUsU0FBUyxJQUFJLEtBQUssQ0FBQyxDQUFDO1FBQ3RELE9BQU8sSUFBSSxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUN0QixDQUFDO0NBQ0Y7QUEvakNELGtCQStqQ0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBBY2Nlc3NMZXZlbExpc3QgfSBmcm9tIFwiLi4vc2hhcmVkL2FjY2Vzcy1sZXZlbFwiO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50IH0gZnJvbSBcIi4uL3NoYXJlZFwiO1xuXG4vKipcbiAqIFN0YXRlbWVudCBwcm92aWRlciBmb3Igc2VydmljZSBbc3NvXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvbGlzdF9hd3Nzc28uaHRtbCkuXG4gKlxuICogQHBhcmFtIHNpZCBbU0lEXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3NpZC5odG1sKSBvZiB0aGUgc3RhdGVtZW50XG4gKi9cbmV4cG9ydCBjbGFzcyBTc28gZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnQge1xuICBwdWJsaWMgc2VydmljZVByZWZpeCA9ICdzc28nO1xuXG4gIC8qKlxuICAgKiBTdGF0ZW1lbnQgcHJvdmlkZXIgZm9yIHNlcnZpY2UgW3Nzb10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2xpc3RfYXdzc3NvLmh0bWwpLlxuICAgKlxuICAgKiBAcGFyYW0gc2lkIFtTSURdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfc2lkLmh0bWwpIG9mIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIGNvbnN0cnVjdG9yIChzaWQ/OiBzdHJpbmcpIHtcbiAgICBzdXBlcihzaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIENvbm5lY3QgYSBkaXJlY3RvcnkgdG8gYmUgdXNlZCBieSBBV1MgU2luZ2xlIFNpZ24tT25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlRGlyZWN0b3J5KCkge1xuICAgIHRoaXMudG8oJ3NzbzpBc3NvY2lhdGVEaXJlY3RvcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gYXNzb2NpYXRpb24gYmV0d2VlbiBhIGRpcmVjdG9yeSB1c2VyIG9yIGdyb3VwIGFuZCBhIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvQXNzb2NpYXRlUHJvZmlsZSgpIHtcbiAgICB0aGlzLnRvKCdzc286QXNzb2NpYXRlUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEF0dGFjaGVzIGFuIEFXUyBtYW5hZ2VkIHBvbGljeSB0byBhIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9BdHRhY2hNYW5hZ2VkUG9saWN5VG9QZXJtaXNzaW9uU2V0KCkge1xuICAgIHRoaXMudG8oJ3NzbzpBdHRhY2hNYW5hZ2VkUG9saWN5VG9QZXJtaXNzaW9uU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQXNzaWducyBhY2Nlc3MgdG8gYSBQcmluY2lwYWwgZm9yIGEgc3BlY2lmaWVkIEFXUyBhY2NvdW50IHVzaW5nIGEgc3BlY2lmaWVkIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9DcmVhdGVBY2NvdW50QXNzaWdubWVudCgpIHtcbiAgICB0aGlzLnRvKCdzc286Q3JlYXRlQWNjb3VudEFzc2lnbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYW4gYXBwbGljYXRpb24gaW5zdGFuY2UgdG8gQVdTIFNpbmdsZSBTaWduLU9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFwcGxpY2F0aW9uSW5zdGFuY2UoKSB7XG4gICAgdGhpcy50bygnc3NvOkNyZWF0ZUFwcGxpY2F0aW9uSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBuZXcgY2VydGlmaWNhdGUgZm9yIGFuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VDZXJ0aWZpY2F0ZSgpIHtcbiAgICB0aGlzLnRvKCdzc286Q3JlYXRlQXBwbGljYXRpb25JbnN0YW5jZUNlcnRpZmljYXRlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgbWFuYWdlZCBhcHBsaWNhdGlvbiBpbnN0YW5jZSB0byBBV1MgU2luZ2xlIFNpZ24tT25cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlTWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2UoKSB7XG4gICAgdGhpcy50bygnc3NvOkNyZWF0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ3JlYXRlIGEgcGVybWlzc2lvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlUGVybWlzc2lvblNldCgpIHtcbiAgICB0aGlzLnRvKCdzc286Q3JlYXRlUGVybWlzc2lvblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhIHByb2ZpbGUgZm9yIGFuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0NyZWF0ZVByb2ZpbGUoKSB7XG4gICAgdGhpcy50bygnc3NvOkNyZWF0ZVByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYSBmZWRlcmF0aW9uIHRydXN0IGluIGEgdGFyZ2V0IGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvQ3JlYXRlVHJ1c3QoKSB7XG4gICAgdGhpcy50bygnc3NvOkNyZWF0ZVRydXN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlcyBhIFByaW5jaXBhbCdzIGFjY2VzcyBmcm9tIGEgc3BlY2lmaWVkIEFXUyBhY2NvdW50IHVzaW5nIGEgc3BlY2lmaWVkIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBY2NvdW50QXNzaWdubWVudCgpIHtcbiAgICB0aGlzLnRvKCdzc286RGVsZXRlQWNjb3VudEFzc2lnbm1lbnQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgdGhlIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZUFwcGxpY2F0aW9uSW5zdGFuY2UoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlbGV0ZUFwcGxpY2F0aW9uSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgYW4gaW5hY3RpdmUgb3IgZXhwaXJlZCBjZXJ0aWZpY2F0ZSBmcm9tIHRoZSBhcHBsaWNhdGlvbiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EZWxldGVBcHBsaWNhdGlvbkluc3RhbmNlQ2VydGlmaWNhdGUoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlbGV0ZUFwcGxpY2F0aW9uSW5zdGFuY2VDZXJ0aWZpY2F0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZXMgdGhlIGlubGluZSBwb2xpY3kgZnJvbSBhIHNwZWNpZmllZCBwZXJtaXNzaW9uIHNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlSW5saW5lUG9saWN5RnJvbVBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlbGV0ZUlubGluZVBvbGljeUZyb21QZXJtaXNzaW9uU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVsZXRlIHRoZSBtYW5hZ2VkIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlKCkge1xuICAgIHRoaXMudG8oJ3NzbzpEZWxldGVNYW5hZ2VkQXBwbGljYXRpb25JbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZSBhIHBlcm1pc3Npb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlbGV0ZVBlcm1pc3Npb25TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZWxldGUgdGhlIHBlcm1pc3Npb24gcG9saWN5IGFzc29jaWF0ZWQgd2l0aCBhIHBlcm1pc3Npb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0RlbGV0ZVBlcm1pc3Npb25zUG9saWN5KCkge1xuICAgIHRoaXMudG8oJ3NzbzpEZWxldGVQZXJtaXNzaW9uc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlbGV0ZSB0aGUgcHJvZmlsZSBmb3IgYW4gYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvRGVsZXRlUHJvZmlsZSgpIHtcbiAgICB0aGlzLnRvKCdzc286RGVsZXRlUHJvZmlsZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERlc2NyaWJlcyB0aGUgc3RhdHVzIG9mIHRoZSBhc3NpZ25tZW50IGNyZWF0aW9uIHJlcXVlc3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBY2NvdW50QXNzaWdubWVudENyZWF0aW9uU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpEZXNjcmliZUFjY291bnRBc3NpZ25tZW50Q3JlYXRpb25TdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgdGhlIHN0YXR1cyBvZiBhbiBhc3NpZ25tZW50IGRlbGV0aW9uIHJlcXVlc3QuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVBY2NvdW50QXNzaWdubWVudERlbGV0aW9uU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpEZXNjcmliZUFjY291bnRBc3NpZ25tZW50RGVsZXRpb25TdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBEZXNjcmliZXMgYSBwZXJtaXNzaW9uIHNldFxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0Rlc2NyaWJlUGVybWlzc2lvblNldCgpIHtcbiAgICB0aGlzLnRvKCdzc286RGVzY3JpYmVQZXJtaXNzaW9uU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogRGVzY3JpYmVzIHRoZSBzdGF0dXMgZm9yIHRoZSBnaXZlbiBQZXJtaXNzaW9uIFNldCBQcm92aXNpb25pbmcgcmVxdWVzdC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVBlcm1pc3Npb25TZXRQcm92aXNpb25pbmdTdGF0dXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlc2NyaWJlUGVybWlzc2lvblNldFByb3Zpc2lvbmluZ1N0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGFsbCB0aGUgcGVybWlzc2lvbnMgcG9saWNpZXMgYXNzb2NpYXRlZCB3aXRoIGEgcGVybWlzc2lvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EZXNjcmliZVBlcm1pc3Npb25zUG9saWNpZXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkRlc2NyaWJlUGVybWlzc2lvbnNQb2xpY2llcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIE9idGFpbnMgdGhlIHJlZ2lvbnMgd2hlcmUgeW91ciBvcmdhbml6YXRpb24gaGFzIGVuYWJsZWQgQVdTIFNpbmdsZSBTaWduLW9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvRGVzY3JpYmVSZWdpc3RlcmVkUmVnaW9ucygpIHtcbiAgICB0aGlzLnRvKCdzc286RGVzY3JpYmVSZWdpc3RlcmVkUmVnaW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERldGFjaGVzIHRoZSBhdHRhY2hlZCBBV1MgbWFuYWdlZCBwb2xpY3kgZnJvbSB0aGUgc3BlY2lmaWVkIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EZXRhY2hNYW5hZ2VkUG9saWN5RnJvbVBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOkRldGFjaE1hbmFnZWRQb2xpY3lGcm9tUGVybWlzc2lvblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2Fzc29jaWF0ZSBhIGRpcmVjdG9yeSB0byBiZSB1c2VkIGJ5IEFXUyBTaW5nbGUgU2lnbi1PblxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9EaXNhc3NvY2lhdGVEaXJlY3RvcnkoKSB7XG4gICAgdGhpcy50bygnc3NvOkRpc2Fzc29jaWF0ZURpcmVjdG9yeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2Fzc29jaWF0ZSBhIGRpcmVjdG9yeSB1c2VyIG9yIGdyb3VwIGZyb20gYSBwcm9maWxlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0Rpc2Fzc29jaWF0ZVByb2ZpbGUoKSB7XG4gICAgdGhpcy50bygnc3NvOkRpc2Fzc29jaWF0ZVByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZSBkZXRhaWxzIGZvciBhbiBhcHBsaWNhdGlvbiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0dldEFwcGxpY2F0aW9uSW5zdGFuY2UoKSB7XG4gICAgdGhpcy50bygnc3NvOkdldEFwcGxpY2F0aW9uSW5zdGFuY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZSBhcHBsaWNhdGlvbiB0ZW1wbGF0ZSBkZXRhaWxzXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvR2V0QXBwbGljYXRpb25UZW1wbGF0ZSgpIHtcbiAgICB0aGlzLnRvKCdzc286R2V0QXBwbGljYXRpb25UZW1wbGF0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIE9idGFpbnMgdGhlIGlubGluZSBwb2xpY3kgYXNzaWduZWQgdG8gdGhlIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0dldElubGluZVBvbGljeUZvclBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOkdldElubGluZVBvbGljeUZvclBlcm1pc3Npb25TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZSBkZXRhaWxzIGZvciBhbiBhcHBsaWNhdGlvbiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0dldE1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlKCkge1xuICAgIHRoaXMudG8oJ3NzbzpHZXRNYW5hZ2VkQXBwbGljYXRpb25JbnN0YW5jZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIE1mYSBEZXZpY2UgTWFuYWdlbWVudCBzZXR0aW5ncyBmb3IgdGhlIGRpcmVjdG9yeVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0dldE1mYURldmljZU1hbmFnZW1lbnRGb3JEaXJlY3RvcnkoKSB7XG4gICAgdGhpcy50bygnc3NvOkdldE1mYURldmljZU1hbmFnZW1lbnRGb3JEaXJlY3RvcnknKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZSBkZXRhaWxzIG9mIGEgcGVybWlzc2lvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQZXJtaXNzaW9uU2V0KCkge1xuICAgIHRoaXMudG8oJ3NzbzpHZXRQZXJtaXNzaW9uU2V0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmUgYWxsIHBlcm1pc3Npb24gcG9saWNpZXMgYXNzb2NpYXRlZCB3aXRoIGEgcGVybWlzc2lvbiBzZXRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHNzbzpEZXNjcmliZVBlcm1pc3Npb25zUG9saWNpZXNcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0dldFBlcm1pc3Npb25zUG9saWN5KCkge1xuICAgIHRoaXMudG8oJ3NzbzpHZXRQZXJtaXNzaW9uc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGEgcHJvZmlsZSBmb3IgYW4gYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRQcm9maWxlKCkge1xuICAgIHRoaXMudG8oJ3NzbzpHZXRQcm9maWxlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQ2hlY2sgaWYgQVdTIFNpbmdsZSBTaWduLU9uIGlzIGVuYWJsZWRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTU09TdGF0dXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkdldFNTT1N0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIHNoYXJlZCBjb25maWd1cmF0aW9uIGZvciB0aGUgY3VycmVudCBTU08gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRTaGFyZWRTc29Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3NzbzpHZXRTaGFyZWRTc29Db25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmUgY29uZmlndXJhdGlvbiBmb3IgdGhlIGN1cnJlbnQgU1NPIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvR2V0U3NvQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdzc286R2V0U3NvQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIHRoZSBmZWRlcmF0aW9uIHRydXN0IGluIGEgdGFyZ2V0IGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9HZXRUcnVzdCgpIHtcbiAgICB0aGlzLnRvKCdzc286R2V0VHJ1c3QnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIGFwcGxpY2F0aW9uIGluc3RhbmNlIGJ5IHVwbG9hZGluZyBhbiBhcHBsaWNhdGlvbiBTQU1MIG1ldGFkYXRhIGZpbGUgcHJvdmlkZWQgYnkgdGhlIHNlcnZpY2UgcHJvdmlkZXJcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvSW1wb3J0QXBwbGljYXRpb25JbnN0YW5jZVNlcnZpY2VQcm92aWRlck1ldGFkYXRhKCkge1xuICAgIHRoaXMudG8oJ3NzbzpJbXBvcnRBcHBsaWNhdGlvbkluc3RhbmNlU2VydmljZVByb3ZpZGVyTWV0YWRhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgc3RhdHVzIG9mIHRoZSBBV1MgYWNjb3VudCBhc3NpZ25tZW50IGNyZWF0aW9uIHJlcXVlc3RzIGZvciBhIHNwZWNpZmllZCBTU08gaW5zdGFuY2UuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFjY291bnRBc3NpZ25tZW50Q3JlYXRpb25TdGF0dXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RBY2NvdW50QXNzaWdubWVudENyZWF0aW9uU3RhdHVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgdGhlIHN0YXR1cyBvZiB0aGUgQVdTIGFjY291bnQgYXNzaWdubWVudCBkZWxldGlvbiByZXF1ZXN0cyBmb3IgYSBzcGVjaWZpZWQgU1NPIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBY2NvdW50QXNzaWdubWVudERlbGV0aW9uU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0QWNjb3VudEFzc2lnbm1lbnREZWxldGlvblN0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBhc3NpZ25lZSBvZiB0aGUgc3BlY2lmaWVkIEFXUyBhY2NvdW50IHdpdGggdGhlIHNwZWNpZmllZCBwZXJtaXNzaW9uIHNldC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QWNjb3VudEFzc2lnbm1lbnRzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0QWNjb3VudEFzc2lnbm1lbnRzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogTGlzdHMgYWxsIHRoZSBBV1MgYWNjb3VudHMgd2hlcmUgdGhlIHNwZWNpZmllZCBwZXJtaXNzaW9uIHNldCBpcyBwcm92aXNpb25lZC5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QWNjb3VudHNGb3JQcm92aXNpb25lZFBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RBY2NvdW50c0ZvclByb3Zpc2lvbmVkUGVybWlzc2lvblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGFsbCBvZiB0aGUgY2VydGlmaWNhdGVzIGZvciBhIGdpdmVuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFwcGxpY2F0aW9uSW5zdGFuY2VDZXJ0aWZpY2F0ZXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RBcHBsaWNhdGlvbkluc3RhbmNlQ2VydGlmaWNhdGVzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmUgYWxsIGFwcGxpY2F0aW9uIGluc3RhbmNlc1xuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IExpc3RcbiAgICpcbiAgICogRGVwZW5kZW50IGFjdGlvbnM6XG4gICAqIC0gc3NvOkdldEFwcGxpY2F0aW9uSW5zdGFuY2VcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RBcHBsaWNhdGlvbkluc3RhbmNlcygpIHtcbiAgICB0aGlzLnRvKCdzc286TGlzdEFwcGxpY2F0aW9uSW5zdGFuY2VzJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0cmlldmUgYWxsIHN1cHBvcnRlZCBhcHBsaWNhdGlvbiB0ZW1wbGF0ZXNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIERlcGVuZGVudCBhY3Rpb25zOlxuICAgKiAtIHNzbzpHZXRBcHBsaWNhdGlvblRlbXBsYXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0QXBwbGljYXRpb25UZW1wbGF0ZXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RBcHBsaWNhdGlvblRlbXBsYXRlcycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGFsbCBzdXBwb3J0ZWQgYXBwbGljYXRpb25zXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdEFwcGxpY2F0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdzc286TGlzdEFwcGxpY2F0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGRldGFpbHMgYWJvdXQgdGhlIGRpcmVjdG9yeSBjb25uZWN0ZWQgdG8gQVdTIFNpbmdsZSBTaWduLU9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdERpcmVjdG9yeUFzc29jaWF0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdzc286TGlzdERpcmVjdG9yeUFzc29jaWF0aW9ucycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBTU08gSW5zdGFuY2VzIHRoYXQgdGhlIGNhbGxlciBoYXMgYWNjZXNzIHRvLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RJbnN0YW5jZXMoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RJbnN0YW5jZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgQVdTIG1hbmFnZWQgcG9saWNpZXMgdGhhdCBhcmUgYXR0YWNoZWQgdG8gYSBzcGVjaWZpZWQgcGVybWlzc2lvbiBzZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdE1hbmFnZWRQb2xpY2llc0luUGVybWlzc2lvblNldCgpIHtcbiAgICB0aGlzLnRvKCdzc286TGlzdE1hbmFnZWRQb2xpY2llc0luUGVybWlzc2lvblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIHRoZSBzdGF0dXMgb2YgdGhlIFBlcm1pc3Npb24gU2V0IFByb3Zpc2lvbmluZyByZXF1ZXN0cyBmb3IgYSBzcGVjaWZpZWQgU1NPIGluc3RhbmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQZXJtaXNzaW9uU2V0UHJvdmlzaW9uaW5nU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0UGVybWlzc2lvblNldFByb3Zpc2lvbmluZ1N0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIGFsbCBwZXJtaXNzaW9uIHNldHNcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UGVybWlzc2lvblNldHMoKSB7XG4gICAgdGhpcy50bygnc3NvOkxpc3RQZXJtaXNzaW9uU2V0cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIExpc3RzIGFsbCB0aGUgcGVybWlzc2lvbiBzZXRzIHRoYXQgYXJlIHByb3Zpc2lvbmVkIHRvIGEgc3BlY2lmaWVkIEFXUyBhY2NvdW50LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFJlYWRcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b0xpc3RQZXJtaXNzaW9uU2V0c1Byb3Zpc2lvbmVkVG9BY2NvdW50KCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0UGVybWlzc2lvblNldHNQcm92aXNpb25lZFRvQWNjb3VudCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHJpZXZlIHRoZSBkaXJlY3RvcnkgdXNlciBvciBncm91cCBhc3NvY2lhdGVkIHdpdGggdGhlIHByb2ZpbGVcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0UHJvZmlsZUFzc29jaWF0aW9ucygpIHtcbiAgICB0aGlzLnRvKCdzc286TGlzdFByb2ZpbGVBc3NvY2lhdGlvbnMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXRyaWV2ZSBhbGwgcHJvZmlsZXMgZm9yIGFuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogUmVhZFxuICAgKlxuICAgKiBEZXBlbmRlbnQgYWN0aW9uczpcbiAgICogLSBzc286R2V0UHJvZmlsZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvTGlzdFByb2ZpbGVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0UHJvZmlsZXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBMaXN0cyB0aGUgdGFncyB0aGF0IGFyZSBhdHRhY2hlZCB0byBhIHNwZWNpZmllZCByZXNvdXJjZS5cbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBSZWFkXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9MaXN0VGFnc0ZvclJlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3NzbzpMaXN0VGFnc0ZvclJlc291cmNlJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVGhlIHByb2Nlc3MgYnkgd2hpY2ggYSBzcGVjaWZpZWQgcGVybWlzc2lvbiBzZXQgaXMgcHJvdmlzaW9uZWQgdG8gdGhlIHNwZWNpZmllZCB0YXJnZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1Byb3Zpc2lvblBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOlByb3Zpc2lvblBlcm1pc3Npb25TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBdHRhY2hlcyBhbiBJQU0gaW5saW5lIHBvbGljeSB0byBhIHBlcm1pc3Npb24gc2V0LlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9QdXRJbmxpbmVQb2xpY3lUb1Blcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOlB1dElubGluZVBvbGljeVRvUGVybWlzc2lvblNldCcpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFB1dCBNZmEgRGV2aWNlIE1hbmFnZW1lbnQgc2V0dGluZ3MgZm9yIHRoZSBkaXJlY3RvcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvUHV0TWZhRGV2aWNlTWFuYWdlbWVudEZvckRpcmVjdG9yeSgpIHtcbiAgICB0aGlzLnRvKCdzc286UHV0TWZhRGV2aWNlTWFuYWdlbWVudEZvckRpcmVjdG9yeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHBvbGljeSB0byBhIHBlcm1pc3Npb24gc2V0XG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1B1dFBlcm1pc3Npb25zUG9saWN5KCkge1xuICAgIHRoaXMudG8oJ3NzbzpQdXRQZXJtaXNzaW9uc1BvbGljeScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEluaXRpYWxpemUgQVdTIFNpbmdsZSBTaWduLU9uXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1N0YXJ0U1NPKCkge1xuICAgIHRoaXMudG8oJ3NzbzpTdGFydFNTTycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIEFzc29jaWF0ZXMgYSBzZXQgb2YgdGFncyB3aXRoIGEgc3BlY2lmaWVkIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1RhZ1Jlc291cmNlKCkge1xuICAgIHRoaXMudG8oJ3NzbzpUYWdSZXNvdXJjZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIERpc2Fzc29jaWF0ZXMgYSBzZXQgb2YgdGFncyBmcm9tIGEgc3BlY2lmaWVkIHJlc291cmNlLlxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFRhZ2dpbmdcbiAgICpcbiAgICogUG9zc2libGUgY29uZGl0aW9uczpcbiAgICogLSAuaWZBd3NSZXF1ZXN0VGFnKClcbiAgICogLSAuaWZBd3NUYWdLZXlzKClcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VudGFnUmVzb3VyY2UoKSB7XG4gICAgdGhpcy50bygnc3NvOlVudGFnUmVzb3VyY2UnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXQgYSBjZXJ0aWZpY2F0ZSBhcyB0aGUgYWN0aXZlIG9uZSBmb3IgdGhpcyBhcHBsaWNhdGlvbiBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlQWN0aXZlQ2VydGlmaWNhdGUoKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VBY3RpdmVDZXJ0aWZpY2F0ZScpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBkaXNwbGF5IGRhdGEgb2YgYW4gYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZURpc3BsYXlEYXRhKCkge1xuICAgIHRoaXMudG8oJ3NzbzpVcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlRGlzcGxheURhdGEnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgZmVkZXJhdGlvbiByZXNwb25zZSBjb25maWd1cmF0aW9uIGZvciB0aGUgYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVJlc3BvbnNlQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdzc286VXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVJlc3BvbnNlQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBmZWRlcmF0aW9uIHJlc3BvbnNlIHNjaGVtYSBjb25maWd1cmF0aW9uIGZvciB0aGUgYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVJlc3BvbnNlU2NoZW1hQ29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdzc286VXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVJlc3BvbnNlU2NoZW1hQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBzZWN1cml0eSBkZXRhaWxzIGZvciB0aGUgYXBwbGljYXRpb24gaW5zdGFuY2VcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVNlY3VyaXR5Q29uZmlndXJhdGlvbigpIHtcbiAgICB0aGlzLnRvKCdzc286VXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVNlY3VyaXR5Q29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSBzZXJ2aWNlIHByb3ZpZGVyIHJlbGF0ZWQgY29uZmlndXJhdGlvbiBmb3IgdGhlIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VTZXJ2aWNlUHJvdmlkZXJDb25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3NzbzpVcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlU2VydmljZVByb3ZpZGVyQ29uZmlndXJhdGlvbicpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSB0aGUgc3RhdHVzIG9mIGFuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VTdGF0dXMoKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VTdGF0dXMnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIHVzZXIgYXR0cmlidXRlIG1hcHBpbmdzIGZvciB5b3VyIGNvbm5lY3RlZCBkaXJlY3RvcnlcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlRGlyZWN0b3J5QXNzb2NpYXRpb24oKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZURpcmVjdG9yeUFzc29jaWF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlIHRoZSBzdGF0dXMgb2YgYSBtYW5hZ2VkIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlU3RhdHVzKCkge1xuICAgIHRoaXMudG8oJ3NzbzpVcGRhdGVNYW5hZ2VkQXBwbGljYXRpb25JbnN0YW5jZVN0YXR1cycpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLyoqXG4gICAqIFVwZGF0ZSB0aGUgcGVybWlzc2lvbiBzZXQuXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVBlcm1pc3Npb25TZXQoKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZVBlcm1pc3Npb25TZXQnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIHByb2ZpbGUgZm9yIGFuIGFwcGxpY2F0aW9uIGluc3RhbmNlXG4gICAqXG4gICAqIEFjY2VzcyBMZXZlbDogV3JpdGVcbiAgICpcbiAgICogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3NpbmdsZXNpZ25vbi9sYXRlc3QvdXNlcmd1aWRlL2lhbS1hdXRoLWFjY2Vzcy11c2luZy1pZC1wb2xpY2llcy5odG1sI3BvbGljeWV4YW1wbGVcbiAgICovXG4gIHB1YmxpYyB0b1VwZGF0ZVByb2ZpbGUoKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZVByb2ZpbGUnKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBVcGRhdGUgdGhlIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBjdXJyZW50IFNTTyBpbnN0YW5jZVxuICAgKlxuICAgKiBBY2Nlc3MgTGV2ZWw6IFdyaXRlXG4gICAqXG4gICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zaW5nbGVzaWdub24vbGF0ZXN0L3VzZXJndWlkZS9pYW0tYXV0aC1hY2Nlc3MtdXNpbmctaWQtcG9saWNpZXMuaHRtbCNwb2xpY3lleGFtcGxlXG4gICAqL1xuICBwdWJsaWMgdG9VcGRhdGVTU09Db25maWd1cmF0aW9uKCkge1xuICAgIHRoaXMudG8oJ3NzbzpVcGRhdGVTU09Db25maWd1cmF0aW9uJyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogVXBkYXRlIHRoZSBmZWRlcmF0aW9uIHRydXN0IGluIGEgdGFyZ2V0IGFjY291bnRcbiAgICpcbiAgICogQWNjZXNzIExldmVsOiBXcml0ZVxuICAgKlxuICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc2luZ2xlc2lnbm9uL2xhdGVzdC91c2VyZ3VpZGUvaWFtLWF1dGgtYWNjZXNzLXVzaW5nLWlkLXBvbGljaWVzLmh0bWwjcG9saWN5ZXhhbXBsZVxuICAgKi9cbiAgcHVibGljIHRvVXBkYXRlVHJ1c3QoKSB7XG4gICAgdGhpcy50bygnc3NvOlVwZGF0ZVRydXN0Jyk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwcm90ZWN0ZWQgYWNjZXNzTGV2ZWxMaXN0OiBBY2Nlc3NMZXZlbExpc3QgPSB7XG4gICAgXCJXcml0ZVwiOiBbXG4gICAgICBcIkFzc29jaWF0ZURpcmVjdG9yeVwiLFxuICAgICAgXCJBc3NvY2lhdGVQcm9maWxlXCIsXG4gICAgICBcIkF0dGFjaE1hbmFnZWRQb2xpY3lUb1Blcm1pc3Npb25TZXRcIixcbiAgICAgIFwiQ3JlYXRlQWNjb3VudEFzc2lnbm1lbnRcIixcbiAgICAgIFwiQ3JlYXRlQXBwbGljYXRpb25JbnN0YW5jZVwiLFxuICAgICAgXCJDcmVhdGVBcHBsaWNhdGlvbkluc3RhbmNlQ2VydGlmaWNhdGVcIixcbiAgICAgIFwiQ3JlYXRlTWFuYWdlZEFwcGxpY2F0aW9uSW5zdGFuY2VcIixcbiAgICAgIFwiQ3JlYXRlUGVybWlzc2lvblNldFwiLFxuICAgICAgXCJDcmVhdGVQcm9maWxlXCIsXG4gICAgICBcIkNyZWF0ZVRydXN0XCIsXG4gICAgICBcIkRlbGV0ZUFjY291bnRBc3NpZ25tZW50XCIsXG4gICAgICBcIkRlbGV0ZUFwcGxpY2F0aW9uSW5zdGFuY2VcIixcbiAgICAgIFwiRGVsZXRlQXBwbGljYXRpb25JbnN0YW5jZUNlcnRpZmljYXRlXCIsXG4gICAgICBcIkRlbGV0ZUlubGluZVBvbGljeUZyb21QZXJtaXNzaW9uU2V0XCIsXG4gICAgICBcIkRlbGV0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlXCIsXG4gICAgICBcIkRlbGV0ZVBlcm1pc3Npb25TZXRcIixcbiAgICAgIFwiRGVsZXRlUGVybWlzc2lvbnNQb2xpY3lcIixcbiAgICAgIFwiRGVsZXRlUHJvZmlsZVwiLFxuICAgICAgXCJEZXRhY2hNYW5hZ2VkUG9saWN5RnJvbVBlcm1pc3Npb25TZXRcIixcbiAgICAgIFwiRGlzYXNzb2NpYXRlRGlyZWN0b3J5XCIsXG4gICAgICBcIkRpc2Fzc29jaWF0ZVByb2ZpbGVcIixcbiAgICAgIFwiSW1wb3J0QXBwbGljYXRpb25JbnN0YW5jZVNlcnZpY2VQcm92aWRlck1ldGFkYXRhXCIsXG4gICAgICBcIlByb3Zpc2lvblBlcm1pc3Npb25TZXRcIixcbiAgICAgIFwiUHV0SW5saW5lUG9saWN5VG9QZXJtaXNzaW9uU2V0XCIsXG4gICAgICBcIlB1dE1mYURldmljZU1hbmFnZW1lbnRGb3JEaXJlY3RvcnlcIixcbiAgICAgIFwiUHV0UGVybWlzc2lvbnNQb2xpY3lcIixcbiAgICAgIFwiU3RhcnRTU09cIixcbiAgICAgIFwiVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZUFjdGl2ZUNlcnRpZmljYXRlXCIsXG4gICAgICBcIlVwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VEaXNwbGF5RGF0YVwiLFxuICAgICAgXCJVcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlUmVzcG9uc2VDb25maWd1cmF0aW9uXCIsXG4gICAgICBcIlVwZGF0ZUFwcGxpY2F0aW9uSW5zdGFuY2VSZXNwb25zZVNjaGVtYUNvbmZpZ3VyYXRpb25cIixcbiAgICAgIFwiVXBkYXRlQXBwbGljYXRpb25JbnN0YW5jZVNlY3VyaXR5Q29uZmlndXJhdGlvblwiLFxuICAgICAgXCJVcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlU2VydmljZVByb3ZpZGVyQ29uZmlndXJhdGlvblwiLFxuICAgICAgXCJVcGRhdGVBcHBsaWNhdGlvbkluc3RhbmNlU3RhdHVzXCIsXG4gICAgICBcIlVwZGF0ZURpcmVjdG9yeUFzc29jaWF0aW9uXCIsXG4gICAgICBcIlVwZGF0ZU1hbmFnZWRBcHBsaWNhdGlvbkluc3RhbmNlU3RhdHVzXCIsXG4gICAgICBcIlVwZGF0ZVBlcm1pc3Npb25TZXRcIixcbiAgICAgIFwiVXBkYXRlUHJvZmlsZVwiLFxuICAgICAgXCJVcGRhdGVTU09Db25maWd1cmF0aW9uXCIsXG4gICAgICBcIlVwZGF0ZVRydXN0XCJcbiAgICBdLFxuICAgIFwiUmVhZFwiOiBbXG4gICAgICBcIkRlc2NyaWJlQWNjb3VudEFzc2lnbm1lbnRDcmVhdGlvblN0YXR1c1wiLFxuICAgICAgXCJEZXNjcmliZUFjY291bnRBc3NpZ25tZW50RGVsZXRpb25TdGF0dXNcIixcbiAgICAgIFwiRGVzY3JpYmVQZXJtaXNzaW9uU2V0XCIsXG4gICAgICBcIkRlc2NyaWJlUGVybWlzc2lvblNldFByb3Zpc2lvbmluZ1N0YXR1c1wiLFxuICAgICAgXCJEZXNjcmliZVBlcm1pc3Npb25zUG9saWNpZXNcIixcbiAgICAgIFwiRGVzY3JpYmVSZWdpc3RlcmVkUmVnaW9uc1wiLFxuICAgICAgXCJHZXRBcHBsaWNhdGlvbkluc3RhbmNlXCIsXG4gICAgICBcIkdldEFwcGxpY2F0aW9uVGVtcGxhdGVcIixcbiAgICAgIFwiR2V0SW5saW5lUG9saWN5Rm9yUGVybWlzc2lvblNldFwiLFxuICAgICAgXCJHZXRNYW5hZ2VkQXBwbGljYXRpb25JbnN0YW5jZVwiLFxuICAgICAgXCJHZXRNZmFEZXZpY2VNYW5hZ2VtZW50Rm9yRGlyZWN0b3J5XCIsXG4gICAgICBcIkdldFBlcm1pc3Npb25TZXRcIixcbiAgICAgIFwiR2V0UGVybWlzc2lvbnNQb2xpY3lcIixcbiAgICAgIFwiR2V0UHJvZmlsZVwiLFxuICAgICAgXCJHZXRTU09TdGF0dXNcIixcbiAgICAgIFwiR2V0U2hhcmVkU3NvQ29uZmlndXJhdGlvblwiLFxuICAgICAgXCJHZXRTc29Db25maWd1cmF0aW9uXCIsXG4gICAgICBcIkdldFRydXN0XCIsXG4gICAgICBcIkxpc3RBY2NvdW50QXNzaWdubWVudENyZWF0aW9uU3RhdHVzXCIsXG4gICAgICBcIkxpc3RBY2NvdW50QXNzaWdubWVudERlbGV0aW9uU3RhdHVzXCIsXG4gICAgICBcIkxpc3RBY2NvdW50QXNzaWdubWVudHNcIixcbiAgICAgIFwiTGlzdEFjY291bnRzRm9yUHJvdmlzaW9uZWRQZXJtaXNzaW9uU2V0XCIsXG4gICAgICBcIkxpc3RBcHBsaWNhdGlvbkluc3RhbmNlQ2VydGlmaWNhdGVzXCIsXG4gICAgICBcIkxpc3RBcHBsaWNhdGlvblRlbXBsYXRlc1wiLFxuICAgICAgXCJMaXN0QXBwbGljYXRpb25zXCIsXG4gICAgICBcIkxpc3REaXJlY3RvcnlBc3NvY2lhdGlvbnNcIixcbiAgICAgIFwiTGlzdEluc3RhbmNlc1wiLFxuICAgICAgXCJMaXN0TWFuYWdlZFBvbGljaWVzSW5QZXJtaXNzaW9uU2V0XCIsXG4gICAgICBcIkxpc3RQZXJtaXNzaW9uU2V0UHJvdmlzaW9uaW5nU3RhdHVzXCIsXG4gICAgICBcIkxpc3RQZXJtaXNzaW9uU2V0c1wiLFxuICAgICAgXCJMaXN0UGVybWlzc2lvblNldHNQcm92aXNpb25lZFRvQWNjb3VudFwiLFxuICAgICAgXCJMaXN0UHJvZmlsZUFzc29jaWF0aW9uc1wiLFxuICAgICAgXCJMaXN0UHJvZmlsZXNcIixcbiAgICAgIFwiTGlzdFRhZ3NGb3JSZXNvdXJjZVwiXG4gICAgXSxcbiAgICBcIkxpc3RcIjogW1xuICAgICAgXCJMaXN0QXBwbGljYXRpb25JbnN0YW5jZXNcIlxuICAgIF0sXG4gICAgXCJUYWdnaW5nXCI6IFtcbiAgICAgIFwiVGFnUmVzb3VyY2VcIixcbiAgICAgIFwiVW50YWdSZXNvdXJjZVwiXG4gICAgXVxuICB9O1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBQZXJtaXNzaW9uU2V0IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGluc3RhbmNlSWQgLSBJZGVudGlmaWVyIGZvciB0aGUgaW5zdGFuY2VJZC5cbiAgICogQHBhcmFtIHBlcm1pc3Npb25TZXRJZCAtIElkZW50aWZpZXIgZm9yIHRoZSBwZXJtaXNzaW9uU2V0SWQuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKlxuICAgKiBQb3NzaWJsZSBjb25kaXRpb25zOlxuICAgKiAtIC5pZkF3c1Jlc291cmNlVGFnKClcbiAgICovXG4gIHB1YmxpYyBvblBlcm1pc3Npb25TZXQoaW5zdGFuY2VJZDogc3RyaW5nLCBwZXJtaXNzaW9uU2V0SWQ6IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnNzbzo6OnBlcm1pc3Npb25TZXQvJHtJbnN0YW5jZUlkfS8ke1Blcm1pc3Npb25TZXRJZH0nO1xuICAgIGFybiA9IGFybi5yZXBsYWNlKCcke0luc3RhbmNlSWR9JywgaW5zdGFuY2VJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGVybWlzc2lvblNldElkfScsIHBlcm1pc3Npb25TZXRJZCk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBBY2NvdW50IHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIGFjY291bnRJZCAtIEFjY291bnQgb2YgdGhlIHJlc291cmNlOyBkZWZhdWx0cyB0byBlbXB0eSBzdHJpbmc6IGFsbCBhY2NvdW50cy5cbiAgICogQHBhcmFtIHBhcnRpdGlvbiAtIFBhcnRpdGlvbiBvZiB0aGUgQVdTIGFjY291bnQgW2F3cywgYXdzLWNuLCBhd3MtdXMtZ292XTsgZGVmYXVsdHMgdG8gYGF3c2AuXG4gICAqL1xuICBwdWJsaWMgb25BY2NvdW50KGFjY291bnRJZD86IHN0cmluZywgcGFydGl0aW9uPzogc3RyaW5nKSB7XG4gICAgdmFyIGFybiA9ICdhcm46JHtQYXJ0aXRpb259OnNzbzo6OmFjY291bnQvJHtBY2NvdW50SWR9JztcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtBY2NvdW50SWR9JywgYWNjb3VudElkIHx8ICcqJyk7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7UGFydGl0aW9ufScsIHBhcnRpdGlvbiB8fCAnYXdzJyk7XG4gICAgcmV0dXJuIHRoaXMub24oYXJuKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgcmVzb3VyY2Ugb2YgdHlwZSBJbnN0YW5jZSB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBpbnN0YW5jZUlkIC0gSWRlbnRpZmllciBmb3IgdGhlIGluc3RhbmNlSWQuXG4gICAqIEBwYXJhbSBwYXJ0aXRpb24gLSBQYXJ0aXRpb24gb2YgdGhlIEFXUyBhY2NvdW50IFthd3MsIGF3cy1jbiwgYXdzLXVzLWdvdl07IGRlZmF1bHRzIHRvIGBhd3NgLlxuICAgKi9cbiAgcHVibGljIG9uSW5zdGFuY2UoaW5zdGFuY2VJZDogc3RyaW5nLCBwYXJ0aXRpb24/OiBzdHJpbmcpIHtcbiAgICB2YXIgYXJuID0gJ2Fybjoke1BhcnRpdGlvbn06c3NvOjo6aW5zdGFuY2UvJHtJbnN0YW5jZUlkfSc7XG4gICAgYXJuID0gYXJuLnJlcGxhY2UoJyR7SW5zdGFuY2VJZH0nLCBpbnN0YW5jZUlkKTtcbiAgICBhcm4gPSBhcm4ucmVwbGFjZSgnJHtQYXJ0aXRpb259JywgcGFydGl0aW9uIHx8ICdhd3MnKTtcbiAgICByZXR1cm4gdGhpcy5vbihhcm4pO1xuICB9XG59XG4iXX0=