"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trustedadvisor = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Trustedadvisor extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [trustedadvisor](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awstrustedadvisor.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'trustedadvisor';
        this.accessLevelList = {
            "Read": [
                "DescribeAccount",
                "DescribeAccountAccess",
                "DescribeCheckItems",
                "DescribeCheckRefreshStatuses",
                "DescribeCheckSummaries",
                "DescribeChecks",
                "DescribeNotificationPreferences",
                "DescribeOrganization",
                "DescribeOrganizationAccounts",
                "DescribeReports",
                "DescribeServiceMetadata"
            ],
            "Write": [
                "ExcludeCheckItems",
                "GenerateReport",
                "IncludeCheckItems",
                "RefreshCheck",
                "SetAccountAccess",
                "SetOrganizationAccess",
                "UpdateNotificationPreferences"
            ]
        };
    }
    /**
     * Grants permission to view the AWS Support plan and various AWS Trusted Advisor preferences
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccount() {
        this.to('trustedadvisor:DescribeAccount');
        return this;
    }
    /**
     * Grants permission to view if the AWS account has enabled or disabled AWS Trusted Advisor
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeAccountAccess() {
        this.to('trustedadvisor:DescribeAccountAccess');
        return this;
    }
    /**
     * Grants permission to view details for the check items
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckItems() {
        this.to('trustedadvisor:DescribeCheckItems');
        return this;
    }
    /**
     * Grants permission to view the refresh statuses for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckRefreshStatuses() {
        this.to('trustedadvisor:DescribeCheckRefreshStatuses');
        return this;
    }
    /**
     * Grants permission to view AWS Trusted Advisor check summaries
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeCheckSummaries() {
        this.to('trustedadvisor:DescribeCheckSummaries');
        return this;
    }
    /**
     * Grants permission to view details for AWS Trusted Advisor checks
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeChecks() {
        this.to('trustedadvisor:DescribeChecks');
        return this;
    }
    /**
     * Grants permission to view the notification preferences for the AWS account
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeNotificationPreferences() {
        this.to('trustedadvisor:DescribeNotificationPreferences');
        return this;
    }
    /**
     * Grants permission to view if the AWS account meets the requirements to enable the organizational view feature
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganization() {
        this.to('trustedadvisor:DescribeOrganization');
        return this;
    }
    /**
     * Grants permission to view the linked AWS accounts that are in the organization
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeOrganizationAccounts() {
        this.to('trustedadvisor:DescribeOrganizationAccounts');
        return this;
    }
    /**
     * Grants permission to view details for organizational view reports, such as the report name, runtime, date created, status, and format
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeReports() {
        this.to('trustedadvisor:DescribeReports');
        return this;
    }
    /**
     * Grants permission to view information about organizational view reports, such as the AWS Regions, check categories, check names, and resource statuses
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toDescribeServiceMetadata() {
        this.to('trustedadvisor:DescribeServiceMetadata');
        return this;
    }
    /**
     * Grants permission to exclude recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toExcludeCheckItems() {
        this.to('trustedadvisor:ExcludeCheckItems');
        return this;
    }
    /**
     * Grants permission to create a report for AWS Trusted Advisor checks in your organization
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toGenerateReport() {
        this.to('trustedadvisor:GenerateReport');
        return this;
    }
    /**
     * Grants permission to include recommendations for AWS Trusted Advisor checks
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toIncludeCheckItems() {
        this.to('trustedadvisor:IncludeCheckItems');
        return this;
    }
    /**
     * Grants permission to refresh an AWS Trusted Advisor check
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toRefreshCheck() {
        this.to('trustedadvisor:RefreshCheck');
        return this;
    }
    /**
     * Grants permission to enable or disable AWS Trusted Advisor for the account
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetAccountAccess() {
        this.to('trustedadvisor:SetAccountAccess');
        return this;
    }
    /**
     * Grants permission to enable the organizational view feature for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toSetOrganizationAccess() {
        this.to('trustedadvisor:SetOrganizationAccess');
        return this;
    }
    /**
     * Grants permission to update notification preferences for AWS Trusted Advisor
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/awssupport/latest/user/security-trusted-advisor.html#trusted-advisor-operations
     */
    toUpdateNotificationPreferences() {
        this.to('trustedadvisor:UpdateNotificationPreferences');
        return this;
    }
    /**
     * Adds a resource of type checks to the statement
     *
     * https://docs.aws.amazon.com/awssupport/latest/APIReference/API_TrustedAdvisorCheckDescription.html
     *
     * @param categoryCode - Identifier for the categoryCode.
     * @param checkId - Identifier for the checkId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onChecks(categoryCode, checkId, account, region, partition) {
        var arn = 'arn:${Partition}:trustedadvisor:${Region}:${Account}:checks/${CategoryCode}/${CheckId}';
        arn = arn.replace('${CategoryCode}', categoryCode);
        arn = arn.replace('${CheckId}', checkId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Trustedadvisor = Trustedadvisor;
//# sourceMappingURL=data:application/json;base64,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