"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wafv2 = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [wafv2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswafv2.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class Wafv2 extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [wafv2](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_awswafv2.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'wafv2';
        this.accessLevelList = {
            "Write": [
                "AssociateWebACL",
                "CreateIPSet",
                "CreateRegexPatternSet",
                "CreateRuleGroup",
                "DeleteFirewallManagerRuleGroups",
                "DeleteIPSet",
                "DeleteLoggingConfiguration",
                "DeleteRegexPatternSet",
                "DeleteRuleGroup",
                "DisassociateFirewallManager",
                "DisassociateWebACL",
                "PutFirewallManagerRuleGroups",
                "PutLoggingConfiguration",
                "UpdateIPSet",
                "UpdateRegexPatternSet",
                "UpdateRuleGroup"
            ],
            "Read": [
                "CheckCapacity",
                "GetIPSet",
                "GetLoggingConfiguration",
                "GetPermissionPolicy",
                "GetRateBasedStatementManagedKeys",
                "GetRegexPatternSet",
                "GetRuleGroup",
                "GetSampledRequests",
                "GetWebACL",
                "GetWebACLForResource",
                "ListTagsForResource"
            ],
            "Permissions management": [
                "CreateWebACL",
                "DeletePermissionPolicy",
                "DeleteWebACL",
                "PutPermissionPolicy",
                "UpdateWebACL"
            ],
            "List": [
                "DescribeManagedRuleGroup",
                "ListAvailableManagedRuleGroups",
                "ListIPSets",
                "ListLoggingConfigurations",
                "ListRegexPatternSets",
                "ListResourcesForWebACL",
                "ListRuleGroups",
                "ListWebACLs"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to associate a WebACL with a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_AssociateWebACL.html
     */
    toAssociateWebACL() {
        this.to('wafv2:AssociateWebACL');
        return this;
    }
    /**
     * Grants permission to calculate web ACL capacity unit (WCU) requirements for a specified scope and set of rules.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CheckCapacity.html
     */
    toCheckCapacity() {
        this.to('wafv2:CheckCapacity');
        return this;
    }
    /**
     * Grants permission to create an IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateIPSet.html
     */
    toCreateIPSet() {
        this.to('wafv2:CreateIPSet');
        return this;
    }
    /**
     * Grants permission to create a RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRegexPatternSet.html
     */
    toCreateRegexPatternSet() {
        this.to('wafv2:CreateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to create a RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateRuleGroup.html
     */
    toCreateRuleGroup() {
        this.to('wafv2:CreateRuleGroup');
        return this;
    }
    /**
     * Grants permission to create a WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_CreateWebACL.html
     */
    toCreateWebACL() {
        this.to('wafv2:CreateWebACL');
        return this;
    }
    /**
     * Grants permission to delete specified FirewallManagedRulesGroups from the specified WebACL if not managed by Firewall Manager anymore.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteFirewallManagerRuleGroups.html
     */
    toDeleteFirewallManagerRuleGroups() {
        this.to('wafv2:DeleteFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to delete the specified IPSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteIPSet.html
     */
    toDeleteIPSet() {
        this.to('wafv2:DeleteIPSet');
        return this;
    }
    /**
     * Grants permission to delete the LoggingConfiguration from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteLoggingConfiguration.html
     */
    toDeleteLoggingConfiguration() {
        this.to('wafv2:DeleteLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to delete the PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeletePermissionPolicy.html
     */
    toDeletePermissionPolicy() {
        this.to('wafv2:DeletePermissionPolicy');
        return this;
    }
    /**
     * Grants permission to delete the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRegexPatternSet.html
     */
    toDeleteRegexPatternSet() {
        this.to('wafv2:DeleteRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to delete the specified RuleGroup.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteRuleGroup.html
     */
    toDeleteRuleGroup() {
        this.to('wafv2:DeleteRuleGroup');
        return this;
    }
    /**
     * Grants permission to delete the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DeleteWebACL.html
     */
    toDeleteWebACL() {
        this.to('wafv2:DeleteWebACL');
        return this;
    }
    /**
     * Grants permission to view high-level information for a managed rule group.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DescribeManagedRuleGroup.html
     */
    toDescribeManagedRuleGroup() {
        this.to('wafv2:DescribeManagedRuleGroup');
        return this;
    }
    /**
     * Grants permission to disassociate Firewall Manager from the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateFirewallManager.html
     */
    toDisassociateFirewallManager() {
        this.to('wafv2:DisassociateFirewallManager');
        return this;
    }
    /**
     * Grants permission disassociate a WebACL from an application resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_DisassociateWebACL.html
     */
    toDisassociateWebACL() {
        this.to('wafv2:DisassociateWebACL');
        return this;
    }
    /**
     * Grants permission to view details about the specified IPSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetIPSet.html
     */
    toGetIPSet() {
        this.to('wafv2:GetIPSet');
        return this;
    }
    /**
     * Grants permission to view LoggingConfiguration about the specified WebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetLoggingConfiguration.html
     */
    toGetLoggingConfiguration() {
        this.to('wafv2:GetLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to view PermissionPolicy on the specified RuleGroup.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetPermissionPolicy.html
     */
    toGetPermissionPolicy() {
        this.to('wafv2:GetPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to view the keys that are currently blocked by a rate-based rule.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRateBasedStatementManagedKeys.html
     */
    toGetRateBasedStatementManagedKeys() {
        this.to('wafv2:GetRateBasedStatementManagedKeys');
        return this;
    }
    /**
     * Grants permission to view details about the specified RegexPatternSet.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRegexPatternSet.html
     */
    toGetRegexPatternSet() {
        this.to('wafv2:GetRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to view details about the specified RuleGroup.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetRuleGroup.html
     */
    toGetRuleGroup() {
        this.to('wafv2:GetRuleGroup');
        return this;
    }
    /**
     * Grants permission to view detailed information about a specified number of requests--a sample--that AWS WAF randomly selects from among the first 5,000 requests that your AWS resource received during a time range that you choose.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetSampledRequests.html
     */
    toGetSampledRequests() {
        this.to('wafv2:GetSampledRequests');
        return this;
    }
    /**
     * Grants permission to view details about the specified GetWebACL.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACL.html
     */
    toGetWebACL() {
        this.to('wafv2:GetWebACL');
        return this;
    }
    /**
     * Grants permission to view the WebACL for the specified resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_GetWebACLForResource.html
     */
    toGetWebACLForResource() {
        this.to('wafv2:GetWebACLForResource');
        return this;
    }
    /**
     * Grants permission to view an array of managed rule groups that are available for you to use.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListAvailableManagedRuleGroups.html
     */
    toListAvailableManagedRuleGroups() {
        this.to('wafv2:ListAvailableManagedRuleGroups');
        return this;
    }
    /**
     * Grants permission to view an array of IPSetSummary objects for the IP sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListIPSets.html
     */
    toListIPSets() {
        this.to('wafv2:ListIPSets');
        return this;
    }
    /**
     * Grants permission to view an array of your LoggingConfiguration objects.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListLoggingConfigurations.html
     */
    toListLoggingConfigurations() {
        this.to('wafv2:ListLoggingConfigurations');
        return this;
    }
    /**
     * Grants permission to view an array of RegexPatternSetSummary objects for the regex pattern sets that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRegexPatternSets.html
     */
    toListRegexPatternSets() {
        this.to('wafv2:ListRegexPatternSets');
        return this;
    }
    /**
     * Grants permission to view an array of the Amazon Resource Names (ARNs) for the resources that are associated with the specified web ACL.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListResourcesForWebACL.html
     */
    toListResourcesForWebACL() {
        this.to('wafv2:ListResourcesForWebACL');
        return this;
    }
    /**
     * Grants permission to view an array of RuleGroupSummary objects for the rule groups that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListRuleGroups.html
     */
    toListRuleGroups() {
        this.to('wafv2:ListRuleGroups');
        return this;
    }
    /**
     * Grants permission to lists tag for the specified resource.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('wafv2:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to view an array of WebACLSummary objects for the web ACLs that you manage.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_ListWebACLs.html
     */
    toListWebACLs() {
        this.to('wafv2:ListWebACLs');
        return this;
    }
    /**
     * Grants permission to create FirewallManagedRulesGroups in the specified WebACL.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutFirewallManagerRuleGroups.html
     */
    toPutFirewallManagerRuleGroups() {
        this.to('wafv2:PutFirewallManagerRuleGroups');
        return this;
    }
    /**
     * Grants permission to enables the specified LoggingConfiguration, to start logging from a web ACL.
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:CreateServiceLinkedRole
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutLoggingConfiguration.html
     */
    toPutLoggingConfiguration() {
        this.to('wafv2:PutLoggingConfiguration');
        return this;
    }
    /**
     * Grants permission to attach the specified IAM policy to the specified resource. The only supported use for this action is to share a RuleGroup across accounts.
     *
     * Access Level: Permissions management
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_PutPermissionPolicy.html
     */
    toPutPermissionPolicy() {
        this.to('wafv2:PutPermissionPolicy');
        return this;
    }
    /**
     * Grants permission to associates tags with the specified AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('wafv2:TagResource');
        return this;
    }
    /**
     * Grants permission to disassociates tags from an AWS resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('wafv2:UntagResource');
        return this;
    }
    /**
     * Grants permission to update the specified IPSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateIPSet.html
     */
    toUpdateIPSet() {
        this.to('wafv2:UpdateIPSet');
        return this;
    }
    /**
     * Grants permission to update the specified RegexPatternSet.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRegexPatternSet.html
     */
    toUpdateRegexPatternSet() {
        this.to('wafv2:UpdateRegexPatternSet');
        return this;
    }
    /**
     * Grants permission to update the specified RuleGroup.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateRuleGroup.html
     */
    toUpdateRuleGroup() {
        this.to('wafv2:UpdateRuleGroup');
        return this;
    }
    /**
     * Grants permission to update the specified WebACL.
     *
     * Access Level: Permissions management
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_UpdateWebACL.html
     */
    toUpdateWebACL() {
        this.to('wafv2:UpdateWebACL');
        return this;
    }
    /**
     * Adds a resource of type webacl to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onWebacl(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/webacl/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ipset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_IPSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIpset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/ipset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type rulegroup to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RuleGroup.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRulegroup(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/rulegroup/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type regexpatternset to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_RegexPatternSet.html
     *
     * @param scope - Identifier for the scope.
     * @param name - Identifier for the name.
     * @param id - Identifier for the id.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRegexpatternset(scope, name, id, account, region, partition) {
        var arn = 'arn:${Partition}:wafv2:${Region}:${Account}:${Scope}/regexpatternset/${Name}/${Id}';
        arn = arn.replace('${Scope}', scope);
        arn = arn.replace('${Name}', name);
        arn = arn.replace('${Id}', id);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type loadbalancer/app/ to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param loadBalancerName - Identifier for the loadBalancerName.
     * @param loadBalancerId - Identifier for the loadBalancerId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onLoadbalancerApp(loadBalancerName, loadBalancerId, account, region, partition) {
        var arn = 'arn:${Partition}:elasticloadbalancing:${Region}:${Account}:loadbalancer/app/${LoadBalancerName}/${LoadBalancerId}';
        arn = arn.replace('${LoadBalancerName}', loadBalancerName);
        arn = arn.replace('${LoadBalancerId}', loadBalancerId);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type apigateway to the statement
     *
     * https://docs.aws.amazon.com/waf/latest/APIReference/API_WebACL.html
     *
     * @param apiId - Identifier for the apiId.
     * @param stageName - Identifier for the stageName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onApigateway(apiId, stageName, account, region, partition) {
        var arn = 'arn:${Partition}:apigateway:${Region}:${Account}:/restapis/${ApiId}/stages/${StageName}';
        arn = arn.replace('${ApiId}', apiId);
        arn = arn.replace('${StageName}', stageName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.Wafv2 = Wafv2;
//# sourceMappingURL=data:application/json;base64,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