"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessAnalyzer = void 0;
const shared_1 = require("../shared");
/**
 * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_iamaccessanalyzer.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
class AccessAnalyzer extends shared_1.PolicyStatement {
    /**
     * Statement provider for service [access-analyzer](https://docs.aws.amazon.com/IAM/latest/UserGuide/list_iamaccessanalyzer.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid) {
        super(sid);
        this.servicePrefix = 'access-analyzer';
        this.accessLevelList = {
            "Write": [
                "CreateAnalyzer",
                "CreateArchiveRule",
                "DeleteAnalyzer",
                "DeleteArchiveRule",
                "StartResourceScan",
                "UpdateArchiveRule",
                "UpdateFindings"
            ],
            "Read": [
                "GetAnalyzedResource",
                "GetAnalyzer",
                "GetArchiveRule",
                "GetFinding",
                "ListAnalyzedResources",
                "ListFindings",
                "ListTagsForResource"
            ],
            "List": [
                "ListAnalyzers",
                "ListArchiveRules"
            ],
            "Tagging": [
                "TagResource",
                "UntagResource"
            ]
        };
    }
    /**
     * Grants permission to create an analyzer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateAnalyzer.html
     */
    toCreateAnalyzer() {
        this.to('access-analyzer:CreateAnalyzer');
        return this;
    }
    /**
     * Grants permission to create an archive rule for the specified analyzer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_CreateArchiveRule.html
     */
    toCreateArchiveRule() {
        this.to('access-analyzer:CreateArchiveRule');
        return this;
    }
    /**
     * Grants permission to delete the specified analyzer.
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteAnalyzer.html
     */
    toDeleteAnalyzer() {
        this.to('access-analyzer:DeleteAnalyzer');
        return this;
    }
    /**
     * Grants permission to delete archive rules for the specified analyzer.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_DeleteArchiveRule.html
     */
    toDeleteArchiveRule() {
        this.to('access-analyzer:DeleteArchiveRule');
        return this;
    }
    /**
     * Grants permission to retrieve information about an analyzed resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzedResource.html
     */
    toGetAnalyzedResource() {
        this.to('access-analyzer:GetAnalyzedResource');
        return this;
    }
    /**
     * Grants permission to retrieve information about analyzers.
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetAnalyzer.html
     */
    toGetAnalyzer() {
        this.to('access-analyzer:GetAnalyzer');
        return this;
    }
    /**
     * Grants permission to retrieve information about archive rules for the specified analyzer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetArchiveRule.html
     */
    toGetArchiveRule() {
        this.to('access-analyzer:GetArchiveRule');
        return this;
    }
    /**
     * Grants permission to retrieve findings.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_GetFinding.html
     */
    toGetFinding() {
        this.to('access-analyzer:GetFinding');
        return this;
    }
    /**
     * Grants permission to retrieve a list of resources that have been analyzed.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzedResources.html
     */
    toListAnalyzedResources() {
        this.to('access-analyzer:ListAnalyzedResources');
        return this;
    }
    /**
     * Grants permission to retrieves a list of analyzers.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListAnalyzers.html
     */
    toListAnalyzers() {
        this.to('access-analyzer:ListAnalyzers');
        return this;
    }
    /**
     * Grants permission to retrieve a list of archive rules from an analyzer.
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListArchiveRules.html
     */
    toListArchiveRules() {
        this.to('access-analyzer:ListArchiveRules');
        return this;
    }
    /**
     * Grants permission to retrieve a list of findings from an analyzer.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListFindings.html
     */
    toListFindings() {
        this.to('access-analyzer:ListFindings');
        return this;
    }
    /**
     * Grants permission to retrieve a list of tags applied to a resource.
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource() {
        this.to('access-analyzer:ListTagsForResource');
        return this;
    }
    /**
     * Grants permission to start a scan of the policies applied to a resource.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_StartResourceScan.html
     */
    toStartResourceScan() {
        this.to('access-analyzer:StartResourceScan');
        return this;
    }
    /**
     * Grants permission to add a tag to a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_TagResource.html
     */
    toTagResource() {
        this.to('access-analyzer:TagResource');
        return this;
    }
    /**
     * Grants permission to remove a tag from a resource.
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UntagResource.html
     */
    toUntagResource() {
        this.to('access-analyzer:UntagResource');
        return this;
    }
    /**
     * Grants permission to modify an archive rule.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateArchiveRule.html
     */
    toUpdateArchiveRule() {
        this.to('access-analyzer:UpdateArchiveRule');
        return this;
    }
    /**
     * Grants permission to modify findings.
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/access-analyzer/latest/APIReference/API_UpdateFindings.html
     */
    toUpdateFindings() {
        this.to('access-analyzer:UpdateFindings');
        return this;
    }
    /**
     * Adds a resource of type Analyzer to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAnalyzer(analyzerName, account, region, partition) {
        var arn = 'arn:${Partition}:access-analyzer:${Region}:${Account}:analyzer/${AnalyzerName}';
        arn = arn.replace('${AnalyzerName}', analyzerName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
    /**
     * Adds a resource of type ArchiveRule to the statement
     *
     * https://docs.aws.amazon.com/IAM/latest/UserGuide/access-analyzer-getting-started.html#permission-resources
     *
     * @param analyzerName - Identifier for the analyzerName.
     * @param ruleName - Identifier for the ruleName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onArchiveRule(analyzerName, ruleName, account, region, partition) {
        var arn = 'arn:${Partition}:access-analyzer:${Region}:${Account}:analyzer/${AnalyzerName}/archive-rule/${RuleName}';
        arn = arn.replace('${AnalyzerName}', analyzerName);
        arn = arn.replace('${RuleName}', ruleName);
        arn = arn.replace('${Account}', account || '*');
        arn = arn.replace('${Region}', region || '*');
        arn = arn.replace('${Partition}', partition || 'aws');
        return this.on(arn);
    }
}
exports.AccessAnalyzer = AccessAnalyzer;
//# sourceMappingURL=data:application/json;base64,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