"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipals = false;
        this.principals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.principals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.principals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.principals)) {
            this.principals[prefix] = [];
        }
        this.principals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds an account principal to the statement
     *
     * @param account ID of the AWS account
     */
    forAccount(account) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`);
    }
    /**
     * Adds a [federated](https://aws.amazon.com/identity/federation/) (web identity) principal to the statement
     *
     * @param provider ID of the AWS account
     */
    forFederated(provider) {
        return this.addPrincipal(PrincipalType.FEDERATED, provider);
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds a canonical user principal to the statement
     *
     * @param userID The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(userID) {
        return this.addPrincipal(PrincipalType.CANONICAL_USER, userID);
    }
    /**
     * Adds federated SAML principal to the statement
     *
     * @param account ID of the AWS account
     * @param providerName Name of the SAML provider
     */
    forSaml(account, providerName) {
        return this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`);
    }
    /**
     * Adds an IAM user principal to the statement
     *
     * @param account ID of the AWS account
     * @param user Name of the IAM user
     */
    forUser(account, user) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`);
    }
    /**
     * Adds an IAM role principal to the statement
     *
     * @param account ID of the AWS account
     * @param role Name of the IAM role
     */
    forRole(account, role) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`);
    }
    /**
     * Adds a specific assumed role session principal to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionName Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, sessionName) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
    }
    /**
     * Adds a service principal to the statement
     *
     * @param service Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(service) {
        return this.addPrincipal(PrincipalType.SERVICE, service);
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiNi1wcmluY2lwYWxzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiNi1wcmluY2lwYWxzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDBDQUF1RDtBQVN2RCxJQUFZLGFBS1g7QUFMRCxXQUFZLGFBQWE7SUFDdkIsNEJBQVcsQ0FBQTtJQUNYLHdDQUF1QixDQUFBO0lBQ3ZCLGlEQUFnQyxDQUFBO0lBQ2hDLG9DQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFMVyxhQUFhLEdBQWIscUJBQWEsS0FBYixxQkFBYSxRQUt4QjtBQUVEOztHQUVHO0FBQ0gsTUFBYSw0QkFBNkIsU0FBUSxxQ0FBeUI7SUFBM0U7O1FBQ1kscUJBQWdCLEdBQUcsS0FBSyxDQUFDO1FBQ3pCLGVBQVUsR0FBZSxFQUFFLENBQUM7SUErTXhDLENBQUM7SUE3TUM7Ozs7T0FJRztJQUNJLE1BQU07UUFDWCxrREFBa0Q7UUFDbEQsSUFBSSxPQUFPLElBQUksQ0FBQyxZQUFZLElBQUksVUFBVSxFQUFFO1lBQzFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxDQUFDO1lBQzFCLE9BQU8sS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDO1NBQ3ZCO1FBQ0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQztRQUNsRSxNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUMsTUFBTSxFQUFFLENBQUM7UUFFakMsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFLEVBQUU7WUFDeEIsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUM7U0FDbkM7UUFFRCxPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRU0sZUFBZTtRQUNwQixJQUFJLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztRQUMxQixrREFBa0Q7UUFDbEQsT0FBTyxLQUFLLENBQUMsZUFBZSxFQUFFLENBQUM7SUFDakMsQ0FBQztJQUVTLGtCQUFrQixLQUFJLENBQUM7SUFFakM7O09BRUc7SUFDSSxhQUFhO1FBQ2xCLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUM7UUFDN0IsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxhQUFhO1FBQ2xCLE9BQU8sTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDTyxZQUFZLENBQUMsTUFBYyxFQUFFLFNBQWlCO1FBQ3RELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUM7UUFDN0IsSUFBSSxDQUFDLENBQUMsTUFBTSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUNoQyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQztTQUM5QjtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3hDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksR0FBRyxDQUFDLEdBQVcsRUFBRSxNQUFlO1FBQ3JDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxNQUFNLElBQUksYUFBYSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFVBQVUsQ0FBQyxPQUFlO1FBQy9CLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFLGdCQUFnQixPQUFPLE9BQU8sQ0FBQyxDQUFDO0lBQzlFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLFFBQWdCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7T0FFRztJQUNJLG1CQUFtQjtRQUN4QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUM3RCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxrQkFBa0I7UUFDdkIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGdCQUFnQixDQUFDLENBQUM7SUFDN0MsQ0FBQztJQUVEOztPQUVHO0lBQ0ksb0JBQW9CO1FBQ3pCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO0lBQ2pELENBQUM7SUFFRDs7T0FFRztJQUVJLGtCQUFrQjtRQUN2QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMscUJBQXFCLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksZ0JBQWdCLENBQUMsTUFBYztRQUNwQyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxPQUFPLENBQUMsT0FBZSxFQUFFLFlBQW9CO1FBQ2xELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FDdEIsZ0JBQWdCLE9BQU8sa0JBQWtCLFlBQVksRUFBRSxDQUN4RCxDQUFDO0lBQ0osQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksT0FBTyxDQUFDLE9BQWUsRUFBRSxJQUFZO1FBQzFDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FDdEIsYUFBYSxDQUFDLEdBQUcsRUFDakIsZ0JBQWdCLE9BQU8sU0FBUyxJQUFJLEVBQUUsQ0FDdkMsQ0FBQztJQUNKLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE9BQU8sQ0FBQyxPQUFlLEVBQUUsSUFBWTtRQUMxQyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQ3RCLGFBQWEsQ0FBQyxHQUFHLEVBQ2pCLGdCQUFnQixPQUFPLFNBQVMsSUFBSSxFQUFFLENBQ3ZDLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0kscUJBQXFCLENBQzFCLE9BQWUsRUFDZixRQUFnQixFQUNoQixXQUFtQjtRQUVuQixPQUFPLElBQUksQ0FBQyxZQUFZLENBQ3RCLGFBQWEsQ0FBQyxHQUFHLEVBQ2pCLGdCQUFnQixPQUFPLGlCQUFpQixRQUFRLElBQUksV0FBVyxFQUFFLENBQ2xFLENBQUM7SUFDSixDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBRUksVUFBVSxDQUFDLE9BQWU7UUFDL0IsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxPQUFPLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLFNBQVM7UUFDZCxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsQ0FBQztJQUNuRCxDQUFDO0NBQ0Y7QUFqTkQsb0VBaU5DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50V2l0aEVmZmVjdCB9IGZyb20gJy4vNS1lZmZlY3QnO1xuXG4vKipcbiAqIEEgY29sbGVjdGlvbiBvZiBQcmluY2lwYWwnc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIFByaW5jaXBhbHMge1xuICBba2V5OiBzdHJpbmddOiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGVudW0gUHJpbmNpcGFsVHlwZSB7XG4gIEFXUyA9ICdBV1MnLFxuICBGRURFUkFURUQgPSAnRmVkZXJhdGVkJyxcbiAgQ0FOT05JQ0FMX1VTRVIgPSAnQ2Fub25pY2FsVXNlcicsXG4gIFNFUlZJQ0UgPSAnU2VydmljZScsXG59XG5cbi8qKlxuICogQWRkcyBcInByaW5jaXBhbFwiIGZ1bmN0aW9uYWxpdHkgdG8gdGhlIFBvbGljeSBTdGF0ZW1lbnRcbiAqL1xuZXhwb3J0IGNsYXNzIFBvbGljeVN0YXRlbWVudFdpdGhQcmluY2lwYWwgZXh0ZW5kcyBQb2xpY3lTdGF0ZW1lbnRXaXRoRWZmZWN0IHtcbiAgcHJvdGVjdGVkIHVzZU5vdFByaW5jaXBhbHMgPSBmYWxzZTtcbiAgcHJvdGVjdGVkIHByaW5jaXBhbHM6IFByaW5jaXBhbHMgPSB7fTtcblxuICAvKipcbiAgICogSW5qZWN0cyBwcmluY2lwYWxzIGludG8gdGhlIHN0YXRlbWVudC5cbiAgICpcbiAgICogT25seSByZWxldmFudCBmb3IgdGhlIG1haW4gcGFja2FnZS4gSW4gQ0RLIG1vZGUgdGhpcyBvbmx5IGNhbGxzIHN1cGVyLlxuICAgKi9cbiAgcHVibGljIHRvSlNPTigpOiBhbnkge1xuICAgIC8vIEB0cy1pZ25vcmUgb25seSBhdmFpbGFibGUgYWZ0ZXIgc3dhcHBpbmcgMS1iYXNlXG4gICAgaWYgKHR5cGVvZiB0aGlzLmFkZFJlc291cmNlcyA9PSAnZnVuY3Rpb24nKSB7XG4gICAgICB0aGlzLmNka0FwcGx5UHJpbmNpcGFscygpO1xuICAgICAgcmV0dXJuIHN1cGVyLnRvSlNPTigpO1xuICAgIH1cbiAgICBjb25zdCBtb2RlID0gdGhpcy51c2VOb3RQcmluY2lwYWxzID8gJ05vdFByaW5jaXBhbCcgOiAnUHJpbmNpcGFsJztcbiAgICBjb25zdCBzdGF0ZW1lbnQgPSBzdXBlci50b0pTT04oKTtcblxuICAgIGlmICh0aGlzLmhhc1ByaW5jaXBhbHMoKSkge1xuICAgICAgc3RhdGVtZW50W21vZGVdID0gdGhpcy5wcmluY2lwYWxzO1xuICAgIH1cblxuICAgIHJldHVybiBzdGF0ZW1lbnQ7XG4gIH1cblxuICBwdWJsaWMgdG9TdGF0ZW1lbnRKc29uKCk6IGFueSB7XG4gICAgdGhpcy5jZGtBcHBseVByaW5jaXBhbHMoKTtcbiAgICAvLyBAdHMtaWdub3JlIG9ubHkgYXZhaWxhYmxlIGFmdGVyIHN3YXBwaW5nIDEtYmFzZVxuICAgIHJldHVybiBzdXBlci50b1N0YXRlbWVudEpzb24oKTtcbiAgfVxuXG4gIHByb3RlY3RlZCBjZGtBcHBseVByaW5jaXBhbHMoKSB7fVxuXG4gIC8qKlxuICAgKiBTd2l0Y2hlcyB0aGUgc3RhdGVtZW50IHRvIHVzZSBbYG5vdFByaW5jaXBhbGBdKGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9JQU0vbGF0ZXN0L1VzZXJHdWlkZS9yZWZlcmVuY2VfcG9saWNpZXNfZWxlbWVudHNfbm90cHJpbmNpcGFsLmh0bWwpLlxuICAgKi9cbiAgcHVibGljIG5vdFByaW5jaXBhbHMoKSB7XG4gICAgdGhpcy51c2VOb3RQcmluY2lwYWxzID0gdHJ1ZTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDaGVja3Mgd2VhdGhlciBhIHByaW5jaXBhbCB3YXMgYXBwbGllZCB0byB0aGUgcG9saWN5XG4gICAqL1xuICBwdWJsaWMgaGFzUHJpbmNpcGFscygpOiBib29sZWFuIHtcbiAgICByZXR1cm4gT2JqZWN0LmtleXModGhpcy5wcmluY2lwYWxzKS5sZW5ndGggPiAwO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcHJlZml4IE9uZSBvZiAqKkFXUyoqLCAqKkZlZGVyYXRlZCoqLCAqKkNhbm9uaWNhbFVzZXIqKiBvciAqKlNlcnZpY2UqKlxuICAgKiBAcGFyYW0gcHJpbmNpcGFsIFRoZSBwcmluY2lwYWwgc3RyaW5nXG4gICAqL1xuICBwcm90ZWN0ZWQgYWRkUHJpbmNpcGFsKHByZWZpeDogc3RyaW5nLCBwcmluY2lwYWw6IHN0cmluZykge1xuICAgIHRoaXMuc2tpcEF1dG9SZXNvdXJjZSA9IHRydWU7XG4gICAgaWYgKCEocHJlZml4IGluIHRoaXMucHJpbmNpcGFscykpIHtcbiAgICAgIHRoaXMucHJpbmNpcGFsc1twcmVmaXhdID0gW107XG4gICAgfVxuICAgIHRoaXMucHJpbmNpcGFsc1twcmVmaXhdLnB1c2gocHJpbmNpcGFsKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGFueSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gYXJuIFRoZSBBUk4gb2YgdGhlIHByaW5jaXBhbFxuICAgKiBAcGFyYW0gcHJlZml4IE9uZSBvZiAqKkFXUyoqLCAqKkZlZGVyYXRlZCoqLCAqKkNhbm9uaWNhbFVzZXIqKiBvciAqKlNlcnZpY2UqKiAtIERlZmF1bHQ6ICoqQVdTKipcbiAgICovXG4gIHB1YmxpYyBmb3IoYXJuOiBzdHJpbmcsIHByZWZpeD86IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmFkZFByaW5jaXBhbChwcmVmaXggfHwgUHJpbmNpcGFsVHlwZS5BV1MsIGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBhY2NvdW50IHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBhY2NvdW50IElEIG9mIHRoZSBBV1MgYWNjb3VudFxuICAgKi9cbiAgcHVibGljIGZvckFjY291bnQoYWNjb3VudDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKFByaW5jaXBhbFR5cGUuQVdTLCBgYXJuOmF3czppYW06OiR7YWNjb3VudH06cm9vdGApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBbZmVkZXJhdGVkXShodHRwczovL2F3cy5hbWF6b24uY29tL2lkZW50aXR5L2ZlZGVyYXRpb24vKSAod2ViIGlkZW50aXR5KSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gcHJvdmlkZXIgSUQgb2YgdGhlIEFXUyBhY2NvdW50XG4gICAqL1xuICBwdWJsaWMgZm9yRmVkZXJhdGVkKHByb3ZpZGVyOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gdGhpcy5hZGRQcmluY2lwYWwoUHJpbmNpcGFsVHlwZS5GRURFUkFURUQsIHByb3ZpZGVyKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgZmVkZXJhdGVkIFtBV1MgQ29nbml0b10oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0lBTS9sYXRlc3QvVXNlckd1aWRlL2lkX3JvbGVzX3Byb3ZpZGVyc19vaWRjX2NvZ25pdG8uaHRtbCkgcHJpbmNpcGFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIHB1YmxpYyBmb3JGZWRlcmF0ZWRDb2duaXRvKCkge1xuICAgIHJldHVybiB0aGlzLmZvckZlZGVyYXRlZCgnY29nbml0by1pZGVudGl0eS5hbWF6b25hd3MuY29tJyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGZlZGVyYXRlZCBbQW1hem9uXShodHRwczovL2xvZ2luLmFtYXpvbi5jb20vKSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKi9cbiAgcHVibGljIGZvckZlZGVyYXRlZEFtYXpvbigpIHtcbiAgICByZXR1cm4gdGhpcy5mb3JGZWRlcmF0ZWQoJ3d3dy5hbWF6b24uY29tJyk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGZlZGVyYXRlZCBbRmFjZWJvb2tdKGh0dHBzOi8vZGV2ZWxvcGVycy5mYWNlYm9vay5jb20vZG9jcy9mYWNlYm9vay1sb2dpbikgcHJpbmNpcGFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICovXG4gIHB1YmxpYyBmb3JGZWRlcmF0ZWRGYWNlYm9vaygpIHtcbiAgICByZXR1cm4gdGhpcy5mb3JGZWRlcmF0ZWQoJ2dyYXBoLmZhY2Vib29rLmNvbScpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBmZWRlcmF0ZWQgW0dvb2dsZV0oaHR0cHM6Ly9kZXZlbG9wZXJzLmdvb2dsZS5jb20vaWRlbnRpdHkvcHJvdG9jb2xzL29hdXRoMi9vcGVuaWQtY29ubmVjdCkgcHJpbmNpcGFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICovXG5cbiAgcHVibGljIGZvckZlZGVyYXRlZEdvb2dsZSgpIHtcbiAgICByZXR1cm4gdGhpcy5mb3JGZWRlcmF0ZWQoJ2FjY291bnRzLmdvb2dsZS5jb20nKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgY2Fub25pY2FsIHVzZXIgcHJpbmNpcGFsIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQHBhcmFtIHVzZXJJRCBUaGUgdXNlciBJRFxuICAgKlxuICAgKiBZb3UgY2FuIFtmaW5kIHRoZSBjYW5vbmljYWwgdXNlciBJRF0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dlbmVyYWwvbGF0ZXN0L2dyL2FjY3QtaWRlbnRpZmllcnMuaHRtbCNGaW5kaW5nQ2Fub25pY2FsSWQpIGZvciB5b3VyIEFXUyBhY2NvdW50IGluIHRoZSBBV1MgTWFuYWdlbWVudCBDb25zb2xlLiBUaGUgY2Fub25pY2FsIHVzZXIgSUQgZm9yIGFuIEFXUyBhY2NvdW50IGlzIHNwZWNpZmljIHRvIHRoZSBhY2NvdW50LiBZb3UgY2FuIHJldHJpZXZlIHRoZSBjYW5vbmljYWwgdXNlciBJRCBmb3IgeW91ciBBV1MgYWNjb3VudCBhcyBlaXRoZXIgdGhlIHJvb3QgdXNlciBvciBhbiBJQU0gdXNlci5cbiAgICovXG4gIHB1YmxpYyBmb3JDYW5vbmljYWxVc2VyKHVzZXJJRDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKFByaW5jaXBhbFR5cGUuQ0FOT05JQ0FMX1VTRVIsIHVzZXJJRCk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBmZWRlcmF0ZWQgU0FNTCBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gYWNjb3VudCBJRCBvZiB0aGUgQVdTIGFjY291bnRcbiAgICogQHBhcmFtIHByb3ZpZGVyTmFtZSBOYW1lIG9mIHRoZSBTQU1MIHByb3ZpZGVyXG4gICAqL1xuICBwdWJsaWMgZm9yU2FtbChhY2NvdW50OiBzdHJpbmcsIHByb3ZpZGVyTmFtZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuZm9yRmVkZXJhdGVkKFxuICAgICAgYGFybjphd3M6aWFtOjoke2FjY291bnR9OnNhbWwtcHJvdmlkZXIvJHtwcm92aWRlck5hbWV9YFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gdXNlciBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gYWNjb3VudCBJRCBvZiB0aGUgQVdTIGFjY291bnRcbiAgICogQHBhcmFtIHVzZXIgTmFtZSBvZiB0aGUgSUFNIHVzZXJcbiAgICovXG4gIHB1YmxpYyBmb3JVc2VyKGFjY291bnQ6IHN0cmluZywgdXNlcjogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKFxuICAgICAgUHJpbmNpcGFsVHlwZS5BV1MsXG4gICAgICBgYXJuOmF3czppYW06OiR7YWNjb3VudH06dXNlci8ke3VzZXJ9YFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiBJQU0gcm9sZSBwcmluY2lwYWwgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAcGFyYW0gYWNjb3VudCBJRCBvZiB0aGUgQVdTIGFjY291bnRcbiAgICogQHBhcmFtIHJvbGUgTmFtZSBvZiB0aGUgSUFNIHJvbGVcbiAgICovXG4gIHB1YmxpYyBmb3JSb2xlKGFjY291bnQ6IHN0cmluZywgcm9sZTogc3RyaW5nKSB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKFxuICAgICAgUHJpbmNpcGFsVHlwZS5BV1MsXG4gICAgICBgYXJuOmF3czppYW06OiR7YWNjb3VudH06cm9sZS8ke3JvbGV9YFxuICAgICk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHNwZWNpZmljIGFzc3VtZWQgcm9sZSBzZXNzaW9uIHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBhY2NvdW50IElEIG9mIHRoZSBBV1MgYWNjb3VudFxuICAgKiBAcGFyYW0gcm9sZU5hbWUgTmFtZSBvZiB0aGUgSUFNIHJvbGVcbiAgICogQHBhcmFtIHNlc3Npb25OYW1lIE5hbWUgb2YgdGhlIHNlc3Npb24uIFlvdSBjYW5ub3QgdXNlIGEgd2lsZGNhcmQgKGAqYCkgdG8gbWVhbiAqYWxsIHNlc3Npb25zKi4gUHJpbmNpcGFscyBtdXN0IGFsd2F5cyBuYW1lIGEgc3BlY2lmaWMgc2Vzc2lvblxuICAgKi9cbiAgcHVibGljIGZvckFzc3VtZWRSb2xlU2Vzc2lvbihcbiAgICBhY2NvdW50OiBzdHJpbmcsXG4gICAgcm9sZU5hbWU6IHN0cmluZyxcbiAgICBzZXNzaW9uTmFtZTogc3RyaW5nXG4gICkge1xuICAgIHJldHVybiB0aGlzLmFkZFByaW5jaXBhbChcbiAgICAgIFByaW5jaXBhbFR5cGUuQVdTLFxuICAgICAgYGFybjphd3M6c3RzOjoke2FjY291bnR9OmFzc3VtZWQtcm9sZS8ke3JvbGVOYW1lfS8ke3Nlc3Npb25OYW1lfWBcbiAgICApO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzZXJ2aWNlIHByaW5jaXBhbCB0byB0aGUgc3RhdGVtZW50XG4gICAqXG4gICAqIEBwYXJhbSBzZXJ2aWNlIExvbmcgdmVyc2lvbiBvZiB0aGUgc2VydmljZSBuYW1lLiBVc3VhbGx5IGluIHRoZSBmb3JtYXQ6IGBsb25nX3NlcnZpY2UtbmFtZS5hbWF6b25hd3MuY29tYFxuICAgKlxuICAgKiBUaGUgc2VydmljZSBwcmluY2lwYWwgaXMgZGVmaW5lZCBieSB0aGUgc2VydmljZS4gVG8gbGVhcm4gdGhlIHNlcnZpY2UgcHJpbmNpcGFsIGZvciBhIHNlcnZpY2UsIHNlZSB0aGUgZG9jdW1lbnRhdGlvbiBmb3IgdGhhdCBzZXJ2aWNlLiBGb3Igc29tZSBzZXJ2aWNlcywgc2VlIFtBV1MgU2VydmljZXMgVGhhdCBXb3JrIHdpdGggSUFNXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX2F3cy1zZXJ2aWNlcy10aGF0LXdvcmstd2l0aC1pYW0uaHRtbCkgYW5kIGxvb2sgZm9yIHRoZSBzZXJ2aWNlcyB0aGF0IGhhdmUgKipZZXMqKiBpbiB0aGUgKipTZXJ2aWNlLUxpbmtlZCBSb2xlKiogY29sdW1uLiBDaG9vc2UgYSAqKlllcyoqIHdpdGggYSBsaW5rIHRvIHZpZXcgdGhlIHNlcnZpY2UtbGlua2VkIHJvbGUgZG9jdW1lbnRhdGlvbiBmb3IgdGhhdCBzZXJ2aWNlLiBWaWV3IHRoZSAqKlNlcnZpY2UtTGlua2VkIFJvbGUgUGVybWlzc2lvbnMqKiBzZWN0aW9uIGZvciB0aGF0IHNlcnZpY2UgdG8gdmlldyB0aGUgc2VydmljZSBwcmluY2lwYWwuXG4gICAqL1xuXG4gIHB1YmxpYyBmb3JTZXJ2aWNlKHNlcnZpY2U6IHN0cmluZykge1xuICAgIHJldHVybiB0aGlzLmFkZFByaW5jaXBhbChQcmluY2lwYWxUeXBlLlNFUlZJQ0UsIHNlcnZpY2UpO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50cyBwdWJsaWMgYXNzZXNcbiAgICpcbiAgICogKipFVkVSWU9ORSBJTiBUSEUgV09STEQgSEFTIEFDQ0VTUyoqXG4gICAqXG4gICAqIFdlIHN0cm9uZ2x5IHJlY29tbWVuZCB0aGF0IHlvdSBkbyBub3QgdXNlIGEgd2lsZGNhcmQgaW4gdGhlIFByaW5jaXBhbCBlbGVtZW50IGluIGEgcm9sZSdzIHRydXN0IHBvbGljeSB1bmxlc3MgeW91IG90aGVyd2lzZSByZXN0cmljdCBhY2Nlc3MgdGhyb3VnaCBhIENvbmRpdGlvbiBlbGVtZW50IGluIHRoZSBwb2xpY3kuIE90aGVyd2lzZSwgYW55IElBTSB1c2VyIGluIGFueSBhY2NvdW50IGluIHlvdXIgcGFydGl0aW9uIGNhbiBhY2Nlc3MgdGhlIHJvbGUuXG4gICAqL1xuICBwdWJsaWMgZm9yUHVibGljKCkge1xuICAgIHJldHVybiB0aGlzLmFkZFByaW5jaXBhbChQcmluY2lwYWxUeXBlLkFXUywgJyonKTtcbiAgfVxufVxuIl19