"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("@aws-cdk/aws-s3-assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
/**
 * Represents an OpenAPI definition asset.
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *
     *   apigateway.ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     */
    static fromAsset(file, options) {
        return new AssetApiDefinition(file, options);
    }
    /**
     * Called after the CFN RestApi resource has been created to allow the Api
     * Definition to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindAfterCreate(_scope, _restApi) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IRestApi(_restApi);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bindAfterCreate);
            }
            throw error;
        }
        return;
    }
}
exports.ApiDefinition = ApiDefinition;
_a = JSII_RTTI_SYMBOL_1;
ApiDefinition[_a] = { fqn: "@aws-cdk/aws-apigateway.ApiDefinition", version: "1.165.0" };
/**
 * OpenAPI specification from an S3 archive.
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
_b = JSII_RTTI_SYMBOL_1;
S3ApiDefinition[_b] = { fqn: "@aws-cdk/aws-apigateway.S3ApiDefinition", version: "1.165.0" };
/**
 * OpenAPI specification from an inline JSON object.
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
_c = JSII_RTTI_SYMBOL_1;
InlineApiDefinition[_c] = { fqn: "@aws-cdk/aws-apigateway.InlineApiDefinition", version: "1.165.0" };
/**
 * OpenAPI specification from a local file.
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
    bindAfterCreate(scope, restApi) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_apigateway_IRestApi(restApi);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bindAfterCreate);
            }
            throw error;
        }
        if (!scope.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const child = constructs_1.Node.of(restApi).defaultChild;
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, this.asset.assetPath);
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'BodyS3Location');
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
_d = JSII_RTTI_SYMBOL_1;
AssetApiDefinition[_d] = { fqn: "@aws-cdk/aws-apigateway.AssetApiDefinition", version: "1.165.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBpLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhcGktZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFDQSxvREFBb0Q7QUFFcEQsaUdBQWlHO0FBQ2pHLDhEQUE4RDtBQUM5RCx5Q0FBeUM7QUFDekMsMkNBQWtDO0FBS2xDOztHQUVHO0FBQ0gsTUFBc0IsYUFBYTtJQUNqQzs7T0FFRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0IsRUFBRSxHQUFXLEVBQUUsYUFBc0I7UUFDOUUsT0FBTyxJQUFJLGVBQWUsQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ3hEO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0E4Q0c7SUFDSSxNQUFNLENBQUMsVUFBVSxDQUFDLFVBQWU7UUFDdEMsT0FBTyxJQUFJLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0tBQzVDO0lBRUQ7O09BRUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLElBQVksRUFBRSxPQUFnQztRQUNwRSxPQUFPLElBQUksa0JBQWtCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQzlDO0lBV0Q7Ozs7T0FJRztJQUNJLGVBQWUsQ0FBQyxNQUFpQixFQUFFLFFBQWtCOzs7Ozs7Ozs7O1FBQzFELE9BQU87S0FDUjs7QUFsRkgsc0NBbUZDOzs7QUFvQ0Q7O0dBRUc7QUFDSCxNQUFhLGVBQWdCLFNBQVEsYUFBYTtJQUdoRCxZQUFZLE1BQWtCLEVBQVUsR0FBVyxFQUFVLGFBQXNCO1FBQ2pGLEtBQUssRUFBRSxDQUFDO1FBRDhCLFFBQUcsR0FBSCxHQUFHLENBQVE7UUFBVSxrQkFBYSxHQUFiLGFBQWEsQ0FBUztRQUdqRixJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLGlEQUFpRCxDQUFDLENBQUM7U0FDcEU7UUFFRCxJQUFJLENBQUMsVUFBVSxHQUFHLE1BQU0sQ0FBQyxVQUFVLENBQUM7S0FDckM7SUFFTSxJQUFJLENBQUMsTUFBaUI7UUFDM0IsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsSUFBSSxDQUFDLFVBQVU7Z0JBQ3ZCLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztnQkFDYixPQUFPLEVBQUUsSUFBSSxDQUFDLGFBQWE7YUFDNUI7U0FDRixDQUFDO0tBQ0g7O0FBckJILDBDQXNCQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLG1CQUFvQixTQUFRLGFBQWE7SUFDcEQsWUFBb0IsVUFBZTtRQUNqQyxLQUFLLEVBQUUsQ0FBQztRQURVLGVBQVUsR0FBVixVQUFVLENBQUs7UUFHakMsSUFBSSxPQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssUUFBUSxFQUFFO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMscUNBQXFDLENBQUMsQ0FBQztTQUN4RDtRQUVELElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLENBQUMsQ0FBQztTQUNwRDtLQUNGO0lBRU0sSUFBSSxDQUFDLE1BQWlCO1FBQzNCLE9BQU87WUFDTCxnQkFBZ0IsRUFBRSxJQUFJLENBQUMsVUFBVTtTQUNsQyxDQUFDO0tBQ0g7O0FBakJILGtEQWtCQzs7O0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGtCQUFtQixTQUFRLGFBQWE7SUFHbkQsWUFBNkIsSUFBWSxFQUFtQixVQUFrQyxFQUFHO1FBQy9GLEtBQUssRUFBRSxDQUFDO1FBRG1CLFNBQUksR0FBSixJQUFJLENBQVE7UUFBbUIsWUFBTyxHQUFQLE9BQU8sQ0FBOEI7S0FFaEc7SUFFTSxJQUFJLENBQUMsS0FBZ0I7UUFDMUIsOEZBQThGO1FBQzlGLElBQUksSUFBSSxDQUFDLEtBQUssS0FBSyxTQUFTLEVBQUU7WUFDNUIsSUFBSSxDQUFDLEtBQUssR0FBRyxJQUFJLFNBQVMsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLGVBQWUsRUFBRTtnQkFDdkQsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO2dCQUNmLEdBQUcsSUFBSSxDQUFDLE9BQU87YUFDaEIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQzNCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZO2dCQUMvQixHQUFHLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO2FBQzVCO1NBQ0YsQ0FBQztLQUNIO0lBRU0sZUFBZSxDQUFDLEtBQWdCLEVBQUUsT0FBaUI7Ozs7Ozs7Ozs7UUFDeEQsSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxFQUFFO1lBQzVFLE9BQU8sQ0FBQyxjQUFjO1NBQ3ZCO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDZixNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7U0FDakU7UUFFRCxNQUFNLEtBQUssR0FBRyxpQkFBSSxDQUFDLEVBQUUsQ0FBQyxPQUFPLENBQUMsQ0FBQyxZQUEwQixDQUFDO1FBQzFELEtBQUssQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLGdDQUFnQyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDaEYsS0FBSyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsb0NBQW9DLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztLQUNqRjs7QUF4Q0gsZ0RBeUNDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIHMzX2Fzc2V0cyBmcm9tICdAYXdzLWNkay9hd3MtczMtYXNzZXRzJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCAqIGFzIGN4YXBpIGZyb20gJ0Bhd3MtY2RrL2N4LWFwaSc7XG5pbXBvcnQgeyBOb2RlIH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5SZXN0QXBpIH0gZnJvbSAnLi9hcGlnYXRld2F5LmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJUmVzdEFwaSB9IGZyb20gJy4vcmVzdGFwaSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuLyoqXG4gKiBSZXByZXNlbnRzIGFuIE9wZW5BUEkgZGVmaW5pdGlvbiBhc3NldC5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEFwaURlZmluaXRpb24ge1xuICAvKipcbiAgICogQ3JlYXRlcyBhbiBBUEkgZGVmaW5pdGlvbiBmcm9tIGEgc3BlY2lmaWNhdGlvbiBmaWxlIGluIGFuIFMzIGJ1Y2tldFxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwga2V5OiBzdHJpbmcsIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpOiBTM0FwaURlZmluaXRpb24ge1xuICAgIHJldHVybiBuZXcgUzNBcGlEZWZpbml0aW9uKGJ1Y2tldCwga2V5LCBvYmplY3RWZXJzaW9uKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gQVBJIGRlZmluaXRpb24gZnJvbSBhbiBpbmxpbmUgb2JqZWN0LiBUaGUgaW5saW5lIG9iamVjdCBtdXN0IGZvbGxvdyB0aGVcbiAgICogc2NoZW1hIG9mIE9wZW5BUEkgMi4wIG9yIE9wZW5BUEkgMy4wXG4gICAqXG4gICAqIEBleGFtcGxlXG4gICAqXG4gICAqICAgYXBpZ2F0ZXdheS5BcGlEZWZpbml0aW9uLmZyb21JbmxpbmUoe1xuICAgKiAgICAgb3BlbmFwaTogJzMuMC4yJyxcbiAgICogICAgIHBhdGhzOiB7XG4gICAqICAgICAgICcvcGV0cyc6IHtcbiAgICogICAgICAgICBnZXQ6IHtcbiAgICogICAgICAgICAgICdyZXNwb25zZXMnOiB7XG4gICAqICAgICAgICAgICAgIDIwMDoge1xuICAgKiAgICAgICAgICAgICAgIGNvbnRlbnQ6IHtcbiAgICogICAgICAgICAgICAgICAgICdhcHBsaWNhdGlvbi9qc29uJzoge1xuICAgKiAgICAgICAgICAgICAgICAgICBzY2hlbWE6IHtcbiAgICogICAgICAgICAgICAgICAgICAgICAkcmVmOiAnIy9jb21wb25lbnRzL3NjaGVtYXMvRW1wdHknLFxuICAgKiAgICAgICAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICAgICAgfSxcbiAgICogICAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgfSxcbiAgICogICAgICAgICAgICd4LWFtYXpvbi1hcGlnYXRld2F5LWludGVncmF0aW9uJzoge1xuICAgKiAgICAgICAgICAgICByZXNwb25zZXM6IHtcbiAgICogICAgICAgICAgICAgICBkZWZhdWx0OiB7XG4gICAqICAgICAgICAgICAgICAgICBzdGF0dXNDb2RlOiAnMjAwJyxcbiAgICogICAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICB9LFxuICAgKiAgICAgICAgICAgICByZXF1ZXN0VGVtcGxhdGVzOiB7XG4gICAqICAgICAgICAgICAgICAgJ2FwcGxpY2F0aW9uL2pzb24nOiAne1wic3RhdHVzQ29kZVwiOiAyMDB9JyxcbiAgICogICAgICAgICAgICAgfSxcbiAgICogICAgICAgICAgICAgcGFzc3Rocm91Z2hCZWhhdmlvcjogJ3doZW5fbm9fbWF0Y2gnLFxuICAgKiAgICAgICAgICAgICB0eXBlOiAnbW9jaycsXG4gICAqICAgICAgICAgICB9LFxuICAgKiAgICAgICAgIH0sXG4gICAqICAgICAgIH0sXG4gICAqICAgICB9LFxuICAgKiAgICAgY29tcG9uZW50czoge1xuICAgKiAgICAgICBzY2hlbWFzOiB7XG4gICAqICAgICAgICAgRW1wdHk6IHtcbiAgICogICAgICAgICAgIHRpdGxlOiAnRW1wdHkgU2NoZW1hJyxcbiAgICogICAgICAgICAgIHR5cGU6ICdvYmplY3QnLFxuICAgKiAgICAgICAgIH0sXG4gICAqICAgICAgIH0sXG4gICAqICAgICB9LFxuICAgKiAgIH0pO1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tSW5saW5lKGRlZmluaXRpb246IGFueSk6IElubGluZUFwaURlZmluaXRpb24ge1xuICAgIHJldHVybiBuZXcgSW5saW5lQXBpRGVmaW5pdGlvbihkZWZpbml0aW9uKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMb2FkcyB0aGUgQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhIGxvY2FsIGRpc2sgYXNzZXQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Bc3NldChmaWxlOiBzdHJpbmcsIG9wdGlvbnM/OiBzM19hc3NldHMuQXNzZXRPcHRpb25zKTogQXNzZXRBcGlEZWZpbml0aW9uIHtcbiAgICByZXR1cm4gbmV3IEFzc2V0QXBpRGVmaW5pdGlvbihmaWxlLCBvcHRpb25zKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBDYWxsZWQgd2hlbiB0aGUgc3BlY2lmaWNhdGlvbiBpcyBpbml0aWFsaXplZCB0byBhbGxvdyB0aGlzIG9iamVjdCB0byBiaW5kXG4gICAqIHRvIHRoZSBzdGFjaywgYWRkIHJlc291cmNlcyBhbmQgaGF2ZSBmdW4uXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZSBUaGUgYmluZGluZyBzY29wZS4gRG9uJ3QgYmUgc21hcnQgYWJvdXQgdHJ5aW5nIHRvIGRvd24tY2FzdCBvclxuICAgKiBhc3N1bWUgaXQncyBpbml0aWFsaXplZC4gWW91IG1heSBqdXN0IHVzZSBpdCBhcyBhIGNvbnN0cnVjdCBzY29wZS5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCBiaW5kKHNjb3BlOiBDb25zdHJ1Y3QpOiBBcGlEZWZpbml0aW9uQ29uZmlnO1xuXG4gIC8qKlxuICAgKiBDYWxsZWQgYWZ0ZXIgdGhlIENGTiBSZXN0QXBpIHJlc291cmNlIGhhcyBiZWVuIGNyZWF0ZWQgdG8gYWxsb3cgdGhlIEFwaVxuICAgKiBEZWZpbml0aW9uIHRvIGJpbmQgdG8gaXQuIFNwZWNpZmljYWxseSBpdCdzIHJlcXVpcmVkIHRvIGFsbG93IGFzc2V0cyB0byBhZGRcbiAgICogbWV0YWRhdGEgZm9yIHRvb2xpbmcgbGlrZSBTQU0gQ0xJIHRvIGJlIGFibGUgdG8gZmluZCB0aGVpciBvcmlnaW5zLlxuICAgKi9cbiAgcHVibGljIGJpbmRBZnRlckNyZWF0ZShfc2NvcGU6IENvbnN0cnVjdCwgX3Jlc3RBcGk6IElSZXN0QXBpKSB7XG4gICAgcmV0dXJuO1xuICB9XG59XG5cbi8qKlxuICogUzMgbG9jYXRpb24gb2YgdGhlIEFQSSBkZWZpbml0aW9uIGZpbGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcGlEZWZpbml0aW9uUzNMb2NhdGlvbiB7XG4gIC8qKiBUaGUgUzMgYnVja2V0ICovXG4gIHJlYWRvbmx5IGJ1Y2tldDogc3RyaW5nO1xuICAvKiogVGhlIFMzIGtleSAqL1xuICByZWFkb25seSBrZXk6IHN0cmluZztcbiAgLyoqXG4gICAqIEFuIG9wdGlvbmFsIHZlcnNpb25cbiAgICogQGRlZmF1bHQgLSBsYXRlc3QgdmVyc2lvblxuICAgKi9cbiAgcmVhZG9ubHkgdmVyc2lvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBQb3N0LUJpbmRpbmcgQ29uZmlndXJhdGlvbiBmb3IgYSBDREsgY29uc3RydWN0XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIHNwZWNpZmljYXRpb24gaW4gUzMgKG11dHVhbGx5IGV4Y2x1c2l2ZSB3aXRoIGBpbmxpbmVEZWZpbml0aW9uYCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQVBJIGRlZmluaXRpb24gaXMgbm90IGFuIFMzIGxvY2F0aW9uXG4gICAqL1xuICByZWFkb25seSBzM0xvY2F0aW9uPzogQXBpRGVmaW5pdGlvblMzTG9jYXRpb247XG5cbiAgLyoqXG4gICAqIElubGluZSBzcGVjaWZpY2F0aW9uIChtdXR1YWxseSBleGNsdXNpdmUgd2l0aCBgczNMb2NhdGlvbmApLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFQSSBkZWZpbml0aW9uIGlzIG5vdCBkZWZpbmVkIGlubGluZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5saW5lRGVmaW5pdGlvbj86IGFueTtcbn1cblxuLyoqXG4gKiBPcGVuQVBJIHNwZWNpZmljYXRpb24gZnJvbSBhbiBTMyBhcmNoaXZlLlxuICovXG5leHBvcnQgY2xhc3MgUzNBcGlEZWZpbml0aW9uIGV4dGVuZHMgQXBpRGVmaW5pdGlvbiB7XG4gIHByaXZhdGUgYnVja2V0TmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKGJ1Y2tldDogczMuSUJ1Y2tldCwgcHJpdmF0ZSBrZXk6IHN0cmluZywgcHJpdmF0ZSBvYmplY3RWZXJzaW9uPzogc3RyaW5nKSB7XG4gICAgc3VwZXIoKTtcblxuICAgIGlmICghYnVja2V0LmJ1Y2tldE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYnVja2V0TmFtZSBpcyB1bmRlZmluZWQgZm9yIHRoZSBwcm92aWRlZCBidWNrZXQnKTtcbiAgICB9XG5cbiAgICB0aGlzLmJ1Y2tldE5hbWUgPSBidWNrZXQuYnVja2V0TmFtZTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogQ29uc3RydWN0KTogQXBpRGVmaW5pdGlvbkNvbmZpZyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHMzTG9jYXRpb246IHtcbiAgICAgICAgYnVja2V0OiB0aGlzLmJ1Y2tldE5hbWUsXG4gICAgICAgIGtleTogdGhpcy5rZXksXG4gICAgICAgIHZlcnNpb246IHRoaXMub2JqZWN0VmVyc2lvbixcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIE9wZW5BUEkgc3BlY2lmaWNhdGlvbiBmcm9tIGFuIGlubGluZSBKU09OIG9iamVjdC5cbiAqL1xuZXhwb3J0IGNsYXNzIElubGluZUFwaURlZmluaXRpb24gZXh0ZW5kcyBBcGlEZWZpbml0aW9uIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSBkZWZpbml0aW9uOiBhbnkpIHtcbiAgICBzdXBlcigpO1xuXG4gICAgaWYgKHR5cGVvZihkZWZpbml0aW9uKSAhPT0gJ29iamVjdCcpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignZGVmaW5pdGlvbiBzaG91bGQgYmUgb2YgdHlwZSBvYmplY3QnKTtcbiAgICB9XG5cbiAgICBpZiAoT2JqZWN0LmtleXMoZGVmaW5pdGlvbikubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0pTT04gZGVmaW5pdGlvbiBjYW5ub3QgYmUgZW1wdHknKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgIHJldHVybiB7XG4gICAgICBpbmxpbmVEZWZpbml0aW9uOiB0aGlzLmRlZmluaXRpb24sXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIE9wZW5BUEkgc3BlY2lmaWNhdGlvbiBmcm9tIGEgbG9jYWwgZmlsZS5cbiAqL1xuZXhwb3J0IGNsYXNzIEFzc2V0QXBpRGVmaW5pdGlvbiBleHRlbmRzIEFwaURlZmluaXRpb24ge1xuICBwcml2YXRlIGFzc2V0PzogczNfYXNzZXRzLkFzc2V0O1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgcGF0aDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMgPSB7IH0pIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCk6IEFwaURlZmluaXRpb25Db25maWcge1xuICAgIC8vIElmIHRoZSBzYW1lIEFzc2V0QVBJRGVmaW5pdGlvbiBpcyB1c2VkIG11bHRpcGxlIHRpbWVzLCByZXRhaW4gb25seSB0aGUgZmlyc3QgaW5zdGFudGlhdGlvbi5cbiAgICBpZiAodGhpcy5hc3NldCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aGlzLmFzc2V0ID0gbmV3IHMzX2Fzc2V0cy5Bc3NldChzY29wZSwgJ0FQSURlZmluaXRpb24nLCB7XG4gICAgICAgIHBhdGg6IHRoaXMucGF0aCxcbiAgICAgICAgLi4udGhpcy5vcHRpb25zLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKHRoaXMuYXNzZXQuaXNaaXBBcmNoaXZlKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEFzc2V0IGNhbm5vdCBiZSBhIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeSAoJHt0aGlzLnBhdGh9KWApO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgIGJ1Y2tldDogdGhpcy5hc3NldC5zM0J1Y2tldE5hbWUsXG4gICAgICAgIGtleTogdGhpcy5hc3NldC5zM09iamVjdEtleSxcbiAgICAgIH0sXG4gICAgfTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kQWZ0ZXJDcmVhdGUoc2NvcGU6IENvbnN0cnVjdCwgcmVzdEFwaTogSVJlc3RBcGkpIHtcbiAgICBpZiAoIXNjb3BlLm5vZGUudHJ5R2V0Q29udGV4dChjeGFwaS5BU1NFVF9SRVNPVVJDRV9NRVRBREFUQV9FTkFCTEVEX0NPTlRFWFQpKSB7XG4gICAgICByZXR1cm47IC8vIG5vdCBlbmFibGVkXG4gICAgfVxuXG4gICAgaWYgKCF0aGlzLmFzc2V0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2JpbmRUb1Jlc291cmNlKCkgbXVzdCBiZSBjYWxsZWQgYWZ0ZXIgYmluZCgpJyk7XG4gICAgfVxuXG4gICAgY29uc3QgY2hpbGQgPSBOb2RlLm9mKHJlc3RBcGkpLmRlZmF1bHRDaGlsZCBhcyBDZm5SZXN0QXBpO1xuICAgIGNoaWxkLmFkZE1ldGFkYXRhKGN4YXBpLkFTU0VUX1JFU09VUkNFX01FVEFEQVRBX1BBVEhfS0VZLCB0aGlzLmFzc2V0LmFzc2V0UGF0aCk7XG4gICAgY2hpbGQuYWRkTWV0YWRhdGEoY3hhcGkuQVNTRVRfUkVTT1VSQ0VfTUVUQURBVEFfUFJPUEVSVFlfS0VZLCAnQm9keVMzTG9jYXRpb24nKTtcbiAgfVxufSJdfQ==