# AUTOGENERATED! DO NOT EDIT! File to edit: notebooks/CLI_Pred.ipynb (unless otherwise specified).

__all__ = ["logger"]

# Cell

from typing import *

# Internal Cell

import os
import typer
from typer import echo
from tabulate import tabulate

from . import helper
from ..logger import get_logger, set_level
from ..constant import CLIENT_DB_USERNAME, CLIENT_DB_PASSWORD

# Internal Cell

app = typer.Typer(
    help="A set of commands for managing and downloading the predictions."
)

# Cell

logger = get_logger(__name__)

# Internal Cell


@app.command()
@helper.requires_auth_token
def ls(
    offset: int = typer.Option(
        0,
        "--offset",
        "-o",
        help="The number of predictions to offset at the beginning. If None, then the default value **0** will be used.",
    ),
    limit: int = typer.Option(
        100,
        "--limit",
        "-l",
        help="The maximum number of predictions to return from the server. If None, "
        "then the default value **100** will be used.",
    ),
    disabled: bool = typer.Option(
        False,
        "--disabled",
        help="If set to **True**, then only the deleted predictions will be returned. Else, the default value "
        "**False** will be used to return only the list of active predictions.",
    ),
    completed: bool = typer.Option(
        False,
        "--completed",
        help="If set to **True**, then only the predictions that are successfully downloaded "
        "to the server will be returned. Else, the default value **False** will be used to "
        "return all the predictions.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output only ids of prediction separated by space.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the list of predictions."""

    from ..client import Prediction

    predx = Prediction.ls(
        offset=offset, limit=limit, disabled=disabled, completed=completed
    )

    df = Prediction.as_df(predx)

    df["created"] = helper.humanize_date(df["created"])

    if quiet:
        ids = df["id"].astype(str).to_list()
        typer.echo("\n".join(ids))
    else:
        typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def details(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the details of a prediction."""

    from ..client import Prediction

    pred = Prediction(id=int(id))
    df = pred.details()

    df["created"] = helper.humanize_date(df["created"])

    typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command()
@helper.requires_auth_token
def rm(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output the deleted prediction id only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Delete a prediction from the server."""

    from ..client import Prediction

    pred = Prediction(id=int(id))
    df = pred.delete()

    df["created"] = helper.humanize_date(df["created"])

    if quiet:
        typer.echo(df.iloc[0]["id"])
    else:
        typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command("to-pandas")
@helper.requires_auth_token
def to_pandas(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Return the prediction results as a pandas DataFrame."""

    from ..client import Prediction

    pred = Prediction(id=int(id))

    df = pred.to_pandas()

    typer.echo(tabulate(df, headers="keys", tablefmt="plain", showindex=False))


# Internal Cell


@app.command("to-s3")
@helper.requires_auth_token
def to_s3(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    uri: str = typer.Option(..., help="The target S3 bucket uri."),
    access_key: Optional[str] = typer.Option(
        None,
        help="Access key for the target S3 bucket. If **None** (default value), then the value from **AWS_ACCESS_KEY_ID** environment variable is used.",
    ),
    secret_key: Optional[str] = typer.Option(
        None,
        help="Secret key for the target S3 bucket. If **None** (default value), then the value from **AWS_SECRET_ACCESS_KEY** environment variable is used.",
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output status only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Push the prediction results to the target AWS S3 bucket."""

    from ..client import Prediction

    pred = Prediction(id=int(id))

    status = pred.to_s3(uri=uri, access_key=access_key, secret_key=secret_key)

    if quiet:
        status.wait()
        typer.echo(
            f"Results for prediction id: {pred.id} are successfully pushed to the s3 bucket."
        )
    else:
        typer.echo(
            f"Pushing the results for prediction id: {pred.id} to the s3 bucket."
        )
        status.progress_bar()


# Internal Cell


@app.command("to-local")
@helper.requires_auth_token
def to_local(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    path: str = typer.Option(..., help="Local directory path."),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output status only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Download the prediction results to a local directory."""

    from ..client import Prediction

    pred = Prediction(id=int(id))

    if quiet:
        pred.to_local(path=path, show_progress=False)
        typer.echo(f"Results for id: {pred.id} has been successfully downloaded.")
    else:
        typer.echo(f"Downloading prediction results for id: {pred.id}.")
        pred.to_local(path=path)


# Internal Cell


@app.command("to-mysql")
@helper.requires_auth_token
def to_mysql(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    host: str = typer.Option(..., help="Database host name."),
    database: str = typer.Option(..., help="Database name."),
    table: str = typer.Option(..., help="Table name."),
    port: int = typer.Option(
        3306,
        help="Host port number. If not passed, then the default value **3306** will be used.",
    ),
    username: Optional[str] = typer.Option(
        None,
        "--username",
        "-u",
        help="Database username. If not passed, then the value set in the environment variable"
        f" **{CLIENT_DB_USERNAME}** will be used else the default value **root** will be used.",
    ),
    password: Optional[str] = typer.Option(
        None,
        "--password",
        "-p",
        help="Database password. If not passed, then the value set in the environment variable"
        f' **{CLIENT_DB_PASSWORD}** will be used else the default value "" will be used.',
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output status only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Push the prediction results to a mysql database.

    If the database requires authentication, pass the username/password as commandline arguments or store it in
    the **AIRT_CLIENT_DB_USERNAME** and **AIRT_CLIENT_DB_PASSWORD** environment variables.
    """

    from ..client import Prediction

    pred = Prediction(id=int(id))

    status = pred.to_mysql(
        host=host,
        database=database,
        table=table,
        port=port,
        username=username,
        password=password,
    )

    if quiet:
        status.wait()
        typer.echo(
            f"Results for prediction id: {pred.id} are successfully pushed to the mysql database."
        )
    else:
        typer.echo(
            f"Pushing the results for prediction id: {pred.id} to the mysql database."
        )
        status.progress_bar()


# Internal Cell


@app.command("to-clickhouse")
@helper.requires_auth_token
def to_clickhouse(
    id: int = typer.Argument(
        ...,
        help="Prediction id.",
    ),
    host: str = typer.Option(..., help="Remote database host name."),
    database: str = typer.Option(..., help="Database name."),
    table: str = typer.Option(..., help="Table name."),
    protocol: str = typer.Option(..., help="Protocol to use (native/http)."),
    port: int = typer.Option(
        0,
        help="Host port number. If not passed, then the default value **0** will be used.",
    ),
    username: Optional[str] = typer.Option(
        None,
        "--username",
        "-u",
        help="Database username. If not passed, then the value set in the environment variable"
        " **CLICKHOUSE_USERNAME** will be used else the default value **root** will be used.",
    ),
    password: Optional[str] = typer.Option(
        None,
        "--password",
        "-p",
        help="Database password. If not passed, then the value set in the environment variable"
        ' **CLICKHOUSE_PASSWORD** will be used else the default value "" will be used.',
    ),
    quiet: bool = typer.Option(
        False,
        "--quiet",
        "-q",
        help="Output status only.",
    ),
    debug: bool = typer.Option(
        False,
        "--debug",
        "-d",
        help="Set logger level to DEBUG and output everything.",
    ),
) -> None:
    """Push the prediction results to a clickhouse database.

    If the database requires authentication, pass the username/password as commandline arguments or store it in
    the **CLICKHOUSE_USERNAME** and **CLICKHOUSE_PASSWORD** environment variables.
    """

    from ..client import Prediction

    pred = Prediction(id=int(id))

    status = pred.to_clickhouse(
        host=host,
        database=database,
        table=table,
        port=port,
        protocol=protocol,
        username=username,
        password=password,
    )

    if quiet:
        status.wait()
        typer.echo(
            f"Results for prediction id: {pred.id} are successfully pushed to the clickhouse database."
        )
    else:
        typer.echo(
            f"Pushing the results for prediction id: {pred.id} to the clickhouse database."
        )
        status.progress_bar()
