import base64
import binascii
import hashlib
import hmac
import html
import json
import os
from Crypto.Hash import SHA
from Crypto.PublicKey import RSA
from Crypto.Signature import PKCS1_v1_5
from urllib.parse import parse_qs, urlparse


class Transaction:
    """A Paybox System transaction, from your server to the customer's browser, and from Paybox server to yours

    Attributes:
        MANDATORY   The values nedded to call for a payment
        ACCESSORY   The values you may add to modify Paybox behavior
        RESPONSE_CODES  Every response code Paybox may return after a payment attempt
    """

    def __init__(
        self,
        PAYBOX_SITE,
        PAYBOX_RANG,
        PAYBOX_IDENTIFIANT,
        PAYBOX_SECRETKEYPROD,
        PAYBOX_SECRETKEYTEST,
        production=False,
        PBX_TOTAL=None,
        PBX_CMD=None,
        PBX_PORTEUR=None,
        PBX_TIME=None,
        PBX_REPONDRE_A=None,
        PBX_REFUSE=None,
        PBX_EFFECTUE=None,
        PBX_ANNULE=None,
        PBX_ATTENTE=None,
        PBX_DEVISE=978,
        firstname=None,
        lastname=None,
        address1=None,
        address2=None,
        zipcode=None,
        city=None,
        countrycode=250,  # iso 3166_1 numeric code (ie: France=250)
        totalquantity=1,  # from 1 to 99
        paybox_path='/cgi/FramepagepaiementRWD.cgi'
    ):
        self.production = production
        self.firstname = firstname
        self.lastname = lastname
        self.address1 = address1
        self.address2 = address2
        self.zipcode = zipcode
        self.city = city
        self.countrycode = countrycode
        self.totalquantity = totalquantity

        if self.production:
            self.action = "https://tpeweb.e-transactions.fr" + paybox_path
            self.SECRET = PAYBOX_SECRETKEYPROD
        else:
            self.action = "https://recette-tpeweb.e-transactions.fr" + paybox_path
            self.SECRET = PAYBOX_SECRETKEYTEST

        self.FIELDS = {
            "PBX_SITE": PAYBOX_SITE,  # SITE NUMBER (given by Paybox)
            "PBX_RANG": PAYBOX_RANG,  # RANG NUMBER (given by Paybox)
            # IDENTIFIANT NUMBER (given by Paybox)
            "PBX_IDENTIFIANT": PAYBOX_IDENTIFIANT,
            "PBX_TOTAL": PBX_TOTAL,  # Total amount of the transaction, in cents
            "PBX_DEVISE": str(PBX_DEVISE),  # Currency of the transaction
            "PBX_CMD": PBX_CMD,  # Transaction reference generated by the ecommerce
            "PBX_PORTEUR": PBX_PORTEUR,  # Customer's email address
            # List of the variables Paybox must return to the IPN url
            "PBX_RETOUR": "amount:M;paymentId:R;transactionId:T;authorizationId:A;cardType:C;"
                          "cardNumber:N;cardExpiration:D;error:E;payboxRef:S;date:W;time:Q;signature:K;",
            "PBX_SOURCE": "RWD",
            "PBX_HASH": "SHA512",  # Hash algorithm used to calculate the Hmac value
            "PBX_TIME": PBX_TIME,  # Time of the transaction (iso 8601 format)
            "PBX_BILLING": self.get_pbx_billing(),
            # 3DSv2 https://www.ca-moncommerce.com/espace-client-mon-commerce/up2pay-e-transactions/ma-documentation/manuel-dintegration-focus-3ds-v2/principes-generaux/#integration-3dsv2-developpeur-webmaster
            "PBX_SHOPPINGCART": self.get_pbx_shoppingcart(),  # 3DSv2
            "PBX_REFUSE": PBX_REFUSE,  # url de retour en cas de refus de paiement
            # url IPN. WARNING. With Trailing slash, otherwise Django 301 to it...
            "PBX_REPONDRE_A": PBX_REPONDRE_A,
            "PBX_EFFECTUE": PBX_EFFECTUE,  # url de retour en cas de succes
            "PBX_ANNULE": PBX_ANNULE,  # url de retour en cas d'abandon
            "PBX_ATTENTE": PBX_ATTENTE,  # url de retour en cas d'abandon
            "PBX_LANGUE": "FRA",
            # 3 Chars. payment language. GBR for English
            # calling method for IPN url (https://www.ca-moncommerce.com/espace-client-mon-commerce/up2pay-e-transactions/ma-documentation/manuel-dintegration-up2pay-e-transactions/chapitre-11-dictionnaire-de-donn%C3%A9es/#affichage-des-pages-de-paiement)
            "PBX_RUF1": "POST",
        }

        # self.ACCESSORY = {
        #     "PBX_REFUSE": PBX_REFUSE,  # url de retour en cas de refus de paiement
        #     # url IPN. WARNING. With Trailing slash, otherwise Django 301 to it...
        #     "PBX_REPONDRE_A": PBX_REPONDRE_A,
        #     "PBX_EFFECTUE": PBX_EFFECTUE,  # url de retour en cas de succes
        #     "PBX_ANNULE": PBX_ANNULE,  # url de retour en cas d'abandon
        #     "PBX_ATTENTE": PBX_ATTENTE,  # url de retour en cas d'abandon
        #     "PBX_LANGUE": "FRA",  # 3 Chars. payment language. GBR for English
        #     # calling method for IPN url (https://www.ca-moncommerce.com/espace-client-mon-commerce/up2pay-e-transactions/ma-documentation/manuel-dintegration-up2pay-e-transactions/chapitre-11-dictionnaire-de-donn%C3%A9es/#affichage-des-pages-de-paiement)
        #     "PBX_RUF1": "POST",
        # }

        self.RESPONSE_CODES = {
            "00000": "Success",
            "00001": "Connection failed. Make a new attempt at tpeweb1.paybox.com",
            "001xx": "Payment rejected",
            "00003": "Paybox Error. Make a new attempt at tpeweb1.paybox.com",
            "00004": "Card Number invalid",
            "00006": "site, rang, or identifiant invalid. Connection rejected",
            "00008": "Card Expiration Date invalid",
            "00009": "Error while creating a subscription",
            "00010": "Unrecognized currency",
            "00011": "Incorrect amount",
            "00015": "Payment already done",
            "00016": "Subscriber already known",
            "00021": "Unauthorized Card",
            "00029": "Incorrect Card Number",
            "00030": "Time Out",
            "00031": "Reserved",
            "00032": "Reserved",
            "00033": "Country Not Supported",
            "00040": "3DSecure validation failed",
            "99999": "Payment on Hold",
        }

    # see https://www.ca-moncommerce.com/espace-client-mon-commerce/up2pay-e-transactions/ma-documentation/manuel-dintegration-focus-3ds-v2/principes-generaux/#integration-3dsv2-developpeur-webmaster
    # PBX_BILLING is an XML document
    # containing the name and the complete address
    # of the card holder
    def get_pbx_billing(self):
        szRet = ('<?xml version="1.0" encoding="utf-8" ?><Billing>'
                 '<Address><FirstName>{firstname}</FirstName>'
                 '<LastName>{lastname}</LastName>'
                 '<Address1>{address1}</Address1>').format(
            firstname=self.firstname,
            lastname=self.lastname,
            address1=self.address1)

        if self.address2 is not None and len(str(self.address2)) > 0:
            szRet += ('<Address2>{address2}</Address2>').format(
                address2=self.address2)

        szRet += ('<ZipCode>{zipcode}</ZipCode>'
                  '<City>{city}</City><CountryCode>{countrycode}</CountryCode>'
                  '</Address></Billing>').format(
            zipcode=self.zipcode,
            city=self.city,
            countrycode=self.countrycode)

        # return html.escape(szRet)
        return szRet

    # see https://www.ca-moncommerce.com/espace-client-mon-commerce/up2pay-e-transactions/ma-documentation/manuel-dintegration-focus-3ds-v2/principes-generaux/#integration-3dsv2-developpeur-webmaster
    # PBX_SHOPPINGCART is an XML document
    # containing the quantity of items in the
    # shopping cart (valid values are 0<v<100)
    def get_pbx_shoppingcart(self):
        totalquantity = self.totalquantity
        if self.totalquantity < 1:
            totalquantity = 1

        if self.totalquantity > 99:
            totalquantity = 99

        szRet = ('<?xml version="1.0" encoding="utf-8" ?>'
                 '<shoppingcart>'
                 '<total>'
                 '<totalQuantity>{quantity}</totalQuantity>'
                 '</total>'
                 '</shoppingcart>').format(quantity=totalquantity)

        # return html.escape(szRet)
        return szRet

    def get_action(self):
        return self.action

    def post_to_paybox(self):
        """Returns three variables ready to be integrated in an hidden form, in a template"""

        tosign = None
        # for the accessory variables, the order is not important
        for name, value in self.FIELDS.items():
            if value:
                if name != 'PBX_HMAC':
                    if tosign is not None:
                        tosign += "&" + name + "=" + str(value)
                    else:
                        tosign = name + "=" + str(value)

        print(tosign)
        binary_key = binascii.unhexlify(self.SECRET)
        signature = (
            hmac.new(binary_key, tosign.encode("ascii"), hashlib.sha512)
            .hexdigest()
            .upper()
        )
        self.FIELDS["PBX_HMAC"] = signature

        return {
            "action": self.action,
            "fields": self.FIELDS,
            "hmac": signature
        }

    def construct_html_form(self):
        """Returns an html form ready to be used (string)"""

        fields = "\n".join(
            [
                "<input type='hidden' name='{0}' value='{1}'>".format(
                    field, html.escape(str(self.FIELDS[field]))
                )
                for field in self.FIELDS
                if self.FIELDS[field]
            ]
        )

        # _html = """<form method="POST" action="{action}">
        #     {fields}
        #     <input type="submit" value="Payer">
        # </form>"""

        return {"fields": fields, "action": self.action}

    def get_html_elements(self):
        form_values = self.post_to_paybox()
        form_list = []
        for item in form_values['fields']:
            form_list.append({"name": item, "value": form_values['fields'][item]})
        return json.dumps(form_list)

    def verify_notification(self, response_url, order_total, verify_certificate=True):
        """Verifies the notification sent by Paybox to your server.

        It verifies :
            - the authenticity of the message
            - the fact that the message has not been altered
            - if not in production, the auth_number must be "XXXXXX"
            - if in production, there must be a Response Code
            - the total returned must be equal to the total of the order you've saved in ddb

        :response_url: (string), the full response url with its encoded args
        :order_total': (int), the total amount required
        :verify_certificate: (bool)

        It returns a dict which contains three variables:
            - success, (bool) True if the payment is valid
            - status, (str) The Paybox Response Code
            - auth_code, (str) The Authorization Code generated by the Authorization Center
        """

        url_parsed = urlparse(response_url)  # object
        message = url_parsed.query  # string
        query = parse_qs(message)  # dictionnary

        if verify_certificate:
            self.verify_certificate(
                message=message, signature=query["SIGN"][0])

        if not self.production:
            assert query["AU"][0] == "XXXXXX", "Incorrect Test Authorization Code"
        else:
            assert "RC" in query, "No Response Code Returned"

        assert query["TO"][0] == str(
            order_total
        ), "Total does not match. PBX: %s - CMD: %s" % (
            query["TO"][0],
            str(order_total),
        )

        return {
            "success": True if query["RC"][0] == "00000" else False,
            "status": self.RESPONSE_CODES.get(
                query["RC"][0][:-2] + "xx",
                self.RESPONSE_CODES.get(
                    query["RC"][0], "Unrecognized Response Code"),
            ),
            "auth_code": query["AU"][0] if "AU" in query else False,
        }

    def verify_certificate(self, message, signature):
        """Verifies the Paybox certificate, authenticity and alteration.
        If everything goes well, returns True. Otherwise raise an Error

        :message: (str), the full url with its args
        :signature: (str), the signature of the message, separated from the url

        Flow:
            - The signature is decoded base64
            - The signature is removed from the message
            - The Paybox pubkey is loaded from an external file
            - it's validity is checked
            - The message is digested by SHA1
            - The SHA1 message is verified against the binary signature
        """

        # detach the signature from the message
        message_without_sign = message.split("&SIGN=")[0]
        # decode base64 the signature
        binary_signature = base64.b64decode(signature)
        # create a pubkey object
        key = RSA.importKey(
            open(os.path.join(os.path.dirname(__file__), "pubkey.pem"), "rb").read()
        )
        # digest the message
        h = SHA.new(bytes(message_without_sign, encoding="utf8"))
        # and verify the signature
        verifier = PKCS1_v1_5.new(key)
        assert verifier.verify(
            h, binary_signature), "Signature Verification Failed"

        return True
