"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Workload = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const WORKLOAD_SYMBOL = Symbol.for("@shady-island/core.Workload");
/**
 * A collection of Stacks in an Environment representing a deployment Tier.
 *
 * Derive a subclass of `Workload` and create your stacks within.
 *
 * The difference between this object and a `Stage` is that a `Stage` is meant
 * to be deployed with CDK Pipelines. This class can be used with `cdk deploy`.
 * This class also provides context loading capabilities.
 */
class Workload extends constructs_1.Construct {
    /**
     * Creates a new Workload.
     *
     * @param scope - The construct scope.
     * @param id - The construct ID.
     * @param props - The constructor options.
     */
    constructor(scope, id, props) {
        super(scope, id);
        Object.defineProperty(this, WORKLOAD_SYMBOL, { value: true });
        const { env: { region, account } = {}, tier, workloadName, contextFile, baseDomainName, } = props;
        this.workloadName = workloadName ?? `${id}`.toLowerCase();
        this.tier = tier;
        this.publicDomainName = baseDomainName
            ? `${tier.id}.${this.workloadName}.${baseDomainName}`.toLowerCase()
            : undefined;
        this._stacks = new Map();
        const { region: stageRegion, account: stageAccount } = aws_cdk_lib_1.Stage.of(this);
        this.region = region || stageRegion;
        this.account = account || stageAccount;
        if (contextFile) {
            this.loadContext(contextFile);
        }
    }
    /**
     * Return the Workload the construct is contained within, fails if there is no
     * workload up the tree.
     *
     * @param construct - The construct whose parent nodes will be searched
     * @returns The Workload containing the construct
     * @throws Error - if none of the construct's parents are a workload
     */
    static of(construct) {
        const workload = construct.node.scopes
            .reverse()
            .slice(1)
            .find(Workload.isWorkload);
        if (!workload) {
            throw new Error("No workload exists in the parentage of the provided construct");
        }
        return workload;
    }
    /**
     * Test whether the given construct is a Workload.
     *
     * @param x - The value to test
     * @returns Whether the value is a Workload object.
     */
    static isWorkload(x) {
        return x !== null && typeof x === "object" && WORKLOAD_SYMBOL in x;
    }
    loadContext(filename) {
        try {
            fs.accessSync(filename, fs.constants.F_OK);
        }
        catch (err) {
            throw new Error(`Context file does not exist: ${filename}`);
        }
        try {
            fs.accessSync(filename, fs.constants.R_OK);
        }
        catch (err) {
            throw new Error(`Context file is not readable: ${filename}`);
        }
        const data = fs.readFileSync(filename, { encoding: "utf8" });
        let defaults = {};
        try {
            defaults = JSON.parse(data);
        }
        catch (e) {
            throw new Error(`Context file contains invalid JSON syntax: ${filename}`);
        }
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
    }
    /**
     * @returns The stacks created by invoking `createStack`
     */
    get stacks() {
        return [...this._stacks.values()];
    }
    /**
     * Adds a stack to the Workload.
     *
     * This method will return a `Stack` with this Workload as its scope. By
     * default, the `stackName` property provided to the `Stack` will be this
     * Workload's `workloadName`, its `tier`, and the value of the `id`
     * parameter separated by hyphens, all in lowercase.
     *
     * @example
     *
     * const exampleDev = new Workload(app, 'Example', {
     *   tier: Tier.DEVELOPMENT,
     *   env: { account: '123456789012', region: 'us-east-1' },
     * });
     * const networkStack = exampleDev.createStack('Network', {});
     * assert.strictEqual(networkStack.stackName, 'example-dev-network').
     *
     * You can override the `env` and `stackName` properties in the `props`
     * argument if desired.
     *
     * The stack will have a `DeploymentTier` tag added, set to the tier label.
     *
     * @param id - The Stack construct id (e.g. "Network")
     * @param props - The new Stack properties
     */
    createStack(id, props) {
        const { stackName, ...options } = props ?? {};
        const newStackName = stackName ?? `${this.workloadName}-${this.tier.id}-${id}`.toLowerCase();
        const stack = new aws_cdk_lib_1.Stack(this, id, {
            stackName: newStackName,
            env: this.account && this.region
                ? { account: this.account, region: this.region }
                : undefined,
            ...options,
        });
        this.tier.applyTags(stack);
        this._stacks.set(newStackName, stack);
        return stack;
    }
}
exports.Workload = Workload;
_a = JSII_RTTI_SYMBOL_1;
Workload[_a] = { fqn: "shady-island.Workload", version: "0.0.9" };
//# sourceMappingURL=data:application/json;base64,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