# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/02_cifar_loader.ipynb (unless otherwise specified).

__all__ = ['download_torch_dsets', 'load_torch_items', 'load_classes', 'CifarNP2ImageTransform', 'Int2TensorTransform',
           'CifarImageTransform', 'CifarImage2FloatTransform', 'make_torch_tfms', 'CifarTupleTransform', 'TupleTorchDS',
           'make_cifar_item_tfm', 'i2t_tfm', 'cfnp2img_tfm', 'cfimg_tfm', 'cfimg2float_tfm', 'make_cifar_tls',
           'make_cifar_dl', 'make_fastai_cifar_dls']

# Cell
def download_torch_dsets(path, torch_dset):
    """Download cifar10 datasets using torchvision utils

       Arguments:
           path (pathlib.Path): path to download the dataset (aka root)
    """
    train_dataset = torch_dset(
        root=path,
        train=True,
        download=True

        )
    test_dataset = torch_dset(
        root=path,
        train=False,
        download=True,
        )
    return train_dataset,test_dataset

# Internal Cell
import numpy as np
import torchvision.datasets.utils as tv_utils
import torchvision.datasets.cifar as cifar_dsets
import pickle

# Cell
def load_torch_items(downloaded_list, path, check=False):
    """loads cifar test/train items into tuple(data, target)

       scrobbled together from torch.data.utils.datasets.CIFAR10 code
       https://pytorch.org/docs/stable/_modules/torchvision/datasets/cifar.html#CIFAR10

       Arguments:
           downloaded_list : a list of file names with their checksum, see CIFAR10.train_list or CIFAR10.test_list.
           path (pathlib.Path): the root path where the dataset was downloaded
           check(bool, optional): whether to perform an integrity check on the downloaded files (default: False)
    """
    data = []
    targets = []
    # now load the picked numpy arrays
    for file_name, checksum in downloaded_list:
        file_path = path/cifar_dsets.CIFAR10.base_folder/file_name
        if check and not tv_utils.check_integrity(file_path, checksum):
            raise RuntimeError(
        f'Data checksum failed for file:{file_path} checksum:{checksum}')
        with open(file_path, 'rb') as f:
            entry = pickle.load(f, encoding='latin1')
            data.append(entry['data'])
            if 'labels' in entry:
                targets.extend(entry['labels'])
            else:
                targets.extend(entry['fine_labels'])

    data = np.vstack(data).reshape(-1, 3, 32, 32)
    data = data.transpose((0, 2, 3, 1))  # convert to HWC

    return data, targets

# Cell
# TODO: incorporate list of classes into dataloaders vocab and decodes

from fastcore.foundation import L
def load_classes(path):
    """Load classes to used to map categories to target labels"""
    base_folder = cifar_dsets.CIFAR10.base_folder
    meta = cifar_dsets.CIFAR10.meta
    file_path = path/base_folder/meta['filename']
    if not tv_utils.check_integrity(file_path, meta['md5']):
        raise RuntimeError('Dataset metadata file not found or corrupted.' +
                           ' You can use download=True to download it')
    data = {}
    with open(file_path, 'rb') as infile:
        data = pickle.load(infile, encoding='latin1')
    classes = data[meta['key']]
#     class_for i, _class in enumerato_idx = {_class: i te(classes)}
    return L(classes)

# Internal Cell
from fastcore.transform import Transform
import torchvision.transforms.functional as TVF
import torch
import PIL

# Cell
class CifarNP2ImageTransform(Transform):
    def encodes(self, o:np.ndarray) -> None:
        return PIL.Image.fromarray(o)

# Cell
class Int2TensorTransform(Transform):
    def encodes(self, o: int) -> None:
        return torch.tensor(o)

# Cell
class CifarImageTransform(Transform):
    def encodes(self, o: PIL.Image) -> None:
        return TVF.to_tensor(o)

# Cell
class CifarImage2FloatTransform(Transform):
    def encodes(self, o: torch.Tensor) -> None:
        return o.float().div_(255.)

# Internal Cell
import torchvision as thv

# Cell
def make_torch_tfms():
    norm = thv.transforms.Normalize(
        mean=(0.4914, 0.4822, 0.4465), std=(0.2023, 0.1994, 0.2010))
    transform_train = thv.transforms.Compose([
        thv.transforms.RandomCrop(32, padding=4),
        thv.transforms.RandomHorizontalFlip(),
        thv.transforms.ToTensor(),
        norm,
    ])
    transform_test = thv.transforms.Compose([
        thv.transforms.ToTensor(),
        norm,
    ])
    return transform_train, transform_test

# Internal Cell
from fastcore.transform import ItemTransform
from fastcore.basics import store_attr

# Cell
class CifarTupleTransform(ItemTransform):
    def __init__(self, x_tfm, y_tfm):
        store_attr()
    def encodes(self,xy):
        return [self.x_tfm(xy[0]), self.y_tfm(xy[1])]

# Internal Cell
import torch.utils.data as th_data
from torch.utils.data import Dataset

# Cell
# TODO: Use TupleTorchDS to create torch dataloaders

class TupleTorchDS(th_data.Dataset):
    def __init__(self, items, x_tfm=None, y_tfm=None):
        store_attr()

    def __len__(self):
        return len(self.items)

    def __getitem__(self, index):
        x,y = self.items[index]
        x = self.x_tfm(x) if self.x_tfm is not None else x
        y = self.y_tfm(y) if self.y_tfm is not None else y
        return (x,y)

# Internal Cell
from fastcore.transform import Pipeline

# Cell
i2t_tfm = Int2TensorTransform() # cnvt int -> torch.tensor
cfnp2img_tfm = CifarNP2ImageTransform() # cnvt ndarray -> PIL.Image
cfimg_tfm = CifarImageTransform() # cnvt PIL.Image -> torch.tensor
cfimg2float_tfm = CifarImage2FloatTransform() # cnvt tensor int -> float + div 255

def make_cifar_item_tfm(th_img_tfms=None):
    img_tfms = [cfnp2img_tfm]
    if th_img_tfms is not None:
        # assumes th_img_tfms incl ToTensor (cnvt2 PIL.Image -> tensor + div by 255)
        img_tfms += [th_img_tfms]
    else:
        img_tfms += [cfimg_tfm, cfimg2float_tfm]

    return CifarTupleTransform(x_tfm=Pipeline(img_tfms), y_tfm=i2t_tfm)

# Internal Cell
from fastai.data.core import TfmdDL
from fastai.data.core import TfmdLists
from fastcore.foundation import L

# Cell
def make_cifar_tls(file_list, path, item_tfm, check=True):
    data, targets = load_torch_items(file_list, path, check=check)
    item_tuples = L(data,targets).zip()
    tls = TfmdLists(item_tuples,[item_tfm])
    return tls

# Cell
def make_cifar_dl(file_list, path, th_img_tfms=None, check=True, bs=64, **kwargs):
    item_tfm = make_cifar_item_tfm(th_img_tfms)
    tls = make_cifar_tls(file_list, path, item_tfm, check=check)
    dl = TfmdDL(tls,bs=bs, **kwargs)
    return dl

# Internal Cell
from fastai.data.core import DataLoaders

# Cell
def make_fastai_cifar_dls(path, bs=64, check=True, device=None, **kwargs):
    train_tfm, test_tfm = make_torch_tfms()
    train_dl = make_cifar_dl(
        cifar_dsets.CIFAR10.train_list,
        path,
        train_tfm,
        check=check, bs=bs,
        shuffle=True)
    test_dl = make_cifar_dl(
        cifar_dsets.CIFAR10.test_list,
        path,
        test_tfm,
        check=check, bs=bs,
        shuffle=False)
    dls = DataLoaders(train_dl, test_dl, device=device)
    return dls