Benchmarks
---

*Dislcaimer: remember benchmarks are generally fraught with issues and their
legitimacy depends on a multitude of factors.*

I've used basic, but seemingly nice, [`pytest-benchmark`](https://pytest-benchmark.readthedocs.io/en/stable/usage.html)
to produce the following benchmarks on a fairly modern machine.

If you'd like to run these yourself, you'll need whatever according system 
dependency, if any, required by the specific algorithm as well as the Python
lib installed. Can install all via `pip install -r benchmark-requirements.txt`. 

The resulting output from benchmarks indicate what file and lib was used, (`cramjam` vs `gzip`)
`test_gzip[urls.10K-cramjam]` indicates `cramjam` and `benchmarks/data/urls.10K` file was used during the gzip benchmark.


Special performance notes:
---
In general, if the length of the output from de/compression is known, for any variant,  
you can provide `output_len=<<some int>>` to get a 1-4x performance increase. This
is allows us to avoid double allocation. 

Furthermore, you can provide `output_len` equal to the "at most" length when 
de/compressing `bytearray` objects; as these can be resized after the de/compression 
is finished and actual size is known for basically the same performance. This is not
true for `bytes` objects, as we cannot resize after de/compression and the result
will be trialing null bytes.

For snappy used in conjunction with `bytearray`s, this is only midly helpful as
we automatically estimate the buffer size and can resize the `bytearray` after.


--- 

#### Gzip

`make bench-gzip`

```bash 
------------------------------------------------------------------------------------------------------------------- benchmark: 28 tests -------------------------------------------------------------------------------------------------------------------
Name (time in us)                                           Min                       Max                      Mean                  StdDev                    Median                     IQR            Outliers         OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_gzip[Mark.Twain-Tom.Sawyer.txt-cramjam]           499.5080 (1.0)            728.4359 (1.0)            512.5226 (1.0)           18.5201 (1.0)            502.8080 (1.0)           19.4933 (1.0)        208;63  1,951.1333 (1.0)        1469           1
test_gzip[Mark.Twain-Tom.Sawyer.txt-gzip]              574.5530 (1.15)           800.9881 (1.10)           590.6570 (1.15)          20.6061 (1.11)           579.6610 (1.15)          21.3466 (1.10)       204;62  1,693.0301 (0.87)       1384           1
test_gzip[alice29.txt-cramjam]                      10,784.0190 (21.59)       11,725.0331 (16.10)       11,019.1375 (21.50)        159.9243 (8.64)        10,992.1545 (21.86)        139.6049 (7.16)         17;7     90.7512 (0.05)         90           1
test_gzip[alice29.txt-gzip]                         11,856.6670 (23.74)       12,723.6000 (17.47)       12,117.6208 (23.64)        154.9040 (8.36)        12,090.8240 (24.05)        148.3495 (7.61)         14;5     82.5245 (0.04)         81           1
test_gzip[asyoulik.txt-cramjam]                      8,255.5760 (16.53)        8,808.0181 (12.09)        8,388.3462 (16.37)         89.2145 (4.82)         8,374.5810 (16.66)        102.5778 (5.26)         30;4    119.2130 (0.06)        119           1
test_gzip[asyoulik.txt-gzip]                         9,073.6510 (18.17)        9,783.6760 (13.43)        9,242.8460 (18.03)        109.5283 (5.91)         9,221.6395 (18.34)        117.2170 (6.01)         23;5    108.1918 (0.06)        106           1
test_gzip[fifty-four-mb-random-cramjam]          1,918,547.3130 (>1000.0)  2,281,174.9650 (>1000.0)  2,015,150.7522 (>1000.0)  151,686.4135 (>1000.0)  1,968,377.9971 (>1000.0)  140,688.2324 (>1000.0)       1;1      0.4962 (0.00)          5           1
test_gzip[fifty-four-mb-random-gzip]             1,665,076.1690 (>1000.0)  1,747,323.1020 (>1000.0)  1,705,157.4009 (>1000.0)   29,886.1207 (>1000.0)  1,704,222.0499 (>1000.0)   35,055.7323 (>1000.0)       2;0      0.5865 (0.00)          5           1
test_gzip[fifty-four-mb-repeating-cramjam]         202,266.0240 (404.93)     203,656.1889 (279.58)     202,876.8688 (395.84)       586.9810 (31.69)      202,665.2470 (403.07)       976.8272 (50.11)         2;0      4.9291 (0.00)          5           1
test_gzip[fifty-four-mb-repeating-gzip]            406,177.8371 (813.16)     408,499.9190 (560.79)     406,954.2060 (794.02)       941.9838 (50.86)      406,857.7340 (809.17)     1,182.0850 (60.64)         1;0      2.4573 (0.00)          5           1
test_gzip[fireworks.jpeg-cramjam]                    3,925.8640 (7.86)         5,138.1841 (7.05)         4,011.1756 (7.83)         110.0326 (5.94)         3,984.3025 (7.92)          83.2741 (4.27)        20;11    249.3035 (0.13)        246           1
test_gzip[fireworks.jpeg-gzip]                       3,184.8371 (6.38)         3,616.7459 (4.97)         3,262.3774 (6.37)          59.6370 (3.22)         3,251.3360 (6.47)          72.2620 (3.71)         62;8    306.5249 (0.16)        302           1
test_gzip[geo.protodata-cramjam]                     1,572.8211 (3.15)         1,828.2511 (2.51)         1,610.3785 (3.14)          36.8006 (1.99)         1,599.6840 (3.18)          44.1979 (2.27)       101;18    620.9720 (0.32)        609           1
test_gzip[geo.protodata-gzip]                        2,136.3030 (4.28)         2,421.2720 (3.32)         2,184.2864 (4.26)          51.3550 (2.77)         2,165.2665 (4.31)          64.8794 (3.33)        64;14    457.8154 (0.23)        436           1
test_gzip[html-cramjam]                              1,950.0050 (3.90)         2,314.5130 (3.18)         2,001.8077 (3.91)          46.9272 (2.53)         1,987.3570 (3.95)          61.8350 (3.17)        81;12    499.5485 (0.26)        480           1
test_gzip[html-gzip]                                 2,466.7480 (4.94)         4,897.7289 (6.72)         2,561.2299 (5.00)         230.8277 (12.46)        2,516.4341 (5.00)          77.6293 (3.98)        10;18    390.4374 (0.20)        391           1
test_gzip[html_x_4-cramjam]                          9,295.9950 (18.61)       10,382.4440 (14.25)        9,466.6361 (18.47)        166.3815 (8.98)         9,422.9169 (18.74)        134.9731 (6.92)         12;9    105.6341 (0.05)        107           1
test_gzip[html_x_4-gzip]                            10,916.7770 (21.86)       11,705.1441 (16.07)       11,102.0056 (21.66)        138.3860 (7.47)        11,072.2100 (22.02)        140.6029 (7.21)         17;4     90.0738 (0.05)         86           1
test_gzip[kppkn.gtb-cramjam]                        38,534.4781 (77.14)       39,374.4000 (54.05)       38,952.7990 (76.00)        200.1030 (10.80)       38,965.1706 (77.50)        238.4051 (12.23)         7;0     25.6721 (0.01)         26           1
test_gzip[kppkn.gtb-gzip]                           62,682.8249 (125.49)      64,644.2850 (88.74)       63,360.6900 (123.63)       646.7011 (34.92)       63,061.5850 (125.42)       938.3130 (48.14)         4;0     15.7827 (0.01)         16           1
test_gzip[lcet10.txt-cramjam]                       26,310.1180 (52.67)       27,208.7040 (37.35)       26,565.8054 (51.83)        223.3611 (12.06)       26,516.8336 (52.74)        235.3310 (12.07)        10;3     37.6424 (0.02)         38           1
test_gzip[lcet10.txt-gzip]                          29,205.7050 (58.47)       30,147.2150 (41.39)       29,597.1343 (57.75)        193.6721 (10.46)       29,561.2030 (58.79)        219.0185 (11.24)         5;2     33.7871 (0.02)         32           1
test_gzip[paper-100k.pdf-cramjam]                    2,809.4910 (5.62)         3,251.7480 (4.46)         2,873.6302 (5.61)          51.4007 (2.78)         2,861.0771 (5.69)          55.6436 (2.85)        50;15    347.9919 (0.18)        343           1
test_gzip[paper-100k.pdf-gzip]                       3,216.7250 (6.44)         3,755.8790 (5.16)         3,286.1528 (6.41)          71.3635 (3.85)         3,266.0410 (6.50)          74.8790 (3.84)        37;12    304.3072 (0.16)        298           1
test_gzip[plrabn12.txt-cramjam]                     47,907.7781 (95.91)       50,276.0150 (69.02)       48,341.7333 (94.32)        489.6980 (26.44)       48,157.5651 (95.78)        338.0665 (17.34)         1;1     20.6861 (0.01)         21           1
test_gzip[plrabn12.txt-gzip]                        52,439.5080 (104.98)      55,409.8539 (76.07)       53,299.8328 (104.00)       872.4323 (47.11)       53,018.4830 (105.44)     1,305.1905 (66.96)         4;0     18.7618 (0.01)         19           1
test_gzip[urls.10K-cramjam]                         35,436.3660 (70.94)       36,588.2240 (50.23)       35,778.6046 (69.81)        306.0787 (16.53)       35,707.5301 (71.02)        246.1017 (12.62)         6;3     27.9497 (0.01)         27           1
test_gzip[urls.10K-gzip]                            40,238.6910 (80.56)       41,588.8891 (57.09)       40,757.2876 (79.52)        266.3417 (14.38)       40,718.7125 (80.98)        206.5550 (10.60)         3;3     24.5355 (0.01)         24           1
```

#### Snappy

There are two types here, `framed` and `raw`; the recommended one being snappy's `framed` format.

`make bench-snappy-framed`

```bash
------------------------------------------------------------------------------------------------------------------------- benchmark: 28 tests -------------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                    Min                       Max                      Mean                    StdDev                    Median                     IQR            Outliers          OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_snappy_framed[Mark.Twain-Tom.Sawyer.txt-cramjam]            67.8750 (1.0)            184.4690 (1.0)             69.9458 (1.0)              5.8189 (1.0)             68.5219 (1.0)            0.4220 (1.72)      394;606  14,296.7757 (1.0)        5157           1
test_snappy_framed[Mark.Twain-Tom.Sawyer.txt-snappy]            127.5920 (1.88)           321.8240 (1.74)           131.6162 (1.88)             9.4019 (1.62)           128.8950 (1.88)           0.7390 (3.01)      407;905   7,597.8511 (0.53)       5234           1
test_snappy_framed[alice29.txt-cramjam]                         676.6730 (9.97)           965.0890 (5.23)           693.1161 (9.91)            27.2101 (4.68)           679.8870 (9.92)          20.7548 (84.45)      143;90   1,442.7596 (0.10)       1423           1
test_snappy_framed[alice29.txt-snappy]                        1,362.6880 (20.08)        1,884.4430 (10.22)        1,394.3984 (19.94)           43.0275 (7.39)         1,385.3500 (20.22)         39.8223 (162.04)      62;37     717.1552 (0.05)        679           1
test_snappy_framed[asyoulik.txt-cramjam]                        593.6730 (8.75)           886.0190 (4.80)           607.4296 (8.68)            24.3360 (4.18)           596.2565 (8.70)          19.8330 (80.70)      174;85   1,646.2814 (0.12)       1612           1
test_snappy_framed[asyoulik.txt-snappy]                       1,154.8221 (17.01)        1,474.2750 (7.99)         1,179.9538 (16.87)           33.2390 (5.71)         1,169.8285 (17.07)         33.0758 (134.59)      86;34     847.4908 (0.06)        814           1
test_snappy_framed[fifty-four-mb-random-cramjam]            106,125.5330 (>1000.0)    122,712.4280 (665.22)     113,907.3788 (>1000.0)      4,144.4500 (712.24)     113,855.9120 (>1000.0)    2,535.2445 (>1000.0)       3;2       8.7791 (0.00)         11           1
test_snappy_framed[fifty-four-mb-random-snappy]           5,736,722.0209 (>1000.0)  8,884,037.3160 (>1000.0)  8,029,947.1742 (>1000.0)  1,294,217.3836 (>1000.0)  8,468,064.0850 (>1000.0)  931,786.2789 (>1000.0)       1;1       0.1245 (0.00)          5           1
test_snappy_framed[fifty-four-mb-repeating-cramjam]          60,967.8230 (898.24)      82,541.3500 (447.45)      68,490.7448 (979.20)       8,907.2626 (>1000.0)     62,633.5774 (914.07)    16,801.8445 (>1000.0)       4;0      14.6005 (0.00)         12           1
test_snappy_framed[fifty-four-mb-repeating-snappy]          416,213.7700 (>1000.0)    445,269.2589 (>1000.0)    425,080.6548 (>1000.0)     12,411.2216 (>1000.0)    418,530.8550 (>1000.0)   16,490.2027 (>1000.0)       1;0       2.3525 (0.00)          5           1
test_snappy_framed[fireworks.jpeg-cramjam]                       90.7030 (1.34)           265.3861 (1.44)            93.6783 (1.34)             6.9734 (1.20)            91.9680 (1.34)           0.3370 (1.37)     696;1196  10,674.8307 (0.75)       8521           1
test_snappy_framed[fireworks.jpeg-snappy]                       634.0879 (9.34)           905.7500 (4.91)           653.3697 (9.34)            24.7685 (4.26)           643.7055 (9.39)          14.7215 (59.90)     151;145   1,530.5271 (0.11)       1472           1
test_snappy_framed[geo.protodata-cramjam]                       204.8140 (3.02)           484.7000 (2.63)           210.1287 (3.00)            12.0245 (2.07)           206.3761 (3.01)           0.9029 (3.67)      410;868   4,758.9888 (0.33)       4299           1
test_snappy_framed[geo.protodata-snappy]                        727.0420 (10.71)          979.4390 (5.31)           744.5916 (10.65)           25.8275 (4.44)           732.3179 (10.69)         22.5262 (91.66)      155;72   1,343.0181 (0.09)       1299           1
test_snappy_framed[html-cramjam]                                211.0000 (3.11)           400.2160 (2.17)           216.8802 (3.10)            12.6234 (2.17)           212.4980 (3.10)           0.7945 (3.23)      435;805   4,610.8402 (0.32)       3736           1
test_snappy_framed[html-snappy]                                 662.6330 (9.76)         1,020.1860 (5.53)           695.4484 (9.94)            29.1009 (5.00)           686.1100 (10.01)         23.4039 (95.23)      345;76   1,437.9211 (0.10)       1350           1
test_snappy_framed[html_x_4-cramjam]                            814.7360 (12.00)        1,156.0221 (6.27)           833.7065 (11.92)           34.9787 (6.01)           819.8969 (11.97)         21.4603 (87.32)       81;76   1,199.4629 (0.08)       1083           1
test_snappy_framed[html_x_4-snappy]                           2,654.4550 (39.11)        3,212.9940 (17.42)        2,729.8276 (39.03)           92.7821 (15.94)        2,701.6510 (39.43)         73.2925 (298.24)      31;25     366.3235 (0.03)        336           1
test_snappy_framed[kppkn.gtb-cramjam]                           582.3770 (8.58)           868.2209 (4.71)           596.1833 (8.52)            23.0826 (3.97)           584.8820 (8.54)          19.5719 (79.64)      194;81   1,677.3366 (0.12)       1515           1
test_snappy_framed[kppkn.gtb-snappy]                          1,413.1500 (20.82)        1,795.0049 (9.73)         1,441.8433 (20.61)           38.8178 (6.67)         1,435.4640 (20.95)         34.1205 (138.84)      54;30     693.5566 (0.05)        603           1
test_snappy_framed[lcet10.txt-cramjam]                        1,766.2169 (26.02)        2,194.9910 (11.90)        1,809.2666 (25.87)           52.9258 (9.10)         1,795.1445 (26.20)         50.4309 (205.21)      42;24     552.7101 (0.04)        526           1
test_snappy_framed[lcet10.txt-snappy]                         3,776.0150 (55.63)        4,416.6800 (23.94)        3,865.1326 (55.26)           90.4870 (15.55)        3,843.4530 (56.09)         87.1853 (354.77)      22;12     258.7233 (0.02)        235           1
test_snappy_framed[paper-100k.pdf-cramjam]                       84.9301 (1.25)           217.1260 (1.18)            87.1440 (1.25)             6.4667 (1.11)            85.6020 (1.25)           0.2458 (1.0)      593;1125  11,475.2531 (0.80)       8555           1
test_snappy_framed[paper-100k.pdf-snappy]                       533.0379 (7.85)           845.9381 (4.59)           546.8427 (7.82)            24.6972 (4.24)           538.5000 (7.86)          16.9175 (68.84)     144;115   1,828.6796 (0.13)       1636           1
test_snappy_framed[plrabn12.txt-cramjam]                      2,386.8741 (35.17)        3,894.9240 (21.11)        2,475.3461 (35.39)          169.0649 (29.05)        2,435.1560 (35.54)         63.3658 (257.84)      20;30     403.9839 (0.03)        339           1
test_snappy_framed[plrabn12.txt-snappy]                       5,019.8400 (73.96)        5,844.5639 (31.68)        5,131.9534 (73.37)          112.6590 (19.36)        5,106.9070 (74.53)         89.4035 (363.79)      15;13     194.8576 (0.01)        188           1
test_snappy_framed[urls.10K-cramjam]                          2,149.3380 (31.67)        2,757.3260 (14.95)        2,204.8847 (31.52)           69.6202 (11.96)        2,185.4395 (31.89)         61.6229 (250.75)      30;15     453.5384 (0.03)        406           1
test_snappy_framed[urls.10K-snappy]                           5,894.0420 (86.84)        6,559.6290 (35.56)        6,029.6206 (86.20)          120.3123 (20.68)        5,994.6314 (87.48)         84.0485 (342.00)      26;18     165.8479 (0.01)        164           1
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

`make bench-snappy-raw`

```bash
------------------------------------------------------------------------------------------------------------ benchmark: 28 tests -------------------------------------------------------------------------------------------------------------
Name (time in us)                                              Min                    Max                   Mean              StdDev                 Median                 IQR            Outliers          OPS            Rounds  Iterations
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_snappy_raw[Mark.Twain-Tom.Sawyer.txt-cramjam]         51.9970 (3.43)        110.7800 (2.43)         53.6142 (3.37)       3.9642 (2.32)         52.7401 (3.38)       0.3098 (2.50)      432;630  18,651.7875 (0.30)       7791           1
test_snappy_raw[Mark.Twain-Tom.Sawyer.txt-snappy]          52.7010 (3.47)        122.6700 (2.69)         54.9113 (3.45)       4.9091 (2.88)         53.6260 (3.44)       0.4690 (3.78)     768;1676  18,211.1850 (0.29)      11113           1
test_snappy_raw[alice29.txt-cramjam]                      599.3680 (39.51)       895.2310 (19.61)       613.8907 (38.59)     22.1703 (13.00)       603.5195 (38.70)     20.4269 (164.76)     133;54   1,628.9544 (0.03)       1158           1
test_snappy_raw[alice29.txt-snappy]                       600.2350 (39.57)       775.9969 (17.00)       613.0591 (38.54)     17.8929 (10.49)       603.6960 (38.71)     19.4444 (156.83)     231;68   1,631.1640 (0.03)       1631           1
test_snappy_raw[asyoulik.txt-cramjam]                     532.1550 (35.08)       831.1180 (18.21)       545.4919 (34.29)     21.3844 (12.54)       535.0631 (34.31)     19.8718 (160.28)     167;63   1,833.2078 (0.03)       1307           1
test_snappy_raw[asyoulik.txt-snappy]                      532.5220 (35.10)       800.6040 (17.54)       544.5727 (34.24)     18.2074 (10.67)       536.1935 (34.38)     12.6080 (101.69)    219;197   1,836.3022 (0.03)       1830           1
test_snappy_raw[fifty-four-mb-random-cramjam]          37,541.5270 (>1000.0)  38,867.6571 (851.52)   37,919.1172 (>1000.0)  307.6292 (180.35)   37,829.9700 (>1000.0)  311.4055 (>1000.0)       6;2      26.3719 (0.00)         27           1
test_snappy_raw[fifty-four-mb-random-snappy]           54,326.6139 (>1000.0)  55,845.7799 (>1000.0)  54,884.3537 (>1000.0)  412.7079 (241.95)   54,832.6219 (>1000.0)  475.0847 (>1000.0)       6;1      18.2201 (0.00)         19           1
test_snappy_raw[fifty-four-mb-repeating-cramjam]       17,394.3370 (>1000.0)  18,066.0550 (395.79)   17,612.4400 (>1000.0)  163.3287 (95.75)    17,561.4910 (>1000.0)  237.1181 (>1000.0)      11;0      56.7780 (0.00)         29           1
test_snappy_raw[fifty-four-mb-repeating-snappy]        31,430.8960 (>1000.0)  32,698.4840 (716.36)   31,772.2997 (>1000.0)  271.8508 (159.38)   31,746.9710 (>1000.0)  266.7012 (>1000.0)       4;2      31.4740 (0.00)         31           1
test_snappy_raw[fireworks.jpeg-cramjam]                    25.3069 (1.67)         59.5120 (1.30)         26.2138 (1.65)       2.6992 (1.58)         25.5830 (1.64)       0.1240 (1.0)       349;901  38,147.7836 (0.61)       7113           1
test_snappy_raw[fireworks.jpeg-snappy]                     15.1701 (1.0)          45.6451 (1.0)          15.9064 (1.0)        1.7057 (1.0)          15.5960 (1.0)        0.1411 (1.14)     988;5393  62,867.9262 (1.0)       35170           1
test_snappy_raw[geo.protodata-cramjam]                    155.7680 (10.27)       250.6210 (5.49)        160.3773 (10.08)      9.2061 (5.40)        157.2450 (10.08)      0.8419 (6.79)      356;768   6,235.2982 (0.10)       3922           1
test_snappy_raw[geo.protodata-snappy]                     142.9900 (9.43)        266.7190 (5.84)        148.3392 (9.33)       8.0790 (4.74)        145.8429 (9.35)       2.2233 (17.93)     471;684   6,741.3054 (0.11)       5261           1
test_snappy_raw[html-cramjam]                             165.5300 (10.91)       266.9621 (5.85)        169.8480 (10.68)      9.1233 (5.35)        166.7660 (10.69)      0.6332 (5.11)      348;702   5,887.6187 (0.09)       3877           1
test_snappy_raw[html-snappy]                              156.3310 (10.31)       247.4729 (5.42)        160.6741 (10.10)      8.7547 (5.13)        157.9490 (10.13)      0.7500 (6.05)      480;904   6,223.7766 (0.10)       5697           1
test_snappy_raw[html_x_4-cramjam]                         660.6469 (43.55)     1,195.4820 (26.19)       678.1065 (42.63)     39.0818 (22.91)       664.1049 (42.58)     20.3864 (164.43)      42;47   1,474.6945 (0.02)       1035           1
test_snappy_raw[html_x_4-snappy]                          633.0810 (41.73)       917.3179 (20.10)       648.8233 (40.79)     21.6437 (12.69)       638.4600 (40.94)     20.0127 (161.42)     193;63   1,541.2517 (0.02)       1451           1
test_snappy_raw[kppkn.gtb-cramjam]                        499.2380 (32.91)       708.5471 (15.52)       510.3782 (32.09)     18.7145 (10.97)       501.3411 (32.15)     14.9945 (120.94)    175;131   1,959.3315 (0.03)       1355           1
test_snappy_raw[kppkn.gtb-snappy]                         503.4071 (33.18)       733.6430 (16.07)       514.8920 (32.37)     18.8445 (11.05)       505.8900 (32.44)     14.7057 (118.61)    223;174   1,942.1549 (0.03)       1907           1
test_snappy_raw[lcet10.txt-cramjam]                     1,585.7200 (104.53)    1,902.8189 (41.69)     1,620.1644 (101.86)    42.2322 (24.76)     1,608.8235 (103.16)    43.0678 (347.37)      57;18     617.2213 (0.01)        486           1
test_snappy_raw[lcet10.txt-snappy]                      1,590.4551 (104.84)    2,116.2720 (46.36)     1,627.2067 (102.30)    47.9340 (28.10)     1,616.7820 (103.67)    41.8220 (337.32)      48;22     614.5501 (0.01)        534           1
test_snappy_raw[paper-100k.pdf-cramjam]                    28.1520 (1.86)         69.1239 (1.51)         29.0821 (1.83)       2.6677 (1.56)         28.4630 (1.83)       0.1448 (1.17)     344;1243  34,385.3648 (0.55)       7894           1
test_snappy_raw[paper-100k.pdf-snappy]                     20.3360 (1.34)         85.0500 (1.86)         21.3502 (1.34)       2.1718 (1.27)         20.9491 (1.34)       0.4959 (4.00)     930;1321  46,838.0012 (0.75)      25100           1
test_snappy_raw[plrabn12.txt-cramjam]                   2,169.4830 (143.01)    4,583.7279 (100.42)    2,236.1366 (140.58)   180.7923 (105.99)    2,212.1780 (141.84)    60.5080 (488.04)       7;10     447.1999 (0.01)        318           1
test_snappy_raw[plrabn12.txt-snappy]                    2,187.1999 (144.18)    2,717.8620 (59.54)     2,246.2940 (141.22)    67.8218 (39.76)     2,230.6465 (143.03)    47.7184 (384.88)      38;30     445.1777 (0.01)        392           1
test_snappy_raw[urls.10K-cramjam]                       1,852.2060 (122.10)    2,464.6381 (54.00)     1,899.2503 (119.40)    65.4195 (38.35)     1,882.6990 (120.72)    52.5715 (424.02)      31;18     526.5235 (0.01)        424           1
test_snappy_raw[urls.10K-snappy]                        1,814.2400 (119.59)    2,366.7030 (51.85)     1,855.7408 (116.67)    63.5185 (37.24)     1,840.4050 (118.00)    45.1091 (363.83)      25;20     538.8684 (0.01)        438           1
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```


##### Benchmarks for using `compress_into` and `decompress_into`
All variants implement a `compress_into` and `decompress_into`
This is where, if you have a Python numpy array pre-allocated to the size of the
de/compress data output, we can place the bytes directly into that buffer.

Again, since basically no variants implement similar functionality as we saw in the Gzip `output_len`
benchmarks, this benchmark is specific to `cramjam`

```bash
--------------------------------------------------------------------------------------------------------------------------- benchmark: 28 tests ----------------------------------------------------------------------------------------------------------------------------
Name (time in us)                                                                           Min                    Max                   Mean              StdDev                 Median                 IQR            Outliers           OPS            Rounds  Iterations
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_cramjam_snappy_de_compress_into[Mark.Twain-Tom.Sawyer.txt-compress_into]           40.5740 (6.10)         80.9841 (4.98)         42.0519 (6.02)       3.3541 (3.54)         41.1549 (5.99)       0.3109 (4.71)    1006;2779   23,780.1522 (0.17)      17961           1
test_cramjam_snappy_de_compress_into[Mark.Twain-Tom.Sawyer.txt-decompress_into]          6.7360 (1.01)         21.8459 (1.34)          6.9848 (1.0)        0.9464 (1.0)           6.8730 (1.0)        0.0660 (1.0)       122;700  143,168.7285 (1.0)        9684           1
test_cramjam_snappy_de_compress_into[alice29.txt-compress_into]                        461.2940 (69.40)       669.6520 (41.18)       473.9482 (67.85)     18.6320 (19.69)       464.7850 (67.62)     15.4326 (233.80)    240;175    2,109.9353 (0.01)       1992           1
test_cramjam_snappy_de_compress_into[alice29.txt-decompress_into]                        6.7560 (1.02)         22.4550 (1.38)          7.1457 (1.02)       0.9911 (1.05)          7.0620 (1.03)       0.1279 (1.94)       51;203  139,943.6715 (0.98)       3866           1
test_cramjam_snappy_de_compress_into[asyoulik.txt-compress_into]                       408.1880 (61.41)       628.2150 (38.64)       420.7408 (60.24)     17.0570 (18.02)       411.6645 (59.90)     19.7020 (298.48)     301;60    2,376.7602 (0.02)       2338           1
test_cramjam_snappy_de_compress_into[asyoulik.txt-decompress_into]                       6.6471 (1.0)          35.4199 (2.18)          7.1268 (1.02)       1.2031 (1.27)          7.0110 (1.02)       0.0992 (1.50)       86;954  140,315.1785 (0.98)       5545           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-random-compress_into]            21,292.6060 (>1000.0)  22,134.7660 (>1000.0)  21,556.6603 (>1000.0)  195.2096 (206.28)   21,524.6500 (>1000.0)  282.1560 (>1000.0)       7;1       46.3894 (0.00)         26           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-random-decompress_into]               6.8980 (1.04)         17.6111 (1.08)          7.4460 (1.07)       1.8980 (2.01)          7.0570 (1.03)       0.2597 (3.93)          1;2  134,299.5668 (0.94)         31           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-repeating-compress_into]         14,725.2050 (>1000.0)  15,681.2500 (964.41)   14,952.5908 (>1000.0)  172.2539 (182.02)   14,909.9910 (>1000.0)  154.4059 (>1000.0)       8;3       66.8780 (0.00)         62           1
test_cramjam_snappy_de_compress_into[fifty-four-mb-repeating-decompress_into]            6.8520 (1.03)         16.2600 (1.0)           7.4351 (1.06)       1.7599 (1.86)          7.0125 (1.02)       0.1526 (2.31)          1;5  134,496.9292 (0.94)         28           1
test_cramjam_snappy_de_compress_into[fireworks.jpeg-compress_into]                      45.8390 (6.90)         94.1610 (5.79)         47.7454 (6.84)       3.5484 (3.75)         46.8700 (6.82)       0.8349 (12.65)   1167;2927   20,944.4108 (0.15)      19418           1
test_cramjam_snappy_de_compress_into[fireworks.jpeg-decompress_into]                     6.7510 (1.02)         41.1160 (2.53)          7.1202 (1.02)       1.1970 (1.26)          6.9160 (1.01)       0.1888 (2.86)      515;974  140,445.6319 (0.98)      25688           1
test_cramjam_snappy_de_compress_into[geo.protodata-compress_into]                      120.1930 (18.08)       224.7400 (13.82)       124.6344 (17.84)      7.8673 (8.31)        122.0175 (17.75)      1.2110 (18.35)    666;1483    8,023.4694 (0.06)       7554           1
test_cramjam_snappy_de_compress_into[geo.protodata-decompress_into]                      6.7500 (1.02)         28.4720 (1.75)          7.2207 (1.03)       1.2935 (1.37)          7.0831 (1.03)       0.2290 (3.47)      260;297  138,490.1876 (0.97)      11741           1
test_cramjam_snappy_de_compress_into[html-compress_into]                               128.7600 (19.37)       240.4600 (14.79)       133.1894 (19.07)      8.4420 (8.92)        130.3609 (18.97)      0.9710 (14.71)    619;1486    7,508.1029 (0.05)       7215           1
test_cramjam_snappy_de_compress_into[html-decompress_into]                               6.7130 (1.01)         29.0590 (1.79)          7.3321 (1.05)       1.5681 (1.66)          6.9850 (1.02)       0.2070 (3.14)      878;946  136,386.3798 (0.95)      12593           1
test_cramjam_snappy_de_compress_into[html_x_4-compress_into]                           510.7940 (76.84)       783.9740 (48.21)       525.5117 (75.24)     21.4496 (22.67)       514.8470 (74.91)     20.4052 (309.13)     227;82    1,902.9072 (0.01)       1807           1
test_cramjam_snappy_de_compress_into[html_x_4-decompress_into]                           6.7271 (1.01)         26.0500 (1.60)          7.2545 (1.04)       1.3682 (1.45)          6.9321 (1.01)       0.3700 (5.60)       89;281  137,844.8162 (0.96)       3292           1
test_cramjam_snappy_de_compress_into[kppkn.gtb-compress_into]                          360.5250 (54.24)       599.6680 (36.88)       371.5869 (53.20)     15.3924 (16.26)       363.2680 (52.85)     20.0116 (303.17)     306;42    2,691.1603 (0.02)       2497           1
test_cramjam_snappy_de_compress_into[kppkn.gtb-decompress_into]                          6.7559 (1.02)         28.5150 (1.75)          7.2983 (1.04)       1.3938 (1.47)          7.1101 (1.03)       0.2340 (3.54)      129;254  137,018.7128 (0.96)       4798           1
test_cramjam_snappy_de_compress_into[lcet10.txt-compress_into]                       1,220.3240 (183.59)    1,442.1020 (88.69)     1,252.7170 (179.35)    34.3613 (36.31)     1,244.1790 (181.02)    44.0781 (667.77)     138;22      798.2649 (0.01)        814           1
test_cramjam_snappy_de_compress_into[lcet10.txt-decompress_into]                         6.7160 (1.01)         27.3180 (1.68)          7.2492 (1.04)       1.3898 (1.47)          7.0255 (1.02)       0.3814 (5.78)       60;125  137,946.6539 (0.96)       1868           1
test_cramjam_snappy_de_compress_into[paper-100k.pdf-compress_into]                      39.6080 (5.96)         91.3910 (5.62)         41.3625 (5.92)       3.3869 (3.58)         40.6440 (5.91)       0.9619 (14.57)   1132;1662   24,176.4958 (0.17)      20478           1
test_cramjam_snappy_de_compress_into[paper-100k.pdf-decompress_into]                     6.7310 (1.01)         31.5481 (1.94)          7.1218 (1.02)       1.1942 (1.26)          6.9310 (1.01)       0.1870 (2.83)      440;725  140,414.1376 (0.98)      23261           1
test_cramjam_snappy_de_compress_into[plrabn12.txt-compress_into]                     1,651.1610 (248.40)    2,189.8210 (134.68)    1,689.8730 (241.94)    46.3566 (48.98)     1,677.2315 (244.03)    50.6585 (767.47)      69;14      591.7604 (0.00)        572           1
test_cramjam_snappy_de_compress_into[plrabn12.txt-decompress_into]                       6.6870 (1.01)         36.4761 (2.24)          7.1576 (1.02)       1.5466 (1.63)          6.8745 (1.00)       0.2451 (3.71)        44;53  139,711.1538 (0.98)       1228           1
test_cramjam_snappy_de_compress_into[urls.10K-compress_into]                         1,445.5540 (217.47)    1,788.2879 (109.98)    1,486.7689 (212.86)    41.1075 (43.44)     1,474.6580 (214.56)    52.1065 (789.40)      88;14      672.5995 (0.00)        636           1
test_cramjam_snappy_de_compress_into[urls.10K-decompress_into]                           6.7020 (1.01)         31.9510 (1.97)          7.3564 (1.05)       1.7521 (1.85)          6.8840 (1.00)       0.2775 (4.20)       85;117  135,935.3108 (0.95)       1117           1
----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Lz4

`make bench-lz4`

```bash
------------------------------------------------------------------------------------------------------------------- benchmark: 28 tests -------------------------------------------------------------------------------------------------------------------
Name (time in us)                                             Min                       Max                      Mean                 StdDev                    Median                    IQR            Outliers         OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_lz4[Mark.Twain-Tom.Sawyer.txt-cramjam]              245.3330 (1.0)          7,722.4160 (21.29)          364.4594 (1.05)        240.3840 (33.07)          368.7030 (1.06)         18.0650 (2.54)        3;265  2,743.7899 (0.95)       1155           1
test_lz4[Mark.Twain-Tom.Sawyer.txt-python-lz4]           335.9900 (1.37)           362.7009 (1.0)            347.2828 (1.0)           7.2700 (1.0)            346.6295 (1.0)           7.1260 (1.0)           2;1  2,879.4977 (1.0)          10           1
test_lz4[alice29.txt-cramjam]                          2,814.1720 (11.47)        4,936.4029 (13.61)        3,046.3559 (8.77)        379.7652 (52.24)        2,920.2300 (8.42)         70.8259 (9.94)        33;46    328.2611 (0.11)        338           1
test_lz4[alice29.txt-python-lz4]                       3,121.6950 (12.72)        5,633.2920 (15.53)        3,441.3794 (9.91)        478.2910 (65.79)        3,252.9815 (9.38)        140.1610 (19.67)       34;50    290.5812 (0.10)        300           1
test_lz4[asyoulik.txt-cramjam]                         2,314.5641 (9.43)         3,103.1220 (8.56)         2,382.4872 (6.86)         92.2261 (12.69)        2,359.6610 (6.81)         74.5750 (10.47)       29;18    419.7294 (0.15)        404           1
test_lz4[asyoulik.txt-python-lz4]                      2,633.2920 (10.73)        4,455.6119 (12.28)        2,722.9006 (7.84)        196.3372 (27.01)        2,682.5680 (7.74)         65.2385 (9.15)        10;21    367.2554 (0.13)        367           1
test_lz4[fifty-four-mb-random-cramjam]               757,487.6130 (>1000.0)    792,758.5000 (>1000.0)    772,111.8064 (>1000.0)  13,251.3858 (>1000.0)    769,733.5170 (>1000.0)  16,313.7843 (>1000.0)       2;0      1.2951 (0.00)          5           1
test_lz4[fifty-four-mb-random-python-lz4]          1,597,680.0489 (>1000.0)  1,637,720.9669 (>1000.0)  1,613,587.9904 (>1000.0)  17,438.2782 (>1000.0)  1,606,424.0520 (>1000.0)  29,249.6240 (>1000.0)       1;0      0.6197 (0.00)          5           1
test_lz4[fifty-four-mb-repeating-cramjam]            222,541.1070 (907.10)     237,449.5791 (654.67)     228,523.9746 (658.03)    6,592.2073 (906.77)     227,023.9190 (654.95)   11,598.6770 (>1000.0)       1;0      4.3759 (0.00)          5           1
test_lz4[fifty-four-mb-repeating-python-lz4]         161,025.9259 (656.36)     356,819.4229 (983.78)     214,613.3904 (617.98)   73,690.5136 (>1000.0)    186,189.4020 (537.14)   90,134.4944 (>1000.0)       1;0      4.6595 (0.00)          7           1
test_lz4[fireworks.jpeg-cramjam]                       1,556.7919 (6.35)         3,704.2680 (10.21)        1,665.9848 (4.80)        218.8115 (30.10)        1,613.6711 (4.66)         33.3220 (4.68)        26;59    600.2456 (0.21)        569           1
test_lz4[fireworks.jpeg-python-lz4]                    2,682.0981 (10.93)        5,521.3780 (15.22)        2,876.1775 (8.28)        411.3732 (56.59)        2,762.2810 (7.97)         68.5747 (9.62)        22;43    347.6837 (0.12)        365           1
test_lz4[geo.protodata-cramjam]                          744.4300 (3.03)         1,499.9070 (4.14)           799.9747 (2.30)         82.7017 (11.38)          779.4825 (2.25)         27.9890 (3.93)        58;87  1,250.0395 (0.43)       1174           1
test_lz4[geo.protodata-python-lz4]                       694.2170 (2.83)         1,546.8430 (4.26)           722.2650 (2.08)         47.9471 (6.60)           718.2080 (2.07)         22.3070 (3.13)        51;71  1,384.5333 (0.48)       1383           1
test_lz4[html-cramjam]                                   778.6801 (3.17)         1,917.2900 (5.29)           832.4302 (2.40)        111.0058 (15.27)          805.6900 (2.32)         13.8602 (1.95)       58;284  1,201.3020 (0.42)       1162           1
test_lz4[html-python-lz4]                                716.7751 (2.92)         1,543.9009 (4.26)           757.5458 (2.18)        105.6609 (14.53)          730.4699 (2.11)         27.0145 (3.79)       63;104  1,320.0522 (0.46)       1340           1
test_lz4[html_x_4-cramjam]                             3,105.4450 (12.66)        4,875.9540 (13.44)        3,301.6318 (9.51)        241.6346 (33.24)        3,248.0890 (9.37)        129.7960 (18.21)       20;21    302.8805 (0.11)        289           1
test_lz4[html_x_4-python-lz4]                          3,016.9110 (12.30)        5,823.1900 (16.06)        3,218.6525 (9.27)        294.3938 (40.49)        3,157.1820 (9.11)        103.9958 (14.59)       20;23    310.6890 (0.11)        255           1
test_lz4[kppkn.gtb-cramjam]                            3,345.9300 (13.64)        5,890.7979 (16.24)        3,538.4708 (10.19)       358.4893 (49.31)        3,434.2399 (9.91)        107.9319 (15.15)       21;28    282.6080 (0.10)        286           1
test_lz4[kppkn.gtb-python-lz4]                         3,323.8300 (13.55)        6,563.6940 (18.10)        3,649.2799 (10.51)       507.6818 (69.83)        3,453.0250 (9.96)        141.2800 (19.83)       26;48    274.0267 (0.10)        274           1
test_lz4[lcet10.txt-cramjam]                           7,444.1610 (30.34)       13,636.0540 (37.60)        8,044.9605 (23.17)     1,071.5044 (147.39)       7,670.1291 (22.13)       302.9506 (42.51)       15;22    124.3014 (0.04)        129           1
test_lz4[lcet10.txt-python-lz4]                        8,787.4030 (35.82)       15,124.7031 (41.70)        9,523.0310 (27.42)     1,039.6363 (143.00)       9,116.1070 (26.30)       404.7097 (56.79)       15;20    105.0086 (0.04)        113           1
test_lz4[paper-100k.pdf-cramjam]                       1,165.2170 (4.75)         3,391.3370 (9.35)         1,238.3142 (3.57)        187.5870 (25.80)        1,197.4330 (3.45)         45.5229 (6.39)        41;60    807.5495 (0.28)        806           1
test_lz4[paper-100k.pdf-python-lz4]                    1,786.4701 (7.28)         3,989.5640 (11.00)        1,930.3730 (5.56)        270.5717 (37.22)        1,852.4720 (5.34)         43.4450 (6.10)        38;75    518.0346 (0.18)        533           1
test_lz4[plrabn12.txt-cramjam]                        10,025.1100 (40.86)       15,823.3760 (43.63)       11,022.9072 (31.74)     1,388.2368 (190.95)      10,650.3405 (30.73)       757.5119 (106.30)        7;7     90.7202 (0.03)         62           1
test_lz4[plrabn12.txt-python-lz4]                     11,853.4301 (48.32)       17,182.6570 (47.37)       13,388.9093 (38.55)     1,159.1757 (159.45)      12,978.2920 (37.44)     1,572.6237 (220.69)       18;1     74.6887 (0.03)         77           1
test_lz4[urls.10K-cramjam]                             8,241.6250 (33.59)       15,744.7390 (43.41)        8,967.4984 (25.82)     1,247.4730 (171.59)       8,477.2279 (24.46)       717.1374 (100.64)        9;9    111.5138 (0.04)        104           1
test_lz4[urls.10K-python-lz4]                         10,320.9730 (42.07)       33,072.6539 (91.18)       14,466.2239 (41.66)     4,628.6794 (636.68)      12,947.5440 (37.35)     4,977.0970 (698.44)        4;4     69.1265 (0.02)         94           1
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### LZ4 _block_ format

`make bench-lz4-block`

```bash
---------------------------------------------------------------------------------------------------------------- benchmark: 28 tests -----------------------------------------------------------------------------------------------------------------
Name (time in us)                                                 Min                     Max                    Mean                StdDev                  Median                 IQR            Outliers          OPS            Rounds  Iterations
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_lz4_block[Mark.Twain-Tom.Sawyer.txt-cramjam]             40.9340 (1.37)         184.3430 (1.19)          42.9931 (1.35)         4.6601 (1.0)           41.5510 (1.37)       0.3530 (1.92)    1034;1427  23,259.5474 (0.74)       9422           1
test_lz4_block[Mark.Twain-Tom.Sawyer.txt-python-lz4]          38.4370 (1.28)         154.2830 (1.0)           40.6069 (1.28)         5.5279 (1.19)          38.9110 (1.28)       0.4095 (2.23)    1107;2431  24,626.3528 (0.78)      10544           1
test_lz4_block[alice29.txt-cramjam]                          472.0840 (15.77)        713.6530 (4.63)         494.8957 (15.56)       25.8485 (5.55)         493.4380 (16.27)     31.5625 (171.54)     190;53   2,020.6276 (0.06)       1839           1
test_lz4_block[alice29.txt-python-lz4]                       513.4790 (17.15)        857.2650 (5.56)         537.8642 (16.91)       27.2345 (5.84)         535.3320 (17.65)     34.5312 (187.68)     215;47   1,859.2053 (0.06)       1863           1
test_lz4_block[asyoulik.txt-cramjam]                         380.3780 (12.71)        635.7260 (4.12)         401.1980 (12.61)       28.4094 (6.10)         393.5930 (12.98)     23.0175 (125.10)    193;141   2,492.5350 (0.08)       2096           1
test_lz4_block[asyoulik.txt-python-lz4]                      416.9540 (13.93)        734.8450 (4.76)         437.0448 (13.74)       23.5893 (5.06)         437.7690 (14.44)     24.1480 (131.24)    251;108   2,288.0950 (0.07)       2290           1
test_lz4_block[fifty-four-mb-random-cramjam]              19,617.3530 (655.40)    20,615.6860 (133.62)    19,903.2510 (625.72)     196.4017 (42.15)     19,867.1210 (655.12)   183.0970 (995.12)       14;3      50.2430 (0.00)         50           1
test_lz4_block[fifty-four-mb-random-python-lz4]          115,706.2050 (>1000.0)  116,848.0440 (757.36)   116,398.5400 (>1000.0)    421.1026 (90.36)    116,555.1350 (>1000.0)  572.2738 (>1000.0)       3;0       8.5912 (0.00)          9           1
test_lz4_block[fifty-four-mb-repeating-cramjam]           16,384.0190 (547.37)    34,818.2670 (225.68)    17,412.4441 (547.42)   3,003.0026 (644.41)    16,854.9620 (555.79)   258.3560 (>1000.0)       1;4      57.4302 (0.00)         36           1
test_lz4_block[fifty-four-mb-repeating-python-lz4]        58,464.7160 (>1000.0)   61,384.1530 (397.87)    59,202.9512 (>1000.0)    776.3388 (166.59)    59,151.1860 (>1000.0)  644.4060 (>1000.0)       2;2      16.8910 (0.00)         17           1
test_lz4_block[fireworks.jpeg-cramjam]                        29.9320 (1.0)          161.8750 (1.05)          31.8084 (1.0)          5.0188 (1.08)          30.3260 (1.0)        0.1840 (1.0)     1946;3770  31,438.2794 (1.0)       18014           1
test_lz4_block[fireworks.jpeg-python-lz4]                     34.4750 (1.15)         169.8460 (1.10)          36.7064 (1.15)         5.2782 (1.13)          35.1650 (1.16)       0.8880 (4.83)    1940;2096  27,243.1909 (0.87)      17242           1
test_lz4_block[geo.protodata-cramjam]                        140.1360 (4.68)         291.7290 (1.89)         147.5534 (4.64)        12.0341 (2.58)         141.8530 (4.68)       9.3140 (50.62)     990;375   6,777.2052 (0.22)       5938           1
test_lz4_block[geo.protodata-python-lz4]                     144.4831 (4.83)         350.4770 (2.27)         152.2133 (4.79)        12.8940 (2.77)         146.0120 (4.81)      10.0460 (54.60)    1043;347   6,569.7283 (0.21)       5972           1
test_lz4_block[html-cramjam]                                 142.9390 (4.78)         277.8630 (1.80)         149.9061 (4.71)        11.7107 (2.51)         144.4170 (4.76)       9.1133 (49.53)     846;321   6,670.8442 (0.21)       5319           1
test_lz4_block[html-python-lz4]                              147.3250 (4.92)         279.1871 (1.81)         154.4610 (4.86)        11.5615 (2.48)         149.0470 (4.91)       9.1137 (49.53)     994;375   6,474.1270 (0.21)       6013           1
test_lz4_block[html_x_4-cramjam]                             600.2710 (20.05)        941.8120 (6.10)         628.8270 (19.77)       29.0365 (6.23)         623.0760 (20.55)     39.2540 (213.34)     186;28   1,590.2625 (0.05)       1491           1
test_lz4_block[html_x_4-python-lz4]                          625.6220 (20.90)        930.9070 (6.03)         655.2924 (20.60)       30.3427 (6.51)         648.3260 (21.38)     39.3870 (214.07)     196;40   1,526.0362 (0.05)       1448           1
test_lz4_block[kppkn.gtb-cramjam]                            446.5680 (14.92)        729.3510 (4.73)         466.9985 (14.68)       24.7114 (5.30)         467.1435 (15.40)     27.8070 (151.13)     217;64   2,141.3345 (0.07)       2110           1
test_lz4_block[kppkn.gtb-python-lz4]                         457.6420 (15.29)        673.1950 (4.36)         477.8109 (15.02)       23.3480 (5.01)         473.5590 (15.62)     26.9885 (146.68)     239;72   2,092.8781 (0.07)       2073           1
test_lz4_block[lcet10.txt-cramjam]                         1,300.3600 (43.44)      1,639.6500 (10.63)      1,358.7622 (42.72)       44.2987 (9.51)       1,350.1540 (44.52)     57.7395 (313.81)     228;11     735.9640 (0.02)        720           1
test_lz4_block[lcet10.txt-python-lz4]                      1,410.9100 (47.14)      1,763.4770 (11.43)      1,474.0569 (46.34)       48.0336 (10.31)      1,465.7810 (48.33)     60.8988 (330.98)     176;13     678.3999 (0.02)        621           1
test_lz4_block[paper-100k.pdf-cramjam]                        75.7430 (2.53)         213.3430 (1.38)          80.0105 (2.52)         8.9087 (1.91)          76.5580 (2.52)       2.2090 (12.01)    848;1656  12,498.3609 (0.40)       8862           1
test_lz4_block[paper-100k.pdf-python-lz4]                     82.7410 (2.76)         217.1710 (1.41)          87.4508 (2.75)         9.4264 (2.02)          83.8570 (2.77)       0.8590 (4.67)     905;2221  11,435.0004 (0.36)       9246           1
test_lz4_block[plrabn12.txt-cramjam]                       1,527.4300 (51.03)      1,917.3070 (12.43)      1,594.6245 (50.13)       55.9814 (12.01)      1,584.3080 (52.24)     64.3090 (349.52)     125;16     627.1069 (0.02)        511           1
test_lz4_block[plrabn12.txt-python-lz4]                    2,011.2020 (67.19)      2,540.4010 (16.47)      2,123.5693 (66.76)       93.4244 (20.05)      2,098.7020 (69.20)    111.1333 (604.00)     105;22     470.9053 (0.01)        459           1
test_lz4_block[urls.10K-cramjam]                           1,511.1370 (50.49)      2,008.8290 (13.02)      1,591.5999 (50.04)       65.2772 (14.01)      1,578.4930 (52.05)     61.4250 (333.84)     120;29     628.2986 (0.02)        511           1
test_lz4_block[urls.10K-python-lz4]                        2,001.9611 (66.88)      2,490.6120 (16.14)      2,103.2583 (66.12)       66.4167 (14.25)      2,090.2720 (68.93)     78.7600 (428.06)     111;12     475.4528 (0.02)        427           1
------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### Brotli

`make bench-brotli`

```bash
----------------------------------------------------------------------------------------------------- benchmark: 24 tests ------------------------------------------------------------------------------------------------------
Name (time in ms)                                         Min                   Max                  Mean             StdDev                Median                IQR            Outliers      OPS            Rounds  Iterations
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_brotli[Mark.Twain-Tom.Sawyer.txt-brotli]         17.8897 (1.0)         21.5322 (1.05)        18.4333 (1.0)       0.6208 (1.44)        18.2443 (1.0)       0.4319 (1.0)           5;5  54.2498 (1.0)          50           1
test_brotli[Mark.Twain-Tom.Sawyer.txt-cramjam]        18.1533 (1.01)        20.4369 (1.0)         18.7759 (1.02)      0.5898 (1.37)        18.6078 (1.02)      0.8347 (1.93)         10;1  53.2598 (0.98)         50           1
test_brotli[alice29.txt-brotli]                      220.3170 (12.32)      230.1574 (11.26)      224.9658 (12.20)     3.8282 (8.88)       223.9629 (12.28)     5.6323 (13.04)         2;0   4.4451 (0.08)          5           1
test_brotli[alice29.txt-cramjam]                     226.4241 (12.66)      228.6188 (11.19)      227.6423 (12.35)     0.9194 (2.13)       227.9714 (12.50)     1.5160 (3.51)          2;0   4.3929 (0.08)          5           1
test_brotli[asyoulik.txt-brotli]                     176.8532 (9.89)       178.5816 (8.74)       177.9033 (9.65)      0.6775 (1.57)       177.9574 (9.75)      0.9254 (2.14)          2;0   5.6210 (0.10)          6           1
test_brotli[asyoulik.txt-cramjam]                    181.3138 (10.14)      185.9610 (9.10)       183.0713 (9.93)      1.8366 (4.26)       182.5439 (10.01)     2.9700 (6.88)          1;0   5.4624 (0.10)          6           1
test_brotli[fireworks.jpeg-brotli]                    70.7069 (3.95)        75.6588 (3.70)        71.9962 (3.91)      1.1901 (2.76)        71.7863 (3.93)      0.6675 (1.55)          2;1  13.8896 (0.26)         14           1
test_brotli[fireworks.jpeg-cramjam]                   71.3536 (3.99)        73.7503 (3.61)        71.8630 (3.90)      0.6269 (1.45)        71.7001 (3.93)      0.5713 (1.32)          1;1  13.9154 (0.26)         14           1
test_brotli[geo.protodata-brotli]                    124.7704 (6.97)       126.1844 (6.17)       125.5772 (6.81)      0.4313 (1.0)        125.5776 (6.88)      0.4719 (1.09)          2;0   7.9632 (0.15)          8           1
test_brotli[geo.protodata-cramjam]                   128.6971 (7.19)       130.4920 (6.39)       129.6153 (7.03)      0.6275 (1.45)       129.5793 (7.10)      1.0149 (2.35)          4;0   7.7151 (0.14)          8           1
test_brotli[html-brotli]                             133.0865 (7.44)       137.1792 (6.71)       134.8985 (7.32)      1.1569 (2.68)       134.9101 (7.39)      0.7996 (1.85)          2;2   7.4130 (0.14)          8           1
test_brotli[html-cramjam]                            136.2971 (7.62)       141.6241 (6.93)       138.3340 (7.50)      2.2464 (5.21)       137.6318 (7.54)      3.8429 (8.90)          2;0   7.2289 (0.13)          8           1
test_brotli[html_x_4-brotli]                         162.2292 (9.07)       164.1336 (8.03)       163.3756 (8.86)      0.7556 (1.75)       163.5865 (8.97)      1.3218 (3.06)          2;0   6.1209 (0.11)          6           1
test_brotli[html_x_4-cramjam]                        166.7431 (9.32)       168.3913 (8.24)       167.3817 (9.08)      0.6918 (1.60)       167.2347 (9.17)      1.1951 (2.77)          1;0   5.9744 (0.11)          6           1
test_brotli[kppkn.gtb-brotli]                        416.0493 (23.26)      420.2574 (20.56)      417.5143 (22.65)     1.6582 (3.84)       417.0724 (22.86)     1.9902 (4.61)          1;0   2.3951 (0.04)          5           1
test_brotli[kppkn.gtb-cramjam]                       432.7377 (24.19)      438.8289 (21.47)      434.3502 (23.56)     2.5329 (5.87)       433.5837 (23.77)     2.0103 (4.65)          1;1   2.3023 (0.04)          5           1
test_brotli[lcet10.txt-brotli]                       689.3127 (38.53)      695.1439 (34.01)      691.5692 (37.52)     2.2512 (5.22)       691.0756 (37.88)     2.8880 (6.69)          2;0   1.4460 (0.03)          5           1
test_brotli[lcet10.txt-cramjam]                      706.3514 (39.48)      730.3781 (35.74)      713.2318 (38.69)     9.7223 (22.54)      709.3515 (38.88)     7.2208 (16.72)         1;1   1.4021 (0.03)          5           1
test_brotli[paper-100k.pdf-brotli]                   397.2979 (22.21)      399.6303 (19.55)      398.6496 (21.63)     0.9994 (2.32)       398.5795 (21.85)     1.6975 (3.93)          1;0   2.5085 (0.05)          5           1
test_brotli[paper-100k.pdf-cramjam]                  397.1881 (22.20)      487.4731 (23.85)      418.1530 (22.68)    38.8914 (90.17)      401.4564 (22.00)    27.9225 (64.65)         1;1   2.3915 (0.04)          5           1
test_brotli[plrabn12.txt-brotli]                     766.4343 (42.84)      965.1578 (47.23)      822.5129 (44.62)    81.3593 (188.62)     785.6867 (43.06)    70.6752 (163.63)        1;1   1.2158 (0.02)          5           1
test_brotli[plrabn12.txt-cramjam]                    776.7893 (43.42)      791.7569 (38.74)      780.8096 (42.36)     6.4344 (14.92)      777.1387 (42.60)     7.2310 (16.74)         1;0   1.2807 (0.02)          5           1
test_brotli[urls.10K-brotli]                       1,232.4364 (68.89)    1,281.6436 (62.71)    1,251.4702 (67.89)    21.9635 (50.92)    1,239.3781 (67.93)    36.2594 (83.95)         1;0   0.7991 (0.01)          5           1
test_brotli[urls.10K-cramjam]                      1,256.7332 (70.25)    1,348.1630 (65.97)    1,284.4820 (69.68)    36.9337 (85.63)    1,274.7599 (69.87)    37.5966 (87.05)         1;0   0.7785 (0.01)          5           1
--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```

#### ZSTD

`make bench-zstd`

```bash 
------------------------------------------------------------------------------------------------------------- benchmark: 28 tests -------------------------------------------------------------------------------------------------------------
Name (time in us)                                         Min                     Max                    Mean                StdDev                  Median                   IQR            Outliers         OPS            Rounds  Iterations
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
test_zstd[Mark.Twain-Tom.Sawyer.txt-cramjam]         162.9620 (1.09)         406.7180 (1.0)          176.6457 (1.0)         18.1423 (1.14)         169.1270 (1.0)         10.4961 (30.47)     201;206  5,661.0504 (1.0)        1524           1
test_zstd[Mark.Twain-Tom.Sawyer.txt-zstd]            150.1160 (1.0)          476.3250 (1.17)         254.5834 (1.44)        73.5374 (4.64)         286.3150 (1.69)       111.8249 (324.63)       15;1  3,927.9858 (0.69)         50           1
test_zstd[alice29.txt-cramjam]                     1,375.7630 (9.16)       1,755.1420 (4.32)       1,409.3451 (7.98)        44.2825 (2.79)       1,399.8351 (8.28)        39.0127 (113.25)      63;29    709.5494 (0.13)        675           1
test_zstd[alice29.txt-zstd]                        1,325.4730 (8.83)       2,010.5341 (4.94)       1,360.4852 (7.70)        62.9252 (3.97)       1,345.8120 (7.96)        39.4245 (114.45)      35;29    735.0319 (0.13)        691           1
test_zstd[asyoulik.txt-cramjam]                    1,215.3659 (8.10)       1,828.1790 (4.49)       1,250.6696 (7.08)        57.2588 (3.61)       1,238.3070 (7.32)        38.2224 (110.96)      53;47    799.5717 (0.14)        759           1
test_zstd[asyoulik.txt-zstd]                       1,159.6669 (7.73)       1,858.6870 (4.57)       1,189.6754 (6.73)        54.2058 (3.42)       1,175.6300 (6.95)        34.5710 (100.36)      54;39    840.5654 (0.15)        816           1
test_zstd[fifty-four-mb-random-cramjam]          126,119.5041 (840.15)   127,714.4219 (314.01)   126,973.2383 (718.80)     588.7249 (37.11)    127,016.0485 (751.01)     922.4786 (>1000.0)       4;0      7.8757 (0.00)          8           1
test_zstd[fifty-four-mb-random-zstd]              54,271.3830 (361.53)    67,254.4950 (165.36)    62,470.3614 (353.65)   3,707.2937 (233.68)    63,691.2219 (376.59)   4,924.4625 (>1000.0)       4;0     16.0076 (0.00)         16           1
test_zstd[fifty-four-mb-repeating-cramjam]        49,074.5769 (326.91)    50,540.2781 (124.26)    49,350.9156 (279.38)     374.5157 (23.61)     49,230.9110 (291.09)     269.6409 (782.76)        1;1     20.2630 (0.00)         14           1
test_zstd[fifty-four-mb-repeating-zstd]           27,456.1089 (182.90)    30,666.1710 (75.40)     28,946.7121 (163.87)     620.6539 (39.12)     29,004.8160 (171.50)     532.4374 (>1000.0)       8;4     34.5462 (0.01)         33           1
test_zstd[fireworks.jpeg-cramjam]                    287.1940 (1.91)         503.2960 (1.24)         296.2061 (1.68)        19.8144 (1.25)         288.1080 (1.70)         8.4531 (24.54)     228;298  3,376.0276 (0.60)       2354           1
test_zstd[fireworks.jpeg-zstd]                       216.6729 (1.44)         594.6030 (1.46)         221.7168 (1.26)        16.1432 (1.02)         217.1519 (1.28)         0.3445 (1.0)       280;881  4,510.2582 (0.80)       4208           1
test_zstd[geo.protodata-cramjam]                     327.6070 (2.18)         517.6039 (1.27)         338.9284 (1.92)        17.7418 (1.12)         332.8320 (1.97)         8.2538 (23.96)     259;399  2,950.4755 (0.52)       2635           1
test_zstd[geo.protodata-zstd]                        280.3099 (1.87)         669.7009 (1.65)         289.7946 (1.64)        15.8649 (1.0)          285.2400 (1.69)         2.6374 (7.66)      288;645  3,450.7199 (0.61)       3200           1
test_zstd[html-cramjam]                              356.0591 (2.37)         772.1799 (1.90)         366.8109 (2.08)        23.6813 (1.49)         359.2240 (2.12)         5.6227 (16.32)     175;413  2,726.2002 (0.48)       2443           1
test_zstd[html-zstd]                                 312.1750 (2.08)         533.3930 (1.31)         321.4469 (1.82)        19.8513 (1.25)         314.5090 (1.86)         3.7401 (10.86)     246;582  3,110.9334 (0.55)       2934           1
test_zstd[html_x_4-cramjam]                          476.6970 (3.18)         719.8390 (1.77)         490.4110 (2.78)        21.8491 (1.38)         481.2561 (2.85)        15.6217 (45.35)     192;152  2,039.1062 (0.36)       1797           1
test_zstd[html_x_4-zstd]                             364.6320 (2.43)         644.5120 (1.58)         374.6641 (2.12)        19.1959 (1.21)         367.3399 (2.17)         5.6954 (16.53)     231;447  2,669.0574 (0.47)       2477           1
test_zstd[kppkn.gtb-cramjam]                       1,093.0240 (7.28)       1,506.8919 (3.71)       1,131.5039 (6.41)        49.7900 (3.14)       1,117.8980 (6.61)        44.4521 (129.04)      92;52    883.7795 (0.16)        843           1
test_zstd[kppkn.gtb-zstd]                          1,044.6580 (6.96)       1,529.9510 (3.76)       1,069.5185 (6.05)        38.3409 (2.42)       1,053.7340 (6.23)        28.2210 (81.93)       77;49    935.0002 (0.17)        896           1
test_zstd[lcet10.txt-cramjam]                      3,546.4830 (23.62)      4,082.9000 (10.04)      3,626.4003 (20.53)       78.0370 (4.92)       3,605.1755 (21.32)       78.6935 (228.45)      31;11    275.7555 (0.05)        260           1
test_zstd[lcet10.txt-zstd]                         3,407.8470 (22.70)      4,383.9460 (10.78)      3,496.2894 (19.79)      105.5919 (6.66)       3,474.2125 (20.54)       70.1590 (203.67)      22;22    286.0175 (0.05)        286           1
test_zstd[paper-100k.pdf-cramjam]                    494.6779 (3.30)         958.8940 (2.36)         510.4041 (2.89)        32.5814 (2.05)         499.8320 (2.96)        13.1470 (38.17)      95;186  1,959.2320 (0.35)       1821           1
test_zstd[paper-100k.pdf-zstd]                       480.3210 (3.20)         801.3280 (1.97)         493.8572 (2.80)        26.0754 (1.64)         484.5071 (2.86)        14.3057 (41.53)     133;137  2,024.8767 (0.36)       1655           1
test_zstd[plrabn12.txt-cramjam]                    4,773.0900 (31.80)      5,415.0301 (13.31)      4,930.6590 (27.91)      134.2914 (8.46)       4,886.1710 (28.89)      140.8120 (408.78)      38;10    202.8126 (0.04)        165           1
test_zstd[plrabn12.txt-zstd]                       4,617.4621 (30.76)      5,391.8620 (13.26)      4,735.0956 (26.81)      101.1250 (6.37)       4,712.4890 (27.86)       77.8873 (226.11)      28;11    211.1890 (0.04)        203           1
test_zstd[urls.10K-cramjam]                        4,454.6180 (29.67)      6,157.7980 (15.14)      4,578.2559 (25.92)      153.7447 (9.69)       4,547.1541 (26.89)       93.7060 (272.03)      12;11    218.4238 (0.04)        194           1
test_zstd[urls.10K-zstd]                           4,201.4410 (27.99)      4,969.7730 (12.22)      4,298.6442 (24.33)      103.9367 (6.55)       4,271.2825 (25.25)       69.8585 (202.80)      18;13    232.6315 (0.04)        228           1
-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
```
