import { aws_ec2 as ec2, aws_ecs as ecs, aws_events as events, aws_kms as kms, aws_rds as rds, aws_stepfunctions as stepfunctions } from 'aws-cdk-lib';
import { Construct } from 'constructs';
export interface IRdsSanitizedSnapshotter {
    /**
     * Database cluster to snapshot and sanitize.
     *
     * Only one of `databaseCluster` and `databaseInstance` can be specified.
     */
    readonly databaseCluster?: rds.IDatabaseCluster;
    /**
     * Database instance to snapshot and sanitize.
     *
     * Only one of `databaseCluster` and `databaseInstance` can be specified.
     */
    readonly databaseInstance?: rds.IDatabaseInstance;
    /**
     * Name of database to connect to inside the RDS cluster or instance. This database will be used to execute the SQL script.
     *
     * @default 'postgres' for PostgreSQL and not set for MySQL
     */
    readonly databaseName?: string;
    /**
     * KMS key used to encrypt original database, if any.
     */
    readonly databaseKey?: kms.IKey;
    /**
     * VPC where temporary database and sanitizing task will be created.
     */
    readonly vpc: ec2.IVpc;
    /**
     * VPC subnets to use for temporary databases.
     *
     * @default ec2.SubnetType.PRIVATE_ISOLATED
     */
    readonly dbSubnets?: ec2.SubnetSelection;
    /**
     * VPC subnets to use for sanitization task.
     *
     * @default ec2.SubnetType.PRIVATE_WITH_NAT
     */
    readonly sanitizeSubnets?: ec2.SubnetSelection;
    /**
     * Cluster where sanitization task will be executed.
     *
     * @default a new cluster running on given VPC
     */
    readonly fargateCluster?: ecs.ICluster;
    /**
     * SQL script used to sanitize the database. It will be executed against the temporary database.
     *
     * You would usually want to start this with `USE mydatabase;`.
     */
    readonly script: string;
    /**
     * Optional KMS key to encrypt target snapshot.
     */
    readonly snapshotKey?: kms.IKey;
    /**
     * Prefix for sanitized snapshot name. The current date and time will be added to it.
     *
     * @default cluster identifier (which might be too long)
     */
    readonly snapshotPrefix?: string;
    /**
     * Prefix for all temporary snapshots and databases. The step function execution id will be added to it.
     *
     * @default 'sanitize'
     */
    readonly tempPrefix?: string;
    /**
     * The schedule or rate (frequency) that determines when the sanitized snapshot runs automatically.
     */
    readonly schedule?: events.Schedule;
    /**
     * Limit the number of snapshot history. Set this to delete old snapshots and only leave a certain number of snapshots.
     */
    readonly snapshotHistoryLimit?: number;
    /**
     * List of accounts the sanitized snapshot should be shared with.
     */
    readonly shareAccounts?: string[];
}
/**
 * A process to create sanitized snapshots of RDS instance or cluster, optionally on a schedule. The process is handled by a step function.
 *
 * 1. Snapshot the source database
 * 2. Optionally re-encrypt the snapshot with a different key in case you want to share it with an account that doesn't have access to the original key
 * 3. Create a temporary database
 * 4. Run a Fargate task to connect to the temporary database and execute an arbitrary SQL script to sanitize it
 * 5. Snapshot the sanitized database
 * 6. Clean-up temporary snapshots and databases
 */
export declare class RdsSanitizedSnapshotter extends Construct {
    readonly props: IRdsSanitizedSnapshotter;
    private waitFn;
    private readonly securityGroup;
    private readonly subnetGroup;
    private readonly subnets;
    private readonly fargateCluster;
    private readonly sqlScript;
    private readonly reencrypt;
    private readonly generalTags;
    private readonly finalSnapshotTags;
    private readonly databaseIdentifier;
    private readonly databaseName?;
    private readonly snapshotPrefix;
    private readonly tempPrefix;
    private readonly isCluster;
    private readonly dbClusterArn;
    private readonly dbInstanceArn;
    private readonly targetSnapshotArn;
    private readonly tempSnapshotArn;
    private readonly tempDbClusterArn;
    private readonly tempDbInstanceArn;
    /**
     * Step function in charge of the entire process including snapshotting, sanitizing, and cleanup. Trigger this step function to get a new snapshot.
     */
    snapshotter: stepfunctions.StateMachine;
    constructor(scope: Construct, id: string, props: IRdsSanitizedSnapshotter);
    private dbParametersTask;
    private createSnapshot;
    private waitForOperation;
    private reencryptSnapshot;
    private createTemporaryDatabase;
    private setPassword;
    private createTemporaryDatabaseInstance;
    private getTempDbClusterEndpoint;
    private getTempDbEndpoint;
    private sanitize;
    private finalSnapshot;
    private shareSnapshot;
    private deleteOldSnapshots;
    private cleanup;
}
