"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RdsSanitizedSnapshotter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const delete_old_function_1 = require("./delete-old-function");
const parameters_function_1 = require("./parameters-function");
const wait_function_1 = require("./wait-function");
/**
 * A process to create sanitized snapshots of RDS instance or cluster, optionally on a schedule. The process is handled by a step function.
 *
 * 1. Snapshot the source database
 * 2. Optionally re-encrypt the snapshot with a different key in case you want to share it with an account that doesn't have access to the original key
 * 3. Create a temporary database
 * 4. Run a Fargate task to connect to the temporary database and execute an arbitrary SQL script to sanitize it
 * 5. Snapshot the sanitized database
 * 6. Clean-up temporary snapshots and databases
 */
class RdsSanitizedSnapshotter extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.securityGroup = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'SG', {
            description: 'Group for communication between sanitizing job and database',
            vpc: props.vpc,
        });
        cdk.Tags.of(this.securityGroup).add('Name', 'RDS-sanitized-snapshots');
        this.subnetGroup = new aws_cdk_lib_1.aws_rds.SubnetGroup(this, 'Subnet group', {
            description: 'Temporary database used for RDS-sanitize-snapshots',
            vpc: props.vpc,
            vpcSubnets: props.vpc.selectSubnets(props.dbSubnets ?? { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED }),
        });
        this.subnets = props.sanitizeSubnets ?? { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT };
        this.fargateCluster = props.fargateCluster ?? new aws_cdk_lib_1.aws_ecs.Cluster(this, 'cluster', { vpc: props.vpc });
        this.sqlScript = props.script;
        if (this.subnets.subnetType === aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED) {
            cdk.Annotations.of(this).addWarning('Isolated subnets may not work for sanitization task as it requires access to public ECR');
        }
        if (props.databaseCluster) {
            this.isCluster = true;
            this.databaseIdentifier = props.databaseCluster.clusterIdentifier;
        }
        else if (props.databaseInstance) {
            this.isCluster = false;
            this.databaseIdentifier = props.databaseInstance.instanceIdentifier;
        }
        else {
            throw new Error('One of `databaseCluster` or `databaseInstance` must be specified');
        }
        this.databaseName = props.databaseName;
        this.tempPrefix = props.tempPrefix ?? 'sanitize';
        this.snapshotPrefix = props.snapshotPrefix ?? this.databaseIdentifier;
        this.reencrypt = props.snapshotKey !== undefined;
        this.dbClusterArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            resourceName: this.databaseIdentifier,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.dbInstanceArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            resourceName: this.databaseIdentifier,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.tempSnapshotArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: this.isCluster ? 'cluster-snapshot' : 'snapshot',
            resourceName: `${this.tempPrefix}-*`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.tempDbClusterArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'cluster',
            resourceName: `${this.tempPrefix}-*`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.tempDbInstanceArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            resourceName: `${this.tempPrefix}-*`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.targetSnapshotArn = cdk.Stack.of(this).formatArn({
            service: 'rds',
            resource: this.isCluster ? 'cluster-snapshot' : 'snapshot',
            resourceName: `${this.snapshotPrefix}-*`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
        this.generalTags = [
            {
                Key: 'RDS-sanitized-snapshots',
                Value: this.databaseIdentifier,
            },
        ];
        this.finalSnapshotTags = this.generalTags.concat([{
                Key: 'Final',
                Value: 'true',
            }]);
        const parametersState = this.dbParametersTask(props.databaseKey);
        const errorCatcher = new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Error Catcher', { resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD });
        let c;
        let s;
        s = c = this.createSnapshot('Create Temporary Snapshot', '$.databaseIdentifier', '$.tempSnapshotId', this.generalTags);
        s = s.next(this.waitForOperation('Wait for Snapshot', 'snapshot', '$.databaseIdentifier', '$.tempSnapshotId'));
        if (props.snapshotKey) {
            s = s.next(this.reencryptSnapshot(props.snapshotKey));
            s = s.next(this.waitForOperation('Wait for Re-encrypt', 'snapshot', '$.databaseIdentifier', '$.tempEncSnapshotId'));
            s = s.next(this.createTemporaryDatabase('$.tempEncSnapshotId'));
        }
        else {
            s = s.next(this.createTemporaryDatabase('$.tempSnapshotId'));
        }
        s = s.next(this.waitForOperation('Wait for Temporary Database', this.isCluster ? 'cluster' : 'instance', '$.tempDbId'));
        s = s.next(this.setPassword());
        s = s.next(this.waitForOperation('Wait for Temporary Password', this.isCluster ? 'cluster' : 'instance', '$.tempDbId'));
        if (this.isCluster) {
            s = s.next(this.createTemporaryDatabaseInstance());
            s = s.next(this.waitForOperation('Wait for Temporary Instance', 'instance', '$.tempDbInstanceId'));
            s = s.next(this.getTempDbClusterEndpoint());
        }
        else {
            s = s.next(this.getTempDbEndpoint());
        }
        s = s.next(this.sanitize());
        s = s.next(this.finalSnapshot());
        s = s.next(this.waitForOperation('Wait for Final Snapshot', 'snapshot', '$.tempDbId', '$.targetSnapshotId'));
        if (props.shareAccounts) {
            s = s.next(this.shareSnapshot(props.shareAccounts));
        }
        if (props.snapshotHistoryLimit) {
            s.next(this.deleteOldSnapshots(props.snapshotHistoryLimit));
        }
        errorCatcher.branch(c);
        const cleanupTasks = this.cleanup();
        this.snapshotter = new aws_cdk_lib_1.aws_stepfunctions.StateMachine(this, 'Director', {
            definition: parametersState.next(errorCatcher.addCatch(cleanupTasks, { resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD })).next(cleanupTasks),
        });
        // needed for creating a snapshot with tags
        this.snapshotter.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['rds:AddTagsToResource'],
            resources: [this.tempSnapshotArn, this.targetSnapshotArn, this.tempDbClusterArn],
        }));
        // key permissions
        if (props.snapshotKey) {
            props.snapshotKey.grant(this.snapshotter, 'kms:CreateGrant', 'kms:DescribeKey');
        }
        if (props.databaseKey) {
            props.databaseKey.grant(this.snapshotter, 'kms:CreateGrant', 'kms:DescribeKey');
        }
        // allow fargate to access RDS
        this.securityGroup.connections.allowFrom(this.securityGroup.connections, aws_cdk_lib_1.aws_ec2.Port.allTcp());
        // schedule
        if (props.schedule) {
            new aws_cdk_lib_1.aws_events.Rule(this, 'Schedule', {
                description: `Sanitized snapshot of RDS ${this.databaseIdentifier}`,
                schedule: props.schedule,
                targets: [
                    new aws_cdk_lib_1.aws_events_targets.SfnStateMachine(this.snapshotter),
                ],
            });
        }
    }
    dbParametersTask(databaseKey) {
        const parametersFn = new parameters_function_1.ParametersFunction(this, 'parameters', { logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH });
        const parametersState = new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Get Parameters', {
            lambdaFunction: parametersFn,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                executionId: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$$.Execution.Id'),
                isCluster: this.isCluster,
                databaseIdentifier: this.databaseIdentifier,
                databaseKey: databaseKey?.keyArn || '',
                snapshotPrefix: this.snapshotPrefix,
                tempPrefix: this.tempPrefix,
            }),
            payloadResponseOnly: true,
        });
        if (this.isCluster) {
            parametersFn.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: ['rds:DescribeDBClusters'],
                resources: [this.dbClusterArn],
            }));
            parametersFn.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: ['rds:DescribeDBInstances'],
                resources: ['*'],
            }));
        }
        else {
            parametersFn.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
                actions: ['rds:DescribeDBInstances'],
                resources: [this.dbInstanceArn],
            }));
        }
        return parametersState;
    }
    createSnapshot(id, databaseId, snapshotId, tags) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, id, {
            service: 'rds',
            action: this.isCluster ? 'createDBClusterSnapshot' : 'createDBSnapshot',
            parameters: {
                DbClusterIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(databaseId) : undefined,
                DbClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(snapshotId) : undefined,
                DbInstanceIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(databaseId),
                DbSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(snapshotId),
                Tags: tags,
            },
            iamResources: ['*'],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    waitForOperation(id, resourceType, databaseIdentifier, snapshotId) {
        this.waitFn = this.waitFn ?? new wait_function_1.WaitFunction(this, 'wait', {
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: ['rds:DescribeDBClusters', 'rds:DescribeDBClusterSnapshots', 'rds:DescribeDBSnapshots', 'rds:DescribeDBInstances'],
                    resources: [this.dbClusterArn, this.dbInstanceArn, this.tempDbClusterArn,
                        this.tempSnapshotArn, this.targetSnapshotArn, this.tempDbInstanceArn],
                }),
            ],
        });
        let payload = {
            resourceType,
            databaseIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(databaseIdentifier),
            snapshotIdentifier: undefined,
            isCluster: this.isCluster,
        };
        if (snapshotId) {
            payload.snapshotIdentifier = aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(snapshotId);
        }
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, id, {
            lambdaFunction: this.waitFn,
            payloadResponseOnly: true,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject(payload),
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        }).addRetry({
            errors: ['NotReady'],
            interval: cdk.Duration.minutes(1),
            maxAttempts: 300,
            backoffRate: 1,
        });
    }
    reencryptSnapshot(key) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Re-encrypt Snapshot', {
            service: 'rds',
            action: this.isCluster ? 'copyDBClusterSnapshot' : 'copyDBSnapshot',
            parameters: {
                SourceDBClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempSnapshotId') : undefined,
                TargetDBClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempEncSnapshotId') : undefined,
                SourceDBSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempSnapshotId'),
                TargetDBSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempEncSnapshotId'),
                KmsKeyId: key.keyId,
                CopyTags: false,
                Tags: this.generalTags,
            },
            iamResources: [this.tempSnapshotArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    createTemporaryDatabase(snapshotIdentifier) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Create Temporary Database', {
            service: 'rds',
            action: this.isCluster ? 'restoreDBClusterFromSnapshot' : 'restoreDBInstanceFromDBSnapshot',
            parameters: {
                DbClusterIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId') : undefined,
                DbInstanceIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
                Engine: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.engine'),
                SnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(snapshotIdentifier) : undefined,
                DbSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(snapshotIdentifier),
                PubliclyAccessible: false,
                VpcSecurityGroupIds: [this.securityGroup.securityGroupId],
                DbSubnetGroupName: this.subnetGroup.subnetGroupName,
                Tags: this.generalTags,
            },
            iamResources: [
                this.tempDbClusterArn,
                this.tempDbInstanceArn,
                this.tempSnapshotArn,
                cdk.Stack.of(this).formatArn({
                    service: 'rds',
                    resource: 'subgrp',
                    resourceName: this.subnetGroup.subnetGroupName,
                    arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    setPassword() {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Set Temporary Password', {
            service: 'rds',
            action: this.isCluster ? 'modifyDBCluster' : 'modifyDBInstance',
            parameters: {
                DbClusterIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId') : undefined,
                DbInstanceIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
                MasterUserPassword: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.password'),
                ApplyImmediately: true,
                BackupRetentionPeriod: this.isCluster ? undefined : 0,
            },
            iamResources: [this.isCluster ? this.tempDbClusterArn : this.tempDbInstanceArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    createTemporaryDatabaseInstance() {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Create Temporary Instance', {
            service: 'rds',
            action: 'createDBInstance',
            parameters: {
                DbClusterIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
                DbInstanceIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbInstanceId'),
                DbInstanceClass: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbInstanceClass'),
                Engine: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.engine'),
            },
            iamResources: [this.tempDbClusterArn, this.tempDbInstanceArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    getTempDbClusterEndpoint() {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Get Temporary Cluster Endpoint', {
            service: 'rds',
            action: 'describeDBClusters',
            parameters: {
                DbClusterIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
            },
            iamResources: [this.tempDbClusterArn],
            resultSelector: {
                endpoint: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.DbClusters[0].Endpoint'),
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.host'),
        });
    }
    getTempDbEndpoint() {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Get Temporary Endpoint', {
            service: 'rds',
            action: 'describeDBInstances',
            parameters: {
                DbInstanceIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
            },
            iamResources: [this.tempDbInstanceArn],
            resultSelector: {
                endpoint: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.DbInstances[0].Endpoint.Address'),
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.host'),
        });
    }
    sanitize() {
        const logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            retention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
        });
        const mysqlTask = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'MySQL Task', {
            volumes: [{ name: 'config', host: {} }],
        });
        let mycnf = '[client]\nuser=$MYSQL_USER\nhost=$MYSQL_HOST\nport=$MYSQL_PORT\npassword=$MYSQL_PASSWORD';
        if (this.databaseName) {
            mycnf += '\ndatabase=$MYSQL_DATABASE';
        }
        const mysqlConfigContainer = mysqlTask.addContainer('config', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromRegistry('public.ecr.aws/docker/library/bash:4-alpine3.15'),
            command: ['bash', '-c', `echo "${mycnf}" > ~/.my.cnf && chmod 700 ~/.my.cnf`],
            logging: aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'mysql-config',
            }),
            essential: false,
        });
        mysqlConfigContainer.addMountPoints({ sourceVolume: 'config', containerPath: '/root', readOnly: false });
        const mysqlContainer = mysqlTask.addContainer('mysql', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromRegistry('public.ecr.aws/lts/mysql:latest'),
            command: ['mysql', '-e', this.sqlScript],
            logging: aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'mysql-sanitize',
            }),
        });
        mysqlContainer.addMountPoints({ sourceVolume: 'config', containerPath: '/root', readOnly: true });
        mysqlContainer.addContainerDependencies({ container: mysqlConfigContainer, condition: aws_cdk_lib_1.aws_ecs.ContainerDependencyCondition.SUCCESS });
        const postgresTask = new aws_cdk_lib_1.aws_ecs.FargateTaskDefinition(this, 'PostreSQL Task');
        const postgresContainer = postgresTask.addContainer('postgres', {
            image: aws_cdk_lib_1.aws_ecs.AssetImage.fromRegistry('public.ecr.aws/lts/postgres:latest'),
            command: ['psql', '-c', this.sqlScript],
            logging: aws_cdk_lib_1.aws_ecs.LogDriver.awsLogs({
                logGroup,
                streamPrefix: 'psql-sanitize',
            }),
        });
        const choice = new aws_cdk_lib_1.aws_stepfunctions.Choice(this, 'Sanitize')
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.stringEquals('$.dockerImage', 'mysql'), new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, 'Sanitize MySQL', {
            integrationPattern: aws_cdk_lib_1.aws_stepfunctions.IntegrationPattern.RUN_JOB,
            launchTarget: new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsFargateLaunchTarget(),
            cluster: this.fargateCluster,
            subnets: this.subnets,
            securityGroups: [this.securityGroup],
            taskDefinition: mysqlTask,
            containerOverrides: [
                {
                    containerDefinition: mysqlConfigContainer,
                    environment: [
                        {
                            name: 'MYSQL_HOST',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.host.endpoint'),
                        },
                        {
                            name: 'MYSQL_PORT',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.port'),
                        },
                        {
                            name: 'MYSQL_USER',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.user'),
                        },
                        {
                            name: 'MYSQL_PASSWORD',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.password'),
                        },
                        {
                            name: 'MYSQL_DATABASE',
                            value: this.databaseName ?? '',
                        },
                    ],
                },
            ],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        }))
            .when(aws_cdk_lib_1.aws_stepfunctions.Condition.stringEquals('$.dockerImage', 'postgres'), new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsRunTask(this, 'Sanitize Postgres', {
            integrationPattern: aws_cdk_lib_1.aws_stepfunctions.IntegrationPattern.RUN_JOB,
            launchTarget: new aws_cdk_lib_1.aws_stepfunctions_tasks.EcsFargateLaunchTarget(),
            cluster: this.fargateCluster,
            subnets: this.subnets,
            securityGroups: [this.securityGroup],
            taskDefinition: postgresTask,
            containerOverrides: [
                {
                    containerDefinition: postgresContainer,
                    environment: [
                        {
                            name: 'PGHOST',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.host.endpoint'),
                        },
                        {
                            name: 'PGPORT',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.port'),
                        },
                        {
                            name: 'PGUSER',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.user'),
                        },
                        {
                            name: 'PGPASSWORD',
                            value: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDb.password'),
                        },
                        {
                            name: 'PGDATABASE',
                            value: this.databaseName ?? 'postgres',
                        },
                        {
                            name: 'PGCONNECT_TIMEOUT',
                            value: '30',
                        },
                    ],
                },
            ],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        }));
        return choice.afterwards();
    }
    finalSnapshot() {
        return this.createSnapshot('Create Final Snapshot', '$.tempDbId', '$.targetSnapshotId', this.finalSnapshotTags);
    }
    // private waitForSnapshotSfn(id: string, snapshotId: string) {
    //   const describeSnapshot = new stepfunctions_tasks.CallAwsService(this, id, {
    //     service: 'rds',
    //     action: 'describeDBClusterSnapshots',
    //     parameters: {
    //       DbClusterIdentifier: stepfunctions.JsonPath.stringAt('$.databaseIdentifier'),
    //       DbClusterSnapshotIdentifier: stepfunctions.JsonPath.stringAt(snapshotId),
    //     },
    //     iamResources: [this.dbArn, this.tempSnapshotArn, this.targetSnapshotArn],
    //     resultPath: '$.results.waitSnapshot',
    //   });
    //
    //   return describeSnapshot.next(
    //     new stepfunctions.Choice(this, `${id} (check)`)
    //       .when(stepfunctions.Condition.or(
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*stop*'),
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*delet*'), // both delete and deleting
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*fail*'),
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*incompatible*'),
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*inaccessible*'),
    //         stepfunctions.Condition.stringMatches('$.results.waitSnapshot', '*error*'),
    //       ), new stepfunctions.Fail(this, `${id} (bad status)`))
    //       .when(
    //         stepfunctions.Condition.stringEquals('$.results.waitSnapshot', 'available'),
    //         new stepfunctions.Succeed(this, `${id} (done)`),
    //       )
    //       .otherwise(describeSnapshot),
    //   );
    // }
    shareSnapshot(accounts) {
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Share Snapshot', {
            service: 'rds',
            action: this.isCluster ? 'modifyDBClusterSnapshotAttribute' : 'modifyDBSnapshotAttribute',
            parameters: {
                DbClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.targetSnapshotId') : undefined,
                DbSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.targetSnapshotId'),
                AttributeName: 'restore',
                ValuesToAdd: accounts,
            },
            iamResources: [this.targetSnapshotArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    deleteOldSnapshots(historyLimit) {
        const deleteOldFn = new delete_old_function_1.DeleteOldFunction(this, 'delete-old', {
            logRetention: aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH,
            timeout: cdk.Duration.minutes(5),
        });
        deleteOldFn.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['tag:GetResources'],
            resources: ['*'],
        }));
        deleteOldFn.addToRolePolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['rds:DeleteDBClusterSnapshot', 'rds:DeleteDBSnapshot'],
            resources: [this.targetSnapshotArn],
        }));
        return new aws_cdk_lib_1.aws_stepfunctions_tasks.LambdaInvoke(this, 'Delete Old Snapshots', {
            lambdaFunction: deleteOldFn,
            payload: aws_cdk_lib_1.aws_stepfunctions.TaskInput.fromObject({
                tags: this.finalSnapshotTags,
                historyLimit: historyLimit,
                resourceType: 'rds:cluster-snapshot',
            }),
            payloadResponseOnly: true,
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        });
    }
    cleanup() {
        // We retry everything because when any branch fails, all other branches are cancelled.
        // Retrying gives other branches an opportunity to start and hopefully at least run.
        const p = new aws_cdk_lib_1.aws_stepfunctions.Parallel(this, 'Cleanup', { resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD });
        p.branch(new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Temporary Snapshot', {
            service: 'rds',
            action: this.isCluster ? 'deleteDBClusterSnapshot' : 'deleteDBSnapshot',
            parameters: {
                DbClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempSnapshotId') : undefined,
                DbSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempSnapshotId'),
            },
            iamResources: [this.tempSnapshotArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        }).addRetry({ maxAttempts: 5, interval: cdk.Duration.seconds(10) }));
        if (this.reencrypt) {
            p.branch(new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Re-encrypted Snapshot', {
                service: 'rds',
                action: this.isCluster ? 'deleteDBClusterSnapshot' : 'deleteDBSnapshot',
                parameters: {
                    DbClusterSnapshotIdentifier: this.isCluster ? aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempEncSnapshotId') : undefined,
                    DbSnapshotIdentifier: this.isCluster ? undefined : aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempEncSnapshotId'),
                },
                iamResources: [this.tempSnapshotArn],
                resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
            }).addRetry({ maxAttempts: 5, interval: cdk.Duration.seconds(10) }));
        }
        p.branch(new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Temporary Database Instance', {
            service: 'rds',
            action: 'deleteDBInstance',
            parameters: {
                DbInstanceIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt(this.isCluster ? '$.tempDbInstanceId' : '$.tempDbId'),
                SkipFinalSnapshot: true,
            },
            iamResources: [this.tempDbInstanceArn],
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
        }).addRetry({
            maxAttempts: 5,
            interval: cdk.Duration.seconds(10),
        }));
        if (this.isCluster) {
            p.branch(new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, 'Temporary Database', {
                service: 'rds',
                action: 'deleteDBCluster',
                parameters: {
                    DbClusterIdentifier: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.tempDbId'),
                    SkipFinalSnapshot: true,
                },
                iamResources: [this.tempDbClusterArn],
                resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.DISCARD,
            }).addRetry({
                maxAttempts: 5,
                interval: cdk.Duration.seconds(10),
            }));
        }
        return p;
    }
}
exports.RdsSanitizedSnapshotter = RdsSanitizedSnapshotter;
_a = JSII_RTTI_SYMBOL_1;
RdsSanitizedSnapshotter[_a] = { fqn: "@cloudsnorkel/cdk-rds-sanitized-snapshots.RdsSanitizedSnapshotter", version: "0.0.2" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxtQ0FBbUM7QUFDbkMsNkNBV3FCO0FBQ3JCLDJDQUF1QztBQUN2QywrREFBMEQ7QUFDMUQsK0RBQTJEO0FBQzNELG1EQUErQztBQWlHL0M7Ozs7Ozs7OztHQVNHO0FBQ0gsTUFBYSx1QkFBd0IsU0FBUSxzQkFBUztJQTZCcEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBVyxLQUErQjtRQUNoRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRGdDLFVBQUssR0FBTCxLQUFLLENBQTBCO1FBR2hGLElBQUksQ0FBQyxhQUFhLEdBQUcsSUFBSSxxQkFBRyxDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFO1lBQ3JELFdBQVcsRUFBRSw2REFBNkQ7WUFDMUUsR0FBRyxFQUFFLEtBQUssQ0FBQyxHQUFHO1NBQ2YsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxNQUFNLEVBQUUseUJBQXlCLENBQUMsQ0FBQztRQUV2RSxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUkscUJBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMzRCxXQUFXLEVBQUUsb0RBQW9EO1lBQ2pFLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztZQUNkLFVBQVUsRUFBRSxLQUFLLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsU0FBUyxJQUFJLEVBQUUsVUFBVSxFQUFFLHFCQUFHLENBQUMsVUFBVSxDQUFDLGdCQUFnQixFQUFFLENBQUM7U0FDeEcsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsZUFBZSxJQUFJLEVBQUUsVUFBVSxFQUFFLHFCQUFHLENBQUMsVUFBVSxDQUFDLGdCQUFnQixFQUFFLENBQUM7UUFDeEYsSUFBSSxDQUFDLGNBQWMsR0FBRyxLQUFLLENBQUMsY0FBYyxJQUFJLElBQUkscUJBQUcsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxFQUFFLEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQztRQUNuRyxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxNQUFNLENBQUM7UUFFOUIsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLFVBQVUsS0FBSyxxQkFBRyxDQUFDLFVBQVUsQ0FBQyxnQkFBZ0IsRUFBRTtZQUMvRCxHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMseUZBQXlGLENBQUMsQ0FBQztTQUNoSTtRQUVELElBQUksS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUN6QixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQztZQUN0QixJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGVBQWUsQ0FBQyxpQkFBaUIsQ0FBQztTQUNuRTthQUFNLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQ2pDLElBQUksQ0FBQyxTQUFTLEdBQUcsS0FBSyxDQUFDO1lBQ3ZCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUM7U0FDckU7YUFBTTtZQUNMLE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQztTQUNyRjtRQUNELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUV2QyxJQUFJLENBQUMsVUFBVSxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksVUFBVSxDQUFDO1FBQ2pELElBQUksQ0FBQyxjQUFjLEdBQUcsS0FBSyxDQUFDLGNBQWMsSUFBSSxJQUFJLENBQUMsa0JBQWtCLENBQUM7UUFFdEUsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsV0FBVyxLQUFLLFNBQVMsQ0FBQztRQUVqRCxJQUFJLENBQUMsWUFBWSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUMvQyxPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxTQUFTO1lBQ25CLFlBQVksRUFBRSxJQUFJLENBQUMsa0JBQWtCO1lBQ3JDLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsYUFBYSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztZQUNoRCxPQUFPLEVBQUUsS0FBSztZQUNkLFFBQVEsRUFBRSxJQUFJO1lBQ2QsWUFBWSxFQUFFLElBQUksQ0FBQyxrQkFBa0I7WUFDckMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO1NBQzdDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxlQUFlLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxVQUFVO1lBQzFELFlBQVksRUFBRSxHQUFHLElBQUksQ0FBQyxVQUFVLElBQUk7WUFDcEMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO1NBQzdDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDbkQsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsU0FBUztZQUNuQixZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUMsVUFBVSxJQUFJO1lBQ3BDLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3BELE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLElBQUk7WUFDZCxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUMsVUFBVSxJQUFJO1lBQ3BDLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO1lBQ3BELE9BQU8sRUFBRSxLQUFLO1lBQ2QsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxVQUFVO1lBQzFELFlBQVksRUFBRSxHQUFHLElBQUksQ0FBQyxjQUFjLElBQUk7WUFDeEMsU0FBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CO1NBQzdDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxXQUFXLEdBQUc7WUFDakI7Z0JBQ0UsR0FBRyxFQUFFLHlCQUF5QjtnQkFDOUIsS0FBSyxFQUFFLElBQUksQ0FBQyxrQkFBa0I7YUFDL0I7U0FDRixDQUFDO1FBQ0YsSUFBSSxDQUFDLGlCQUFpQixHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLENBQUM7Z0JBQ2hELEdBQUcsRUFBRSxPQUFPO2dCQUNaLEtBQUssRUFBRSxNQUFNO2FBQ2QsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLGVBQWUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBQ2pFLE1BQU0sWUFBWSxHQUFHLElBQUksK0JBQWEsQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLGVBQWUsRUFBRSxFQUFFLFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBRXZILElBQUksQ0FBMkIsQ0FBQztRQUNoQyxJQUFJLENBQTBCLENBQUM7UUFDL0IsQ0FBQyxHQUFHLENBQUMsR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLDJCQUEyQixFQUFFLHNCQUFzQixFQUFFLGtCQUFrQixFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUN2SCxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsbUJBQW1CLEVBQUUsVUFBVSxFQUFFLHNCQUFzQixFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQztRQUMvRyxJQUFJLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDckIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1lBQ3RELENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxxQkFBcUIsRUFBRSxVQUFVLEVBQUUsc0JBQXNCLEVBQUUscUJBQXFCLENBQUMsQ0FBQyxDQUFDO1lBQ3BILENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUM7U0FDakU7YUFBTTtZQUNMLENBQUMsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7U0FDOUQ7UUFDRCxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsNkJBQTZCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUN4SCxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztRQUMvQixDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsNkJBQTZCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztRQUN4SCxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLCtCQUErQixFQUFFLENBQUMsQ0FBQztZQUNuRCxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsNkJBQTZCLEVBQUUsVUFBVSxFQUFFLG9CQUFvQixDQUFDLENBQUMsQ0FBQztZQUNuRyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsd0JBQXdCLEVBQUUsQ0FBQyxDQUFDO1NBQzdDO2FBQU07WUFDTCxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO1NBQ3RDO1FBQ0QsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7UUFDNUIsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUM7UUFDakMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLHlCQUF5QixFQUFFLFVBQVUsRUFBRSxZQUFZLEVBQUUsb0JBQW9CLENBQUMsQ0FBQyxDQUFDO1FBQzdHLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUN2QixDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDO1NBQ3JEO1FBRUQsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLG9CQUFvQixDQUFDLENBQUMsQ0FBQztTQUM3RDtRQUVELFlBQVksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFdkIsTUFBTSxZQUFZLEdBQUcsSUFBSSxDQUFDLE9BQU8sRUFBRSxDQUFDO1FBQ3BDLElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSwrQkFBYSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2xFLFVBQVUsRUFBRSxlQUFlLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxRQUFRLENBQUMsWUFBWSxFQUFFLEVBQUUsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO1NBQ3pJLENBQUMsQ0FBQztRQUVILDJDQUEyQztRQUMzQyxJQUFJLENBQUMsV0FBVyxDQUFDLGVBQWUsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO1lBQ3ZELE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQztTQUNqRixDQUFDLENBQUMsQ0FBQztRQUVKLGtCQUFrQjtRQUNsQixJQUFJLEtBQUssQ0FBQyxXQUFXLEVBQUU7WUFDckIsS0FBSyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxpQkFBaUIsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1NBQ2pGO1FBQ0QsSUFBSSxLQUFLLENBQUMsV0FBVyxFQUFFO1lBQ3JCLEtBQUssQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztTQUNqRjtRQUVELDhCQUE4QjtRQUM5QixJQUFJLENBQUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxXQUFXLEVBQUUscUJBQUcsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLENBQUMsQ0FBQztRQUU1RixXQUFXO1FBQ1gsSUFBSSxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ2xCLElBQUksd0JBQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtnQkFDaEMsV0FBVyxFQUFFLDZCQUE2QixJQUFJLENBQUMsa0JBQWtCLEVBQUU7Z0JBQ25FLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtnQkFDeEIsT0FBTyxFQUFFO29CQUNQLElBQUksZ0NBQWMsQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQztpQkFDckQ7YUFDRixDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7SUFFTyxnQkFBZ0IsQ0FBQyxXQUFzQjtRQUM3QyxNQUFNLFlBQVksR0FBRyxJQUFJLHdDQUFrQixDQUFDLElBQUksRUFBRSxZQUFZLEVBQUUsRUFBRSxZQUFZLEVBQUUsc0JBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxFQUFFLENBQUMsQ0FBQztRQUNoSCxNQUFNLGVBQWUsR0FBRyxJQUFJLHFDQUFtQixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDbkYsY0FBYyxFQUFFLFlBQVk7WUFDNUIsT0FBTyxFQUFFLCtCQUFhLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQztnQkFDMUMsV0FBVyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQztnQkFDL0QsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO2dCQUN6QixrQkFBa0IsRUFBRSxJQUFJLENBQUMsa0JBQWtCO2dCQUMzQyxXQUFXLEVBQUUsV0FBVyxFQUFFLE1BQU0sSUFBSSxFQUFFO2dCQUN0QyxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7Z0JBQ25DLFVBQVUsRUFBRSxJQUFJLENBQUMsVUFBVTthQUM1QixDQUFDO1lBQ0YsbUJBQW1CLEVBQUUsSUFBSTtTQUMxQixDQUFDLENBQUM7UUFDSCxJQUFJLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDbEIsWUFBWSxDQUFDLGVBQWUsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO2dCQUNuRCxPQUFPLEVBQUUsQ0FBQyx3QkFBd0IsQ0FBQztnQkFDbkMsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQzthQUMvQixDQUFDLENBQUMsQ0FBQztZQUNKLFlBQVksQ0FBQyxlQUFlLENBQUMsSUFBSSxxQkFBRyxDQUFDLGVBQWUsQ0FBQztnQkFDbkQsT0FBTyxFQUFFLENBQUMseUJBQXlCLENBQUM7Z0JBQ3BDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQzthQUNqQixDQUFDLENBQUMsQ0FBQztTQUNMO2FBQU07WUFDTCxZQUFZLENBQUMsZUFBZSxDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ25ELE9BQU8sRUFBRSxDQUFDLHlCQUF5QixDQUFDO2dCQUNwQyxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDO2FBQ2hDLENBQUMsQ0FBQyxDQUFDO1NBQ0w7UUFDRCxPQUFPLGVBQWUsQ0FBQztJQUN6QixDQUFDO0lBRU8sY0FBYyxDQUFDLEVBQVUsRUFBRSxVQUFrQixFQUFFLFVBQWtCLEVBQUUsSUFBc0M7UUFDL0csT0FBTyxJQUFJLHFDQUFtQixDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3RELE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLHlCQUF5QixDQUFDLENBQUMsQ0FBQyxrQkFBa0I7WUFDdkUsVUFBVSxFQUFFO2dCQUNWLG1CQUFtQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDN0YsMkJBQTJCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNyRyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7Z0JBQzlGLG9CQUFvQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQztnQkFDOUYsSUFBSSxFQUFFLElBQUk7YUFDWDtZQUNELFlBQVksRUFBRSxDQUFDLEdBQUcsQ0FBQztZQUNuQixVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sZ0JBQWdCLENBQUMsRUFBVSxFQUFFLFlBQWlELEVBQUUsa0JBQTBCLEVBQUUsVUFBbUI7UUFDckksSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUMsTUFBTSxJQUFJLElBQUksNEJBQVksQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQzFELFlBQVksRUFBRSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1lBQzFDLGFBQWEsRUFBRTtnQkFDYixJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO29CQUN0QixPQUFPLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSxnQ0FBZ0MsRUFBRSx5QkFBeUIsRUFBRSx5QkFBeUIsQ0FBQztvQkFDM0gsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksRUFBRSxJQUFJLENBQUMsYUFBYSxFQUFFLElBQUksQ0FBQyxnQkFBZ0I7d0JBQ3RFLElBQUksQ0FBQyxlQUFlLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixFQUFFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztpQkFDeEUsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsSUFBSSxPQUFPLEdBQUc7WUFDWixZQUFZO1lBQ1osa0JBQWtCLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO1lBQ3ZFLGtCQUFrQixFQUFFLFNBQStCO1lBQ25ELFNBQVMsRUFBRSxJQUFJLENBQUMsU0FBUztTQUMxQixDQUFDO1FBQ0YsSUFBSSxVQUFVLEVBQUU7WUFDZCxPQUFPLENBQUMsa0JBQWtCLEdBQUcsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsT0FBTyxJQUFJLHFDQUFtQixDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3BELGNBQWMsRUFBRSxJQUFJLENBQUMsTUFBTTtZQUMzQixtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLE9BQU8sRUFBRSwrQkFBYSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDO1lBQ3BELFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPO1NBQzNDLENBQUMsQ0FBQyxRQUFRLENBQUM7WUFDVixNQUFNLEVBQUUsQ0FBQyxVQUFVLENBQUM7WUFDcEIsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNqQyxXQUFXLEVBQUUsR0FBRztZQUNoQixXQUFXLEVBQUUsQ0FBQztTQUNmLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxpQkFBaUIsQ0FBQyxHQUFhO1FBQ3JDLE9BQU8sSUFBSSxxQ0FBbUIsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQ3pFLE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQyxnQkFBZ0I7WUFDbkUsVUFBVSxFQUFFO2dCQUNWLGlDQUFpQyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNuSCxpQ0FBaUMsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQywrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDdEgsMEJBQTBCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQywrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUM7Z0JBQzVHLDBCQUEwQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHFCQUFxQixDQUFDO2dCQUMvRyxRQUFRLEVBQUUsR0FBRyxDQUFDLEtBQUs7Z0JBQ25CLFFBQVEsRUFBRSxLQUFLO2dCQUNmLElBQUksRUFBRSxJQUFJLENBQUMsV0FBVzthQUN2QjtZQUNELFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUM7WUFDcEMsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU87U0FDM0MsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLHVCQUF1QixDQUFDLGtCQUEwQjtRQUN4RCxPQUFPLElBQUkscUNBQW1CLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSwyQkFBMkIsRUFBRTtZQUMvRSxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDLENBQUMsaUNBQWlDO1lBQzNGLFVBQVUsRUFBRTtnQkFDVixtQkFBbUIsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQywrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQy9GLG9CQUFvQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQztnQkFDaEcsTUFBTSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7Z0JBQ25ELGtCQUFrQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNwRyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQztnQkFDdEcsa0JBQWtCLEVBQUUsS0FBSztnQkFDekIsbUJBQW1CLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQztnQkFDekQsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxlQUFlO2dCQUNuRCxJQUFJLEVBQUUsSUFBSSxDQUFDLFdBQVc7YUFDdkI7WUFDRCxZQUFZLEVBQUU7Z0JBQ1osSUFBSSxDQUFDLGdCQUFnQjtnQkFDckIsSUFBSSxDQUFDLGlCQUFpQjtnQkFDdEIsSUFBSSxDQUFDLGVBQWU7Z0JBQ3BCLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLFNBQVMsQ0FBQztvQkFDM0IsT0FBTyxFQUFFLEtBQUs7b0JBQ2QsUUFBUSxFQUFFLFFBQVE7b0JBQ2xCLFlBQVksRUFBRSxJQUFJLENBQUMsV0FBVyxDQUFDLGVBQWU7b0JBQzlDLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtpQkFDN0MsQ0FBQzthQUNIO1lBQ0QsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU87U0FDM0MsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLFdBQVc7UUFDakIsT0FBTyxJQUFJLHFDQUFtQixDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsd0JBQXdCLEVBQUU7WUFDNUUsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLGtCQUFrQjtZQUMvRCxVQUFVLEVBQUU7Z0JBQ1YsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMvRixvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7Z0JBQ2hHLGtCQUFrQixFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDeEUsZ0JBQWdCLEVBQUUsSUFBSTtnQkFDdEIscUJBQXFCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3REO1lBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUM7WUFDL0UsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU87U0FDM0MsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLCtCQUErQjtRQUNyQyxPQUFPLElBQUkscUNBQW1CLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSwyQkFBMkIsRUFBRTtZQUMvRSxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSxrQkFBa0I7WUFDMUIsVUFBVSxFQUFFO2dCQUNWLG1CQUFtQixFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7Z0JBQ2xFLG9CQUFvQixFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDM0UsZUFBZSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyx1QkFBdUIsQ0FBQztnQkFDekUsTUFBTSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7YUFDcEQ7WUFDRCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDO1lBQzdELFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPO1NBQzNDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyx3QkFBd0I7UUFDOUIsT0FBTyxJQUFJLHFDQUFtQixDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsZ0NBQWdDLEVBQUU7WUFDcEYsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsb0JBQW9CO1lBQzVCLFVBQVUsRUFBRTtnQkFDVixtQkFBbUIsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO2FBQ25FO1lBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDO1lBQ3JDLGNBQWMsRUFBRTtnQkFDZCxRQUFRLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLDBCQUEwQixDQUFDO2FBQ3RFO1lBQ0QsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUM7U0FDN0QsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVPLGlCQUFpQjtRQUN2QixPQUFPLElBQUkscUNBQW1CLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUM1RSxPQUFPLEVBQUUsS0FBSztZQUNkLE1BQU0sRUFBRSxxQkFBcUI7WUFDN0IsVUFBVSxFQUFFO2dCQUNWLG9CQUFvQixFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUM7YUFDcEU7WUFDRCxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsaUJBQWlCLENBQUM7WUFDdEMsY0FBYyxFQUFFO2dCQUNkLFFBQVEsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsbUNBQW1DLENBQUM7YUFDL0U7WUFDRCxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQztTQUM3RCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sUUFBUTtRQUNkLE1BQU0sUUFBUSxHQUFHLElBQUksc0JBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLE1BQU0sRUFBRTtZQUMvQyxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsQ0FBQyxPQUFPO1lBQ3hDLFNBQVMsRUFBRSxzQkFBSSxDQUFDLGFBQWEsQ0FBQyxTQUFTO1NBQ3hDLENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLElBQUkscUJBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ2xFLE9BQU8sRUFBRSxDQUFDLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLENBQUM7U0FDeEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxLQUFLLEdBQUcsMEZBQTBGLENBQUM7UUFDdkcsSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3JCLEtBQUssSUFBSSw0QkFBNEIsQ0FBQztTQUN2QztRQUNELE1BQU0sb0JBQW9CLEdBQUcsU0FBUyxDQUFDLFlBQVksQ0FBQyxRQUFRLEVBQUU7WUFDNUQsS0FBSyxFQUFFLHFCQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxpREFBaUQsQ0FBQztZQUNyRixPQUFPLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLFNBQVMsS0FBSyxzQ0FBc0MsQ0FBQztZQUM3RSxPQUFPLEVBQUUscUJBQUcsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDO2dCQUM3QixRQUFRO2dCQUNSLFlBQVksRUFBRSxjQUFjO2FBQzdCLENBQUM7WUFDRixTQUFTLEVBQUUsS0FBSztTQUNqQixDQUFDLENBQUM7UUFDSCxvQkFBb0IsQ0FBQyxjQUFjLENBQUMsRUFBRSxZQUFZLEVBQUUsUUFBUSxFQUFFLGFBQWEsRUFBRSxPQUFPLEVBQUUsUUFBUSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFDekcsTUFBTSxjQUFjLEdBQUcsU0FBUyxDQUFDLFlBQVksQ0FBQyxPQUFPLEVBQUU7WUFDckQsS0FBSyxFQUFFLHFCQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FBQyxpQ0FBaUMsQ0FBQztZQUNyRSxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDeEMsT0FBTyxFQUFFLHFCQUFHLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQztnQkFDN0IsUUFBUTtnQkFDUixZQUFZLEVBQUUsZ0JBQWdCO2FBQy9CLENBQUM7U0FDSCxDQUFDLENBQUM7UUFDSCxjQUFjLENBQUMsY0FBYyxDQUFDLEVBQUUsWUFBWSxFQUFFLFFBQVEsRUFBRSxhQUFhLEVBQUUsT0FBTyxFQUFFLFFBQVEsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ2xHLGNBQWMsQ0FBQyx3QkFBd0IsQ0FBQyxFQUFFLFNBQVMsRUFBRSxvQkFBb0IsRUFBRSxTQUFTLEVBQUUscUJBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDO1FBRWxJLE1BQU0sWUFBWSxHQUFHLElBQUkscUJBQUcsQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUMzRSxNQUFNLGlCQUFpQixHQUFHLFlBQVksQ0FBQyxZQUFZLENBQUMsVUFBVSxFQUFFO1lBQzlELEtBQUssRUFBRSxxQkFBRyxDQUFDLFVBQVUsQ0FBQyxZQUFZLENBQUMsb0NBQW9DLENBQUM7WUFDeEUsT0FBTyxFQUFFLENBQUMsTUFBTSxFQUFFLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ3ZDLE9BQU8sRUFBRSxxQkFBRyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUM7Z0JBQzdCLFFBQVE7Z0JBQ1IsWUFBWSxFQUFFLGVBQWU7YUFDOUIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUVILE1BQU0sTUFBTSxHQUFHLElBQUksK0JBQWEsQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFVBQVUsQ0FBQzthQUN0RCxJQUFJLENBQ0gsK0JBQWEsQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLGVBQWUsRUFBRSxPQUFPLENBQUMsRUFDOUQsSUFBSSxxQ0FBbUIsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQ3pELGtCQUFrQixFQUFFLCtCQUFhLENBQUMsa0JBQWtCLENBQUMsT0FBTztZQUM1RCxZQUFZLEVBQUUsSUFBSSxxQ0FBbUIsQ0FBQyxzQkFBc0IsRUFBRTtZQUM5RCxPQUFPLEVBQUUsSUFBSSxDQUFDLGNBQWM7WUFDNUIsT0FBTyxFQUFFLElBQUksQ0FBQyxPQUFPO1lBQ3JCLGNBQWMsRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUM7WUFDcEMsY0FBYyxFQUFFLFNBQVM7WUFDekIsa0JBQWtCLEVBQUU7Z0JBQ2xCO29CQUNFLG1CQUFtQixFQUFFLG9CQUFvQjtvQkFDekMsV0FBVyxFQUFFO3dCQUNYOzRCQUNFLElBQUksRUFBRSxZQUFZOzRCQUNsQixLQUFLLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLHdCQUF3QixDQUFDO3lCQUNqRTt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsWUFBWTs0QkFDbEIsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUM7eUJBQ3hEO3dCQUNEOzRCQUNFLElBQUksRUFBRSxZQUFZOzRCQUNsQixLQUFLLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGVBQWUsQ0FBQzt5QkFDeEQ7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLGdCQUFnQjs0QkFDdEIsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQzt5QkFDNUQ7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLGdCQUFnQjs0QkFDdEIsS0FBSyxFQUFFLElBQUksQ0FBQyxZQUFZLElBQUksRUFBRTt5QkFDL0I7cUJBQ0Y7aUJBQ0Y7YUFDRjtZQUNELFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPO1NBQzNDLENBQUMsQ0FDSDthQUNBLElBQUksQ0FDSCwrQkFBYSxDQUFDLFNBQVMsQ0FBQyxZQUFZLENBQUMsZUFBZSxFQUFFLFVBQVUsQ0FBQyxFQUNqRSxJQUFJLHFDQUFtQixDQUFDLFVBQVUsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7WUFDNUQsa0JBQWtCLEVBQUUsK0JBQWEsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPO1lBQzVELFlBQVksRUFBRSxJQUFJLHFDQUFtQixDQUFDLHNCQUFzQixFQUFFO1lBQzlELE9BQU8sRUFBRSxJQUFJLENBQUMsY0FBYztZQUM1QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsY0FBYyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQztZQUNwQyxjQUFjLEVBQUUsWUFBWTtZQUM1QixrQkFBa0IsRUFBRTtnQkFDbEI7b0JBQ0UsbUJBQW1CLEVBQUUsaUJBQWlCO29CQUN0QyxXQUFXLEVBQUU7d0JBQ1g7NEJBQ0UsSUFBSSxFQUFFLFFBQVE7NEJBQ2QsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyx3QkFBd0IsQ0FBQzt5QkFDakU7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLFFBQVE7NEJBQ2QsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxlQUFlLENBQUM7eUJBQ3hEO3dCQUNEOzRCQUNFLElBQUksRUFBRSxRQUFROzRCQUNkLEtBQUssRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsZUFBZSxDQUFDO3lCQUN4RDt3QkFDRDs0QkFDRSxJQUFJLEVBQUUsWUFBWTs0QkFDbEIsS0FBSyxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxtQkFBbUIsQ0FBQzt5QkFDNUQ7d0JBQ0Q7NEJBQ0UsSUFBSSxFQUFFLFlBQVk7NEJBQ2xCLEtBQUssRUFBRSxJQUFJLENBQUMsWUFBWSxJQUFJLFVBQVU7eUJBQ3ZDO3dCQUNEOzRCQUNFLElBQUksRUFBRSxtQkFBbUI7NEJBQ3pCLEtBQUssRUFBRSxJQUFJO3lCQUNaO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQ0gsQ0FBQztRQUVKLE9BQU8sTUFBTSxDQUFDLFVBQVUsRUFBRSxDQUFDO0lBQzdCLENBQUM7SUFFTyxhQUFhO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyx1QkFBdUIsRUFBRSxZQUFZLEVBQUUsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLENBQUM7SUFDbEgsQ0FBQztJQUVELCtEQUErRDtJQUMvRCxnRkFBZ0Y7SUFDaEYsc0JBQXNCO0lBQ3RCLDRDQUE0QztJQUM1QyxvQkFBb0I7SUFDcEIsc0ZBQXNGO0lBQ3RGLGtGQUFrRjtJQUNsRixTQUFTO0lBQ1QsZ0ZBQWdGO0lBQ2hGLDRDQUE0QztJQUM1QyxRQUFRO0lBQ1IsRUFBRTtJQUNGLGtDQUFrQztJQUNsQyxzREFBc0Q7SUFDdEQsMENBQTBDO0lBQzFDLHFGQUFxRjtJQUNyRixrSEFBa0g7SUFDbEgscUZBQXFGO0lBQ3JGLDZGQUE2RjtJQUM3Riw2RkFBNkY7SUFDN0Ysc0ZBQXNGO0lBQ3RGLCtEQUErRDtJQUMvRCxlQUFlO0lBQ2YsdUZBQXVGO0lBQ3ZGLDJEQUEyRDtJQUMzRCxVQUFVO0lBQ1Ysc0NBQXNDO0lBQ3RDLE9BQU87SUFDUCxJQUFJO0lBRUksYUFBYSxDQUFDLFFBQWtCO1FBQ3RDLE9BQU8sSUFBSSxxQ0FBbUIsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQ3BFLE9BQU8sRUFBRSxLQUFLO1lBQ2QsTUFBTSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLGtDQUFrQyxDQUFDLENBQUMsQ0FBQywyQkFBMkI7WUFDekYsVUFBVSxFQUFFO2dCQUNWLDJCQUEyQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUMvRyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztnQkFDeEcsYUFBYSxFQUFFLFNBQVM7Z0JBQ3hCLFdBQVcsRUFBRSxRQUFRO2FBQ3RCO1lBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGlCQUFpQixDQUFDO1lBQ3RDLFVBQVUsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxPQUFPO1NBQzNDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTyxrQkFBa0IsQ0FBQyxZQUFvQjtRQUM3QyxNQUFNLFdBQVcsR0FBRyxJQUFJLHVDQUFpQixDQUFDLElBQUksRUFBRSxZQUFZLEVBQUU7WUFDNUQsWUFBWSxFQUFFLHNCQUFJLENBQUMsYUFBYSxDQUFDLFNBQVM7WUFDMUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUNqQyxDQUFDLENBQUM7UUFDSCxXQUFXLENBQUMsZUFBZSxDQUFDLElBQUkscUJBQUcsQ0FBQyxlQUFlLENBQUM7WUFDbEQsT0FBTyxFQUFFLENBQUMsa0JBQWtCLENBQUM7WUFDN0IsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1NBQ2pCLENBQUMsQ0FBQyxDQUFDO1FBQ0osV0FBVyxDQUFDLGVBQWUsQ0FBQyxJQUFJLHFCQUFHLENBQUMsZUFBZSxDQUFDO1lBQ2xELE9BQU8sRUFBRSxDQUFDLDZCQUE2QixFQUFFLHNCQUFzQixDQUFDO1lBQ2hFLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztTQUNwQyxDQUFDLENBQUMsQ0FBQztRQUNKLE9BQU8sSUFBSSxxQ0FBbUIsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLHNCQUFzQixFQUFFO1lBQ3hFLGNBQWMsRUFBRSxXQUFXO1lBQzNCLE9BQU8sRUFBRSwrQkFBYSxDQUFDLFNBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQzFDLElBQUksRUFBRSxJQUFJLENBQUMsaUJBQWlCO2dCQUM1QixZQUFZLEVBQUUsWUFBWTtnQkFDMUIsWUFBWSxFQUFFLHNCQUFzQjthQUNyQyxDQUFDO1lBQ0YsbUJBQW1CLEVBQUUsSUFBSTtZQUN6QixVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU8sT0FBTztRQUNiLHVGQUF1RjtRQUN2RixvRkFBb0Y7UUFDcEYsTUFBTSxDQUFDLEdBQUcsSUFBSSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFLEVBQUUsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUM7UUFDdEcsQ0FBQyxDQUFDLE1BQU0sQ0FDTixJQUFJLHFDQUFtQixDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEVBQUU7WUFDakUsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDLGtCQUFrQjtZQUN2RSxVQUFVLEVBQUU7Z0JBQ1YsMkJBQTJCLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQzdHLG9CQUFvQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDO2FBQ3ZHO1lBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNwQyxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsV0FBVyxFQUFFLENBQUMsRUFBRSxRQUFRLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUNwRSxDQUFDO1FBQ0YsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2xCLENBQUMsQ0FBQyxNQUFNLENBQ04sSUFBSSxxQ0FBbUIsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO2dCQUNwRSxPQUFPLEVBQUUsS0FBSztnQkFDZCxNQUFNLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMseUJBQXlCLENBQUMsQ0FBQyxDQUFDLGtCQUFrQjtnQkFDdkUsVUFBVSxFQUFFO29CQUNWLDJCQUEyQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTO29CQUNoSCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLCtCQUFhLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztpQkFDMUc7Z0JBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQztnQkFDcEMsVUFBVSxFQUFFLCtCQUFhLENBQUMsUUFBUSxDQUFDLE9BQU87YUFDM0MsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsUUFBUSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FDcEUsQ0FBQztTQUNIO1FBQ0QsQ0FBQyxDQUFDLE1BQU0sQ0FDTixJQUFJLHFDQUFtQixDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsNkJBQTZCLEVBQUU7WUFDMUUsT0FBTyxFQUFFLEtBQUs7WUFDZCxNQUFNLEVBQUUsa0JBQWtCO1lBQzFCLFVBQVUsRUFBRTtnQkFDVixvQkFBb0IsRUFBRSwrQkFBYSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLFlBQVksQ0FBQztnQkFDM0csaUJBQWlCLEVBQUUsSUFBSTthQUN4QjtZQUNELFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQztZQUN0QyxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTztTQUMzQyxDQUFDLENBQUMsUUFBUSxDQUFDO1lBQ1YsV0FBVyxFQUFFLENBQUM7WUFDZCxRQUFRLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQ25DLENBQUMsQ0FDSCxDQUFDO1FBQ0YsSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2xCLENBQUMsQ0FBQyxNQUFNLENBQ04sSUFBSSxxQ0FBbUIsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO2dCQUNqRSxPQUFPLEVBQUUsS0FBSztnQkFDZCxNQUFNLEVBQUUsaUJBQWlCO2dCQUN6QixVQUFVLEVBQUU7b0JBQ1YsbUJBQW1CLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQztvQkFDbEUsaUJBQWlCLEVBQUUsSUFBSTtpQkFDeEI7Z0JBQ0QsWUFBWSxFQUFFLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDO2dCQUNyQyxVQUFVLEVBQUUsK0JBQWEsQ0FBQyxRQUFRLENBQUMsT0FBTzthQUMzQyxDQUFDLENBQUMsUUFBUSxDQUFDO2dCQUNWLFdBQVcsRUFBRSxDQUFDO2dCQUNkLFFBQVEsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7YUFDbkMsQ0FBQyxDQUNILENBQUM7U0FDSDtRQUNELE9BQU8sQ0FBQyxDQUFDO0lBQ1gsQ0FBQzs7QUF2b0JILDBEQXdvQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHtcbiAgYXdzX2VjMiBhcyBlYzIsXG4gIGF3c19lY3MgYXMgZWNzLFxuICBhd3NfZXZlbnRzIGFzIGV2ZW50cyxcbiAgYXdzX2V2ZW50c190YXJnZXRzIGFzIGV2ZW50c190YXJnZXRzLFxuICBhd3NfaWFtIGFzIGlhbSxcbiAgYXdzX2ttcyBhcyBrbXMsXG4gIGF3c19sb2dzIGFzIGxvZ3MsXG4gIGF3c19yZHMgYXMgcmRzLFxuICBhd3Nfc3RlcGZ1bmN0aW9ucyBhcyBzdGVwZnVuY3Rpb25zLFxuICBhd3Nfc3RlcGZ1bmN0aW9uc190YXNrcyBhcyBzdGVwZnVuY3Rpb25zX3Rhc2tzLFxufSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IERlbGV0ZU9sZEZ1bmN0aW9uIH0gZnJvbSAnLi9kZWxldGUtb2xkLWZ1bmN0aW9uJztcbmltcG9ydCB7IFBhcmFtZXRlcnNGdW5jdGlvbiB9IGZyb20gJy4vcGFyYW1ldGVycy1mdW5jdGlvbic7XG5pbXBvcnQgeyBXYWl0RnVuY3Rpb24gfSBmcm9tICcuL3dhaXQtZnVuY3Rpb24nO1xuXG5leHBvcnQgaW50ZXJmYWNlIElSZHNTYW5pdGl6ZWRTbmFwc2hvdHRlciB7XG4gIC8qKlxuICAgKiBEYXRhYmFzZSBjbHVzdGVyIHRvIHNuYXBzaG90IGFuZCBzYW5pdGl6ZS5cbiAgICpcbiAgICogT25seSBvbmUgb2YgYGRhdGFiYXNlQ2x1c3RlcmAgYW5kIGBkYXRhYmFzZUluc3RhbmNlYCBjYW4gYmUgc3BlY2lmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2VDbHVzdGVyPzogcmRzLklEYXRhYmFzZUNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIERhdGFiYXNlIGluc3RhbmNlIHRvIHNuYXBzaG90IGFuZCBzYW5pdGl6ZS5cbiAgICpcbiAgICogT25seSBvbmUgb2YgYGRhdGFiYXNlQ2x1c3RlcmAgYW5kIGBkYXRhYmFzZUluc3RhbmNlYCBjYW4gYmUgc3BlY2lmaWVkLlxuICAgKi9cbiAgcmVhZG9ubHkgZGF0YWJhc2VJbnN0YW5jZT86IHJkcy5JRGF0YWJhc2VJbnN0YW5jZTtcblxuICAvKipcbiAgICogTmFtZSBvZiBkYXRhYmFzZSB0byBjb25uZWN0IHRvIGluc2lkZSB0aGUgUkRTIGNsdXN0ZXIgb3IgaW5zdGFuY2UuIFRoaXMgZGF0YWJhc2Ugd2lsbCBiZSB1c2VkIHRvIGV4ZWN1dGUgdGhlIFNRTCBzY3JpcHQuXG4gICAqXG4gICAqIEBkZWZhdWx0ICdwb3N0Z3JlcycgZm9yIFBvc3RncmVTUUwgYW5kIG5vdCBzZXQgZm9yIE15U1FMXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZU5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEtNUyBrZXkgdXNlZCB0byBlbmNyeXB0IG9yaWdpbmFsIGRhdGFiYXNlLCBpZiBhbnkuXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZUtleT86IGttcy5JS2V5O1xuXG4gIC8qKlxuICAgKiBWUEMgd2hlcmUgdGVtcG9yYXJ5IGRhdGFiYXNlIGFuZCBzYW5pdGl6aW5nIHRhc2sgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjOiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVlBDIHN1Ym5ldHMgdG8gdXNlIGZvciB0ZW1wb3JhcnkgZGF0YWJhc2VzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBlYzIuU3VibmV0VHlwZS5QUklWQVRFX0lTT0xBVEVEXG4gICAqL1xuICByZWFkb25seSBkYlN1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBWUEMgc3VibmV0cyB0byB1c2UgZm9yIHNhbml0aXphdGlvbiB0YXNrLlxuICAgKlxuICAgKiBAZGVmYXVsdCBlYzIuU3VibmV0VHlwZS5QUklWQVRFX1dJVEhfTkFUXG4gICAqL1xuICByZWFkb25seSBzYW5pdGl6ZVN1Ym5ldHM/OiBlYzIuU3VibmV0U2VsZWN0aW9uO1xuXG4gIC8qKlxuICAgKiBDbHVzdGVyIHdoZXJlIHNhbml0aXphdGlvbiB0YXNrIHdpbGwgYmUgZXhlY3V0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IGEgbmV3IGNsdXN0ZXIgcnVubmluZyBvbiBnaXZlbiBWUENcbiAgICovXG4gIHJlYWRvbmx5IGZhcmdhdGVDbHVzdGVyPzogZWNzLklDbHVzdGVyO1xuXG4gIC8qKlxuICAgKiBTUUwgc2NyaXB0IHVzZWQgdG8gc2FuaXRpemUgdGhlIGRhdGFiYXNlLiBJdCB3aWxsIGJlIGV4ZWN1dGVkIGFnYWluc3QgdGhlIHRlbXBvcmFyeSBkYXRhYmFzZS5cbiAgICpcbiAgICogWW91IHdvdWxkIHVzdWFsbHkgd2FudCB0byBzdGFydCB0aGlzIHdpdGggYFVTRSBteWRhdGFiYXNlO2AuXG4gICAqL1xuICByZWFkb25seSBzY3JpcHQ6IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uYWwgS01TIGtleSB0byBlbmNyeXB0IHRhcmdldCBzbmFwc2hvdC5cbiAgICovXG4gIHJlYWRvbmx5IHNuYXBzaG90S2V5Pzoga21zLklLZXk7XG5cbiAgLyoqXG4gICAqIFByZWZpeCBmb3Igc2FuaXRpemVkIHNuYXBzaG90IG5hbWUuIFRoZSBjdXJyZW50IGRhdGUgYW5kIHRpbWUgd2lsbCBiZSBhZGRlZCB0byBpdC5cbiAgICpcbiAgICogQGRlZmF1bHQgY2x1c3RlciBpZGVudGlmaWVyICh3aGljaCBtaWdodCBiZSB0b28gbG9uZylcbiAgICovXG4gIHJlYWRvbmx5IHNuYXBzaG90UHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBQcmVmaXggZm9yIGFsbCB0ZW1wb3Jhcnkgc25hcHNob3RzIGFuZCBkYXRhYmFzZXMuIFRoZSBzdGVwIGZ1bmN0aW9uIGV4ZWN1dGlvbiBpZCB3aWxsIGJlIGFkZGVkIHRvIGl0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAnc2FuaXRpemUnXG4gICAqL1xuICByZWFkb25seSB0ZW1wUHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgc2NoZWR1bGUgb3IgcmF0ZSAoZnJlcXVlbmN5KSB0aGF0IGRldGVybWluZXMgd2hlbiB0aGUgc2FuaXRpemVkIHNuYXBzaG90IHJ1bnMgYXV0b21hdGljYWxseS5cbiAgICovXG4gIHJlYWRvbmx5IHNjaGVkdWxlPzogZXZlbnRzLlNjaGVkdWxlO1xuXG4gIC8qKlxuICAgKiBMaW1pdCB0aGUgbnVtYmVyIG9mIHNuYXBzaG90IGhpc3RvcnkuIFNldCB0aGlzIHRvIGRlbGV0ZSBvbGQgc25hcHNob3RzIGFuZCBvbmx5IGxlYXZlIGEgY2VydGFpbiBudW1iZXIgb2Ygc25hcHNob3RzLlxuICAgKi9cbiAgcmVhZG9ubHkgc25hcHNob3RIaXN0b3J5TGltaXQ/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgYWNjb3VudHMgdGhlIHNhbml0aXplZCBzbmFwc2hvdCBzaG91bGQgYmUgc2hhcmVkIHdpdGguXG4gICAqL1xuICByZWFkb25seSBzaGFyZUFjY291bnRzPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogQSBwcm9jZXNzIHRvIGNyZWF0ZSBzYW5pdGl6ZWQgc25hcHNob3RzIG9mIFJEUyBpbnN0YW5jZSBvciBjbHVzdGVyLCBvcHRpb25hbGx5IG9uIGEgc2NoZWR1bGUuIFRoZSBwcm9jZXNzIGlzIGhhbmRsZWQgYnkgYSBzdGVwIGZ1bmN0aW9uLlxuICpcbiAqIDEuIFNuYXBzaG90IHRoZSBzb3VyY2UgZGF0YWJhc2VcbiAqIDIuIE9wdGlvbmFsbHkgcmUtZW5jcnlwdCB0aGUgc25hcHNob3Qgd2l0aCBhIGRpZmZlcmVudCBrZXkgaW4gY2FzZSB5b3Ugd2FudCB0byBzaGFyZSBpdCB3aXRoIGFuIGFjY291bnQgdGhhdCBkb2Vzbid0IGhhdmUgYWNjZXNzIHRvIHRoZSBvcmlnaW5hbCBrZXlcbiAqIDMuIENyZWF0ZSBhIHRlbXBvcmFyeSBkYXRhYmFzZVxuICogNC4gUnVuIGEgRmFyZ2F0ZSB0YXNrIHRvIGNvbm5lY3QgdG8gdGhlIHRlbXBvcmFyeSBkYXRhYmFzZSBhbmQgZXhlY3V0ZSBhbiBhcmJpdHJhcnkgU1FMIHNjcmlwdCB0byBzYW5pdGl6ZSBpdFxuICogNS4gU25hcHNob3QgdGhlIHNhbml0aXplZCBkYXRhYmFzZVxuICogNi4gQ2xlYW4tdXAgdGVtcG9yYXJ5IHNuYXBzaG90cyBhbmQgZGF0YWJhc2VzXG4gKi9cbmV4cG9ydCBjbGFzcyBSZHNTYW5pdGl6ZWRTbmFwc2hvdHRlciBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHByaXZhdGUgd2FpdEZuOiBXYWl0RnVuY3Rpb24gfCB1bmRlZmluZWQ7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBzZWN1cml0eUdyb3VwOiBlYzIuU2VjdXJpdHlHcm91cDtcbiAgcHJpdmF0ZSByZWFkb25seSBzdWJuZXRHcm91cDogcmRzLlN1Ym5ldEdyb3VwO1xuICBwcml2YXRlIHJlYWRvbmx5IHN1Ym5ldHM6IGVjMi5TdWJuZXRTZWxlY3Rpb247XG4gIHByaXZhdGUgcmVhZG9ubHkgZmFyZ2F0ZUNsdXN0ZXI6IGVjcy5JQ2x1c3RlcjtcbiAgcHJpdmF0ZSByZWFkb25seSBzcWxTY3JpcHQ6IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSByZWVuY3J5cHQ6IGJvb2xlYW47XG5cbiAgcHJpdmF0ZSByZWFkb25seSBnZW5lcmFsVGFnczoge0tleTogc3RyaW5nOyBWYWx1ZTogc3RyaW5nfVtdO1xuICBwcml2YXRlIHJlYWRvbmx5IGZpbmFsU25hcHNob3RUYWdzOiB7S2V5OiBzdHJpbmc7IFZhbHVlOiBzdHJpbmd9W107XG4gIHByaXZhdGUgcmVhZG9ubHkgZGF0YWJhc2VJZGVudGlmaWVyOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgZGF0YWJhc2VOYW1lPzogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IHNuYXBzaG90UHJlZml4OiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdGVtcFByZWZpeDogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGlzQ2x1c3RlcjogYm9vbGVhbjtcbiAgcHJpdmF0ZSByZWFkb25seSBkYkNsdXN0ZXJBcm46IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBkYkluc3RhbmNlQXJuOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdGFyZ2V0U25hcHNob3RBcm46IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSB0ZW1wU25hcHNob3RBcm46IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSB0ZW1wRGJDbHVzdGVyQXJuOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgdGVtcERiSW5zdGFuY2VBcm46IHN0cmluZztcblxuICAvKipcbiAgICogU3RlcCBmdW5jdGlvbiBpbiBjaGFyZ2Ugb2YgdGhlIGVudGlyZSBwcm9jZXNzIGluY2x1ZGluZyBzbmFwc2hvdHRpbmcsIHNhbml0aXppbmcsIGFuZCBjbGVhbnVwLiBUcmlnZ2VyIHRoaXMgc3RlcCBmdW5jdGlvbiB0byBnZXQgYSBuZXcgc25hcHNob3QuXG4gICAqL1xuICBwdWJsaWMgc25hcHNob3R0ZXI6IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHJlYWRvbmx5IHByb3BzOiBJUmRzU2FuaXRpemVkU25hcHNob3R0ZXIpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5zZWN1cml0eUdyb3VwID0gbmV3IGVjMi5TZWN1cml0eUdyb3VwKHRoaXMsICdTRycsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiAnR3JvdXAgZm9yIGNvbW11bmljYXRpb24gYmV0d2VlbiBzYW5pdGl6aW5nIGpvYiBhbmQgZGF0YWJhc2UnLFxuICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgfSk7XG4gICAgY2RrLlRhZ3Mub2YodGhpcy5zZWN1cml0eUdyb3VwKS5hZGQoJ05hbWUnLCAnUkRTLXNhbml0aXplZC1zbmFwc2hvdHMnKTtcblxuICAgIHRoaXMuc3VibmV0R3JvdXAgPSBuZXcgcmRzLlN1Ym5ldEdyb3VwKHRoaXMsICdTdWJuZXQgZ3JvdXAnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ1RlbXBvcmFyeSBkYXRhYmFzZSB1c2VkIGZvciBSRFMtc2FuaXRpemUtc25hcHNob3RzJyxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgdnBjU3VibmV0czogcHJvcHMudnBjLnNlbGVjdFN1Ym5ldHMocHJvcHMuZGJTdWJuZXRzID8/IHsgc3VibmV0VHlwZTogZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCB9KSxcbiAgICB9KTtcblxuICAgIHRoaXMuc3VibmV0cyA9IHByb3BzLnNhbml0aXplU3VibmV0cyA/PyB7IHN1Ym5ldFR5cGU6IGVjMi5TdWJuZXRUeXBlLlBSSVZBVEVfV0lUSF9OQVQgfTtcbiAgICB0aGlzLmZhcmdhdGVDbHVzdGVyID0gcHJvcHMuZmFyZ2F0ZUNsdXN0ZXIgPz8gbmV3IGVjcy5DbHVzdGVyKHRoaXMsICdjbHVzdGVyJywgeyB2cGM6IHByb3BzLnZwYyB9KTtcbiAgICB0aGlzLnNxbFNjcmlwdCA9IHByb3BzLnNjcmlwdDtcblxuICAgIGlmICh0aGlzLnN1Ym5ldHMuc3VibmV0VHlwZSA9PT0gZWMyLlN1Ym5ldFR5cGUuUFJJVkFURV9JU09MQVRFRCkge1xuICAgICAgY2RrLkFubm90YXRpb25zLm9mKHRoaXMpLmFkZFdhcm5pbmcoJ0lzb2xhdGVkIHN1Ym5ldHMgbWF5IG5vdCB3b3JrIGZvciBzYW5pdGl6YXRpb24gdGFzayBhcyBpdCByZXF1aXJlcyBhY2Nlc3MgdG8gcHVibGljIEVDUicpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5kYXRhYmFzZUNsdXN0ZXIpIHtcbiAgICAgIHRoaXMuaXNDbHVzdGVyID0gdHJ1ZTtcbiAgICAgIHRoaXMuZGF0YWJhc2VJZGVudGlmaWVyID0gcHJvcHMuZGF0YWJhc2VDbHVzdGVyLmNsdXN0ZXJJZGVudGlmaWVyO1xuICAgIH0gZWxzZSBpZiAocHJvcHMuZGF0YWJhc2VJbnN0YW5jZSkge1xuICAgICAgdGhpcy5pc0NsdXN0ZXIgPSBmYWxzZTtcbiAgICAgIHRoaXMuZGF0YWJhc2VJZGVudGlmaWVyID0gcHJvcHMuZGF0YWJhc2VJbnN0YW5jZS5pbnN0YW5jZUlkZW50aWZpZXI7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignT25lIG9mIGBkYXRhYmFzZUNsdXN0ZXJgIG9yIGBkYXRhYmFzZUluc3RhbmNlYCBtdXN0IGJlIHNwZWNpZmllZCcpO1xuICAgIH1cbiAgICB0aGlzLmRhdGFiYXNlTmFtZSA9IHByb3BzLmRhdGFiYXNlTmFtZTtcblxuICAgIHRoaXMudGVtcFByZWZpeCA9IHByb3BzLnRlbXBQcmVmaXggPz8gJ3Nhbml0aXplJztcbiAgICB0aGlzLnNuYXBzaG90UHJlZml4ID0gcHJvcHMuc25hcHNob3RQcmVmaXggPz8gdGhpcy5kYXRhYmFzZUlkZW50aWZpZXI7XG5cbiAgICB0aGlzLnJlZW5jcnlwdCA9IHByb3BzLnNuYXBzaG90S2V5ICE9PSB1bmRlZmluZWQ7XG5cbiAgICB0aGlzLmRiQ2x1c3RlckFybiA9IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICByZXNvdXJjZTogJ2NsdXN0ZXInLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmRhdGFiYXNlSWRlbnRpZmllcixcbiAgICAgIGFybkZvcm1hdDogY2RrLkFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgIH0pO1xuICAgIHRoaXMuZGJJbnN0YW5jZUFybiA9IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICByZXNvdXJjZTogJ2RiJyxcbiAgICAgIHJlc291cmNlTmFtZTogdGhpcy5kYXRhYmFzZUlkZW50aWZpZXIsXG4gICAgICBhcm5Gb3JtYXQ6IGNkay5Bcm5Gb3JtYXQuQ09MT05fUkVTT1VSQ0VfTkFNRSxcbiAgICB9KTtcbiAgICB0aGlzLnRlbXBTbmFwc2hvdEFybiA9IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICByZXNvdXJjZTogdGhpcy5pc0NsdXN0ZXIgPyAnY2x1c3Rlci1zbmFwc2hvdCcgOiAnc25hcHNob3QnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLnRlbXBQcmVmaXh9LSpgLFxuICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUsXG4gICAgfSk7XG4gICAgdGhpcy50ZW1wRGJDbHVzdGVyQXJuID0gY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgIHJlc291cmNlOiAnY2x1c3RlcicsXG4gICAgICByZXNvdXJjZU5hbWU6IGAke3RoaXMudGVtcFByZWZpeH0tKmAsXG4gICAgICBhcm5Gb3JtYXQ6IGNkay5Bcm5Gb3JtYXQuQ09MT05fUkVTT1VSQ0VfTkFNRSxcbiAgICB9KTtcbiAgICB0aGlzLnRlbXBEYkluc3RhbmNlQXJuID0gY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgIHJlc291cmNlOiAnZGInLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLnRlbXBQcmVmaXh9LSpgLFxuICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUsXG4gICAgfSk7XG4gICAgdGhpcy50YXJnZXRTbmFwc2hvdEFybiA9IGNkay5TdGFjay5vZih0aGlzKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICByZXNvdXJjZTogdGhpcy5pc0NsdXN0ZXIgPyAnY2x1c3Rlci1zbmFwc2hvdCcgOiAnc25hcHNob3QnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLnNuYXBzaG90UHJlZml4fS0qYCxcbiAgICAgIGFybkZvcm1hdDogY2RrLkFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgIH0pO1xuXG4gICAgdGhpcy5nZW5lcmFsVGFncyA9IFtcbiAgICAgIHtcbiAgICAgICAgS2V5OiAnUkRTLXNhbml0aXplZC1zbmFwc2hvdHMnLFxuICAgICAgICBWYWx1ZTogdGhpcy5kYXRhYmFzZUlkZW50aWZpZXIsXG4gICAgICB9LFxuICAgIF07XG4gICAgdGhpcy5maW5hbFNuYXBzaG90VGFncyA9IHRoaXMuZ2VuZXJhbFRhZ3MuY29uY2F0KFt7XG4gICAgICBLZXk6ICdGaW5hbCcsXG4gICAgICBWYWx1ZTogJ3RydWUnLFxuICAgIH1dKTtcblxuICAgIGNvbnN0IHBhcmFtZXRlcnNTdGF0ZSA9IHRoaXMuZGJQYXJhbWV0ZXJzVGFzayhwcm9wcy5kYXRhYmFzZUtleSk7XG4gICAgY29uc3QgZXJyb3JDYXRjaGVyID0gbmV3IHN0ZXBmdW5jdGlvbnMuUGFyYWxsZWwodGhpcywgJ0Vycm9yIENhdGNoZXInLCB7IHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCB9KTtcblxuICAgIGxldCBjOiBzdGVwZnVuY3Rpb25zLklDaGFpbmFibGU7XG4gICAgbGV0IHM6IHN0ZXBmdW5jdGlvbnMuSU5leHRhYmxlO1xuICAgIHMgPSBjID0gdGhpcy5jcmVhdGVTbmFwc2hvdCgnQ3JlYXRlIFRlbXBvcmFyeSBTbmFwc2hvdCcsICckLmRhdGFiYXNlSWRlbnRpZmllcicsICckLnRlbXBTbmFwc2hvdElkJywgdGhpcy5nZW5lcmFsVGFncyk7XG4gICAgcyA9IHMubmV4dCh0aGlzLndhaXRGb3JPcGVyYXRpb24oJ1dhaXQgZm9yIFNuYXBzaG90JywgJ3NuYXBzaG90JywgJyQuZGF0YWJhc2VJZGVudGlmaWVyJywgJyQudGVtcFNuYXBzaG90SWQnKSk7XG4gICAgaWYgKHByb3BzLnNuYXBzaG90S2V5KSB7XG4gICAgICBzID0gcy5uZXh0KHRoaXMucmVlbmNyeXB0U25hcHNob3QocHJvcHMuc25hcHNob3RLZXkpKTtcbiAgICAgIHMgPSBzLm5leHQodGhpcy53YWl0Rm9yT3BlcmF0aW9uKCdXYWl0IGZvciBSZS1lbmNyeXB0JywgJ3NuYXBzaG90JywgJyQuZGF0YWJhc2VJZGVudGlmaWVyJywgJyQudGVtcEVuY1NuYXBzaG90SWQnKSk7XG4gICAgICBzID0gcy5uZXh0KHRoaXMuY3JlYXRlVGVtcG9yYXJ5RGF0YWJhc2UoJyQudGVtcEVuY1NuYXBzaG90SWQnKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHMgPSBzLm5leHQodGhpcy5jcmVhdGVUZW1wb3JhcnlEYXRhYmFzZSgnJC50ZW1wU25hcHNob3RJZCcpKTtcbiAgICB9XG4gICAgcyA9IHMubmV4dCh0aGlzLndhaXRGb3JPcGVyYXRpb24oJ1dhaXQgZm9yIFRlbXBvcmFyeSBEYXRhYmFzZScsIHRoaXMuaXNDbHVzdGVyID8gJ2NsdXN0ZXInIDogJ2luc3RhbmNlJywgJyQudGVtcERiSWQnKSk7XG4gICAgcyA9IHMubmV4dCh0aGlzLnNldFBhc3N3b3JkKCkpO1xuICAgIHMgPSBzLm5leHQodGhpcy53YWl0Rm9yT3BlcmF0aW9uKCdXYWl0IGZvciBUZW1wb3JhcnkgUGFzc3dvcmQnLCB0aGlzLmlzQ2x1c3RlciA/ICdjbHVzdGVyJyA6ICdpbnN0YW5jZScsICckLnRlbXBEYklkJykpO1xuICAgIGlmICh0aGlzLmlzQ2x1c3Rlcikge1xuICAgICAgcyA9IHMubmV4dCh0aGlzLmNyZWF0ZVRlbXBvcmFyeURhdGFiYXNlSW5zdGFuY2UoKSk7XG4gICAgICBzID0gcy5uZXh0KHRoaXMud2FpdEZvck9wZXJhdGlvbignV2FpdCBmb3IgVGVtcG9yYXJ5IEluc3RhbmNlJywgJ2luc3RhbmNlJywgJyQudGVtcERiSW5zdGFuY2VJZCcpKTtcbiAgICAgIHMgPSBzLm5leHQodGhpcy5nZXRUZW1wRGJDbHVzdGVyRW5kcG9pbnQoKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHMgPSBzLm5leHQodGhpcy5nZXRUZW1wRGJFbmRwb2ludCgpKTtcbiAgICB9XG4gICAgcyA9IHMubmV4dCh0aGlzLnNhbml0aXplKCkpO1xuICAgIHMgPSBzLm5leHQodGhpcy5maW5hbFNuYXBzaG90KCkpO1xuICAgIHMgPSBzLm5leHQodGhpcy53YWl0Rm9yT3BlcmF0aW9uKCdXYWl0IGZvciBGaW5hbCBTbmFwc2hvdCcsICdzbmFwc2hvdCcsICckLnRlbXBEYklkJywgJyQudGFyZ2V0U25hcHNob3RJZCcpKTtcbiAgICBpZiAocHJvcHMuc2hhcmVBY2NvdW50cykge1xuICAgICAgcyA9IHMubmV4dCh0aGlzLnNoYXJlU25hcHNob3QocHJvcHMuc2hhcmVBY2NvdW50cykpO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5zbmFwc2hvdEhpc3RvcnlMaW1pdCkge1xuICAgICAgcy5uZXh0KHRoaXMuZGVsZXRlT2xkU25hcHNob3RzKHByb3BzLnNuYXBzaG90SGlzdG9yeUxpbWl0KSk7XG4gICAgfVxuXG4gICAgZXJyb3JDYXRjaGVyLmJyYW5jaChjKTtcblxuICAgIGNvbnN0IGNsZWFudXBUYXNrcyA9IHRoaXMuY2xlYW51cCgpO1xuICAgIHRoaXMuc25hcHNob3R0ZXIgPSBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUodGhpcywgJ0RpcmVjdG9yJywge1xuICAgICAgZGVmaW5pdGlvbjogcGFyYW1ldGVyc1N0YXRlLm5leHQoZXJyb3JDYXRjaGVyLmFkZENhdGNoKGNsZWFudXBUYXNrcywgeyByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQgfSkpLm5leHQoY2xlYW51cFRhc2tzKSxcbiAgICB9KTtcblxuICAgIC8vIG5lZWRlZCBmb3IgY3JlYXRpbmcgYSBzbmFwc2hvdCB3aXRoIHRhZ3NcbiAgICB0aGlzLnNuYXBzaG90dGVyLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ3JkczpBZGRUYWdzVG9SZXNvdXJjZSddLFxuICAgICAgcmVzb3VyY2VzOiBbdGhpcy50ZW1wU25hcHNob3RBcm4sIHRoaXMudGFyZ2V0U25hcHNob3RBcm4sIHRoaXMudGVtcERiQ2x1c3RlckFybl0sXG4gICAgfSkpO1xuXG4gICAgLy8ga2V5IHBlcm1pc3Npb25zXG4gICAgaWYgKHByb3BzLnNuYXBzaG90S2V5KSB7XG4gICAgICBwcm9wcy5zbmFwc2hvdEtleS5ncmFudCh0aGlzLnNuYXBzaG90dGVyLCAna21zOkNyZWF0ZUdyYW50JywgJ2ttczpEZXNjcmliZUtleScpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuZGF0YWJhc2VLZXkpIHtcbiAgICAgIHByb3BzLmRhdGFiYXNlS2V5LmdyYW50KHRoaXMuc25hcHNob3R0ZXIsICdrbXM6Q3JlYXRlR3JhbnQnLCAna21zOkRlc2NyaWJlS2V5Jyk7XG4gICAgfVxuXG4gICAgLy8gYWxsb3cgZmFyZ2F0ZSB0byBhY2Nlc3MgUkRTXG4gICAgdGhpcy5zZWN1cml0eUdyb3VwLmNvbm5lY3Rpb25zLmFsbG93RnJvbSh0aGlzLnNlY3VyaXR5R3JvdXAuY29ubmVjdGlvbnMsIGVjMi5Qb3J0LmFsbFRjcCgpKTtcblxuICAgIC8vIHNjaGVkdWxlXG4gICAgaWYgKHByb3BzLnNjaGVkdWxlKSB7XG4gICAgICBuZXcgZXZlbnRzLlJ1bGUodGhpcywgJ1NjaGVkdWxlJywge1xuICAgICAgICBkZXNjcmlwdGlvbjogYFNhbml0aXplZCBzbmFwc2hvdCBvZiBSRFMgJHt0aGlzLmRhdGFiYXNlSWRlbnRpZmllcn1gLFxuICAgICAgICBzY2hlZHVsZTogcHJvcHMuc2NoZWR1bGUsXG4gICAgICAgIHRhcmdldHM6IFtcbiAgICAgICAgICBuZXcgZXZlbnRzX3RhcmdldHMuU2ZuU3RhdGVNYWNoaW5lKHRoaXMuc25hcHNob3R0ZXIpLFxuICAgICAgICBdLFxuICAgICAgfSk7XG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBkYlBhcmFtZXRlcnNUYXNrKGRhdGFiYXNlS2V5Pzoga21zLklLZXkpIHtcbiAgICBjb25zdCBwYXJhbWV0ZXJzRm4gPSBuZXcgUGFyYW1ldGVyc0Z1bmN0aW9uKHRoaXMsICdwYXJhbWV0ZXJzJywgeyBsb2dSZXRlbnRpb246IGxvZ3MuUmV0ZW50aW9uRGF5cy5PTkVfTU9OVEggfSk7XG4gICAgY29uc3QgcGFyYW1ldGVyc1N0YXRlID0gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuTGFtYmRhSW52b2tlKHRoaXMsICdHZXQgUGFyYW1ldGVycycsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uOiBwYXJhbWV0ZXJzRm4sXG4gICAgICBwYXlsb2FkOiBzdGVwZnVuY3Rpb25zLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgZXhlY3V0aW9uSWQ6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQkLkV4ZWN1dGlvbi5JZCcpLFxuICAgICAgICBpc0NsdXN0ZXI6IHRoaXMuaXNDbHVzdGVyLFxuICAgICAgICBkYXRhYmFzZUlkZW50aWZpZXI6IHRoaXMuZGF0YWJhc2VJZGVudGlmaWVyLFxuICAgICAgICBkYXRhYmFzZUtleTogZGF0YWJhc2VLZXk/LmtleUFybiB8fCAnJyxcbiAgICAgICAgc25hcHNob3RQcmVmaXg6IHRoaXMuc25hcHNob3RQcmVmaXgsXG4gICAgICAgIHRlbXBQcmVmaXg6IHRoaXMudGVtcFByZWZpeCxcbiAgICAgIH0pLFxuICAgICAgcGF5bG9hZFJlc3BvbnNlT25seTogdHJ1ZSxcbiAgICB9KTtcbiAgICBpZiAodGhpcy5pc0NsdXN0ZXIpIHtcbiAgICAgIHBhcmFtZXRlcnNGbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3JkczpEZXNjcmliZURCQ2x1c3RlcnMnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5kYkNsdXN0ZXJBcm5dLFxuICAgICAgfSkpO1xuICAgICAgcGFyYW1ldGVyc0ZuLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsncmRzOkRlc2NyaWJlREJJbnN0YW5jZXMnXSxcbiAgICAgICAgcmVzb3VyY2VzOiBbJyonXSwgLy8gVE9ETyBjYW4gd2UgZG8gYmV0dGVyIHdpdGhvdXQga25vd2luZyB0aGUgY2x1c3RlciBpbnN0YW5jZSBuYW1lIGFoZWFkIG9mIHRpbWU/XG4gICAgICB9KSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHBhcmFtZXRlcnNGbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICBhY3Rpb25zOiBbJ3JkczpEZXNjcmliZURCSW5zdGFuY2VzJ10sXG4gICAgICAgIHJlc291cmNlczogW3RoaXMuZGJJbnN0YW5jZUFybl0sXG4gICAgICB9KSk7XG4gICAgfVxuICAgIHJldHVybiBwYXJhbWV0ZXJzU3RhdGU7XG4gIH1cblxuICBwcml2YXRlIGNyZWF0ZVNuYXBzaG90KGlkOiBzdHJpbmcsIGRhdGFiYXNlSWQ6IHN0cmluZywgc25hcHNob3RJZDogc3RyaW5nLCB0YWdzOiB7IEtleTogc3RyaW5nOyBWYWx1ZTogc3RyaW5nIH1bXSkge1xuICAgIHJldHVybiBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCBpZCwge1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICBhY3Rpb246IHRoaXMuaXNDbHVzdGVyID8gJ2NyZWF0ZURCQ2x1c3RlclNuYXBzaG90JyA6ICdjcmVhdGVEQlNuYXBzaG90JyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgRGJDbHVzdGVySWRlbnRpZmllcjogdGhpcy5pc0NsdXN0ZXIgPyBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KGRhdGFiYXNlSWQpIDogdW5kZWZpbmVkLFxuICAgICAgICBEYkNsdXN0ZXJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdChzbmFwc2hvdElkKSA6IHVuZGVmaW5lZCxcbiAgICAgICAgRGJJbnN0YW5jZUlkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gdW5kZWZpbmVkIDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdChkYXRhYmFzZUlkKSxcbiAgICAgICAgRGJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gdW5kZWZpbmVkIDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdChzbmFwc2hvdElkKSxcbiAgICAgICAgVGFnczogdGFncyxcbiAgICAgIH0sXG4gICAgICBpYW1SZXNvdXJjZXM6IFsnKiddLFxuICAgICAgcmVzdWx0UGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5ESVNDQVJELFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSB3YWl0Rm9yT3BlcmF0aW9uKGlkOiBzdHJpbmcsIHJlc291cmNlVHlwZTogJ3NuYXBzaG90JyB8ICdjbHVzdGVyJyB8ICdpbnN0YW5jZScsIGRhdGFiYXNlSWRlbnRpZmllcjogc3RyaW5nLCBzbmFwc2hvdElkPzogc3RyaW5nKSB7XG4gICAgdGhpcy53YWl0Rm4gPSB0aGlzLndhaXRGbiA/PyBuZXcgV2FpdEZ1bmN0aW9uKHRoaXMsICd3YWl0Jywge1xuICAgICAgbG9nUmV0ZW50aW9uOiBsb2dzLlJldGVudGlvbkRheXMuT05FX01PTlRILFxuICAgICAgaW5pdGlhbFBvbGljeTogW1xuICAgICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogWydyZHM6RGVzY3JpYmVEQkNsdXN0ZXJzJywgJ3JkczpEZXNjcmliZURCQ2x1c3RlclNuYXBzaG90cycsICdyZHM6RGVzY3JpYmVEQlNuYXBzaG90cycsICdyZHM6RGVzY3JpYmVEQkluc3RhbmNlcyddLFxuICAgICAgICAgIHJlc291cmNlczogW3RoaXMuZGJDbHVzdGVyQXJuLCB0aGlzLmRiSW5zdGFuY2VBcm4sIHRoaXMudGVtcERiQ2x1c3RlckFybixcbiAgICAgICAgICAgIHRoaXMudGVtcFNuYXBzaG90QXJuLCB0aGlzLnRhcmdldFNuYXBzaG90QXJuLCB0aGlzLnRlbXBEYkluc3RhbmNlQXJuXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgbGV0IHBheWxvYWQgPSB7XG4gICAgICByZXNvdXJjZVR5cGUsXG4gICAgICBkYXRhYmFzZUlkZW50aWZpZXI6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoZGF0YWJhc2VJZGVudGlmaWVyKSxcbiAgICAgIHNuYXBzaG90SWRlbnRpZmllcjogdW5kZWZpbmVkIGFzIFN0cmluZyB8IHVuZGVmaW5lZCxcbiAgICAgIGlzQ2x1c3RlcjogdGhpcy5pc0NsdXN0ZXIsXG4gICAgfTtcbiAgICBpZiAoc25hcHNob3RJZCkge1xuICAgICAgcGF5bG9hZC5zbmFwc2hvdElkZW50aWZpZXIgPSBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KHNuYXBzaG90SWQpO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5MYW1iZGFJbnZva2UodGhpcywgaWQsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uOiB0aGlzLndhaXRGbixcbiAgICAgIHBheWxvYWRSZXNwb25zZU9ubHk6IHRydWUsXG4gICAgICBwYXlsb2FkOiBzdGVwZnVuY3Rpb25zLlRhc2tJbnB1dC5mcm9tT2JqZWN0KHBheWxvYWQpLFxuICAgICAgcmVzdWx0UGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5ESVNDQVJELFxuICAgIH0pLmFkZFJldHJ5KHtcbiAgICAgIGVycm9yczogWydOb3RSZWFkeSddLFxuICAgICAgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5taW51dGVzKDEpLFxuICAgICAgbWF4QXR0ZW1wdHM6IDMwMCxcbiAgICAgIGJhY2tvZmZSYXRlOiAxLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSByZWVuY3J5cHRTbmFwc2hvdChrZXk6IGttcy5JS2V5KSB7XG4gICAgcmV0dXJuIG5ldyBzdGVwZnVuY3Rpb25zX3Rhc2tzLkNhbGxBd3NTZXJ2aWNlKHRoaXMsICdSZS1lbmNyeXB0IFNuYXBzaG90Jywge1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICBhY3Rpb246IHRoaXMuaXNDbHVzdGVyID8gJ2NvcHlEQkNsdXN0ZXJTbmFwc2hvdCcgOiAnY29weURCU25hcHNob3QnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBTb3VyY2VEQkNsdXN0ZXJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wU25hcHNob3RJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICBUYXJnZXREQkNsdXN0ZXJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRW5jU25hcHNob3RJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICBTb3VyY2VEQlNuYXBzaG90SWRlbnRpZmllcjogdGhpcy5pc0NsdXN0ZXIgPyB1bmRlZmluZWQgOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBTbmFwc2hvdElkJyksXG4gICAgICAgIFRhcmdldERCU25hcHNob3RJZGVudGlmaWVyOiB0aGlzLmlzQ2x1c3RlciA/IHVuZGVmaW5lZCA6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcEVuY1NuYXBzaG90SWQnKSxcbiAgICAgICAgS21zS2V5SWQ6IGtleS5rZXlJZCxcbiAgICAgICAgQ29weVRhZ3M6IGZhbHNlLFxuICAgICAgICBUYWdzOiB0aGlzLmdlbmVyYWxUYWdzLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGVtcFNuYXBzaG90QXJuXSxcbiAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlVGVtcG9yYXJ5RGF0YWJhc2Uoc25hcHNob3RJZGVudGlmaWVyOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0NyZWF0ZSBUZW1wb3JhcnkgRGF0YWJhc2UnLCB7XG4gICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgIGFjdGlvbjogdGhpcy5pc0NsdXN0ZXIgPyAncmVzdG9yZURCQ2x1c3RlckZyb21TbmFwc2hvdCcgOiAncmVzdG9yZURCSW5zdGFuY2VGcm9tREJTbmFwc2hvdCcsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIERiQ2x1c3RlcklkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGJJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICBEYkluc3RhbmNlSWRlbnRpZmllcjogdGhpcy5pc0NsdXN0ZXIgPyB1bmRlZmluZWQgOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYklkJyksXG4gICAgICAgIEVuZ2luZTogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5lbmdpbmUnKSxcbiAgICAgICAgU25hcHNob3RJZGVudGlmaWVyOiB0aGlzLmlzQ2x1c3RlciA/IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoc25hcHNob3RJZGVudGlmaWVyKSA6IHVuZGVmaW5lZCxcbiAgICAgICAgRGJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gdW5kZWZpbmVkIDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdChzbmFwc2hvdElkZW50aWZpZXIpLFxuICAgICAgICBQdWJsaWNseUFjY2Vzc2libGU6IGZhbHNlLFxuICAgICAgICBWcGNTZWN1cml0eUdyb3VwSWRzOiBbdGhpcy5zZWN1cml0eUdyb3VwLnNlY3VyaXR5R3JvdXBJZF0sXG4gICAgICAgIERiU3VibmV0R3JvdXBOYW1lOiB0aGlzLnN1Ym5ldEdyb3VwLnN1Ym5ldEdyb3VwTmFtZSxcbiAgICAgICAgVGFnczogdGhpcy5nZW5lcmFsVGFncyxcbiAgICAgIH0sXG4gICAgICBpYW1SZXNvdXJjZXM6IFtcbiAgICAgICAgdGhpcy50ZW1wRGJDbHVzdGVyQXJuLFxuICAgICAgICB0aGlzLnRlbXBEYkluc3RhbmNlQXJuLFxuICAgICAgICB0aGlzLnRlbXBTbmFwc2hvdEFybixcbiAgICAgICAgY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICAgICAgcmVzb3VyY2U6ICdzdWJncnAnLFxuICAgICAgICAgIHJlc291cmNlTmFtZTogdGhpcy5zdWJuZXRHcm91cC5zdWJuZXRHcm91cE5hbWUsXG4gICAgICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUsXG4gICAgICAgIH0pLFxuICAgICAgXSxcbiAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgc2V0UGFzc3dvcmQoKSB7XG4gICAgcmV0dXJuIG5ldyBzdGVwZnVuY3Rpb25zX3Rhc2tzLkNhbGxBd3NTZXJ2aWNlKHRoaXMsICdTZXQgVGVtcG9yYXJ5IFBhc3N3b3JkJywge1xuICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICBhY3Rpb246IHRoaXMuaXNDbHVzdGVyID8gJ21vZGlmeURCQ2x1c3RlcicgOiAnbW9kaWZ5REJJbnN0YW5jZScsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIERiQ2x1c3RlcklkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGJJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICBEYkluc3RhbmNlSWRlbnRpZmllcjogdGhpcy5pc0NsdXN0ZXIgPyB1bmRlZmluZWQgOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYklkJyksXG4gICAgICAgIE1hc3RlclVzZXJQYXNzd29yZDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGIucGFzc3dvcmQnKSxcbiAgICAgICAgQXBwbHlJbW1lZGlhdGVseTogdHJ1ZSxcbiAgICAgICAgQmFja3VwUmV0ZW50aW9uUGVyaW9kOiB0aGlzLmlzQ2x1c3RlciA/IHVuZGVmaW5lZCA6IDAsXG4gICAgICB9LFxuICAgICAgaWFtUmVzb3VyY2VzOiBbdGhpcy5pc0NsdXN0ZXIgPyB0aGlzLnRlbXBEYkNsdXN0ZXJBcm4gOiB0aGlzLnRlbXBEYkluc3RhbmNlQXJuXSxcbiAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgY3JlYXRlVGVtcG9yYXJ5RGF0YWJhc2VJbnN0YW5jZSgpIHtcbiAgICByZXR1cm4gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0NyZWF0ZSBUZW1wb3JhcnkgSW5zdGFuY2UnLCB7XG4gICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgIGFjdGlvbjogJ2NyZWF0ZURCSW5zdGFuY2UnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBEYkNsdXN0ZXJJZGVudGlmaWVyOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYklkJyksXG4gICAgICAgIERiSW5zdGFuY2VJZGVudGlmaWVyOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYkluc3RhbmNlSWQnKSxcbiAgICAgICAgRGJJbnN0YW5jZUNsYXNzOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYkluc3RhbmNlQ2xhc3MnKSxcbiAgICAgICAgRW5naW5lOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLmVuZ2luZScpLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGVtcERiQ2x1c3RlckFybiwgdGhpcy50ZW1wRGJJbnN0YW5jZUFybl0sXG4gICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGdldFRlbXBEYkNsdXN0ZXJFbmRwb2ludCgpIHtcbiAgICByZXR1cm4gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0dldCBUZW1wb3JhcnkgQ2x1c3RlciBFbmRwb2ludCcsIHtcbiAgICAgIHNlcnZpY2U6ICdyZHMnLFxuICAgICAgYWN0aW9uOiAnZGVzY3JpYmVEQkNsdXN0ZXJzJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgRGJDbHVzdGVySWRlbnRpZmllcjogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGJJZCcpLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGVtcERiQ2x1c3RlckFybl0sXG4gICAgICByZXN1bHRTZWxlY3Rvcjoge1xuICAgICAgICBlbmRwb2ludDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC5EYkNsdXN0ZXJzWzBdLkVuZHBvaW50JyksXG4gICAgICB9LFxuICAgICAgcmVzdWx0UGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGIuaG9zdCcpLFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRUZW1wRGJFbmRwb2ludCgpIHtcbiAgICByZXR1cm4gbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0dldCBUZW1wb3JhcnkgRW5kcG9pbnQnLCB7XG4gICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgIGFjdGlvbjogJ2Rlc2NyaWJlREJJbnN0YW5jZXMnLFxuICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICBEYkluc3RhbmNlSWRlbnRpZmllcjogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGJJZCcpLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGVtcERiSW5zdGFuY2VBcm5dLFxuICAgICAgcmVzdWx0U2VsZWN0b3I6IHtcbiAgICAgICAgZW5kcG9pbnQ6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQuRGJJbnN0YW5jZXNbMF0uRW5kcG9pbnQuQWRkcmVzcycpLFxuICAgICAgfSxcbiAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLmhvc3QnKSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgc2FuaXRpemUoKTogc3RlcGZ1bmN0aW9ucy5JQ2hhaW5hYmxlIHtcbiAgICBjb25zdCBsb2dHcm91cCA9IG5ldyBsb2dzLkxvZ0dyb3VwKHRoaXMsICdMb2dzJywge1xuICAgICAgcmVtb3ZhbFBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIHJldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICB9KTtcblxuICAgIGNvbnN0IG15c3FsVGFzayA9IG5ldyBlY3MuRmFyZ2F0ZVRhc2tEZWZpbml0aW9uKHRoaXMsICdNeVNRTCBUYXNrJywge1xuICAgICAgdm9sdW1lczogW3sgbmFtZTogJ2NvbmZpZycsIGhvc3Q6IHt9IH1dLFxuICAgIH0pO1xuICAgIGxldCBteWNuZiA9ICdbY2xpZW50XVxcbnVzZXI9JE1ZU1FMX1VTRVJcXG5ob3N0PSRNWVNRTF9IT1NUXFxucG9ydD0kTVlTUUxfUE9SVFxcbnBhc3N3b3JkPSRNWVNRTF9QQVNTV09SRCc7XG4gICAgaWYgKHRoaXMuZGF0YWJhc2VOYW1lKSB7XG4gICAgICBteWNuZiArPSAnXFxuZGF0YWJhc2U9JE1ZU1FMX0RBVEFCQVNFJztcbiAgICB9XG4gICAgY29uc3QgbXlzcWxDb25maWdDb250YWluZXIgPSBteXNxbFRhc2suYWRkQ29udGFpbmVyKCdjb25maWcnLCB7XG4gICAgICBpbWFnZTogZWNzLkFzc2V0SW1hZ2UuZnJvbVJlZ2lzdHJ5KCdwdWJsaWMuZWNyLmF3cy9kb2NrZXIvbGlicmFyeS9iYXNoOjQtYWxwaW5lMy4xNScpLFxuICAgICAgY29tbWFuZDogWydiYXNoJywgJy1jJywgYGVjaG8gXCIke215Y25mfVwiID4gfi8ubXkuY25mICYmIGNobW9kIDcwMCB+Ly5teS5jbmZgXSxcbiAgICAgIGxvZ2dpbmc6IGVjcy5Mb2dEcml2ZXIuYXdzTG9ncyh7XG4gICAgICAgIGxvZ0dyb3VwLFxuICAgICAgICBzdHJlYW1QcmVmaXg6ICdteXNxbC1jb25maWcnLFxuICAgICAgfSksXG4gICAgICBlc3NlbnRpYWw6IGZhbHNlLFxuICAgIH0pO1xuICAgIG15c3FsQ29uZmlnQ29udGFpbmVyLmFkZE1vdW50UG9pbnRzKHsgc291cmNlVm9sdW1lOiAnY29uZmlnJywgY29udGFpbmVyUGF0aDogJy9yb290JywgcmVhZE9ubHk6IGZhbHNlIH0pO1xuICAgIGNvbnN0IG15c3FsQ29udGFpbmVyID0gbXlzcWxUYXNrLmFkZENvbnRhaW5lcignbXlzcWwnLCB7XG4gICAgICBpbWFnZTogZWNzLkFzc2V0SW1hZ2UuZnJvbVJlZ2lzdHJ5KCdwdWJsaWMuZWNyLmF3cy9sdHMvbXlzcWw6bGF0ZXN0JyksXG4gICAgICBjb21tYW5kOiBbJ215c3FsJywgJy1lJywgdGhpcy5zcWxTY3JpcHRdLFxuICAgICAgbG9nZ2luZzogZWNzLkxvZ0RyaXZlci5hd3NMb2dzKHtcbiAgICAgICAgbG9nR3JvdXAsXG4gICAgICAgIHN0cmVhbVByZWZpeDogJ215c3FsLXNhbml0aXplJyxcbiAgICAgIH0pLFxuICAgIH0pO1xuICAgIG15c3FsQ29udGFpbmVyLmFkZE1vdW50UG9pbnRzKHsgc291cmNlVm9sdW1lOiAnY29uZmlnJywgY29udGFpbmVyUGF0aDogJy9yb290JywgcmVhZE9ubHk6IHRydWUgfSk7XG4gICAgbXlzcWxDb250YWluZXIuYWRkQ29udGFpbmVyRGVwZW5kZW5jaWVzKHsgY29udGFpbmVyOiBteXNxbENvbmZpZ0NvbnRhaW5lciwgY29uZGl0aW9uOiBlY3MuQ29udGFpbmVyRGVwZW5kZW5jeUNvbmRpdGlvbi5TVUNDRVNTIH0pO1xuXG4gICAgY29uc3QgcG9zdGdyZXNUYXNrID0gbmV3IGVjcy5GYXJnYXRlVGFza0RlZmluaXRpb24odGhpcywgJ1Bvc3RyZVNRTCBUYXNrJyk7XG4gICAgY29uc3QgcG9zdGdyZXNDb250YWluZXIgPSBwb3N0Z3Jlc1Rhc2suYWRkQ29udGFpbmVyKCdwb3N0Z3JlcycsIHtcbiAgICAgIGltYWdlOiBlY3MuQXNzZXRJbWFnZS5mcm9tUmVnaXN0cnkoJ3B1YmxpYy5lY3IuYXdzL2x0cy9wb3N0Z3JlczpsYXRlc3QnKSxcbiAgICAgIGNvbW1hbmQ6IFsncHNxbCcsICctYycsIHRoaXMuc3FsU2NyaXB0XSxcbiAgICAgIGxvZ2dpbmc6IGVjcy5Mb2dEcml2ZXIuYXdzTG9ncyh7XG4gICAgICAgIGxvZ0dyb3VwLFxuICAgICAgICBzdHJlYW1QcmVmaXg6ICdwc3FsLXNhbml0aXplJyxcbiAgICAgIH0pLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY2hvaWNlID0gbmV3IHN0ZXBmdW5jdGlvbnMuQ2hvaWNlKHRoaXMsICdTYW5pdGl6ZScpXG4gICAgICAud2hlbihcbiAgICAgICAgc3RlcGZ1bmN0aW9ucy5Db25kaXRpb24uc3RyaW5nRXF1YWxzKCckLmRvY2tlckltYWdlJywgJ215c3FsJyksXG4gICAgICAgIG5ldyBzdGVwZnVuY3Rpb25zX3Rhc2tzLkVjc1J1blRhc2sodGhpcywgJ1Nhbml0aXplIE15U1FMJywge1xuICAgICAgICAgIGludGVncmF0aW9uUGF0dGVybjogc3RlcGZ1bmN0aW9ucy5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQiwgLy8gc3luY1xuICAgICAgICAgIGxhdW5jaFRhcmdldDogbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuRWNzRmFyZ2F0ZUxhdW5jaFRhcmdldCgpLFxuICAgICAgICAgIGNsdXN0ZXI6IHRoaXMuZmFyZ2F0ZUNsdXN0ZXIsXG4gICAgICAgICAgc3VibmV0czogdGhpcy5zdWJuZXRzLFxuICAgICAgICAgIHNlY3VyaXR5R3JvdXBzOiBbdGhpcy5zZWN1cml0eUdyb3VwXSxcbiAgICAgICAgICB0YXNrRGVmaW5pdGlvbjogbXlzcWxUYXNrLFxuICAgICAgICAgIGNvbnRhaW5lck92ZXJyaWRlczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBjb250YWluZXJEZWZpbml0aW9uOiBteXNxbENvbmZpZ0NvbnRhaW5lcixcbiAgICAgICAgICAgICAgZW52aXJvbm1lbnQ6IFtcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBuYW1lOiAnTVlTUUxfSE9TVCcsXG4gICAgICAgICAgICAgICAgICB2YWx1ZTogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGIuaG9zdC5lbmRwb2ludCcpLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgbmFtZTogJ01ZU1FMX1BPUlQnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLnBvcnQnKSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIG5hbWU6ICdNWVNRTF9VU0VSJyxcbiAgICAgICAgICAgICAgICAgIHZhbHVlOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYi51c2VyJyksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBuYW1lOiAnTVlTUUxfUEFTU1dPUkQnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLnBhc3N3b3JkJyksXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBuYW1lOiAnTVlTUUxfREFUQUJBU0UnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHRoaXMuZGF0YWJhc2VOYW1lID8/ICcnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgcmVzdWx0UGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5ESVNDQVJELFxuICAgICAgICB9KSxcbiAgICAgIClcbiAgICAgIC53aGVuKFxuICAgICAgICBzdGVwZnVuY3Rpb25zLkNvbmRpdGlvbi5zdHJpbmdFcXVhbHMoJyQuZG9ja2VySW1hZ2UnLCAncG9zdGdyZXMnKSxcbiAgICAgICAgbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuRWNzUnVuVGFzayh0aGlzLCAnU2FuaXRpemUgUG9zdGdyZXMnLCB7XG4gICAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBzdGVwZnVuY3Rpb25zLkludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CLCAvLyBzeW5jXG4gICAgICAgICAgbGF1bmNoVGFyZ2V0OiBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5FY3NGYXJnYXRlTGF1bmNoVGFyZ2V0KCksXG4gICAgICAgICAgY2x1c3RlcjogdGhpcy5mYXJnYXRlQ2x1c3RlcixcbiAgICAgICAgICBzdWJuZXRzOiB0aGlzLnN1Ym5ldHMsXG4gICAgICAgICAgc2VjdXJpdHlHcm91cHM6IFt0aGlzLnNlY3VyaXR5R3JvdXBdLFxuICAgICAgICAgIHRhc2tEZWZpbml0aW9uOiBwb3N0Z3Jlc1Rhc2ssXG4gICAgICAgICAgY29udGFpbmVyT3ZlcnJpZGVzOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIGNvbnRhaW5lckRlZmluaXRpb246IHBvc3RncmVzQ29udGFpbmVyLFxuICAgICAgICAgICAgICBlbnZpcm9ubWVudDogW1xuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIG5hbWU6ICdQR0hPU1QnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLmhvc3QuZW5kcG9pbnQnKSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIG5hbWU6ICdQR1BPUlQnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLnBvcnQnKSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIG5hbWU6ICdQR1VTRVInLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcERiLnVzZXInKSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgIG5hbWU6ICdQR1BBU1NXT1JEJyxcbiAgICAgICAgICAgICAgICAgIHZhbHVlOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRlbXBEYi5wYXNzd29yZCcpLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgbmFtZTogJ1BHREFUQUJBU0UnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6IHRoaXMuZGF0YWJhc2VOYW1lID8/ICdwb3N0Z3JlcycsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICBuYW1lOiAnUEdDT05ORUNUX1RJTUVPVVQnLFxuICAgICAgICAgICAgICAgICAgdmFsdWU6ICczMCcsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgXSxcbiAgICAgICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQsXG4gICAgICAgIH0pLFxuICAgICAgKTtcblxuICAgIHJldHVybiBjaG9pY2UuYWZ0ZXJ3YXJkcygpO1xuICB9XG5cbiAgcHJpdmF0ZSBmaW5hbFNuYXBzaG90KCkge1xuICAgIHJldHVybiB0aGlzLmNyZWF0ZVNuYXBzaG90KCdDcmVhdGUgRmluYWwgU25hcHNob3QnLCAnJC50ZW1wRGJJZCcsICckLnRhcmdldFNuYXBzaG90SWQnLCB0aGlzLmZpbmFsU25hcHNob3RUYWdzKTtcbiAgfVxuXG4gIC8vIHByaXZhdGUgd2FpdEZvclNuYXBzaG90U2ZuKGlkOiBzdHJpbmcsIHNuYXBzaG90SWQ6IHN0cmluZykge1xuICAvLyAgIGNvbnN0IGRlc2NyaWJlU25hcHNob3QgPSBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCBpZCwge1xuICAvLyAgICAgc2VydmljZTogJ3JkcycsXG4gIC8vICAgICBhY3Rpb246ICdkZXNjcmliZURCQ2x1c3RlclNuYXBzaG90cycsXG4gIC8vICAgICBwYXJhbWV0ZXJzOiB7XG4gIC8vICAgICAgIERiQ2x1c3RlcklkZW50aWZpZXI6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQuZGF0YWJhc2VJZGVudGlmaWVyJyksXG4gIC8vICAgICAgIERiQ2x1c3RlclNuYXBzaG90SWRlbnRpZmllcjogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdChzbmFwc2hvdElkKSxcbiAgLy8gICAgIH0sXG4gIC8vICAgICBpYW1SZXNvdXJjZXM6IFt0aGlzLmRiQXJuLCB0aGlzLnRlbXBTbmFwc2hvdEFybiwgdGhpcy50YXJnZXRTbmFwc2hvdEFybl0sXG4gIC8vICAgICByZXN1bHRQYXRoOiAnJC5yZXN1bHRzLndhaXRTbmFwc2hvdCcsXG4gIC8vICAgfSk7XG4gIC8vXG4gIC8vICAgcmV0dXJuIGRlc2NyaWJlU25hcHNob3QubmV4dChcbiAgLy8gICAgIG5ldyBzdGVwZnVuY3Rpb25zLkNob2ljZSh0aGlzLCBgJHtpZH0gKGNoZWNrKWApXG4gIC8vICAgICAgIC53aGVuKHN0ZXBmdW5jdGlvbnMuQ29uZGl0aW9uLm9yKFxuICAvLyAgICAgICAgIHN0ZXBmdW5jdGlvbnMuQ29uZGl0aW9uLnN0cmluZ01hdGNoZXMoJyQucmVzdWx0cy53YWl0U25hcHNob3QnLCAnKnN0b3AqJyksXG4gIC8vICAgICAgICAgc3RlcGZ1bmN0aW9ucy5Db25kaXRpb24uc3RyaW5nTWF0Y2hlcygnJC5yZXN1bHRzLndhaXRTbmFwc2hvdCcsICcqZGVsZXQqJyksIC8vIGJvdGggZGVsZXRlIGFuZCBkZWxldGluZ1xuICAvLyAgICAgICAgIHN0ZXBmdW5jdGlvbnMuQ29uZGl0aW9uLnN0cmluZ01hdGNoZXMoJyQucmVzdWx0cy53YWl0U25hcHNob3QnLCAnKmZhaWwqJyksXG4gIC8vICAgICAgICAgc3RlcGZ1bmN0aW9ucy5Db25kaXRpb24uc3RyaW5nTWF0Y2hlcygnJC5yZXN1bHRzLndhaXRTbmFwc2hvdCcsICcqaW5jb21wYXRpYmxlKicpLFxuICAvLyAgICAgICAgIHN0ZXBmdW5jdGlvbnMuQ29uZGl0aW9uLnN0cmluZ01hdGNoZXMoJyQucmVzdWx0cy53YWl0U25hcHNob3QnLCAnKmluYWNjZXNzaWJsZSonKSxcbiAgLy8gICAgICAgICBzdGVwZnVuY3Rpb25zLkNvbmRpdGlvbi5zdHJpbmdNYXRjaGVzKCckLnJlc3VsdHMud2FpdFNuYXBzaG90JywgJyplcnJvcionKSxcbiAgLy8gICAgICAgKSwgbmV3IHN0ZXBmdW5jdGlvbnMuRmFpbCh0aGlzLCBgJHtpZH0gKGJhZCBzdGF0dXMpYCkpXG4gIC8vICAgICAgIC53aGVuKFxuICAvLyAgICAgICAgIHN0ZXBmdW5jdGlvbnMuQ29uZGl0aW9uLnN0cmluZ0VxdWFscygnJC5yZXN1bHRzLndhaXRTbmFwc2hvdCcsICdhdmFpbGFibGUnKSxcbiAgLy8gICAgICAgICBuZXcgc3RlcGZ1bmN0aW9ucy5TdWNjZWVkKHRoaXMsIGAke2lkfSAoZG9uZSlgKSxcbiAgLy8gICAgICAgKVxuICAvLyAgICAgICAub3RoZXJ3aXNlKGRlc2NyaWJlU25hcHNob3QpLFxuICAvLyAgICk7XG4gIC8vIH1cblxuICBwcml2YXRlIHNoYXJlU25hcHNob3QoYWNjb3VudHM6IHN0cmluZ1tdKSB7XG4gICAgcmV0dXJuIG5ldyBzdGVwZnVuY3Rpb25zX3Rhc2tzLkNhbGxBd3NTZXJ2aWNlKHRoaXMsICdTaGFyZSBTbmFwc2hvdCcsIHtcbiAgICAgIHNlcnZpY2U6ICdyZHMnLFxuICAgICAgYWN0aW9uOiB0aGlzLmlzQ2x1c3RlciA/ICdtb2RpZnlEQkNsdXN0ZXJTbmFwc2hvdEF0dHJpYnV0ZScgOiAnbW9kaWZ5REJTbmFwc2hvdEF0dHJpYnV0ZScsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIERiQ2x1c3RlclNuYXBzaG90SWRlbnRpZmllcjogdGhpcy5pc0NsdXN0ZXIgPyBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLnN0cmluZ0F0KCckLnRhcmdldFNuYXBzaG90SWQnKSA6IHVuZGVmaW5lZCxcbiAgICAgICAgRGJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gdW5kZWZpbmVkIDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50YXJnZXRTbmFwc2hvdElkJyksXG4gICAgICAgIEF0dHJpYnV0ZU5hbWU6ICdyZXN0b3JlJyxcbiAgICAgICAgVmFsdWVzVG9BZGQ6IGFjY291bnRzLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGFyZ2V0U25hcHNob3RBcm5dLFxuICAgICAgcmVzdWx0UGF0aDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5ESVNDQVJELFxuICAgIH0pO1xuICB9XG5cbiAgcHJpdmF0ZSBkZWxldGVPbGRTbmFwc2hvdHMoaGlzdG9yeUxpbWl0OiBudW1iZXIpIHtcbiAgICBjb25zdCBkZWxldGVPbGRGbiA9IG5ldyBEZWxldGVPbGRGdW5jdGlvbih0aGlzLCAnZGVsZXRlLW9sZCcsIHtcbiAgICAgIGxvZ1JldGVudGlvbjogbG9ncy5SZXRlbnRpb25EYXlzLk9ORV9NT05USCxcbiAgICAgIHRpbWVvdXQ6IGNkay5EdXJhdGlvbi5taW51dGVzKDUpLFxuICAgIH0pO1xuICAgIGRlbGV0ZU9sZEZuLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ3RhZzpHZXRSZXNvdXJjZXMnXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgfSkpO1xuICAgIGRlbGV0ZU9sZEZuLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ3JkczpEZWxldGVEQkNsdXN0ZXJTbmFwc2hvdCcsICdyZHM6RGVsZXRlREJTbmFwc2hvdCddLFxuICAgICAgcmVzb3VyY2VzOiBbdGhpcy50YXJnZXRTbmFwc2hvdEFybl0sXG4gICAgfSkpO1xuICAgIHJldHVybiBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5MYW1iZGFJbnZva2UodGhpcywgJ0RlbGV0ZSBPbGQgU25hcHNob3RzJywge1xuICAgICAgbGFtYmRhRnVuY3Rpb246IGRlbGV0ZU9sZEZuLFxuICAgICAgcGF5bG9hZDogc3RlcGZ1bmN0aW9ucy5UYXNrSW5wdXQuZnJvbU9iamVjdCh7XG4gICAgICAgIHRhZ3M6IHRoaXMuZmluYWxTbmFwc2hvdFRhZ3MsXG4gICAgICAgIGhpc3RvcnlMaW1pdDogaGlzdG9yeUxpbWl0LFxuICAgICAgICByZXNvdXJjZVR5cGU6ICdyZHM6Y2x1c3Rlci1zbmFwc2hvdCcsXG4gICAgICB9KSxcbiAgICAgIHBheWxvYWRSZXNwb25zZU9ubHk6IHRydWUsXG4gICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQsXG4gICAgfSk7XG4gIH1cblxuICBwcml2YXRlIGNsZWFudXAoKSB7XG4gICAgLy8gV2UgcmV0cnkgZXZlcnl0aGluZyBiZWNhdXNlIHdoZW4gYW55IGJyYW5jaCBmYWlscywgYWxsIG90aGVyIGJyYW5jaGVzIGFyZSBjYW5jZWxsZWQuXG4gICAgLy8gUmV0cnlpbmcgZ2l2ZXMgb3RoZXIgYnJhbmNoZXMgYW4gb3Bwb3J0dW5pdHkgdG8gc3RhcnQgYW5kIGhvcGVmdWxseSBhdCBsZWFzdCBydW4uXG4gICAgY29uc3QgcCA9IG5ldyBzdGVwZnVuY3Rpb25zLlBhcmFsbGVsKHRoaXMsICdDbGVhbnVwJywgeyByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQgfSk7XG4gICAgcC5icmFuY2goXG4gICAgICBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCAnVGVtcG9yYXJ5IFNuYXBzaG90Jywge1xuICAgICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgICAgYWN0aW9uOiB0aGlzLmlzQ2x1c3RlciA/ICdkZWxldGVEQkNsdXN0ZXJTbmFwc2hvdCcgOiAnZGVsZXRlREJTbmFwc2hvdCcsXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBEYkNsdXN0ZXJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wU25hcHNob3RJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICAgIERiU25hcHNob3RJZGVudGlmaWVyOiB0aGlzLmlzQ2x1c3RlciA/IHVuZGVmaW5lZCA6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQoJyQudGVtcFNuYXBzaG90SWQnKSxcbiAgICAgICAgfSxcbiAgICAgICAgaWFtUmVzb3VyY2VzOiBbdGhpcy50ZW1wU25hcHNob3RBcm5dLFxuICAgICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQsXG4gICAgICB9KS5hZGRSZXRyeSh7IG1heEF0dGVtcHRzOiA1LCBpbnRlcnZhbDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMTApIH0pLFxuICAgICk7XG4gICAgaWYgKHRoaXMucmVlbmNyeXB0KSB7XG4gICAgICBwLmJyYW5jaChcbiAgICAgICAgbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ1JlLWVuY3J5cHRlZCBTbmFwc2hvdCcsIHtcbiAgICAgICAgICBzZXJ2aWNlOiAncmRzJyxcbiAgICAgICAgICBhY3Rpb246IHRoaXMuaXNDbHVzdGVyID8gJ2RlbGV0ZURCQ2x1c3RlclNuYXBzaG90JyA6ICdkZWxldGVEQlNuYXBzaG90JyxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBEYkNsdXN0ZXJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRW5jU25hcHNob3RJZCcpIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgRGJTbmFwc2hvdElkZW50aWZpZXI6IHRoaXMuaXNDbHVzdGVyID8gdW5kZWZpbmVkIDogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRW5jU25hcHNob3RJZCcpLFxuICAgICAgICAgIH0sXG4gICAgICAgICAgaWFtUmVzb3VyY2VzOiBbdGhpcy50ZW1wU25hcHNob3RBcm5dLFxuICAgICAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCxcbiAgICAgICAgfSkuYWRkUmV0cnkoeyBtYXhBdHRlbXB0czogNSwgaW50ZXJ2YWw6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDEwKSB9KSxcbiAgICAgICk7XG4gICAgfVxuICAgIHAuYnJhbmNoKFxuICAgICAgbmV3IHN0ZXBmdW5jdGlvbnNfdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ1RlbXBvcmFyeSBEYXRhYmFzZSBJbnN0YW5jZScsIHtcbiAgICAgICAgc2VydmljZTogJ3JkcycsXG4gICAgICAgIGFjdGlvbjogJ2RlbGV0ZURCSW5zdGFuY2UnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgRGJJbnN0YW5jZUlkZW50aWZpZXI6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguc3RyaW5nQXQodGhpcy5pc0NsdXN0ZXIgPyAnJC50ZW1wRGJJbnN0YW5jZUlkJyA6ICckLnRlbXBEYklkJyksXG4gICAgICAgICAgU2tpcEZpbmFsU25hcHNob3Q6IHRydWUsXG4gICAgICAgIH0sXG4gICAgICAgIGlhbVJlc291cmNlczogW3RoaXMudGVtcERiSW5zdGFuY2VBcm5dLFxuICAgICAgICByZXN1bHRQYXRoOiBzdGVwZnVuY3Rpb25zLkpzb25QYXRoLkRJU0NBUkQsXG4gICAgICB9KS5hZGRSZXRyeSh7XG4gICAgICAgIG1heEF0dGVtcHRzOiA1LFxuICAgICAgICBpbnRlcnZhbDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMTApLFxuICAgICAgfSksXG4gICAgKTtcbiAgICBpZiAodGhpcy5pc0NsdXN0ZXIpIHtcbiAgICAgIHAuYnJhbmNoKFxuICAgICAgICBuZXcgc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCAnVGVtcG9yYXJ5IERhdGFiYXNlJywge1xuICAgICAgICAgIHNlcnZpY2U6ICdyZHMnLFxuICAgICAgICAgIGFjdGlvbjogJ2RlbGV0ZURCQ2x1c3RlcicsXG4gICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgRGJDbHVzdGVySWRlbnRpZmllcjogc3RlcGZ1bmN0aW9ucy5Kc29uUGF0aC5zdHJpbmdBdCgnJC50ZW1wRGJJZCcpLFxuICAgICAgICAgICAgU2tpcEZpbmFsU25hcHNob3Q6IHRydWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBpYW1SZXNvdXJjZXM6IFt0aGlzLnRlbXBEYkNsdXN0ZXJBcm5dLFxuICAgICAgICAgIHJlc3VsdFBhdGg6IHN0ZXBmdW5jdGlvbnMuSnNvblBhdGguRElTQ0FSRCxcbiAgICAgICAgfSkuYWRkUmV0cnkoe1xuICAgICAgICAgIG1heEF0dGVtcHRzOiA1LFxuICAgICAgICAgIGludGVydmFsOiBjZGsuRHVyYXRpb24uc2Vjb25kcygxMCksXG4gICAgICAgIH0pLFxuICAgICAgKTtcbiAgICB9XG4gICAgcmV0dXJuIHA7XG4gIH1cbn1cbiJdfQ==