"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable import/no-extraneous-dependencies */
const crypto = require("crypto");
const AWS = require("aws-sdk");
const rds = new AWS.RDS();
function getDockerImage(engine) {
    if (engine.match(/(^aurora$|mysql|mariadb)/)) {
        return 'mysql';
    }
    else if (engine.match(/postgres/)) {
        return 'postgres';
        // } else if (engine.match(/oracle/)) {
        // } else if (engine.match(/sqlserver/)) {
    }
    else {
        throw new Error(`"${engine}" is not a supported database engine`);
    }
}
function confirmLength(name, value) {
    let error;
    if (value.length > 63) {
        error = 'is too long';
    }
    if (!value.charAt(0).match(/[a-z]/i)) {
        error = 'does not start with a letter';
    }
    if (value.indexOf('--') >= 0) {
        error = 'contains two consecutive hyphens';
    }
    if (error) {
        throw new Error(`"${name}" ${error}. Try adjusting 'tempPrefix' and/or 'snapshotPrefix'. Current value: ${value}`);
    }
}
exports.handler = async function (input) {
    let port;
    let user;
    let engine;
    let kmsKeyId;
    let instanceClass;
    if (input.isCluster) {
        const origDb = await rds.describeDBClusters({ DBClusterIdentifier: input.databaseIdentifier }).promise();
        if (!origDb.DBClusters || origDb.DBClusters.length != 1) {
            throw new Error(`Unable to find ${input.databaseIdentifier}`);
        }
        const cluster = origDb.DBClusters[0];
        if (!cluster.Port || !cluster.MasterUsername || !cluster.DBClusterMembers) {
            throw new Error(`Database missing some required parameters: ${JSON.stringify(cluster)}`);
        }
        const origInstances = await rds.describeDBInstances({ DBInstanceIdentifier: cluster.DBClusterMembers[0].DBInstanceIdentifier }).promise();
        if (!origInstances.DBInstances || origInstances.DBInstances.length < 1) {
            throw new Error(`Unable to find instances for ${input.databaseIdentifier}`);
        }
        const instance = origInstances.DBInstances[0];
        if (!instance.DBInstanceClass) {
            throw new Error(`Database instance missing class: ${JSON.stringify(instance)}`);
        }
        port = cluster.Port;
        user = cluster.MasterUsername;
        engine = cluster.Engine;
        kmsKeyId = cluster.KmsKeyId;
        instanceClass = instance.DBInstanceClass;
    }
    else {
        const origDb = await rds.describeDBInstances({ DBInstanceIdentifier: input.databaseIdentifier }).promise();
        if (!origDb.DBInstances || origDb.DBInstances.length != 1) {
            throw new Error(`Unable to find ${input.databaseIdentifier}`);
        }
        const instance = origDb.DBInstances[0];
        if (!instance.Endpoint?.Address || !instance.Endpoint?.Port || !instance.MasterUsername) {
            throw new Error(`Database missing some required parameters: ${JSON.stringify(instance)}`);
        }
        port = instance.Endpoint.Port;
        user = instance.MasterUsername;
        engine = instance.Engine;
        kmsKeyId = instance.KmsKeyId;
        instanceClass = instance.DBInstanceClass ?? 'db.m5.large';
    }
    if (input.databaseKey && input.databaseKey !== '') {
        if (input.databaseKey !== kmsKeyId) {
            throw new Error(`Database key (${kmsKeyId}) doesn't match databaseKey parameter (${input.databaseKey})`);
        }
    }
    const timestamp = new Date();
    const snapshotSuffix = `-${timestamp.getUTCFullYear()}${timestamp.getUTCMonth().toString().padStart(2, '0')}${timestamp.getUTCDay().toString().padStart(2, '0')}${timestamp.getUTCHours().toString().padStart(2, '0')}${timestamp.getUTCMinutes().toString().padStart(2, '0')}`;
    const targetSnapshotId = `${input.snapshotPrefix}${snapshotSuffix}`;
    const tempSuffix = crypto.randomBytes(8).toString('hex');
    const result = {
        databaseIdentifier: input.databaseIdentifier,
        isCluster: input.isCluster,
        engine: engine ?? 'unknown',
        tempSnapshotId: `${input.tempPrefix}-${tempSuffix}`,
        tempEncSnapshotId: `${input.tempPrefix}-enc-${tempSuffix}`,
        tempDbId: `${input.tempPrefix}-${tempSuffix}`,
        tempDbInstanceId: `${input.tempPrefix}-inst-${tempSuffix}`,
        tempDbInstanceClass: instanceClass,
        targetSnapshotId,
        dockerImage: getDockerImage(engine ?? ''),
        tempDb: {
            host: {
                endpoint: 'NOT KNOWN YET',
            },
            port: port.toString(),
            user: user,
            password: crypto.randomBytes(16).toString('hex'),
        },
    };
    confirmLength('tempSnapshotId', result.tempSnapshotId);
    confirmLength('tempEncSnapshotId', result.tempEncSnapshotId);
    confirmLength('tempDbId', result.tempDbId);
    confirmLength('tempDbInstanceId', result.tempDbInstanceId);
    confirmLength('targetSnapshotId', result.targetSnapshotId);
    return result;
};
//# sourceMappingURL=data:application/json;base64,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