"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'Bus',
        }));
        test.done();
    },
    'named event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'myEventBus',
        }));
        test.done();
    },
    'partner event bus'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'aws.partner/PartnerName/acct1/repo1',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::EventBus', {
            Name: 'aws.partner/PartnerName/acct1/repo1',
            EventSourceName: 'aws.partner/PartnerName/acct1/repo1',
        }));
        test.done();
    },
    'can get bus name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus',
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusName: bus.eventBusName,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusName: { Ref: 'BusEA82B648' },
        }));
        test.done();
    },
    'can get bus arn'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const bus = new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myEventBus',
        });
        // WHEN
        new core_1.CfnResource(stack, 'Res', {
            type: 'Test::Resource',
            properties: {
                EventBusArn: bus.eventBusArn,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Test::Resource', {
            EventBusArn: { 'Fn::GetAtt': ['BusEA82B648', 'Arn'] },
        }));
        test.done();
    },
    'event bus name cannot be default'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'default',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not be 'default'/);
        test.done();
    },
    'event bus name cannot contain slash'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'my/bus',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must not contain '\/'/);
        test.done();
    },
    'event bus cannot have name and source name'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: 'myBus',
            eventSourceName: 'myBus',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' and 'eventSourceName' cannot both be provided/);
        test.done();
    },
    'event bus name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventBusName: '',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventBusName' must satisfy: /);
        test.done();
    },
    'event bus source name must follow pattern'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: 'invalid-partner',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: \/\^aws/);
        test.done();
    },
    'event bus source name cannot be empty string'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const createInvalidBus = () => new lib_1.EventBus(stack, 'Bus', {
            eventSourceName: '',
        });
        // THEN
        test.throws(() => {
            createInvalidBus();
        }, /'eventSourceName' must satisfy: /);
        test.done();
    },
    'can grant PutEvents'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const role = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        // WHEN
        lib_1.EventBus.grantPutEvents(role);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'events:PutEvents',
                        Effect: 'Allow',
                        Resource: '*',
                    },
                ],
                Version: '2012-10-17',
            },
            Roles: [
                {
                    Ref: 'Role1ABCC5F0',
                },
            ],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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