import events = require('@aws-cdk/aws-events');
import iam = require('@aws-cdk/aws-iam');
import { Construct, IConstruct, IResource, Resource } from '@aws-cdk/cdk';
import { LifecycleRule } from './lifecycle';
/**
 * Represents an ECR repository.
 */
export interface IRepository extends IResource {
    /**
     * The name of the repository
     */
    readonly repositoryName: string;
    /**
     * The ARN of the repository
     */
    readonly repositoryArn: string;
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    readonly repositoryUri: string;
    /**
     * Returns the URI of the repository for a certain tag. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Image tag to use (tools usually default to "latest" if omitted)
     */
    repositoryUriForTag(tag?: string): string;
    /**
     * Add a policy statement to the repository's resource policy
     */
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to pull images in this repository.
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     * @param name The name of the rule
     * @param target An IEventRuleTarget to invoke when this event happens (you can add more targets using `addTarget`)
     * @param imageTag Only trigger on the specific image tag
     */
    onImagePushed(name: string, target?: events.IEventRuleTarget, imageTag?: string): events.EventRule;
    /**
     * Export this repository from the stack
     */
    export(): RepositoryImportProps;
}
export interface RepositoryImportProps {
    /**
     * The ARN of the repository to import.
     *
     * At least one of `repositoryArn` or `repositoryName` is required.
     *
     * @default If you only have a repository name and the repository is in the same
     * account/region as the current stack, you can set `repositoryName` instead
     * and the ARN will be formatted with the current region and account.
     */
    readonly repositoryArn?: string;
    /**
     * The full name of the repository to import.
     *
     * This is only needed if the repository ARN is not a concrete string, in which
     * case it is impossible to safely parse the ARN and extract full repository
     * names from it if it includes multiple components (e.g. `foo/bar/myrepo`).
     *
     * If the repository is in the same region/account as the stack, it is sufficient
     * to only specify the repository name.
     */
    readonly repositoryName?: string;
}
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
export declare abstract class RepositoryBase extends Resource implements IRepository {
    /**
     * Import a repository
     */
    static import(scope: Construct, id: string, props: RepositoryImportProps): IRepository;
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName: string, scope: IConstruct): string;
    /**
     * The name of the repository
     */
    abstract readonly repositoryName: string;
    /**
     * The ARN of the repository
     */
    abstract readonly repositoryArn: string;
    /**
     * Add a policy statement to the repository's resource policy
     */
    abstract addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    readonly repositoryUri: string;
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag?: string): string;
    /**
     * Export this repository from the stack
     */
    abstract export(): RepositoryImportProps;
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     * @param name The name of the rule
     * @param target An IEventRuleTarget to invoke when this event happens (you can add more targets using `addTarget`)
     * @param imageTag Only trigger on the specific image tag
     */
    onImagePushed(name: string, target?: events.IEventRuleTarget, imageTag?: string): events.EventRule;
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee: iam.IGrantable, ...actions: string[]): iam.Grant;
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee: iam.IGrantable): iam.Grant;
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee: iam.IGrantable): iam.Grant;
}
export interface RepositoryProps {
    /**
     * Name for this repository
     *
     * @default Automatically generated name.
     */
    readonly repositoryName?: string;
    /**
     * Life cycle rules to apply to this registry
     *
     * @default No life cycle rules
     */
    readonly lifecycleRules?: LifecycleRule[];
    /**
     * The AWS account ID associated with the registry that contains the repository.
     *
     * @see https://docs.aws.amazon.com/AmazonECR/latest/APIReference/API_PutLifecyclePolicy.html
     * @default The default registry is assumed.
     */
    readonly lifecycleRegistryId?: string;
    /**
     * Retain the repository on stack deletion
     *
     * If you don't set this to true, the registry must be empty, otherwise
     * your stack deletion will fail.
     *
     * @default false
     */
    readonly retain?: boolean;
}
/**
 * Define an ECR repository
 */
export declare class Repository extends RepositoryBase {
    readonly repositoryName: string;
    readonly repositoryArn: string;
    private readonly lifecycleRules;
    private readonly registryId?;
    private policyDocument?;
    constructor(scope: Construct, id: string, props?: RepositoryProps);
    /**
     * Export this repository from the stack
     */
    export(): RepositoryImportProps;
    addToResourcePolicy(statement: iam.PolicyStatement): void;
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule: LifecycleRule): void;
    /**
     * Render the life cycle policy object
     */
    private renderLifecyclePolicy;
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    private orderedLifecycleRules;
}
