"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository. Reused between imported repositories and owned repositories.
 */
class RepositoryBase extends cdk_1.Resource {
    /**
     * Import a repository
     */
    static import(scope, id, props) {
        return new ImportedRepository(scope, id, props);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region
     * as the current stack.
     */
    static arnForLocalRepository(repositoryName, scope) {
        return scope.node.stack.formatArn({
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName
        });
    }
    /**
     * The URI of this repository (represents the latest image):
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     *    ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        const parts = this.node.stack.parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.amazonaws.com/${this.repositoryName}${tagSuffix}`;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this
     * repository.
     * @param name The name of the rule
     * @param target An IEventRuleTarget to invoke when this event happens (you can add more targets using `addTarget`)
     * @param imageTag Only trigger on the specific image tag
     */
    onImagePushed(name, target, imageTag) {
        return new events.EventRule(this, name, {
            targets: target ? [target] : undefined,
            eventPattern: {
                source: ['aws.ecr'],
                detail: {
                    eventName: [
                        'PutImage',
                    ],
                    requestParameters: {
                        repositoryName: [
                            this.repositoryName,
                        ],
                        imageTag: imageTag ? [imageTag] : undefined,
                    },
                },
            },
        });
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, "ecr:BatchCheckLayerAvailability", "ecr:GetDownloadUrlForLayer", "ecr:BatchGetImage");
        iam.Grant.addToPrincipal({
            grantee,
            actions: ["ecr:GetAuthorizationToken"],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, "ecr:PutImage", "ecr:InitiateLayerUpload", "ecr:UploadLayerPart", "ecr:CompleteLayerUpload");
    }
}
exports.RepositoryBase = RepositoryBase;
/**
 * An already existing repository
 */
class ImportedRepository extends RepositoryBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        if (props.repositoryArn) {
            this.repositoryArn = props.repositoryArn;
        }
        else {
            if (!props.repositoryName) {
                throw new Error('If "repositoryArn" is not specified, you must specify "repositoryName", ' +
                    'which also implies that the repository resides in the same region/account as this stack');
            }
            this.repositoryArn = RepositoryBase.arnForLocalRepository(props.repositoryName, this);
        }
        if (props.repositoryName) {
            this.repositoryName = props.repositoryName;
        }
        else {
            // if repositoryArn is a token, the repository name is also required. this is because
            // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
            // parse the name from an ARN using CloudFormation's split/select.
            if (cdk_1.Token.unresolved(this.repositoryArn)) {
                throw new Error('repositoryArn is a late-bound value, and therefore repositoryName is required');
            }
            this.repositoryName = this.repositoryArn.split('/').slice(1).join('/');
        }
    }
    export() {
        return this.props;
    }
    addToResourcePolicy(_statement) {
        // FIXME: Add annotation about policy we dropped on the floor
    }
}
/**
 * Define an ECR repository
 */
class Repository extends RepositoryBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.lifecycleRules = new Array();
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: props.repositoryName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: new cdk_1.Token(() => this.policyDocument),
            lifecyclePolicy: new cdk_1.Token(() => this.renderLifecyclePolicy()),
        });
        if (props.retain) {
            resource.options.deletionPolicy = cdk_1.DeletionPolicy.Retain;
        }
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = resource.repositoryName;
        this.repositoryArn = resource.repositoryArn;
    }
    /**
     * Export this repository from the stack
     */
    export() {
        return {
            repositoryArn: new cdk_1.CfnOutput(this, 'RepositoryArn', { value: this.repositoryArn }).makeImportValue().toString(),
            repositoryName: new cdk_1.CfnOutput(this, 'RepositoryName', { value: this.repositoryName }).makeImportValue().toString()
        };
    }
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatement(statement);
    }
    /**
     * Add a life cycle rule to the repository
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.Any : lifecycle_1.TagStatus.Tagged };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.Tagged && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.Tagged && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAgeDays !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAgeDays' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.Any && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.Any).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(this.node.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.Any);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.Any);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.Any);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: rule.rulePriority !== undefined ? rule.rulePriority : autoPrio++
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.Any);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.Any,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAgeDays !== undefined ? lifecycle_1.CountType.SinceImagePushed : lifecycle_1.CountType.ImageCountMoreThan,
            countNumber: rule.maxImageAgeDays !== undefined ? rule.maxImageAgeDays : rule.maxImageCount,
            countUnit: rule.maxImageAgeDays !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire'
        }
    };
}
//# sourceMappingURL=data:application/json;base64,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