/**
 * Copyright (c) 2018, Ouster, Inc.
 * All rights reserved.
 */

#pragma once

#include <cstddef>

namespace ouster {
namespace viz {

inline float** genPalette(const int n, const float from[3], const float to[3]) {
    float** palette = new float*[n];
    for (int i = 0; i < n; i++) {
        palette[i] = new float[3];
        for (int j = 0; j < 3; j++) {
            palette[i][j] = (to[j] * i + from[j] * (n - 1 - i)) / (n - 1);
        }
    }
    return palette;
}

// generated from:
// https://daniel.lawrence.lu/public/colortransform/#0_2423_964_352_6_2624_1000_513_11_3248_1000_617_15_415_1000_774
const size_t spezia_n = 256;
const float spezia_palette[spezia_n][3] = {
    {0.04890922165917825f, 0.34265700288230266f, 0.5139042200196196f},
    {0.04895672077739804f, 0.34399228711079705f, 0.5173325088859984f},
    {0.04899969158023907f, 0.34532432182766976f, 0.5207851330769154f},
    {0.049038068929181285f, 0.34665300013643424f, 0.5242624999557384f},
    {0.0490717860366443f, 0.3479782119131098f, 0.5277650273921529f},
    {0.04910077440233592f, 0.34929984367863964f, 0.5312931441090918f},
    {0.04912496374647964f, 0.35061777846523556f, 0.5348472900437968f},
    {0.049144281939685876f, 0.35193189567631167f, 0.5384279167237124f},
    {0.04915865492929047f, 0.3532420709396423f, 0.5420354876579142f},
    {0.04916800666192803f, 0.3545481759533582f, 0.5456704787448663f},
    {0.04917225900211732f, 0.3558500783243678f, 0.5493333786972924f},
    {0.04917133164659893f, 0.35714764139876426f, 0.553024689485032f},
    {0.0491651420341628f, 0.35844072408375016f, 0.5567449267967906f},
    {0.049153605250673076f, 0.35972918066057785f, 0.5604946205217287f},
    {0.04913663392897654f, 0.36101286058797066f, 0.5642743152519267f},
    {0.04911413814335756f, 0.36229160829545354f, 0.5680845708067875f},
    {0.04908602529819959f, 0.36356526296598163f, 0.5719259627805287f},
    {0.04905220001042406f, 0.36483365830721187f, 0.5757990831139734f},
    {0.04901256398533129f, 0.36609662231071893f, 0.5797045406919258f},
    {0.04896701588534969f, 0.36735397699840217f, 0.5836429619674972f},
    {0.04891545119124254f, 0.36860553815528246f, 0.5876149916148347f},
    {0.04885776205520153f, 0.36985111504782353f, 0.5916212932117864f},
    {0.048793837145294165f, 0.371090510126853f, 0.5956625499541581f},
    {0.048723561480604215f, 0.37232351871408936f, 0.5997394654032839f},
    {0.04864681625641982f, 0.37354992867120285f, 0.6038527642687842f},
    {0.0485634786587359f, 0.37476952005026626f, 0.6080031932284756f},
    {0.04847342166723854f, 0.3759820647243526f, 0.6121915217875443f},
    {0.04837651384597603f, 0.37718732599695254f, 0.6164185431792271f},
    {0.04827261912068898f, 0.3783850581887729f, 0.6206850753093874f},
    {0.04816159654185025f, 0.37957500620037093f, 0.6249919617475522f},
    {0.04804330003224206f, 0.38075690504895116f, 0.6293400727671268f},
    {0.047917578117875524f, 0.3819304793775204f, 0.633730306437712f},
    {0.04778427364089425f, 0.38309544293445374f, 0.6381635897726399f},
    {0.04764322345301101f, 0.38425149802135766f, 0.6426408799350484f},
    {0.04749425808786458f, 0.385398334906948f, 0.6471631655060938f},
    {0.04733720141054259f, 0.3865356312044689f, 0.6517314678190856f},
    {0.04717187024231324f, 0.3876630512099673f, 0.6563468423636755f},
    {0.046998073958454976f, 0.38878024519851034f, 0.6610103802644818f},
    {0.046815614056824016f, 0.3898868486751851f, 0.6657232098388559f},
    {0.04662428369457814f, 0.3909824815774357f, 0.6704864982388766f},
    {0.04642386719018477f, 0.39206674742499825f, 0.6753014531830023f},
    {0.04621413948754389f, 0.39313923241335524f, 0.6801693247832367f},
    {0.045994865578738504f, 0.3941995044462622f, 0.6850914074741193f},
    {0.04576579988147745f, 0.39524711210249736f, 0.6900690420503143f},
    {0.04552668556693947f, 0.3962815835315315f, 0.6951036178201221f},
    {0.04527725383318241f, 0.39730242527232407f, 0.7001965748827989f},
    {0.04501722311872807f, 0.39830912098889804f, 0.7053494065382041f},
    {0.04474629825033485f, 0.39930113011574186f, 0.7105636618379779f},
    {0.044464169518219306f, 0.4002778864054065f, 0.7158409482881979f},
    {0.044170511671191286f, 0.4012387963699213f, 0.7211829347142875f},
    {0.04386498282321687f, 0.4021832376068135f, 0.7265913542998228f},
    {0.04354722326188234f, 0.4031105569995846f, 0.7320680078119023f},
    {0.04321685414797862f, 0.40402006878146585f, 0.7376147670267773f},
    {0.0428734760940282f, 0.40491105245010933f, 0.743233578370643f},
    {0.042516667607970175f, 0.40578275051957646f, 0.748926466791789f},
    {0.04214598338630927f, 0.4066343660945334f, 0.7546955398817109f},
    {0.04176095243886018f, 0.40746506024993384f, 0.7605429922643745f},
    {0.04136107602475044f, 0.40827394919762916f, 0.766471110274553f},
    {0.04094582537627162f, 0.4090601012192915f, 0.7724822769480404f},
    {0.04051463918382638f, 0.40982253334270374f, 0.7785789773486957f},
    {0.040061502782456945f, 0.4105602077358398f, 0.7847638042595603f},
    {0.03959294089889664f, 0.41127202779018696f, 0.7910394642679004f},
    {0.039109793546916495f, 0.4119568338613871f, 0.7974087842769024f},
    {0.03861172210191932f, 0.41261339863144436f, 0.803874718479878f},
    {0.0380983735795864f, 0.4132404220523802f, 0.8104403558364525f},
    {0.03756937968562651f, 0.4138365258262561f, 0.8171089280940507f},
    {0.03702435578736771f, 0.4144002473707861f, 0.8238838184024792f},
    {0.0364628997996382f, 0.4149300332132621f, 0.8307685705742502f},
    {0.03588459097638143f, 0.4154242317480496f, 0.8377668990487521f},
    {0.035288988598694025f, 0.4158810852842974f, 0.844882699624589f},
    {0.03467563054866628f, 0.4162987213006144f, 0.8521200610312002f},
    {0.03404403175731731f, 0.41667514281199364f, 0.8594832774186676f},
    {0.033393682513460185f, 0.41700821774098445f, 0.8669768618532854f},
    {0.03272404661867004f, 0.41729566716967786f, 0.8746055609162682f},
    {0.032034559371859575f, 0.4175350523310705f, 0.8823743705140761f},
    {0.03132462536474723f, 0.41772376017735885f, 0.8902885530212784f},
    {0.03059361606719027f, 0.417858987338036f, 0.8983536558911435f},
    {0.029840867178669222f, 0.41793772225168413f, 0.9065755318852089f},
    {0.02906567571902483f, 0.4179567252211435f, 0.9149603610913213f},
    {0.028267296828018075f, 0.41791250610119823f, 0.9235146749206897f},
    {0.027444940239127507f, 0.41780129927982523f, 0.9322453822980893f},
    {0.026597766388240202f, 0.4176190355565933f, 0.9411597982868389f},
    {0.02572488211232861f, 0.41736131045306674f, 0.9502656754213602f},
    {0.02482533588680886f, 0.41702334840740857f, 0.9595712380560552f},
    {0.023898112542860842f, 0.416599962205498f, 0.9690852200808441f},
    {0.02294212739712791f, 0.41608550687982504f, 0.9788169064013666f},
    {0.02195621971619119f, 0.4154738271597193f, 0.9887761786374855f},
    {0.03533572637548167f, 0.4150344767837667f, 0.9966419438918287f},
    {0.08206748636661013f, 0.4154760610454022f, 0.996875442497312f},
    {0.1131664468320158f, 0.4159292422424467f, 0.9971067037505105f},
    {0.1377759789309851f, 0.4163940123475041f, 0.9973357493609963f},
    {0.1586260932452447f, 0.4168703621191211f, 0.9975626007042689f},
    {0.17695881259992585f, 0.41735828111703227f, 0.997787278826484f},
    {0.19346029551091778f, 0.4178577577177723f, 0.9980098044491156f},
    {0.2085556849234767f, 0.4183687791306285f, 0.9982301979735458f},
    {0.22252938052310162f, 0.41889133141394447f, 0.9984484794855942f},
    {0.2355824089832244f, 0.4194253994917421f, 0.9986646687599702f},
    {0.24786290560296725f, 0.4199709671706614f, 0.9988787852646682f},
    {0.25948364869956886f, 0.42052801715720073f, 0.9990908481652964f},
    {0.2705327829044692f, 0.42109653107524325f, 0.9993008763293371f},
    {0.2810807045979947f, 0.4216764894838623f, 0.9995088883303488f},
    {0.2911846624744039f, 0.4222678718953844f, 0.9997149024521047f},
    {0.30089193496804306f, 0.4228706567937021f, 0.9999189366926701f},
    {0.3199598560384707f, 0.4211529467871777f, 1.0000000000000044f},
    {0.3436114893370144f, 0.4178742172053897f, 1.0000000000000047f},
    {0.36539676089694495f, 0.41458308629177515f, 1.0000000000000044f},
    {0.3856661632570949f, 0.41127775518053283f, 1.0000000000000042f},
    {0.404675301565696f, 0.407956362084171f, 1.0000000000000044f},
    {0.4226172861700883f, 0.4046169767859018f, 1.0000000000000047f},
    {0.43964219386021874f, 0.40125759469274436f, 1.0000000000000047f},
    {0.45586938841351193f, 0.3978761303980185f, 1.0000000000000047f},
    {0.47139565849043324f, 0.39447041069519134f, 1.0000000000000047f},
    {0.4863007849418988f, 0.3910381669772773f, 1.0000000000000047f},
    {0.5006514638539757f, 0.3875770269469873f, 1.0000000000000044f},
    {0.5145041416968924f, 0.3840845055522841f, 1.0000000000000047f},
    {0.5279071095300848f, 0.3805579950497078f, 1.0000000000000047f},
    {0.5409020797263486f, 0.3769947540834305f, 1.0000000000000044f},
    {0.5535253932438766f, 0.3733918956509583f, 1.0000000000000044f},
    {0.5658089579546876f, 0.3697463738064324f, 1.0000000000000042f},
    {0.577780987780821f, 0.366054968928604f, 1.0000000000000049f},
    {0.589466591997403f, 0.3623142713523205f, 1.0000000000000047f},
    {0.6008882502481963f, 0.35852066312849035f, 1.0000000000000044f},
    {0.6120661992793963f, 0.3546702976368881f, 1.0000000000000047f},
    {0.6230187506929341f, 0.35075907672718176f, 1.0000000000000047f},
    {0.6337625542333337f, 0.34678262500419443f, 1.0000000000000047f},
    {0.6443128176539651f, 0.3427362608011279f, 1.0000000000000044f},
    {0.6546834916623888f, 0.33861496329592544f, 1.0000000000000047f},
    {0.664887426552217f, 0.3344133351169368f, 1.0000000000000044f},
    {0.6749365057066918f, 0.3301255596489445f, 1.0000000000000047f},
    {0.6848417600790246f, 0.32574535208217403f, 1.0000000000000047f},
    {0.6946134669261637f, 0.32126590303548275f, 1.0000000000000049f},
    {0.7042612354316643f, 0.31667981331755896f, 1.0000000000000047f},
    {0.7137940813531695f, 0.3119790180493533f, 1.0000000000000049f},
    {0.7232204924365964f, 0.3071546979334297f, 1.0000000000000049f},
    {0.7325484860275505f, 0.30219717488892517f, 1.0000000000000047f},
    {0.7417856600618409f, 0.2970957885292609f, 1.000000000000005f},
    {0.7509392384175178f, 0.2918387489798506f, 1.0000000000000047f},
    {0.760016111449703f, 0.28641296022435003f, 1.0000000000000047f},
    {0.7690228723986646f, 0.2808038063993306f, 1.0000000000000049f},
    {0.7779658502549104f, 0.27499489103633235f, 1.0000000000000049f},
    {0.7868511395774846f, 0.2689677158905533f, 1.0000000000000047f},
    {0.7956846276897148f, 0.26270128126132847f, 1.0000000000000047f},
    {0.804472019617065f, 0.2561715829275765f, 1.0000000000000047f},
    {0.8132188610824966f, 0.2493509709254887f, 1.0000000000000047f},
    {0.8219305598337341f, 0.24220732066040862f, 1.0000000000000049f},
    {0.8306124055427538f, 0.23470294440057987f, 1.0000000000000049f},
    {0.8392695884894237f, 0.2267931361345682f, 1.0000000000000047f},
    {0.847907217217596f, 0.21842418639150069f, 1.0000000000000047f},
    {0.8565303353323375f, 0.20953060994411976f, 1.0000000000000049f},
    {0.8651439375907393f, 0.20003116767718654f, 1.0000000000000049f},
    {0.8737529854254381f, 0.18982297245453064f, 1.0000000000000049f},
    {0.8823624220291222f, 0.17877241522237444f, 1.0000000000000047f},
    {0.8909771871196978f, 0.1667005280966983f, 1.0000000000000047f},
    {0.8996022314990386f, 0.15335795616479617f, 1.000000000000005f},
    {0.9082425315133318f, 0.13837882372526109f, 1.0000000000000049f},
    {0.9169031035195819f, 0.12118667725012405f, 1.0000000000000049f},
    {0.9255890184609986f, 0.10077304980525353f, 1.0000000000000047f},
    {0.9343054166534386f, 0.07504334998300113f, 1.0000000000000049f},
    {0.9430575228859241f, 0.03781952178921804f, 1.000000000000005f},
    {0.9509350420238839f, 1.4218570765223148e-13f, 0.9989984483716071f},
    {0.9554497353124459f, 1.4191675612451605e-13f, 0.9943640499109371f},
    {0.9599176427714787f, 1.4433731987395504e-13f, 0.9897799632511853f},
    {0.9643412154073002f, 1.4245465917994694e-13f, 0.9852425190239346f},
    {0.9687227616942858f, 1.4191675612451605e-13f, 0.9807481714229297f},
    {0.9730644583865243f, 1.411995520506082e-13f, 0.9762934885028384f},
    {0.9773683603724937f, 1.3931689135660008e-13f, 0.9718751430792824f},
    {0.9816364096714153f, 1.3886863881040766e-13f, 0.9674899041721569f},
    {0.9858704436584534f, 1.4039269746746187e-13f, 0.9631346289394122f},
    {0.9900722025959202f, 1.4397871783700112e-13f, 0.9588062550529955f},
    {0.9942433365389557f, 1.4155815408756212e-13f, 0.954501793472642f},
    {0.9983854116765075f, 1.3752388117183045e-13f, 0.9502183215767478f},
    {0.9999999999999819f, 0.02804423714351181f, 0.9437140548413381f},
    {0.9999999999999823f, 0.0675265531658979f, 0.9359017685954015f},
    {0.9999999999999826f, 0.09447578037166751f, 0.9282451825736049f},
    {0.9999999999999823f, 0.11567880450339993f, 0.920737795368809f},
    {0.9999999999999826f, 0.13352190503381375f, 0.9133734552831144f},
    {0.9999999999999823f, 0.1491028314594674f, 0.906146335428585f},
    {0.9999999999999826f, 0.16303259275115084f, 0.8990509109121838f},
    {0.9999999999999826f, 0.17569199214531872f, 0.8920819378992011f},
    {0.9999999999999826f, 0.18733702217610845f, 0.8852344343724449f},
    {0.9999999999999826f, 0.19814940356609517f, 0.8785036624245576f},
    {0.9999999999999823f, 0.20826355122506324f, 0.8718851119384158f},
    {0.9999999999999823f, 0.21778214249596284f, 0.8653744855260821f},
    {0.9999999999999826f, 0.22678566871532468f, 0.8589676846103573f},
    {0.9999999999999823f, 0.2353385863611125f, 0.8526607965450058f},
    {0.9999999999999828f, 0.24349343831907827f, 0.8464500826803465f},
    {0.9999999999999826f, 0.2512937077092952f, 0.840331967290248f},
    {0.9999999999999826f, 0.2587758499993201f, 0.8343030272849384f},
    {0.999999999999983f, 0.26739099502162367f, 0.8275538904243963f},
    {0.999999999999983f, 0.2793555475103376f, 0.8187524096848618f},
    {0.9999999999999828f, 0.29067538241472596f, 0.810154074771914f},
    {0.999999999999983f, 0.3014349177286362f, 0.8017491111724352f},
    {0.9999999999999826f, 0.31170258039783083f, 0.7935283442712853f},
    {0.9999999999999826f, 0.3215347049761315f, 0.7854831467895685f},
    {0.9999999999999826f, 0.3309782925632311f, 0.7776053911816436f},
    {0.9999999999999826f, 0.3400730122474594f, 0.7698874064041857f},
    {0.9999999999999826f, 0.34885268450644075f, 0.7623219385454285f},
    {0.999999999999983f, 0.35734640143399626f, 0.7549021148665397f},
    {0.9999999999999826f, 0.3655793867737775f, 0.7476214108616114f},
    {0.9999999999999826f, 0.3735736659274856f, 0.7404736199894286f},
    {0.9999999999999828f, 0.381348594792351f, 0.7334528257702123f},
    {0.9999999999999826f, 0.38892128210540905f, 0.7265533759748873f},
    {0.9999999999999823f, 0.3963069303390571f, 0.7197698586639263f},
    {0.9999999999999823f, 0.4035191135203492f, 0.7130970798581467f},
    {0.9999999999999823f, 0.410570005644612f, 0.7065300426455539f},
    {0.9999999999999821f, 0.4174705699878856f, 0.700063927546916f},
    {0.9999999999999819f, 0.4242307171780247f, 0.6936940739785828f},
    {0.9999999999999821f, 0.4308594380852102f, 0.6874159626644994f},
    {0.9999999999999821f, 0.4373649162525338f, 0.6812251988606219f},
    {0.9999999999999819f, 0.44375462357781925f, 0.6751174962642902f},
    {0.9999999999999819f, 0.4500354021895003f, 0.6690886614886871f},
    {0.9999999999999821f, 0.45621353486890187f, 0.6631345789884755f},
    {0.9999999999999817f, 0.4622948059133914f, 0.657251196327135f},
    {0.9999999999999817f, 0.4682845539768576f, 0.6514345096795133f},
    {0.9999999999999817f, 0.474187718141824f, 0.645680549464667f},
    {0.9999999999999817f, 0.4800088782535285f, 0.6399853660042518f},
    {0.9999999999999815f, 0.4857522903672667f, 0.6343450151004509f},
    {0.9999999999999815f, 0.4914219180162633f, 0.6287555434246979f},
    {0.9999999999999815f, 0.497021459890778f, 0.6232129736041581f},
    {0.9999999999999815f, 0.5025543744242497f, 0.6177132888869281f},
    {0.9999999999999815f, 0.5080239017046412f, 0.6122524172590773f},
    {0.999999999999981f, 0.5134330830652836f, 0.606826214876734f},
    {0.9999999999999808f, 0.518784778656747f, 0.6014304486641499f},
    {0.9999999999999808f, 0.5240816832574693f, 0.5960607779137368f},
    {0.9999999999999806f, 0.5293263405443853f, 0.5907127347060119f},
    {0.9999999999999806f, 0.5345211560142691f, 0.5853817029456958f},
    {0.9999999999999808f, 0.5396684087209026f, 0.580062895784249f},
    {0.9999999999999808f, 0.5447702619716198f, 0.5747513311680923f},
    {0.9999999999999806f, 0.5498287731085955f, 0.5694418052146554f},
    {0.9999999999999803f, 0.5548459024848833f, 0.5641288630740176f},
    {0.9999999999999801f, 0.5598235217321937f, 0.5588067668806895f},
    {0.9999999999999799f, 0.5647634214064047f, 0.5534694603362047f},
    {0.9999999999999799f, 0.569667318087479f, 0.5481105293861371f},
    {0.9999999999999801f, 0.5745368610026079f, 0.5427231583620321f},
    {0.9999999999999797f, 0.5793736382348097f, 0.5373000808456486f},
    {0.9999999999999797f, 0.5841791825736894f, 0.5318335243749407f},
    {0.9999999999999797f, 0.58895497706055f, 0.5263151479421893f},
    {0.9999999999999795f, 0.5937024602763533f, 0.5207359710263567f},
    {0.9999999999999795f, 0.5984230314181602f, 0.5150862926436902f},
    {0.9999999999999792f, 0.6031180552074987f, 0.5093555985787912f},
    {0.9999999999999792f, 0.607788866672662f, 0.5035324545546109f},
    {0.999999999999979f, 0.6124367758461117f, 0.4976043825895365f},
    {0.999999999999979f, 0.6170630724180334f, 0.4915577171399405f},
    {0.9999999999999788f, 0.6216690303876014f, 0.48537743679248463f},
    {0.9999999999999788f, 0.6262559127547657f, 0.4790469661903673f},
    {0.9999999999999784f, 0.6308249762973255f, 0.4725479414659382f},
    {0.9999999999999786f, 0.6353774764808859f, 0.46585993058805514f},
    {0.9999999999999784f, 0.6399146725529954f, 0.45896009754439654f},
    {0.9999999999999784f, 0.644437832877538f, 0.45182279591800384f},
    {0.9999999999999781f, 0.6489482405714118f, 0.4444190728188997f},
    {0.9999999999999779f, 0.6534471995128909f, 0.4367160577509657f},
    {0.9999999999999779f, 0.6579360408000906f, 0.4286762020035964f},
    {0.9999999999999779f, 0.6624161297489367f, 0.42025632127341656f},
    {0.9999999999999777f, 0.6668888735333959f, 0.41140637540952824f},
    {0.9999999999999777f, 0.6713557295869282f, 0.40206789113388525f},
    {0.9999999999999775f, 0.6758182149038043f, 0.3921718908087272f}};

const size_t calref_n = 256;
const float calref_palette[calref_n][3] = {
    {0.1f, 0.1f, 0.1f},
    {0.36862745098039246f, 0.30980392156862746f, 0.6352941176470588f},
    {0.3618608227604765f, 0.31856978085351784f, 0.6394463667820068f},
    {0.3550941945405613f, 0.3273356401384083f, 0.643598615916955f},
    {0.3483275663206459f, 0.3361014994232987f, 0.647750865051903f},
    {0.3415609381007305f, 0.3448673587081891f, 0.6519031141868512f},
    {0.33479430988081516f, 0.35363321799307956f, 0.6560553633217993f},
    {0.3280276816608997f, 0.36239907727796994f, 0.6602076124567474f},
    {0.3212610534409842f, 0.3711649365628603f, 0.6643598615916955f},
    {0.31449442522106885f, 0.3799307958477509f, 0.6685121107266436f},
    {0.3077277970011534f, 0.38869665513264134f, 0.6726643598615917f},
    {0.300961168781238f, 0.3974625144175317f, 0.6768166089965398f},
    {0.29419454056132255f, 0.4062283737024219f, 0.6809688581314879f},
    {0.2874279123414072f, 0.4149942329873126f, 0.685121107266436f},
    {0.2806612841214917f, 0.4237600922722031f, 0.6892733564013841f},
    {0.27389465590157624f, 0.4325259515570933f, 0.6934256055363321f},
    {0.2671280276816609f, 0.4412918108419839f, 0.6975778546712803f},
    {0.2603613994617455f, 0.45005767012687425f, 0.7017301038062282f},
    {0.25359477124183005f, 0.4588235294117643f, 0.7058823529411765f},
    {0.24682814302191458f, 0.46758938869665506f, 0.7100346020761246f},
    {0.24006151480199922f, 0.4763552479815456f, 0.7141868512110727f},
    {0.23329488658208386f, 0.485121107266436f, 0.7183391003460207f},
    {0.2265282583621684f, 0.49388696655132636f, 0.7224913494809689f},
    {0.21976163014225292f, 0.5026528258362168f, 0.726643598615917f},
    {0.21299500192233756f, 0.5114186851211073f, 0.7307958477508651f},
    {0.20622837370242209f, 0.5201845444059976f, 0.7349480968858132f},
    {0.19946174548250672f, 0.5289504036908883f, 0.7391003460207612f},
    {0.20007689350249913f, 0.5377931564782777f, 0.7393310265282583f},
    {0.2080738177623992f, 0.5467128027681663f, 0.7356401384083046f},
    {0.21607074202229903f, 0.5556324490580544f, 0.7319492502883508f},
    {0.2240676662821992f, 0.5645520953479432f, 0.7282583621683968f},
    {0.23206459054209927f, 0.5734717416378313f, 0.7245674740484429f},
    {0.24006151480199933f, 0.58239138792772f, 0.720876585928489f},
    {0.2480584390618994f, 0.5913110342176088f, 0.7171856978085351f},
    {0.25605536332179935f, 0.6002306805074966f, 0.7134948096885814f},
    {0.2640522875816994f, 0.6091503267973857f, 0.7098039215686275f},
    {0.27204921184159947f, 0.6180699730872741f, 0.7061130334486736f},
    {0.28004613610149953f, 0.6269896193771626f, 0.7024221453287197f},
    {0.2880430603613995f, 0.6359092656670511f, 0.6987312572087658f},
    {0.29603998462129966f, 0.6448289119569397f, 0.695040369088812f},
    {0.3040369088811996f, 0.6537485582468282f, 0.6913494809688581f},
    {0.3120338331410998f, 0.6626682045367166f, 0.6876585928489042f},
    {0.32003075740099973f, 0.671587850826605f, 0.6839677047289503f},
    {0.3280276816608998f, 0.6805074971164937f, 0.6802768166089965f},
    {0.33602460592079986f, 0.6894271434063821f, 0.6765859284890426f},
    {0.3440215301806999f, 0.6983467896962707f, 0.6728950403690888f},
    {0.35201845444059976f, 0.7072664359861591f, 0.6692041522491351f},
    {0.36001537870050004f, 0.7161860822760477f, 0.6655132641291811f},
    {0.3680123029604f, 0.7251057285659362f, 0.6618223760092272f},
    {0.37600922722029995f, 0.7340253748558248f, 0.6581314878892734f},
    {0.3840061514802f, 0.7429450211457131f, 0.6544405997693193f},
    {0.39200307574010007f, 0.7518646674356018f, 0.6507497116493657f},
    {0.40000000000000036f, 0.7607843137254902f, 0.6470588235294117f},
    {0.4106113033448675f, 0.7649365628604383f, 0.6469050365244137f},
    {0.42122260668973477f, 0.7690888119953864f, 0.6467512495194156f},
    {0.43183391003460214f, 0.7732410611303345f, 0.6465974625144175f},
    {0.4424452133794696f, 0.7773933102652826f, 0.6464436755094196f},
    {0.4530565167243371f, 0.7815455594002306f, 0.6462898885044215f},
    {0.46366782006920415f, 0.7856978085351789f, 0.6461361014994234f},
    {0.4742791234140715f, 0.7898500576701271f, 0.6459823144944252f},
    {0.4848904267589389f, 0.794002306805075f, 0.6458285274894271f},
    {0.49550173010380627f, 0.7981545559400232f, 0.645674740484429f},
    {0.5061130334486739f, 0.8023068050749711f, 0.6455209534794312f},
    {0.5167243367935411f, 0.8064590542099194f, 0.645367166474433f},
    {0.5273356401384084f, 0.8106113033448674f, 0.6452133794694349f},
    {0.5379469434832758f, 0.8147635524798154f, 0.6450595924644369f},
    {0.548558246828143f, 0.8189158016147636f, 0.6449058054594388f},
    {0.5591695501730105f, 0.8230680507497117f, 0.6447520184544406f},
    {0.5697808535178779f, 0.8272202998846598f, 0.6445982314494427f},
    {0.5803921568627453f, 0.831372549019608f, 0.6444444444444446f},
    {0.5910034602076126f, 0.8355247981545562f, 0.6442906574394465f},
    {0.60161476355248f, 0.8396770472895041f, 0.6441368704344483f},
    {0.6122260668973473f, 0.8438292964244521f, 0.6439830834294502f},
    {0.6228373702422147f, 0.8479815455594002f, 0.6438292964244523f},
    {0.633448673587082f, 0.8521337946943485f, 0.6436755094194541f},
    {0.6440599769319493f, 0.8562860438292964f, 0.6435217224144562f},
    {0.6546712802768165f, 0.8604382929642447f, 0.6433679354094579f},
    {0.6652825836216838f, 0.8645905420991928f, 0.6432141484044598f},
    {0.675124951941561f, 0.8685121107266438f, 0.6422145328719724f},
    {0.6841983852364476f, 0.8722029988465975f, 0.6403690888119954f},
    {0.6932718185313342f, 0.8758938869665513f, 0.6385236447520186f},
    {0.7023452518262208f, 0.8795847750865051f, 0.6366782006920415f},
    {0.7114186851211074f, 0.8832756632064591f, 0.6348327566320646f},
    {0.7204921184159938f, 0.8869665513264131f, 0.6329873125720877f},
    {0.7295655517108806f, 0.890657439446367f, 0.6311418685121105f},
    {0.7386389850057672f, 0.8943483275663208f, 0.6292964244521339f},
    {0.7477124183006536f, 0.8980392156862746f, 0.6274509803921569f},
    {0.7567858515955403f, 0.9017301038062284f, 0.62560553633218f},
    {0.7658592848904268f, 0.9054209919261822f, 0.6237600922722031f},
    {0.7749327181853134f, 0.909111880046136f, 0.6219146482122262f},
    {0.7840061514802001f, 0.9128027681660901f, 0.6200692041522492f},
    {0.7930795847750867f, 0.916493656286044f, 0.618223760092272f},
    {0.8021530180699734f, 0.920184544405998f, 0.6163783160322951f},
    {0.8112264513648599f, 0.9238754325259518f, 0.6145328719723183f},
    {0.8202998846597466f, 0.9275663206459055f, 0.6126874279123413f},
    {0.8293733179546331f, 0.9312572087658594f, 0.6108419838523645f},
    {0.8384467512495197f, 0.9349480968858133f, 0.6089965397923875f},
    {0.8475201845444063f, 0.9386389850057671f, 0.6071510957324106f},
    {0.8565936178392928f, 0.9423298731257211f, 0.6053056516724337f},
    {0.8656670511341793f, 0.9460207612456747f, 0.6034602076124568f},
    {0.874740484429066f, 0.9497116493656288f, 0.6016147635524798f},
    {0.8838139177239525f, 0.9534025374855826f, 0.5997693194925027f},
    {0.8928873510188393f, 0.9570934256055367f, 0.5979238754325257f},
    {0.9019607843137256f, 0.9607843137254903f, 0.5960784313725491f},
    {0.9058054594386773f, 0.962322183775471f, 0.6020761245674742f},
    {0.9096501345636295f, 0.9638600538254517f, 0.6080738177623993f},
    {0.9134948096885813f, 0.9653979238754326f, 0.6140715109573244f},
    {0.9173394848135333f, 0.9669357939254133f, 0.6200692041522493f},
    {0.9211841599384853f, 0.9684736639753941f, 0.6260668973471741f},
    {0.9250288350634372f, 0.9700115340253751f, 0.6320645905420991f},
    {0.9288735101883892f, 0.9715494040753557f, 0.6380622837370243f},
    {0.932718185313341f, 0.9730872741253366f, 0.6440599769319492f},
    {0.9365628604382931f, 0.9746251441753172f, 0.6500576701268744f},
    {0.9404075355632451f, 0.9761630142252982f, 0.6560553633217994f},
    {0.9442522106881969f, 0.9777008842752788f, 0.6620530565167244f},
    {0.9480968858131487f, 0.9792387543252595f, 0.6680507497116493f},
    {0.9519415609381008f, 0.9807766243752404f, 0.6740484429065746f},
    {0.9557862360630527f, 0.9823144944252212f, 0.6800461361014994f},
    {0.9596309111880046f, 0.9838523644752019f, 0.6860438292964245f},
    {0.9634755863129567f, 0.9853902345251826f, 0.6920415224913494f},
    {0.9673202614379086f, 0.9869281045751634f, 0.6980392156862747f},
    {0.9711649365628605f, 0.9884659746251442f, 0.7040369088811996f},
    {0.9750096116878124f, 0.9900038446751249f, 0.7100346020761246f},
    {0.9788542868127644f, 0.9915417147251058f, 0.7160322952710494f},
    {0.9826989619377164f, 0.9930795847750866f, 0.7220299884659747f},
    {0.9865436370626683f, 0.9946174548250674f, 0.7280276816608996f},
    {0.9903883121876201f, 0.9961553248750481f, 0.7340253748558248f},
    {0.9942329873125721f, 0.9976931949250287f, 0.7400230680507498f},
    {0.9980776624375239f, 0.9992310649750095f, 0.746020761245675f},
    {0.9999231064975008f, 0.9976163014225297f, 0.7450211457131873f},
    {0.9997693194925027f, 0.9928489042675892f, 0.7370242214532873f},
    {0.9996155324875048f, 0.988081507112649f, 0.729027297193387f},
    {0.9994617454825068f, 0.9833141099577085f, 0.7210303729334873f},
    {0.9993079584775085f, 0.9785467128027682f, 0.7130334486735873f},
    {0.9991541714725107f, 0.9737793156478278f, 0.7050365244136869f},
    {0.9990003844675125f, 0.9690119184928874f, 0.697039600153787f},
    {0.9988465974625144f, 0.9642445213379468f, 0.6890426758938869f},
    {0.9986928104575163f, 0.9594771241830067f, 0.681045751633987f},
    {0.9985390234525182f, 0.9547097270280661f, 0.6730488273740869f},
    {0.9983852364475202f, 0.9499423298731258f, 0.6650519031141869f},
    {0.9982314494425222f, 0.9451749327181854f, 0.6570549788542868f},
    {0.998077662437524f, 0.9404075355632449f, 0.6490580545943867f},
    {0.9979238754325258f, 0.9356401384083044f, 0.6410611303344868f},
    {0.9977700884275279f, 0.930872741253364f, 0.6330642060745867f},
    {0.9976163014225298f, 0.9261053440984237f, 0.6250672818146866f},
    {0.9974625144175316f, 0.9213379469434833f, 0.6170703575547868f},
    {0.9973087274125335f, 0.9165705497885427f, 0.6090734332948867f},
    {0.9971549404075356f, 0.9118031526336023f, 0.6010765090349864f},
    {0.9970011534025374f, 0.907035755478662f, 0.5930795847750865f},
    {0.9968473663975395f, 0.9022683583237218f, 0.5850826605151866f},
    {0.9966935793925413f, 0.8975009611687812f, 0.5770857362552864f},
    {0.9965397923875433f, 0.892733564013841f, 0.5690888119953864f},
    {0.9963860053825454f, 0.8879661668589005f, 0.5610918877354861f},
    {0.9962322183775473f, 0.88319876970396f, 0.5530949634755861f},
    {0.996078431372549f, 0.8784313725490196f, 0.5450980392156861f},
    {0.9959246443675508f, 0.8707420222991156f, 0.538638985005767f},
    {0.9957708573625528f, 0.8630526720492118f, 0.5321799307958477f},
    {0.9956170703575548f, 0.855363321799308f, 0.5257208765859284f},
    {0.9954632833525567f, 0.847673971549404f, 0.519261822376009f},
    {0.9953094963475586f, 0.8399846212995001f, 0.5128027681660898f},
    {0.9951557093425605f, 0.8322952710495963f, 0.5063437139561706f},
    {0.9950019223375625f, 0.8246059207996924f, 0.4998846597462513f},
    {0.9948481353325646f, 0.8169165705497885f, 0.4934256055363321f},
    {0.9946943483275664f, 0.8092272202998847f, 0.48696655132641264f},
    {0.9945405613225683f, 0.8015378700499808f, 0.48050749711649365f},
    {0.9943867743175702f, 0.7938485198000771f, 0.47404844290657466f},
    {0.9942329873125721f, 0.7861591695501731f, 0.4675893886966551f},
    {0.994079200307574f, 0.7784698193002692f, 0.4611303344867359f},
    {0.993925413302576f, 0.7707804690503652f, 0.4546712802768166f},
    {0.9937716262975778f, 0.7630911188004613f, 0.4482122260668975f},
    {0.99361783929258f, 0.7554017685505575f, 0.44175317185697793f},
    {0.9934640522875816f, 0.7477124183006536f, 0.43529411764705894f},
    {0.9933102652825835f, 0.7400230680507496f, 0.4288350634371395f},
    {0.9931564782775857f, 0.7323337178008458f, 0.4223760092272202f},
    {0.9930026912725872f, 0.724644367550942f, 0.4159169550173013f},
    {0.9928489042675894f, 0.716955017301038f, 0.40945790080738176f},
    {0.9926951172625912f, 0.7092656670511341f, 0.40299884659746266f},
    {0.9925413302575933f, 0.7015763168012303f, 0.3965397923875432f},
    {0.9923875432525952f, 0.6938869665513263f, 0.390080738177624f},
    {0.992233756247597f, 0.6861976163014225f, 0.3836216839677048f},
    {0.9914648212226067f, 0.677354863514033f, 0.3780853517877738f},
    {0.990080738177624f, 0.6673587081891583f, 0.3734717416378317f},
    {0.9886966551326414f, 0.6573625528642829f, 0.36885813148788904f},
    {0.9873125720876587f, 0.647366397539408f, 0.36424452133794694f},
    {0.985928489042676f, 0.6373702422145329f, 0.3596309111880045f},
    {0.9845444059976933f, 0.6273740868896578f, 0.35501730103806217f},
    {0.9831603229527106f, 0.6173779315647828f, 0.35040369088811985f},
    {0.981776239907728f, 0.6073817762399077f, 0.3457900807381774f},
    {0.9803921568627451f, 0.5973856209150328f, 0.3411764705882355f},
    {0.9790080738177624f, 0.5873894655901575f, 0.33656286043829287f},
    {0.9776239907727798f, 0.5773933102652827f, 0.33194925028835065f},
    {0.976239907727797f, 0.5673971549404075f, 0.3273356401384082f},
    {0.9748558246828143f, 0.5574009996155325f, 0.3227220299884661f},
    {0.9734717416378316f, 0.5474048442906574f, 0.3181084198385238f},
    {0.9720876585928488f, 0.5374086889657824f, 0.31349480968858146f},
    {0.9707035755478661f, 0.5274125336409075f, 0.30888119953863913f},
    {0.9693194925028835f, 0.5174163783160323f, 0.3042675893886967f},
    {0.9679354094579009f, 0.5074202229911575f, 0.2996539792387545f},
    {0.9665513264129182f, 0.4974240676662822f, 0.2950403690888119f},
    {0.9651672433679354f, 0.4874279123414072f, 0.2904267589388697f},
    {0.9637831603229527f, 0.47743175701653207f, 0.2858131487889273f},
    {0.9623990772779699f, 0.4674356016916571f, 0.28119953863898506f},
    {0.9610149942329872f, 0.4574394463667821f, 0.27658592848904273f},
    {0.9596309111880046f, 0.447443291041907f, 0.2719723183391005f},
    {0.958246828143022f, 0.43744713571703187f, 0.2673587081891581f},
    {0.9568627450980394f, 0.42745098039215673f, 0.26274509803921564f},
    {0.9520953479430986f, 0.42022299115724726f, 0.26459054209919286f},
    {0.9473279507881583f, 0.4129950019223377f, 0.26643598615916975f},
    {0.9425605536332179f, 0.40576701268742793f, 0.26828143021914663f},
    {0.9377931564782777f, 0.39853902345251835f, 0.2701268742791235f},
    {0.9330257593233372f, 0.3913110342176086f, 0.2719723183391004f},
    {0.928258362168397f, 0.38408304498269874f, 0.27381776239907707f},
    {0.9234909650134564f, 0.3768550557477892f, 0.2756632064590543f},
    {0.9187235678585162f, 0.36962706651287985f, 0.2775086505190312f},
    {0.9139561707035755f, 0.36239907727797027f, 0.2793540945790083f},
    {0.9091887735486351f, 0.3551710880430604f, 0.28119953863898506f},
    {0.9044213763936948f, 0.34794309880815055f, 0.28304498269896183f},
    {0.8996539792387543f, 0.340715109573241f, 0.28489042675893894f},
    {0.8948865820838141f, 0.3334871203383314f, 0.28673587081891583f},
    {0.8901191849288733f, 0.326259131103422f, 0.28858131487889305f},
    {0.8853517877739333f, 0.319031141868512f, 0.2904267589388695f},
    {0.8805843906189926f, 0.3118031526336025f, 0.2922722029988466f},
    {0.8758169934640523f, 0.30457516339869284f, 0.2941176470588236f},
    {0.871049596309112f, 0.2973471741637831f, 0.2959630911188005f},
    {0.8662821991541714f, 0.29011918492887356f, 0.2978085351787776f},
    {0.8615148019992313f, 0.2828911956939638f, 0.2996539792387542f},
    {0.8567474048442908f, 0.2756632064590542f, 0.30149942329873114f},
    {0.8519800076893502f, 0.26843521722414465f, 0.3033448673587078f},
    {0.84721261053441f, 0.2612072279892348f, 0.3051903114186846f},
    {0.8424452133794698f, 0.2539792387543254f, 0.3070357554786618f},
    {0.8376778162245291f, 0.24675124951941552f, 0.3088811995386387f},
    {0.8310649750096116f, 0.23844675124951953f, 0.30880430603613984f},
    {0.8226066897347173f, 0.22906574394463686f, 0.3068050749711647f},
    {0.8141484044598231f, 0.21968473663975407f, 0.30480584390618953f},
    {0.8056901191849288f, 0.2103037293348713f, 0.30280661284121513f},
    {0.7972318339100345f, 0.20092272202998862f, 0.30080738177624006f},
    {0.7887735486351404f, 0.19154171472510573f, 0.2988081507112643f},
    {0.7803152633602461f, 0.18216070742022294f, 0.2968089196462894f},
    {0.7718569780853518f, 0.17277970011534027f, 0.2948096885813144f},
    {0.7633986928104576f, 0.1633986928104575f, 0.29281045751633983f},
    {0.7549404075355632f, 0.15401768550557482f, 0.29081122645136503f},
    {0.746482122260669f, 0.14463667820069204f, 0.2888119953863893f},
    {0.7380238369857748f, 0.13525567089580925f, 0.2868127643214149f},
    {0.7295655517108804f, 0.12587466359092658f, 0.2848135332564399f},
    {0.7211072664359862f, 0.1164936562860438f, 0.28281430219146436f},
    {0.7126489811610921f, 0.10711264898116135f, 0.2808150711264894f},
    {0.7041906958861976f, 0.09773164167627835f, 0.2788158400615146f},
    {0.6957324106113034f, 0.08835063437139556f, 0.27681660899653987f},
    {0.6872741253364091f, 0.07896962706651289f, 0.2748173779315646f},
    {0.6788158400615149f, 0.06958861976163011f, 0.27281814686658995f},
    {0.6703575547866205f, 0.06020761245674744f, 0.2708189158016141f},
    {0.6618992695117263f, 0.05082660515186466f, 0.26881968473663936f},
    {0.6534409842368321f, 0.041445597846981874f, 0.2668204536716644f},
    {0.6449826989619377f, 0.0320645905420992f, 0.26482122260668983f},
    {0.6365244136870435f, 0.02268358323721642f, 0.2628219915417146f},
    {0.6280661284121491f, 0.013302575932333749f, 0.26082276047673913f}};

}  // namespace viz
}  // namespace ouster
