/**
 * Copyright (c) 2021, Ouster, Inc.
 * All rights reserved.
 *
 * @file
 * @brief ouster_pyclient_pcap python module
 */
#include <pybind11/pybind11.h>

#include <cmath>
#include <sstream>
#include <string>

#include "ouster/impl/build.h"
#include "ouster/os_pcap.h"

using namespace ouster::sensor_utils;
namespace py = pybind11;

PYBIND11_MAKE_OPAQUE(std::shared_ptr<playback_handle>);
PYBIND11_MAKE_OPAQUE(std::shared_ptr<record_handle>);

PYBIND11_PLUGIN(_pcap) {
    py::module m("_pcap", R"(Pcap bindings generated by pybind11.

This module is generated from the C++ code and not meant to be used directly.
)");

    // turn off signatures in docstrings: mypy stubs provide better types
    py::options options;
    options.disable_function_signatures();

    py::class_<packet_info, std::shared_ptr<packet_info>>(m, "packet_info")
        .def(py::init<>())
        .def("__repr__",
             [](const packet_info& data) {
                 std::stringstream result;
                 result << data;
                 return result.str();
             })
        .def_readonly("dst_ip", &packet_info::dst_ip)
        .def_readonly("src_ip", &packet_info::src_ip)
        .def_readonly("dst_port", &packet_info::dst_port)
        .def_readonly("src_port", &packet_info::src_port)
        .def_readonly("payload_size", &packet_info::payload_size)
        .def_property_readonly("timestamp",
                               [](packet_info& packet_info) -> double {
                                   return packet_info.timestamp.count() / 1e6;
                               })
        .def_readonly("fragments_in_packet", &packet_info::fragments_in_packet)
        .def_readonly("ip_version", &packet_info::ip_version)
        .def_readonly("encapsulation_protocol",
                      &packet_info::encapsulation_protocol)
        .def_readonly("network_protocol", &packet_info::network_protocol);

    // pcap reading
    py::class_<std::shared_ptr<playback_handle>>(m, "playback_handle");

    m.def("replay_initialize", &replay_initialize);

    m.def("replay_uninitialize", [](std::shared_ptr<playback_handle>& handle) {
        replay_uninitialize(*handle);
    });

    m.def("next_packet_info",
          [](std::shared_ptr<playback_handle>& handle, packet_info& packet_info)
              -> bool { return next_packet_info(*handle, packet_info); });

    m.def(
        "read_packet",
        [](std::shared_ptr<playback_handle>& handle, py::buffer buf) -> size_t {
            auto info = buf.request();
            if (info.format != py::format_descriptor<uint8_t>::format()) {
                throw std::invalid_argument(
                    "Incompatible argument: expected a bytearray");
            }
            return read_packet(*handle, static_cast<uint8_t*>(info.ptr),
                               info.size);
        });

    m.def("replay_reset", [](std::shared_ptr<playback_handle>& handle) {
        replay_reset(*handle);
    });

    // pcap writing
    py::class_<std::shared_ptr<record_handle>>(m, "record_handle");

    m.def("record_initialize",
          py::overload_cast<const std::string&, const std::string&,
                            const std::string&, int, bool>(&record_initialize),
          py::arg("file_name"), py::arg("src_ip"), py::arg("dst_ip"),
          py::arg("frag_size"), py::arg("use_sll_encapsulation") = false,
          R"(
                ``def record_initialize(file_name: str, src_ip: str, dst_ip: str, frag_size: int,
                      use_sll_encapsulation: bool = ...) -> record_handle:``
                  
                  Initialize record handle for single sensor pcap files

            )");

    m.def("record_uninitialize", [](std::shared_ptr<record_handle>& handle) {
        record_uninitialize(*handle);
    });

    m.def("record_packet",
          [](std::shared_ptr<record_handle>& handle, int src_port, int dst_port,
             py::buffer buf, double timestamp) {
              auto info = buf.request();
              if (info.format != py::format_descriptor<uint8_t>::format()) {
                  throw std::invalid_argument(
                      "Incompatible argument: expected a bytearray");
              }
              record_packet(*handle, src_port, dst_port,
                            static_cast<uint8_t*>(info.ptr), info.size,
                            llround(timestamp * 1e6));
          });

    m.attr("__version__") = ouster::SDK_VERSION;

    return m.ptr();
}
