"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const tmp = require("tmp");
const fs = require("fs");
class Seeder extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.setup || !Array.isArray(props.setup))
            throw new Error('setup value must be an array of JSON objects');
        this.props = props;
        const destinationBucket = new aws_s3_1.Bucket(this, 'acds-bucket', {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        tmp.setGracefulCleanup();
        tmp.dir((err, dir) => {
            if (err)
                throw err;
            this.writeTempFile(dir, 'setup.json', props.setup);
            if (props.teardown) {
                this.writeTempFile(dir, 'teardown.json', props.teardown);
            }
            new aws_s3_deployment_1.BucketDeployment(this, id, {
                sources: [aws_s3_deployment_1.Source.asset(dir)],
                destinationBucket,
                retainOnDelete: false,
            });
        });
        const fn = new aws_lambda_1.Function(this, 'handler', {
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            code: aws_lambda_1.Code.fromInline(`
console.log('function loaded');

const AWS = require('aws-sdk');
const s3 = new AWS.S3();

const writeTypeFromAction = (action) => {
  if (action === "put")
    return "Item";
  if (action === "delete")
    return "Key";
}

const run = async (filename, action) => {
  console.log('reading from s3');
  const data = await s3.getObject({
    Bucket: "${destinationBucket.bucketName}", 
    Key: filename
  }).promise();
  console.log('finished reading from s3');
  
  console.log('transforming seed data');
  const seed = JSON.parse(data.Body.toString());
  console.log('finished transforming seed data');
  
  const documentClient = new AWS.DynamoDB.DocumentClient();
  console.log('sending data to dynamodb');
  for(let i = 0; i < seed.length;i++) {
    await documentClient[action]({
      TableName: '${props.table.tableName}',
      [writeTypeFromAction(action)]: seed[i]
    }).promise();
  };
  console.log('finished sending data to dynamodb');
}

exports.handler = async (event) => {
  if (event.mode === "delete" || event.mode === "update")
    await run("teardown.json", "delete");
  if (event.mode === "create" || event.mode === "update")
    await run("setup.json", "put");
}`),
        });
        destinationBucket.grantRead(fn);
        props.table.grantWriteData(fn);
        const onEvent = new custom_resources_1.AwsCustomResource(this, 'on-event', {
            onCreate: {
                ...this.callLambdaOptions(),
                parameters: {
                    FunctionName: fn.functionArn,
                    InvokeArgs: JSON.stringify({
                        mode: 'create',
                    }),
                },
            },
            onDelete: props.teardown
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'delete',
                        }),
                    },
                }
                : undefined,
            onUpdate: props.refreshOnUpdate
                ? {
                    ...this.callLambdaOptions(),
                    parameters: {
                        FunctionName: fn.functionArn,
                        InvokeArgs: JSON.stringify({
                            mode: 'update',
                        }),
                    },
                }
                : undefined,
        });
        fn.grantInvoke(onEvent);
    }
    callLambdaOptions() {
        return {
            service: 'Lambda',
            action: 'invokeAsync',
            apiVersion: '2015-03-31',
            physicalResourceId: `${this.props.table.tableArn}-seeder`,
        };
    }
    writeTempFile(dir, filename, data) {
        const buffer = Buffer.from(JSON.stringify(data));
        const filepath = dir + '/' + filename;
        fs.writeFileSync(filepath, buffer);
    }
}
exports.Seeder = Seeder;
//# sourceMappingURL=data:application/json;base64,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