"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const have_resource_1 = require("@aws-cdk/assert/lib/assertions/have-resource");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('User Pool', () => {
    test('default setup', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: true,
                InviteMessageTemplate: have_resource_1.ABSENT,
            },
            EmailVerificationMessage: 'The verification code to your new account is {####}',
            EmailVerificationSubject: 'Verify your new account',
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_CODE',
                EmailMessage: 'The verification code to your new account is {####}',
                EmailSubject: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
            SmsConfiguration: {
                SnsCallerArn: {
                    'Fn::GetAtt': ['PoolsmsRoleC3352CE6', 'Arn'],
                },
                ExternalId: 'Pool',
            },
            lambdaTriggers: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: 'sts:AssumeRole',
                        Condition: {
                            StringEquals: {
                                'sts:ExternalId': 'Pool',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cognito-idp.amazonaws.com',
                        },
                    },
                ],
                Version: '2012-10-17',
            },
            Policies: [
                {
                    PolicyDocument: {
                        Statement: [
                            {
                                Action: 'sns:Publish',
                                Effect: 'Allow',
                                Resource: '*',
                            },
                        ],
                        Version: '2012-10-17',
                    },
                    PolicyName: 'sns-publish',
                },
            ],
        });
    });
    test('self sign up option is correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            selfSignUpEnabled: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                AllowAdminCreateUserOnly: false,
            },
        });
    });
    test('email verification via link is configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userVerification: {
                emailStyle: lib_1.VerificationEmailStyle.LINK,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailVerificationMessage: have_resource_1.ABSENT,
            EmailVerificationSubject: have_resource_1.ABSENT,
            SmsVerificationMessage: 'The verification code to your new account is {####}',
            VerificationMessageTemplate: {
                DefaultEmailOption: 'CONFIRM_WITH_LINK',
                EmailMessageByLink: 'Verify your account by clicking on {##Verify Email##}',
                EmailSubjectByLink: 'Verify your new account',
                SmsMessage: 'The verification code to your new account is {####}',
            },
        });
    }),
        test('email and sms verification messages are validated', () => {
            const stack = new core_1.Stack();
            expect(() => new lib_1.UserPool(stack, 'Pool1', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'invalid email body',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool2', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    emailBody: 'valid email body {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool3', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message',
                },
            })).toThrow(/SMS message/);
            expect(() => new lib_1.UserPool(stack, 'Pool4', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.CODE,
                    smsMessage: 'invalid sms message {####}',
                },
            })).not.toThrow();
            expect(() => new lib_1.UserPool(stack, 'Pool5', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {####}',
                },
            })).toThrow(/Verification email body/);
            expect(() => new lib_1.UserPool(stack, 'Pool6', {
                userVerification: {
                    emailStyle: lib_1.VerificationEmailStyle.LINK,
                    emailBody: 'invalid email body {##Verify Email##}',
                },
            })).not.toThrow();
        });
    test('user invitation messages are configured correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            userInvitation: {
                emailBody: 'invitation email body',
                emailSubject: 'invitation email subject',
                smsMessage: 'invitation sms',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AdminCreateUserConfig: {
                InviteMessageTemplate: {
                    EmailMessage: 'invitation email body',
                    EmailSubject: 'invitation email subject',
                    SMSMessage: 'invitation sms',
                },
            },
        });
    });
    test('smsRole property is recognized', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = aws_iam_1.Role.fromRoleArn(stack, 'smsRole', 'arn:aws:iam::664773442901:role/sms-role');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            smsRole: role,
            smsRoleExternalId: 'test-external-id',
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            SmsConfiguration: {
                ExternalId: 'test-external-id',
                SnsCallerArn: role.roleArn,
            },
        });
    });
    test('import using id', () => {
        // GIVEN
        const stack = new core_1.Stack(undefined, undefined, {
            env: { region: 'some-region-1', account: '0123456789012' },
        });
        const userPoolId = 'test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolId(stack, 'userpool', userPoolId);
        expect(pool.userPoolId).toEqual(userPoolId);
        expect(pool.userPoolArn).toMatch(/cognito-idp:some-region-1:0123456789012:userpool\/test-user-pool/);
    });
    test('import using arn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userPoolArn = 'arn:aws:cognito-idp:us-east-1:0123456789012:userpool/test-user-pool';
        // WHEN
        const pool = lib_1.UserPool.fromUserPoolArn(stack, 'userpool', userPoolArn);
        expect(pool.userPoolId).toEqual('test-user-pool');
        expect(stack.resolve(pool.userPoolArn)).toEqual({
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':cognito-idp:',
                    { Ref: 'AWS::Region' },
                    ':',
                    { Ref: 'AWS::AccountId' },
                    ':userpool/test-user-pool',
                ]],
        });
    });
    test('support tags', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool', {
            userPoolName: 'myPool',
        });
        core_1.Tag.add(pool, 'PoolTag', 'PoolParty');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'myPool',
            UserPoolTags: {
                PoolTag: 'PoolParty',
            },
        });
    });
    test('lambda triggers via properties are correctly configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = fooFunction(stack, 'preSignUp');
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            lambdaTriggers: {
                preSignUp: fn,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                PreSignUp: stack.resolve(fn.functionArn),
            },
        });
        expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            FunctionName: stack.resolve(fn.functionArn),
            Principal: 'cognito-idp.amazonaws.com',
        });
    });
    test('add* API correctly appends triggers', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const createAuthChallenge = fooFunction(stack, 'createAuthChallenge');
        const customMessage = fooFunction(stack, 'customMessage');
        const defineAuthChallenge = fooFunction(stack, 'defineAuthChallenge');
        const postAuthentication = fooFunction(stack, 'postAuthentication');
        const postConfirmation = fooFunction(stack, 'postConfirmation');
        const preAuthentication = fooFunction(stack, 'preAuthentication');
        const preSignUp = fooFunction(stack, 'preSignUp');
        const preTokenGeneration = fooFunction(stack, 'preTokenGeneration');
        const userMigration = fooFunction(stack, 'userMigration');
        const verifyAuthChallengeResponse = fooFunction(stack, 'verifyAuthChallengeResponse');
        // WHEN
        const pool = new lib_1.UserPool(stack, 'Pool');
        pool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, createAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.CUSTOM_MESSAGE, customMessage);
        pool.addTrigger(lib_1.UserPoolOperation.DEFINE_AUTH_CHALLENGE, defineAuthChallenge);
        pool.addTrigger(lib_1.UserPoolOperation.POST_AUTHENTICATION, postAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.POST_CONFIRMATION, postConfirmation);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_AUTHENTICATION, preAuthentication);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_SIGN_UP, preSignUp);
        pool.addTrigger(lib_1.UserPoolOperation.PRE_TOKEN_GENERATION, preTokenGeneration);
        pool.addTrigger(lib_1.UserPoolOperation.USER_MIGRATION, userMigration);
        pool.addTrigger(lib_1.UserPoolOperation.VERIFY_AUTH_CHALLENGE_RESPONSE, verifyAuthChallengeResponse);
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            LambdaConfig: {
                CreateAuthChallenge: stack.resolve(createAuthChallenge.functionArn),
                CustomMessage: stack.resolve(customMessage.functionArn),
                DefineAuthChallenge: stack.resolve(defineAuthChallenge.functionArn),
                PostAuthentication: stack.resolve(postAuthentication.functionArn),
                PostConfirmation: stack.resolve(postConfirmation.functionArn),
                PreAuthentication: stack.resolve(preAuthentication.functionArn),
                PreSignUp: stack.resolve(preSignUp.functionArn),
                PreTokenGeneration: stack.resolve(preTokenGeneration.functionArn),
                UserMigration: stack.resolve(userMigration.functionArn),
                VerifyAuthChallengeResponse: stack.resolve(verifyAuthChallengeResponse.functionArn),
            },
        });
        [createAuthChallenge, customMessage, defineAuthChallenge, postAuthentication,
            postConfirmation, preAuthentication, preSignUp, preTokenGeneration, userMigration,
            verifyAuthChallengeResponse].forEach((fn) => {
            expect(stack).toHaveResourceLike('AWS::Lambda::Permission', {
                Action: 'lambda:InvokeFunction',
                FunctionName: stack.resolve(fn.functionArn),
                Principal: 'cognito-idp.amazonaws.com',
            });
        });
    });
    test('fails when the same trigger is added twice', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const userpool = new lib_1.UserPool(stack, 'Pool');
        const fn1 = new lambda.Function(stack, 'fn1', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        const fn2 = new lambda.Function(stack, 'fn2', {
            code: lambda.Code.fromInline('foo'),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
        });
        // WHEN
        userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn1);
        expect(() => userpool.addTrigger(lib_1.UserPoolOperation.CREATE_AUTH_CHALLENGE, fn2)).toThrow(/already exists/);
    });
    test('no username aliases specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool');
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('fails when preferredUsername is used without username', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { preferredUsername: true },
        })).toThrow(/username/);
    });
    test('username and email are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true, email: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: have_resource_1.ABSENT,
            AliasAttributes: ['email'],
        });
    });
    test('email and phone number are specified as the username aliases', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameAttributes: ['email', 'phone_number'],
            AliasAttributes: have_resource_1.ABSENT,
        });
    });
    test('email and phone number are autoverified, by default, if they are specified as signIn', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            signInAliases: { email: true },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            signInAliases: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            AutoVerifiedAttributes: ['email'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('explicit autoverify are correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInAliases: { username: true },
            autoVerify: { email: true, phone: true },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            AutoVerifiedAttributes: ['email', 'phone_number'],
        });
    });
    test('sign in case sensitive is correctly picked up', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            signInCaseSensitive: false,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: {
                CaseSensitive: false,
            },
        });
    });
    test('sign in case sensitive is absent by default', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {});
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UsernameConfiguration: have_resource_1.ABSENT,
        });
    });
    test('required attributes', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            requiredAttributes: {
                fullname: true,
                timezone: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'name',
                    Required: true,
                },
                {
                    Name: 'zoneinfo',
                    Required: true,
                },
            ],
        });
    });
    test('schema is absent when required attributes are specified but as false', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            requiredAttributes: {
                familyName: false,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            Schema: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            Schema: have_resource_1.ABSENT,
        });
    });
    test('custom attributes with default constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute(),
                'custom-number-attr': new lib_1.NumberAttribute(),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    Name: 'custom-string-attr',
                    AttributeDataType: 'String',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
                {
                    Name: 'custom-number-attr',
                    AttributeDataType: 'Number',
                    StringAttributeConstraints: have_resource_1.ABSENT,
                    NumberAttributeConstraints: have_resource_1.ABSENT,
                },
            ],
        });
    });
    test('custom attributes with constraints', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            customAttributes: {
                'custom-string-attr': new lib_1.StringAttribute({ minLen: 5, maxLen: 50 }),
                'custom-number-attr': new lib_1.NumberAttribute({ min: 500, max: 2000 }),
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Schema: [
                {
                    AttributeDataType: 'String',
                    Name: 'custom-string-attr',
                    StringAttributeConstraints: {
                        MaxLength: '50',
                        MinLength: '5',
                    },
                },
                {
                    AttributeDataType: 'Number',
                    Name: 'custom-number-attr',
                    NumberAttributeConstraints: {
                        MaxValue: '2000',
                        MinValue: '500',
                    },
                },
            ],
        });
    });
    test('mfaTypes is ignored when mfaEnforcement is undefined or set to OFF', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.OFF,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: have_resource_1.ABSENT,
            EnabledMfas: have_resource_1.ABSENT,
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'OFF',
            EnabledMfas: have_resource_1.ABSENT,
        });
    });
    test('sms mfa type is the default when mfaEnforcement is set to REQUIRED or OPTIONAL', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool1', {
            userPoolName: 'Pool1',
            mfa: lib_1.Mfa.OPTIONAL,
        });
        new lib_1.UserPool(stack, 'Pool2', {
            userPoolName: 'Pool2',
            mfa: lib_1.Mfa.REQUIRED,
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool1',
            MfaConfiguration: 'OPTIONAL',
            EnabledMfas: ['SMS_MFA'],
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            UserPoolName: 'Pool2',
            MfaConfiguration: 'ON',
            EnabledMfas: ['SMS_MFA'],
        });
    });
    test('mfa type is correctly picked up when specified', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            mfa: lib_1.Mfa.REQUIRED,
            mfaSecondFactor: {
                sms: true,
                otp: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EnabledMfas: ['SMS_MFA', 'SOFTWARE_TOKEN_MFA'],
        });
    });
    test('password policy is correctly set', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                minLength: 15,
                requireDigits: true,
                requireLowercase: true,
                requireUppercase: true,
                requireSymbols: true,
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    TemporaryPasswordValidityDays: 2,
                    MinimumLength: 15,
                    RequireLowercase: true,
                    RequireUppercase: true,
                    RequireNumbers: true,
                    RequireSymbols: true,
                },
            },
        });
    });
    test('password minimum length is set to the default when other parts of the policy is configured', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(2),
                requireDigits: true,
            },
        });
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            Policies: {
                PasswordPolicy: {
                    MinimumLength: 8,
                },
            },
        });
    });
    test('throws when tempPassword validity is not in round days', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.hours(30),
            },
        })).toThrow();
    });
    test('temp password throws an error when above the max', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool', {
            passwordPolicy: {
                tempPasswordValidity: core_1.Duration.days(400),
            },
        })).toThrow(/tempPasswordValidity cannot be greater than/);
    });
    test('throws when minLength is out of range', () => {
        const stack = new core_1.Stack();
        expect(() => new lib_1.UserPool(stack, 'Pool1', {
            passwordPolicy: {
                minLength: 5,
            },
        })).toThrow(/minLength for password must be between/);
        expect(() => new lib_1.UserPool(stack, 'Pool2', {
            passwordPolicy: {
                minLength: 100,
            },
        })).toThrow(/minLength for password must be between/);
    });
    test('email transmission settings are recognized correctly', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.UserPool(stack, 'Pool', {
            emailSettings: {
                from: 'from@myawesomeapp.com',
                replyTo: 'replyTo@myawesomeapp.com',
            },
        });
        // THEN
        expect(stack).toHaveResourceLike('AWS::Cognito::UserPool', {
            EmailConfiguration: {
                From: 'from@myawesomeapp.com',
                ReplyToEmailAddress: 'replyTo@myawesomeapp.com',
            },
        });
    });
});
test('addClient', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const userpool = new lib_1.UserPool(stack, 'Pool');
    userpool.addClient('UserPoolClient', {
        userPoolClientName: 'userpoolclient',
    });
    const imported = lib_1.UserPool.fromUserPoolId(stack, 'imported', 'imported-userpool-id');
    imported.addClient('UserPoolImportedClient', {
        userPoolClientName: 'userpoolimportedclient',
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolclient',
        UserPoolId: stack.resolve(userpool.userPoolId),
    });
    expect(stack).toHaveResourceLike('AWS::Cognito::UserPoolClient', {
        ClientName: 'userpoolimportedclient',
        UserPoolId: stack.resolve(imported.userPoolId),
    });
});
function fooFunction(scope, name) {
    return new lambda.Function(scope, name, {
        functionName: name,
        code: lambda.Code.inline('foo'),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
    });
}
//# sourceMappingURL=data:application/json;base64,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