import json
import os
from typing import Any, Dict, Optional

from click import ClickException
import jsonschema
import yaml

ANYSCALE_PROJECT_FILE = ".anyscale.yaml"
ANYSCALE_AUTOSCALER_FILE = "session-default.yaml"


CLUSTER_YAML_TEMPLATE = """
# This file was generated by `anyscale init`.

# The maximum number of workers nodes to launch in addition to the head
# node. This takes precedence over min_workers. min_workers defaults to 0.
max_workers: 1

docker:
    image: anyscale/ray-ml:latest
    container_name: ray_container
    pull_before_run: False

setup_commands: []

head_setup_commands: []

# Cloud-provider specific configuration.
provider:
    type: aws
    region: us-west-2
    availability_zone: us-west-2a

# How Ray will authenticate with newly launched nodes.
auth:
    ssh_user: ubuntu
"""

CLUSTER_CONFIG_TEMPLATE_STR = json.dumps(
    yaml.load(CLUSTER_YAML_TEMPLATE, Loader=yaml.SafeLoader)
)


def validate_project_schema(project_config: Dict[str, str]) -> Any:
    """Validate a project config against the project schema.
    Args:
        project_config (dict): Parsed project yaml.
    Raises:
        jsonschema.exceptions.ValidationError: This exception is raised
            if the project file is not valid.
    """
    dir = os.path.dirname(os.path.abspath(__file__))
    with open(os.path.join(dir, "ProjectConfig.json")) as f:
        schema = json.load(f)

    jsonschema.validate(instance=project_config, schema=schema)


def find_project_root(directory: str) -> Optional[str]:
    """Find root directory of the project.

    Args:
        directory (str): Directory to start the search in.

    Returns:
        Path of the parent directory containing the project
        or None if no such project is found.
    """
    prev, directory = None, os.path.abspath(directory)
    while prev != directory:
        if os.path.exists(ANYSCALE_PROJECT_FILE):
            return directory
        prev, directory = directory, os.path.abspath(os.path.join(directory, os.pardir))
    return None


class ProjectDefinition(object):
    def __init__(self, root_dir: str):
        self.root = os.path.join(root_dir, "")
        anyscale_yaml = os.path.join(root_dir, ANYSCALE_PROJECT_FILE)
        if os.path.exists(anyscale_yaml):
            with open(anyscale_yaml) as f:
                self.config = yaml.safe_load(f)
        else:
            self.config = {}

        if "cluster" not in self.config:
            self.config["cluster"] = {
                "config": os.path.join(self.root, ANYSCALE_AUTOSCALER_FILE)
            }

    def cluster_yaml(self) -> str:
        if "config" in self.config["cluster"]:
            return str(self.config["cluster"]["config"])
        return os.path.join(self.root, ANYSCALE_AUTOSCALER_FILE)


def load_project_or_throw() -> ProjectDefinition:
    # First check if there is a .anyscale.yaml.
    root_dir = find_project_root(os.getcwd())
    if not root_dir:
        raise ClickException("No project directory found")
    return ProjectDefinition(root_dir)


def get_project_id(project_dir: str) -> str:
    """
    Args:
        project_dir: Project root directory.

    Returns:
        The ID of the associated Project in the database.

    Raises:
        ValueError: If the current project directory does
            not contain a project ID.
    """
    project_filename = os.path.join(project_dir, ANYSCALE_PROJECT_FILE)
    if os.path.isfile(project_filename):
        with open(project_filename) as f:
            config = yaml.safe_load(f)
            project_id = config["project_id"]
    else:
        # TODO(pcm): Consider doing this for the user and retrying the command
        # they were trying to run.
        raise ClickException(
            "Ray project in {} not registered yet. "
            "Did you run 'anyscale init'?".format(project_dir)
        )
    try:
        result = str(project_id)
    except ValueError:
        # TODO(pcm): Tell the user what to do here.
        raise ClickException(
            "{} does not contain a valid project ID".format(project_filename)
        )
    return result


def validate_project_name(project_name: str) -> bool:
    return " " not in project_name.strip()
