"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const s3 = require("@aws-cdk/aws-s3");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const lambda_helper_1 = require("./lambda-helper");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    const cfnCfDistribution = cfDistribution.node.defaultChild;
    cfnCfDistribution.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W70',
                    reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
                }]
        }
    };
    return cfDistribution;
}
function createCloudfrontLoggingBucket(scope) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, 'CloudfrontLoggingBucket', s3_bucket_defaults_1.DefaultS3Props());
    s3_bucket_helper_1.applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: `This S3 bucket is used as the access logging bucket for CloudFront Distribution`
                }]
        }
    };
    return loggingBucket;
}
// Lambda@Edge function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultLambdaEdgeFunction(scope) {
    const edgeLambdaFunc = lambda_helper_1.deployLambdaFunction(scope, {
        code: new lambda.InlineCode("exports.handler = (event, context, callback) => { \
          const response = event.Records[0].cf.response; \
          const headers = response.headers; \
          headers['x-xss-protection'] = [ \
            { \
              key: 'X-XSS-Protection', \
              value: '1; mode=block' \
            } \
          ]; \
          headers['x-frame-options'] = [ \
            { \
              key: 'X-Frame-Options', \
              value: 'DENY' \
            } \
          ]; \
          headers['x-content-type-options'] = [ \
            { \
              key: 'X-Content-Type-Options', \
              value: 'nosniff' \
            } \
          ]; \
          headers['strict-transport-security'] = [ \
            { \
              key: 'Strict-Transport-Security', \
              value: 'max-age=63072000; includeSubdomains; preload' \
            } \
          ]; \
          headers['referrer-policy'] = [ \
            { \
              key: 'Referrer-Policy', \
              value: 'same-origin' \
            } \
          ]; \
          headers['content-security-policy'] = [ \
            { \
              key: 'Content-Security-Policy', \
              value: \"default-src 'none'; base-uri 'self'; img-src 'self'; script-src 'self'; style-src 'self' https:; object-src 'none'; frame-ancestors 'none'; font-src 'self' https:; form-action 'self'; manifest-src 'self'; connect-src 'self'\" \
             } \
          ]; \
          callback(null, response); \
        };"),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    }, 'SetHttpSecurityHeaders');
    // Remove any environment variables as it is not supported by the Lambda@Edge functions :(
    const cfnEdgeLambdaFunction = edgeLambdaFunc.node.defaultChild;
    cfnEdgeLambdaFunction.addDeletionOverride('Properties.Environment');
    return edgeLambdaFunc;
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders) {
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, cloudFrontDistributionProps.loggingConfig.bucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = createCloudfrontLoggingBucket(scope);
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders) {
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(scope, 'OAIImported', cfnOrigAccessId.ref);
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, cloudFrontDistributionProps.loggingConfig.bucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = createCloudfrontLoggingBucket(scope);
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Add S3 Bucket Policy to allow s3:GetObject for CloudFront Origin Access Identity User
    sourceBucket.addToResourcePolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject'],
        resources: [sourceBucket.arnForObjects('*')],
        principals: [new iam.CanonicalUserPrincipal(cfnOrigAccessId.attrS3CanonicalUserId)]
    }));
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    const sourceBucketPolicy = bucketPolicy.node.findChild('Resource');
    sourceBucketPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'F16',
                    reason: `Public website bucket policy requires a wildcard principal`
                }]
        }
    };
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
//# sourceMappingURL=data:application/json;base64,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