"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("@aws-cdk/aws-sns");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": core_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    // If props is undefined, define it
    props = (props === undefined) ? {} : props;
    // Setup the topic properties
    let snsTopicProps;
    if (props.topicProps) {
        // If property overrides have been provided, incorporate them and deploy
        snsTopicProps = utils_1.overrideProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        snsTopicProps = sns_defaults_1.DefaultSnsTopicProps;
    }
    // Set encryption properties
    // TODO: Look into using the AWS managed CMK by using 'alias/aws/sns'
    if (!props.enableEncryption || props.enableEncryption === true) {
        if (props.encryptionKey) {
            snsTopicProps.masterKey = props.encryptionKey;
        }
        else {
            snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope);
        }
    }
    // Create the SNS Topic
    const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
    applySecureTopicPolicy(topic);
    return [topic, snsTopicProps.masterKey];
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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