"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
// Imports
const aws_logs_1 = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps) {
    let logGroup;
    // Configure Cloudwatch log group for Step function State Machine
    if (!stateMachineProps.logs) {
        logGroup = new aws_logs_1.LogGroup(scope, 'StateMachineLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    }
    else {
        logGroup = stateMachineProps.logs.destination;
    }
    // Override the defaults with the user provided props
    const _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    cfnDefaultPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `The 'LogDelivery' actions do not support resource-level authorizations`
                }]
        }
    };
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:aws:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RlcC1mdW5jdGlvbi1oZWxwZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJzdGVwLWZ1bmN0aW9uLWhlbHBlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7OztBQUVILFVBQVU7QUFDVixnREFBNkM7QUFDN0MscUNBQXFDO0FBQ3JDLHVEQUF1RDtBQUN2RCxtRkFBdUU7QUFDdkUsa0RBQWtEO0FBQ2xELG1DQUF3QztBQUN4Qyx3Q0FBd0M7QUFDeEMsc0RBQXNEO0FBRXREOzs7O0dBSUc7QUFDSCxTQUFnQixpQkFBaUIsQ0FBQyxLQUFvQixFQUFFLGlCQUF3QztJQUU1RixJQUFJLFFBQWtCLENBQUM7SUFFdkIsaUVBQWlFO0lBQ2pFLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxJQUFJLEVBQUU7UUFDekIsUUFBUSxHQUFHLElBQUksbUJBQVEsQ0FBQyxLQUFLLEVBQUUsc0JBQXNCLEVBQUUsb0RBQW9CLEVBQUUsQ0FBQyxDQUFDO0tBQ2xGO1NBQU07UUFDSCxRQUFRLEdBQUcsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFdBQXVCLENBQUM7S0FDN0Q7SUFFRCxxREFBcUQ7SUFDckQsTUFBTSxRQUFRLEdBQUcscUJBQWEsQ0FBQyxVQUFVLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUVqRyxtRUFBbUU7SUFDbkUsTUFBTSxHQUFHLEdBQUcsSUFBSSxHQUFHLENBQUMsWUFBWSxDQUFDLEtBQUssRUFBRSxjQUFjLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDbEUsTUFBTSxJQUFJLEdBQUcsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFhLENBQUM7SUFDcEQsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxJQUFJLENBQUMsWUFBNkIsQ0FBQztJQUVqRyw2REFBNkQ7SUFDN0QsZ0JBQWdCLENBQUMsbUJBQW1CLENBQUMsbUNBQW1DLEVBQ3RFO1FBQ0Usd0JBQXdCO1FBQ3hCLHFCQUFxQjtRQUNyQix3QkFBd0I7UUFDeEIsd0JBQXdCO1FBQ3hCLHdCQUF3QjtLQUN6QixDQUFDLENBQUM7SUFFTCx1RUFBdUU7SUFDdkUsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLFFBQVEsR0FBRztRQUNyQyxPQUFPLEVBQUU7WUFDUCxpQkFBaUIsRUFBRSxDQUFDO29CQUNoQixFQUFFLEVBQUUsS0FBSztvQkFDVCxNQUFNLEVBQUUsd0VBQXdFO2lCQUNuRixDQUFDO1NBQ0g7S0FDRixDQUFDO0lBRUYsK0VBQStFO0lBQy9FLEdBQUcsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQzFDLE9BQU8sRUFBRTtZQUNQLHdCQUF3QjtZQUN4QiwrQkFBK0I7WUFDL0Isd0JBQXdCO1NBQ3pCO1FBQ0QsU0FBUyxFQUFFLENBQUMsZ0JBQWdCLEdBQUcsQ0FBQyxHQUFHLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUM7S0FDdEUsQ0FBQyxDQUFDLENBQUM7SUFFSixPQUFPLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0FBQzNCLENBQUM7QUFsREQsOENBa0RDO0FBRUQsU0FBZ0IseUJBQXlCLENBQUMsS0FBb0IsRUFBRSxFQUFvQjtJQUNsRixvQ0FBb0M7SUFDcEMsTUFBTSxNQUFNLEdBQXVCLElBQUksS0FBSyxFQUFFLENBQUM7SUFFL0Msb0ZBQW9GO0lBQ3BGLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxzQkFBc0IsRUFBRTtRQUM5RCxNQUFNLEVBQUUsRUFBRSxDQUFDLFlBQVksRUFBRTtRQUN6QixTQUFTLEVBQUUsQ0FBQztRQUNaLGlCQUFpQixFQUFFLENBQUM7UUFDcEIsU0FBUyxFQUFFLEtBQUs7UUFDaEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztRQUNqQyxrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLDhFQUE4RTtLQUNqRyxDQUFDLENBQUMsQ0FBQztJQUVKLDJGQUEyRjtJQUMzRixNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUseUJBQXlCLEVBQUU7UUFDakUsTUFBTSxFQUFFLEVBQUUsQ0FBQyxlQUFlLEVBQUU7UUFDNUIsU0FBUyxFQUFFLENBQUM7UUFDWixpQkFBaUIsRUFBRSxDQUFDO1FBQ3BCLFNBQVMsRUFBRSxLQUFLO1FBQ2hCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUM7UUFDakMsa0JBQWtCLEVBQUUsVUFBVSxDQUFDLGtCQUFrQixDQUFDLGtDQUFrQztRQUNwRixnQkFBZ0IsRUFBRSxpRkFBaUY7S0FDcEcsQ0FBQyxDQUFDLENBQUM7SUFFSixxRkFBcUY7SUFDckYsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLFVBQVUsQ0FBQyxLQUFLLENBQUMsS0FBSyxFQUFFLHVCQUF1QixFQUFFO1FBQy9ELE1BQU0sRUFBRSxFQUFFLENBQUMsYUFBYSxFQUFFO1FBQzFCLFNBQVMsRUFBRSxDQUFDO1FBQ1osaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixTQUFTLEVBQUUsU0FBUztRQUNwQixNQUFNLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1FBQ2pDLGtCQUFrQixFQUFFLFVBQVUsQ0FBQyxrQkFBa0IsQ0FBQyxrQ0FBa0M7UUFDcEYsZ0JBQWdCLEVBQUUsK0VBQStFO0tBQ2xHLENBQUMsQ0FBQyxDQUFDO0lBRUosT0FBTyxNQUFNLENBQUM7QUFDaEIsQ0FBQztBQXRDRCw4REFzQ0MiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAxOSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbi8vIEltcG9ydHNcbmltcG9ydCB7IExvZ0dyb3VwIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc21EZWZhdWx0cyBmcm9tICcuL3N0ZXAtZnVuY3Rpb24tZGVmYXVsdHMnO1xuaW1wb3J0IHsgRGVmYXVsdExvZ0dyb3VwUHJvcHMgfSBmcm9tICcuL2Nsb3Vkd2F0Y2gtbG9nLWdyb3VwLWRlZmF1bHRzJztcbmltcG9ydCAqIGFzIHNmbiBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBvdmVycmlkZVByb3BzIH0gZnJvbSAnLi91dGlscyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcblxuLyoqXG4gKiBCdWlsZHMgYW5kIHJldHVybnMgYSBTdGF0ZU1hY2hpbmUuXG4gKiBAcGFyYW0gc2NvcGUgLSB0aGUgY29uc3RydWN0IHRvIHdoaWNoIHRoZSBTdGF0ZU1hY2hpbmUgc2hvdWxkIGJlIGF0dGFjaGVkIHRvLlxuICogQHBhcmFtIHN0YXRlTWFjaGluZVByb3BzIC0gdXNlci1zcGVjaWZpZWQgcHJvcGVydGllcyB0byBvdmVycmlkZSB0aGUgZGVmYXVsdCBwcm9wZXJ0aWVzLlxuICovXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRTdGF0ZU1hY2hpbmUoc2NvcGU6IGNkay5Db25zdHJ1Y3QsIHN0YXRlTWFjaGluZVByb3BzOiBzZm4uU3RhdGVNYWNoaW5lUHJvcHMpOiBbc2ZuLlN0YXRlTWFjaGluZSwgTG9nR3JvdXBdIHtcblxuICAgIGxldCBsb2dHcm91cDogTG9nR3JvdXA7XG5cbiAgICAvLyBDb25maWd1cmUgQ2xvdWR3YXRjaCBsb2cgZ3JvdXAgZm9yIFN0ZXAgZnVuY3Rpb24gU3RhdGUgTWFjaGluZVxuICAgIGlmICghc3RhdGVNYWNoaW5lUHJvcHMubG9ncykge1xuICAgICAgICBsb2dHcm91cCA9IG5ldyBMb2dHcm91cChzY29wZSwgJ1N0YXRlTWFjaGluZUxvZ0dyb3VwJywgRGVmYXVsdExvZ0dyb3VwUHJvcHMoKSk7XG4gICAgfSBlbHNlIHtcbiAgICAgICAgbG9nR3JvdXAgPSBzdGF0ZU1hY2hpbmVQcm9wcy5sb2dzLmRlc3RpbmF0aW9uIGFzIExvZ0dyb3VwO1xuICAgIH1cblxuICAgIC8vIE92ZXJyaWRlIHRoZSBkZWZhdWx0cyB3aXRoIHRoZSB1c2VyIHByb3ZpZGVkIHByb3BzXG4gICAgY29uc3QgX3NtUHJvcHMgPSBvdmVycmlkZVByb3BzKHNtRGVmYXVsdHMuRGVmYXVsdFN0YXRlTWFjaGluZVByb3BzKGxvZ0dyb3VwKSwgc3RhdGVNYWNoaW5lUHJvcHMpO1xuXG4gICAgLy8gT3ZlcnJpZGUgdGhlIENsb3Vkd2F0Y2ggcGVybWlzc2lvbnMgdG8gbWFrZSBpdCBtb3JlIGZpbmUgZ3JhaW5lZFxuICAgIGNvbnN0IF9zbSA9IG5ldyBzZm4uU3RhdGVNYWNoaW5lKHNjb3BlLCAnU3RhdGVNYWNoaW5lJywgX3NtUHJvcHMpO1xuICAgIGNvbnN0IHJvbGUgPSBfc20ubm9kZS5maW5kQ2hpbGQoJ1JvbGUnKSBhcyBpYW0uUm9sZTtcbiAgICBjb25zdCBjZm5EZWZhdWx0UG9saWN5ID0gcm9sZS5ub2RlLmZpbmRDaGlsZCgnRGVmYXVsdFBvbGljeScpLm5vZGUuZGVmYXVsdENoaWxkIGFzIGlhbS5DZm5Qb2xpY3k7XG5cbiAgICAvLyBSZWR1Y2UgdGhlIHNjb3BlIG9mIGFjdGlvbnMgZm9yIHRoZSBleGlzdGluZyBEZWZhdWx0UG9saWN5XG4gICAgY2ZuRGVmYXVsdFBvbGljeS5hZGRQcm9wZXJ0eU92ZXJyaWRlKCdQb2xpY3lEb2N1bWVudC5TdGF0ZW1lbnQuMC5BY3Rpb24nLFxuICAgICAgW1xuICAgICAgICBcImxvZ3M6Q3JlYXRlTG9nRGVsaXZlcnlcIixcbiAgICAgICAgJ2xvZ3M6R2V0TG9nRGVsaXZlcnknLFxuICAgICAgICAnbG9nczpVcGRhdGVMb2dEZWxpdmVyeScsXG4gICAgICAgICdsb2dzOkRlbGV0ZUxvZ0RlbGl2ZXJ5JyxcbiAgICAgICAgJ2xvZ3M6TGlzdExvZ0RlbGl2ZXJpZXMnXG4gICAgICBdKTtcblxuICAgIC8vIE92ZXJyaWRlIENmbiBOYWcgd2FybmluZyBXMTI6IElBTSBwb2xpY3kgc2hvdWxkIG5vdCBhbGxvdyAqIHJlc291cmNlXG4gICAgY2ZuRGVmYXVsdFBvbGljeS5jZm5PcHRpb25zLm1ldGFkYXRhID0ge1xuICAgICAgY2ZuX25hZzoge1xuICAgICAgICBydWxlc190b19zdXBwcmVzczogW3tcbiAgICAgICAgICAgIGlkOiAnVzEyJyxcbiAgICAgICAgICAgIHJlYXNvbjogYFRoZSAnTG9nRGVsaXZlcnknIGFjdGlvbnMgZG8gbm90IHN1cHBvcnQgcmVzb3VyY2UtbGV2ZWwgYXV0aG9yaXphdGlvbnNgXG4gICAgICAgIH1dXG4gICAgICB9XG4gICAgfTtcblxuICAgIC8vIEFkZCBhIG5ldyBwb2xpY3kgd2l0aCBsb2dnaW5nIHBlcm1pc3Npb25zIGZvciB0aGUgZ2l2ZW4gY2xvdWR3YXRjaCBsb2cgZ3JvdXBcbiAgICBfc20uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ2xvZ3M6UHV0UmVzb3VyY2VQb2xpY3knLFxuICAgICAgICAnbG9nczpEZXNjcmliZVJlc291cmNlUG9saWNpZXMnLFxuICAgICAgICAnbG9nczpEZXNjcmliZUxvZ0dyb3VwcydcbiAgICAgIF0sXG4gICAgICByZXNvdXJjZXM6IFtgYXJuOmF3czpsb2dzOiR7Y2RrLkF3cy5SRUdJT059OiR7Y2RrLkF3cy5BQ0NPVU5UX0lEfToqYF1cbiAgICB9KSk7XG5cbiAgICByZXR1cm4gW19zbSwgbG9nR3JvdXBdO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRTdGVwRnVuY3Rpb25DV0FsYXJtcyhzY29wZTogY2RrLkNvbnN0cnVjdCwgc206IHNmbi5TdGF0ZU1hY2hpbmUpOiBjbG91ZHdhdGNoLkFsYXJtW10ge1xuICAvLyBTZXR1cCBDVyBBbGFybXMgZm9yIFN0YXRlIE1hY2hpbmVcbiAgY29uc3QgYWxhcm1zOiBjbG91ZHdhdGNoLkFsYXJtW10gPSBuZXcgQXJyYXkoKTtcblxuICAvLyBTdW0gb2YgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgaXMgPj0gMSBmb3IgNSBtaW51dGVzLCAxIGNvbnNlY3V0aXZlIHRpbWVcbiAgYWxhcm1zLnB1c2gobmV3IGNsb3Vkd2F0Y2guQWxhcm0oc2NvcGUsICdFeGVjdXRpb25GYWlsZWRBbGFybScsIHtcbiAgICBtZXRyaWM6IHNtLm1ldHJpY0ZhaWxlZCgpLFxuICAgIHRocmVzaG9sZDogMSxcbiAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICBzdGF0aXN0aWM6ICdTdW0nLFxuICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLnNlY29uZHMoMzAwKSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0FsYXJtIGZvciB0aGUgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgZXhjZWVkZWQgdGhlIHRocmVzaG9sZCBvZiAxLiAnXG4gIH0pKTtcblxuICAvLyBTdW0gb2YgbnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBmYWlsZWQgbWF4aW11bSBpcyA+PSAxIGZvciA1IG1pbnV0ZSwgMSBjb25zZWN1dGl2ZSB0aW1lXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnRXhlY3V0aW9uVGhyb3R0bGVkQWxhcm0nLCB7XG4gICAgbWV0cmljOiBzbS5tZXRyaWNUaHJvdHRsZWQoKSxcbiAgICB0aHJlc2hvbGQ6IDEsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnU3VtJyxcbiAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5zZWNvbmRzKDMwMCksXG4gICAgY29tcGFyaXNvbk9wZXJhdG9yOiBjbG91ZHdhdGNoLkNvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgIGFsYXJtRGVzY3JpcHRpb246ICdBbGFybSBmb3IgdGhlIG51bWJlciBvZiBleGVjdXRpb25zIHRoYXQgdGhyb3R0bGVkIGV4Y2VlZGVkIHRoZSB0aHJlc2hvbGQgb2YgMS4gJ1xuICB9KSk7XG5cbiAgLy8gTnVtYmVyIG9mIGV4ZWN1dGlvbnMgdGhhdCBhYm9ydGVkIG1heGltdW0gaXMgPj0gMSBmb3IgNSBtaW51dGUsIDEgY29uc2VjdXRpdmUgdGltZVxuICBhbGFybXMucHVzaChuZXcgY2xvdWR3YXRjaC5BbGFybShzY29wZSwgJ0V4ZWN1dGlvbkFib3J0ZWRBbGFybScsIHtcbiAgICBtZXRyaWM6IHNtLm1ldHJpY0Fib3J0ZWQoKSxcbiAgICB0aHJlc2hvbGQ6IDEsXG4gICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgc3RhdGlzdGljOiAnTWF4aW11bScsXG4gICAgcGVyaW9kOiBjZGsuRHVyYXRpb24uc2Vjb25kcygzMDApLFxuICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY2xvdWR3YXRjaC5Db21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICBhbGFybURlc2NyaXB0aW9uOiAnQWxhcm0gZm9yIHRoZSBudW1iZXIgb2YgZXhlY3V0aW9ucyB0aGF0IGFib3J0ZWQgZXhjZWVkZWQgdGhlIHRocmVzaG9sZCBvZiAxLiAnXG4gIH0pKTtcblxuICByZXR1cm4gYWxhcm1zO1xufSJdfQ==