"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
function setupRestApi(stack, apiProps) {
    const [restApi] = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '12345678' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
test('snapshot test RegionalApiGateway default params', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "LambdaRestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/o ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
test('Test default RestApi deployment for Cloudwatch loggroup', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        UpdateReplacePolicy: "Retain",
        DeletionPolicy: "Retain"
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::ApiGateway::Stage', {
        AccessLogSetting: {
            DestinationArn: {
                "Fn::GetAtt": [
                    "ApiAccessLogGroupCEA70788",
                    "Arn"
                ]
            },
            Format: "{\"requestId\":\"$context.requestId\",\"ip\":\"$context.identity.sourceIp\",\"user\":\"$context.identity.user\",\"caller\":\"$context.identity.caller\",\"requestTime\":\"$context.requestTime\",\"httpMethod\":\"$context.httpMethod\",\"resourcePath\":\"$context.resourcePath\",\"status\":\"$context.status\",\"protocol\":\"$context.protocol\",\"responseLength\":\"$context.responseLength\"}",
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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