"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const rule_1 = require("../lib/rule");
class SomeTarget {
    constructor(input) {
        this.input = input;
    }
    bind() {
        return { id: 'T1', arn: 'ARN1', input: this.input };
    }
}
module.exports = {
    'json template': {
        'can just be a JSON object'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(core_1.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ SomeObject: 'withAValue' })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "{\"SomeObject\":\"withAValue\"}"
                    }
                ]
            }));
            test.done();
        },
        'can use token'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(core_1.Duration.minutes(1)),
            });
            const user = new aws_iam_1.User(stack, 'User');
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromObject({ userArn: user.userArn })));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: {
                            'Fn::Join': [
                                '',
                                [
                                    '{\"userArn\":\"',
                                    {
                                        'Fn::GetAtt': [
                                            'User00B015A1',
                                            'Arn'
                                        ]
                                    },
                                    '\"}'
                                ]
                            ]
                        }
                    }
                ]
            }));
            test.done();
        },
    },
    'text templates': {
        'strings with newlines are serialized to a newline-delimited list of JSON strings'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(core_1.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('I have\nmultiple lines')));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "\"I have\"\n\"multiple lines\""
                    }
                ]
            }));
            test.done();
        },
        'escaped newlines are not interpreted as newlines'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(core_1.Duration.minutes(1)),
            });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromMultilineText('this is not\\na real newline'))),
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                    Targets: [
                        {
                            Input: "\"this is not\\\\na real newline\""
                        }
                    ]
                }));
            test.done();
        },
        'can use Tokens in text templates'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const rule = new rule_1.Rule(stack, 'Rule', {
                schedule: lib_1.Schedule.rate(core_1.Duration.minutes(1)),
            });
            const world = cdk.Lazy.stringValue({ produce: () => 'world' });
            // WHEN
            rule.addTarget(new SomeTarget(lib_1.RuleTargetInput.fromText(`hello ${world}`)));
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Events::Rule', {
                Targets: [
                    {
                        Input: "\"hello world\""
                    }
                ]
            }));
            test.done();
        }
    },
};
//# sourceMappingURL=data:application/json;base64,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